#include "make_incident_ticket.h"

#include <drive/backend/incident/incident_context.h>
#include <drive/backend/incident/contexts/incident_ticket.h>


#include <drive/backend/abstract/frontend.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/transaction/tx.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/util/algorithm/ptr.h>

namespace NDrive {
    TMakeIncidentTicketTransition::TRegistrator TMakeIncidentTicketTransition::Registrator;

    TMakeIncidentTicketTransition::TMakeIncidentTicketTransition()
        : TBase(MakeAtomicShared<TIncidentTicketIncidentContext>(), MakeAtomicShared<TNotifyHandlers>("incident_transition-" + GetTypeName()))
    {
    }

    NDrive::TScheme TMakeIncidentTicketTransition::GetScheme(const TIncidentStateContext& context) const {
        TMaybe<EIncidentType> incidentType;
        IIncidentContext::TPtr existingContextPtr;
        if (context.HasInstance()) {
            incidentType = context.GetInstanceRef().GetIncidentType();
            existingContextPtr = context.GetInstanceRef().GetContext(TIncidentTicketIncidentContext::GetTypeName());
        }
        return TIncidentTicketIncidentContext::GetScheme(*context.GetServer(), incidentType, existingContextPtr);
    }

    bool TMakeIncidentTicketTransition::Initialize(const NJson::TJsonValue& data, const TIncidentStateContext& context, TMessagesCollector& errors) {
        if (!TBase::Initialize(data, context, errors)) {
            return false;
        }

        auto nativeContextPtr = std::dynamic_pointer_cast<TIncidentTicketIncidentContext>(GetDataContextPtr());
        if (!nativeContextPtr) {
            return false;
        }

        TMaybe<EIncidentType> dataIncidentType = nativeContextPtr->GetIncidentType();
        if (!context.HasInstance() || !dataIncidentType || context.OptionalInstance()->GetIncidentType() != *dataIncidentType) {
            errors.AddMessage(__LOCATION__, "Data incident type differs from instance's one");
            return false;
        }

        return true;
    }

    TString TMakeIncidentTicketTransition::GetSettingKey(TStringBuf settingKeySuffix) const {
        return GenericIncidentSettingPrefix + "." + settingKeySuffix;
    }

    TMaybe<bool> TMakeIncidentTicketTransition::DoCheckIsApplicable(const TIncidentStateContext& context, NDrive::TEntitySession& /* session */) const {
        return (context.HasInstance()) ? (context.OptionalInstance()->GetIncidentType() != EIncidentType::Evacuation) : false;
    }

    bool TMakeIncidentTicketTransition::HandleIssueAttachments(const TIncidentStateContext& context, NJson::TJsonValue& contextEntry, NDrive::TEntitySession& session) const {
        const auto& contextAttachments = contextEntry["photos"];

        if (contextAttachments.IsDefined()) {
            NJson::TJsonValue existingAttachments(NJson::JSON_NULL);
            if (auto existingContext = context.GetInstanceRef().GetContext<TIncidentTicketIncidentContext>()) {
                existingAttachments = existingContext->GetData()["photos"];
            }

            auto updatedContextAttachments = PrepareIssueAttachments(contextAttachments, existingAttachments, session);
            if (updatedContextAttachments) {
                contextEntry["photos"] = *updatedContextAttachments;  // overwrite
            } else {
                return false;  // failed to process
            }
        } else {
            contextEntry["photos"] = NJson::JSON_ARRAY;
        }

        return true;
    }
    bool TMakeIncidentTicketTransition::HandleComponents(const TIncidentStateContext& context, TJsonFetchContext::TDynamicContext& dynamicContext, NDrive::TEntitySession& session) const {
        if (!context.HasInstance()) {
            return false;
        }
        const TIncidentData& incidentInstance = context.GetInstanceRef();
        if (!incidentInstance.GetCarId()) {
            dynamicContext["components"] = "";
            return true;
        }
        const NDrive::IServer& server = *context.GetServer();
        TSet<TString> components;
        {
            NJson::TJsonValue json;
            auto settings = server.GetSettings().GetValueDef<TString>(GetSettingKey("tag_components_map"), "{}");
            if (!NJson::ReadJsonTree(settings, &json)) {
                session.SetErrorInfo("HandleComponents", "Fail to parse settings: " + settings, EDriveSessionResult::IncorrectRequest);
                return false;
            }
            if (json.IsMap() && !json.GetMap().empty()) {
                TDBTags tags;
                if (!server.GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({ incidentInstance.GetCarId() }, {}, tags, session)) {
                    return false;
                }
                TSet<TString> knownTags;
                for (const auto& tag : tags) {
                    if (tag && json.Has(tag->GetName()) && knownTags.insert(tag->GetName()).second) {
                        if (json[tag->GetName()].IsString()) {
                            components.insert(json[tag->GetName()].GetString());
                        } else if (json[tag->GetName()].IsArray()) {
                            for (const auto& item : json[tag->GetName()].GetArray()) {
                                if (item.IsString()) {
                                    components.insert(item.GetString());
                                }
                            }
                        }
                    }
                }
            }
        }
        dynamicContext["components"] = components.empty() ? "" : "\"" + JoinSeq("\",\"", components) + "\"";
        return true;
    }

    void TMakeIncidentTicketTransition::InitializeLocalizedVariants(const TIncidentStateContext& context, TDynamicContext& dynamicContext, const ELocalization locale) const {
        TIncidentTicketIncidentContext::InitializeLocalizedVariants(*context.GetServer(), dynamicContext, locale);
    }
}
