#include "make_incident_ticket.h"

#include <drive/backend/incident/incident_context.h>

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/database/transaction/tx.h>
#include <drive/backend/notifications/startrek/startrek.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/parse.h>
#include <rtline/util/algorithm/ptr.h>

#include <util/generic/algorithm.h>
#include <util/string/cast.h>

namespace NDrive {
    TMakeTicketBaseTransition::TMakeTicketBaseTransition(IIncidentContext::TPtr dataContextPtr, TNotifyHandlers::TPtr notifyHandlersPtr)
        : TBase()
        , DataContextPtr(dataContextPtr)
        , NotifyHandlersPtr(notifyHandlersPtr)
    {
        static_assert(std::is_same<TDynamicContext, TJsonFetchContext::TDynamicContext>());
    }

    bool TMakeTicketBaseTransition::Initialize(const NJson::TJsonValue& data, const TIncidentStateContext& context, TMessagesCollector& errors) {
        {
            const NDrive::IServer& server = *context.GetServer();

            TNotifyHandlers::THandler handler;
            handler.SetEventType(GetTransitionType());
            handler.SetNotifierName(server.GetSettings().GetValueDef<TString>(GetSettingKey("notifier_name"), ""));
            handler.SetMessageTemplate(server.GetSettings().GetValueDef<TString>(GetSettingKey("template"), ""));
            NotifyHandlersPtr->AddHandler(std::move(handler));
        }

        if (!DataContextPtr || !DataContextPtr->DeserializeFromJson(data, errors)) {
            return false;
        }

        return true;
    }

    TSet<EIncidentStatus> TMakeTicketBaseTransition::GetAllowedSourceStatuses() const {
        return { EIncidentStatus::New, EIncidentStatus::StartrekTicketsProcessed, EIncidentStatus::StartrekTicketsProcessingError };
    }

    EIncidentStatus TMakeTicketBaseTransition::GetDestinationStatus(const TIncidentStateContext& context) const {
        return (context.GetIsPerformSuccessfulDef(false)) ? EIncidentStatus::StartrekTicketsProcessed : EIncidentStatus::StartrekTicketsProcessingError;
    }

    bool TMakeTicketBaseTransition::DoPerform(TIncidentStateContext& context, NDrive::TEntitySession& session) const {
        auto fetchContext = PrepareFetchContext(context, session);
        if (!fetchContext) {
            return false;
        }

        auto resultPtr = NotifyHandlersPtr->Handle(GetTransitionType(), *fetchContext);
        if (!resultPtr || resultPtr->HasErrors()) {
            TString errorMessage = "Error creating ticket: " + ((resultPtr) ? resultPtr->SerializeToJson().GetStringRobust() : "unknown error");
            session.SetErrorInfo(TIncidentData::GetTableName(), errorMessage, EDriveSessionResult::IncorrectRequest);
            return false;
        }

        context.MutableInstance()->UpsertContext(DataContextPtr);  // save provided context

        return HandleNotifyResult(resultPtr, context, session);
    }

    TMaybe<TJsonFetchContext> TMakeTicketBaseTransition::PrepareFetchContext(TIncidentStateContext& context, NDrive::TEntitySession& session) const {
        TIncidentData& incidentInstance = context.MutableInstanceRef();

        auto contextEntry = DataContextPtr->SerializeToJson();
        if (!HandleIssueAttachments(context, contextEntry, session)) {
            return {};
        }

        TJsonFetchContext::TDynamicContext dynamicContext = {
            {"performer_id", context.GetPerformerId()},
            {"car_id", incidentInstance.GetCarId()},
            {"user_id", incidentInstance.GetUserId()},
            {"session_id", incidentInstance.GetSessionId()}
        };

        if (!HandleComponents(context, dynamicContext, session)) {
            return {};
        }

        InitializeLocalizedVariants(context, dynamicContext);
        InitializeExistingIssue(context, dynamicContext);

        return TJsonFetchContext(context.GetServer(), std::move(contextEntry), dynamicContext, NotifyHandlersPtr);
    }

    TMaybe<NJson::TJsonValue> TMakeTicketBaseTransition::PrepareIssueAttachments(const NJson::TJsonValue& contextAttachments, const NJson::TJsonValue& existingAttachments, NDrive::TEntitySession& session) const {
        // startrek requires 1) attachment ids to be integers but strings and 2) to pass only files not attached already (bug?)
        TVector<TString> strAttachmentIds;
        if (!NJson::ParseField(contextAttachments, strAttachmentIds, /* required = */ true)) {
            session.SetErrorInfo(TIncidentData::GetTableName(), "Incorrect photo ids: " + contextAttachments.GetStringRobust(), EDriveSessionResult::IncorrectRequest);
            return {};
        }

        TSet<TString> existingStrAttachmentIds;
        if (existingAttachments.IsDefined() && !NJson::ParseField(existingAttachments, existingStrAttachmentIds)) {
            session.SetErrorInfo(TIncidentData::GetTableName(), "Incorrect photo ids stored already", EDriveSessionResult::InconsistencySystem);
            return {};
        }

        TVector<ui64> attachmentIds;
        for (auto&& strAttachmentId: strAttachmentIds) {
            if (existingStrAttachmentIds.contains(strAttachmentId)) {
                continue;
            }
            ui64 attachmentId;
            if (!TryFromString(strAttachmentId, attachmentId)) {
                session.SetErrorInfo(TIncidentData::GetTableName(), "Incorrect photo id type: " + strAttachmentId, EDriveSessionResult::IncorrectRequest);
                return {};
            }
            attachmentIds.push_back(attachmentId);
        }

        return NJson::ToJson(attachmentIds);
    }

    void TMakeTicketBaseTransition::InitializeExistingIssue(const TIncidentStateContext& context, TDynamicContext& dynamicContext) const {
        TString issueKey;
        if (auto existingLink = context.OptionalInstance()->GetStartrekTicketLink(GetTransitionType())) {
            issueKey = existingLink->GetTicketKey();
            dynamicContext.emplace("issue_key", issueKey);
        }
        dynamicContext.emplace("issue_exists", NJson::ToJson(!issueKey.empty()).GetStringRobust());
    }

    bool TMakeTicketBaseTransition::HandleNotifyResult(TNotifierResultPtr resultPtr, TIncidentStateContext& context, NDrive::TEntitySession& session) const {
        TAtomicSharedPtr<TStartrekNotifierResult> nativeResultPtr = std::dynamic_pointer_cast<TStartrekNotifierResult>(resultPtr);
        if (!nativeResultPtr) {
            session.SetErrorInfo(TIncidentData::GetTableName(), "Incorrect notify result type", EDriveSessionResult::IncorrectRequest);
            return false;
        }

        TString ticketKey = nativeResultPtr->GetTicket().GetKey();
        if (!ticketKey) {
            session.SetErrorInfo(TIncidentData::GetTableName(), "Invalid ticket key", EDriveSessionResult::IncorrectRequest);
            return false;
        }

        context.MutableInstance()->UpsertStartrekTicketLink(TIncidentStartrekTicketLink(ticketKey, GetTransitionType(), context.GetPerformerId()), GetTransitionType());

        return true;
    }
}
