#include "update_data_composite.h"

#include "update_data.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/incident/incident_context.h>
#include <drive/backend/incident/manager.h>

#include <util/generic/algorithm.h>

namespace NDrive {
    TUpdateDataCompositeIncidentTransition::TRegistrator TUpdateDataCompositeIncidentTransition::Registrator;

    TUpdateDataCompositeIncidentTransition::TUpdateDataCompositeIncidentTransition()
        : TBase()
        , UpdateDataTransitionPtr(MakeAtomicShared<TUpdateDataIncidentTransition>())
    {
    }

    NDrive::TScheme TUpdateDataCompositeIncidentTransition::GetScheme(const TIncidentStateContext& context) const {
        return UpdateDataTransitionPtr->GetScheme(context);
    }

    bool TUpdateDataCompositeIncidentTransition::Initialize(const NJson::TJsonValue& data, const TIncidentStateContext& context, TMessagesCollector& errors) {
        if (!context.HasInstance()) {
            errors.AddMessage(__LOCATION__, "Incident instance is not initialized");
            return false;
        }

        IIncidentContext::TPtr contextPtr;
        EIncidentTransition makeTicketTransitionType, handleObjectTagsTransitionType;

        if (contextPtr = context.GetInstanceRef().GetContext(EIncidentContextType::IncidentTicket)) {
            makeTicketTransitionType = EIncidentTransition::MakeIncidentTicket;
            handleObjectTagsTransitionType = EIncidentTransition::HandleIncidentObjectTags;
        } else if (contextPtr = context.GetInstanceRef().GetContext(EIncidentContextType::EvacuationTicket)) {
            makeTicketTransitionType = EIncidentTransition::MakeEvacuationTicket;
            handleObjectTagsTransitionType = EIncidentTransition::HandleEvacuationObjectTags;
        } else {
            errors.AddMessage(__LOCATION__, "Inconsistent incident context type");
            return false;
        }

        const TIncidentsManager& incidentManager = context.GetServer()->GetDriveAPI()->GetIncidentsManager();

        if (!UpdateDataTransitionPtr->Initialize(data, context, errors)) {
            return false;
        }

        auto ticketData = contextPtr->SerializeToJson();
        MakeTicketTransitionPtr = incidentManager.ConstructTransition(makeTicketTransitionType, ticketData, context, errors);
        if (!MakeTicketTransitionPtr) {
            return false;
        }

        if (context.GetInstanceRef().GetStatus() == EIncidentStatus::ObjectTagsProcessed) {
            HandleObjectTagsTransitionPtr = incidentManager.ConstructTransition(handleObjectTagsTransitionType, ticketData, context, errors);
            if (!HandleObjectTagsTransitionPtr) {
                return false;
            }
        }

        return true;
    }

    TSet<EIncidentStatus> TUpdateDataCompositeIncidentTransition::GetAllowedSourceStatuses() const {
        return { EIncidentStatus::StartrekTicketsProcessed, EIncidentStatus::ObjectTagsProcessed };
    }

    EIncidentStatus TUpdateDataCompositeIncidentTransition::GetDestinationStatus(const TIncidentStateContext& context) const {
        return context.GetCurrentStatus();
    }

    bool TUpdateDataCompositeIncidentTransition::DoPerform(TIncidentStateContext& context, NDrive::TEntitySession& session) const {
        return UpdateDataTransitionPtr->Perform(context, session) &&
               ((MakeTicketTransitionPtr) ? MakeTicketTransitionPtr->Perform(context, session) : true) &&
               ((HandleObjectTagsTransitionPtr) ? HandleObjectTagsTransitionPtr->Perform(context, session) : true);
    }
}
