#include "update_renins_claim_docs.h"

#include <drive/backend/incident/contexts/renins_kasko_claim.h>

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/incident/renins_claims/client.h>
#include <drive/backend/incident/renins_claims/claim_entity_kasko.h>
#include <drive/backend/incident/renins_claims/common_entity.h>

namespace NDrive {
    TUpdateReninsClaimDocsIncidentTransition::TRegistrator TUpdateReninsClaimDocsIncidentTransition::Registrator;

    NDrive::TScheme TUpdateReninsClaimDocsIncidentTransition::GetScheme(const TIncidentStateContext& /* context */) const {
        return {};
    }

    bool TUpdateReninsClaimDocsIncidentTransition::Initialize(const NJson::TJsonValue& /* data */, const TIncidentStateContext& context, TMessagesCollector& errors) {
        if (!context.GetServer()->GetDriveAPI()->HasReninsClaimClient()) {
            errors.AddMessage(__LOCATION__, "No renins claim client configured");
            return false;
        }

        if (!context.HasInstance()) {
            errors.AddMessage(__LOCATION__, "Incident instance is not initialized");
            return false;
        }

        auto claimContextPtr = context.GetInstanceRef().GetContext<TReninsKaskoClaimIncidentContext>();
        if (!claimContextPtr) {
            errors.AddMessage(__LOCATION__, "Incident instance is not initialized");
            return false;
        }

        auto claimEntryPtr = claimContextPtr->GetClaimEntryPtr();
        if (claimEntryPtr->IsInitialClaimApplication()) {
            errors.AddMessage(__LOCATION__, "No claim number specified in claim");
            return false;
        }

        ClaimNumber = claimEntryPtr->GetClaim().GetClaimNumber();
        return true;
    }

    TSet<EIncidentStatus> TUpdateReninsClaimDocsIncidentTransition::GetAllowedSourceStatuses() const {
        return { EIncidentStatus::KaskoClaimApplied };
    }

    EIncidentStatus TUpdateReninsClaimDocsIncidentTransition::GetDestinationStatus(const TIncidentStateContext& context) const {
        return context.GetCurrentStatus();
    }

    TMaybe<bool> TUpdateReninsClaimDocsIncidentTransition::DoCheckIsApplicable(const TIncidentStateContext& context, NDrive::TEntitySession& /* session */) const {
        return (context.HasInstance()) ? (context.OptionalInstance()->GetIncidentType() != EIncidentType::Evacuation) : false;
    }

    bool TUpdateReninsClaimDocsIncidentTransition::DoPerform(TIncidentStateContext& context, NDrive::TEntitySession& session) const {
        auto& incident = context.MutableInstanceRef();

        NDrive::NRenins::TDocumentsEntry documentsEntry;
        documentsEntry.SetClaimNumber(ClaimNumber);
        {
            // collect documents to upload and update sent timestamp
            const auto timestamp = Now();

            TVector<NDrive::NRenins::TDocument> documents;

            for (auto& documentLink: incident.MutableDocumentLinks()) {
                const bool updateRequired = documentLink.GetLastModifiedAt() > documentLink.OptionalLastSentAt().GetOrElse(TInstant::Zero());
                if (updateRequired) {
                    documents.emplace_back(NDrive::NRenins::TDocument(documentLink.GetDocumentUrl(), documentLink.GetAttachmentCode(), documentLink.GetDocumentType()));
                    documentLink.SetLastSentAt(timestamp);
                }
            }

            for (auto& photoLink: incident.MutablePhotoLinks()) {
                const bool updateRequired = photoLink.GetLastModifiedAt() > photoLink.OptionalLastSentAt().GetOrElse(TInstant::Zero());
                TString photoUrl = photoLink.GetImageUrl();
                if (updateRequired && photoUrl) {
                    documents.emplace_back(NDrive::NRenins::TDocument(photoUrl, photoLink.GetAttachmentCode(), ""));
                    photoLink.SetLastSentAt(timestamp);
                }
            }

            if (!documents) {
                session.SetErrorInfo(TIncidentData::GetTableName(), "All documents has been already uploaded", EDriveSessionResult::IncorrectRequest);
                return false;
            }

            NDrive::NRenins::TDocuments documentsInfo;
            documentsInfo.SetDocuments(std::move(documents));

            documentsEntry.SetDocuments(std::move(documentsInfo));
        }

        const auto& reninsClient = context.GetServer()->GetDriveAPI()->GetReninsClaimClient();
        TMessagesCollector errors;
        if (!reninsClient.UploadDocumentsSync(documentsEntry, errors)) {
            session.SetErrorInfo(TIncidentData::GetTableName(), errors.GetStringReport(), EDriveSessionResult::InternalError);
            return false;
        }

        return true;
    }
}
