#include "update_ticket_composite.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/incident/incident_context.h>
#include <drive/backend/incident/manager.h>

#include <util/generic/algorithm.h>

namespace NDrive {
    TUpdateTicketCompositeIncidentTransition::TRegistrator TUpdateTicketCompositeIncidentTransition::Registrator;

    NDrive::TScheme TUpdateTicketCompositeIncidentTransition::GetScheme(const TIncidentStateContext& context) const {
        if (!context.HasInstance()) {
            return {};
        }

        EIncidentTransition makeTicketTransitionType;
        if (context.GetInstanceRef().HasContext(EIncidentContextType::IncidentTicket)) {
            makeTicketTransitionType = EIncidentTransition::MakeIncidentTicket;
        } else if (context.GetInstanceRef().HasContext(EIncidentContextType::EvacuationTicket)) {
            makeTicketTransitionType = EIncidentTransition::MakeEvacuationTicket;
        } else {
            return {};
        }

        TIncidentStateManager::TTransitionPtr transitionPtr = TIncidentStateManager::TFactory::Construct(::ToString(makeTicketTransitionType));
        return (transitionPtr) ? transitionPtr->GetScheme(context) : NDrive::TScheme();
    }

    bool TUpdateTicketCompositeIncidentTransition::Initialize(const NJson::TJsonValue& data, const TIncidentStateContext& context, TMessagesCollector& errors) {
        if (!context.HasInstance()) {
            errors.AddMessage(__LOCATION__, "Incident instance is not initialized");
            return false;
        }

        EIncidentTransition makeTicketTransitionType, handleObjectTagsTransitionType;

        if (context.GetInstanceRef().HasContext(EIncidentContextType::IncidentTicket)) {
            makeTicketTransitionType = EIncidentTransition::MakeIncidentTicket;
            handleObjectTagsTransitionType = EIncidentTransition::HandleIncidentObjectTags;
        } else if (context.GetInstanceRef().HasContext(EIncidentContextType::EvacuationTicket)) {
            makeTicketTransitionType = EIncidentTransition::MakeEvacuationTicket;
            handleObjectTagsTransitionType = EIncidentTransition::HandleEvacuationObjectTags;
        } else {
            errors.AddMessage(__LOCATION__, "Inconsistent incident context type");
            return false;
        }

        const TIncidentsManager& incidentManager = context.GetServer()->GetDriveAPI()->GetIncidentsManager();

        MakeTicketTransitionPtr = incidentManager.ConstructTransition(makeTicketTransitionType, data, context, errors);
        if (!MakeTicketTransitionPtr) {
            return false;
        }

        HandleObjectTagsTransitionPtr = incidentManager.ConstructTransition(handleObjectTagsTransitionType, data, context, errors);
        if (!HandleObjectTagsTransitionPtr) {
            return false;
        }

        return true;
    }

    TSet<EIncidentStatus> TUpdateTicketCompositeIncidentTransition::GetAllowedSourceStatuses() const {
        return { EIncidentStatus::ObjectTagsProcessed };
    }

    EIncidentStatus TUpdateTicketCompositeIncidentTransition::GetDestinationStatus(const TIncidentStateContext& context) const {
        return context.GetCurrentStatus();
    }

    bool TUpdateTicketCompositeIncidentTransition::DoPerform(TIncidentStateContext& context, NDrive::TEntitySession& session) const {
        return MakeTicketTransitionPtr->Perform(context, session) &&
               HandleObjectTagsTransitionPtr->Perform(context, session);
    }
}
