#include "client.h"

#include <drive/backend/logging/events.h>

#include <library/cpp/json/json_reader.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/builder.h>
#include <rtline/library/unistat/cache.h>

#include <util/stream/file.h>

namespace NDrive {

    void TIngosClientConfig::Init(const TYandexConfig::Section* section, const TMap<TString, NSimpleMeta::TConfig>* requestPolicy) {
        const TYandexConfig::TSectionsMap children = section->GetAllChildren();
        {
            auto it = children.find("AuthModule");
            if (it != children.end()) {
                AuthModuleConfig.Init(it->second, requestPolicy);
            }
        }
        {
            auto it = children.find("Client");
            if (it != children.end()) {
                ClientConfig.Init(it->second, requestPolicy);
            }
        }

        AuthPath = section->GetDirectives().Value("AuthPath", AuthPath);
        AssertCorrectConfig(!!AuthPath, "no 'AuthPath' field");
        PutRidesPath = section->GetDirectives().Value("PutRidesPath", PutRidesPath);
        AssertCorrectConfig(!!PutRidesPath, "no 'PutRidesPath' field");
        AuthSecret = section->GetDirectives().Value("Secret", AuthSecret);
        SecretFile = section->GetDirectives().Value("SecretFile", SecretFile);
        if (!AuthSecret && SecretFile) {
            AuthSecret = TIFStream(SecretFile).ReadAll();
        }
        AssertCorrectConfig(!!AuthSecret, "neither 'Secret' nor 'SecretFile' field is present");
    }

    void TIngosClientConfig::ToString(IOutputStream& os) const {
        os << "<AuthModule>" << Endl;
            AuthModuleConfig.ToString(os);
        os << "</AuthModule>" << Endl;

        os << "<Client>" << Endl;
            ClientConfig.ToString(os);
        os << "</Client>" << Endl;

        os << "AuthPath: " << AuthPath << Endl;
        os << "PutRidesPath: " << PutRidesPath << Endl;
        os << "SecretFile: " << SecretFile << Endl;
    }

    TIngosClient::TAuthClient::TAuthClient(const TIngosClientConfig& config)
        : Config(config)
        , Impl(Config.GetAuthModuleConfig(), "ingos_auth")
    {}

    bool TIngosClient::TAuthClient::RefreshToken(TDuration reqTimeout, bool force) {
        TInstant now = Now();
        if (!force && !!AuthToken && NextRefresh >= now) {
            return true;
        }

        TStringStream postData;
        postData << "grant_type=client_credentials&scope=auth.carsharing&client_id=Yandex.Drive&client_secret="<< Config.GetAuthSecret();

        NNeh::THttpRequest req;
        req.SetUri(Config.GetAuthPath());
        req.SetPostData(TBlob::FromString(postData.Str())).SetRequestType("POST").AddHeader("Content-Type", "application/x-www-form-urlencoded");

        auto reply = Impl->SendMessageSync(req, Now() + reqTimeout);

        NDrive::TEventLog::Log("IngosRequest", NJson::TMapBuilder
            ("http_code", reply.Code())
            ("reply", NJson::ToJson(NJson::JsonString(reply.Content())))
            ("key", "auth_token")
            ("request", postData.Str())
        );

        TUnistatSignalsCache::SignalAdd("ingos-auth-reply_code", ::ToString(reply.Code()), 1);
        if (reply.Code() == 200) {
            NJson::TJsonValue replyJson;
            if (!ReadJsonFastTree(reply.Content(), &replyJson)) {
                return false;
            }

            if (!replyJson.Has("access_token")) {
                return false;
            }
            TGuard<TMutex> g(TokenLock);
            AuthToken = replyJson["access_token"].GetString();
            NextRefresh = now + TDuration::Seconds(replyJson["expires_in"].GetUInteger());
            return true;
        }
        return false;
    }

    TString TIngosClient::TAuthClient::GetAuthToken() const {
        TGuard<TMutex> g(TokenLock);
        return AuthToken;
    }

    bool TIngosClient::PushData(TVector<TInsuranceInfo>& ridings, TDuration reqTimeout, bool withBaseCost, const std::function<void(bool, const TInsuranceInfo&)>& callback) const {
        if (ridings.empty()) {
            return true;
        }

        TMap<TString, ui32> indexes;
        AuthModule.RefreshToken(reqTimeout, false);
        NJson::TJsonValue postData(NJson::JSON_MAP);
        for (ui32 index = 0; index < ridings.size(); ++index) {
            const auto& riding = ridings[index];
            NJson::TJsonValue ridingReport;
            ridingReport["partnerAgreementId"] = riding.GetPolicy().GetAgreementPartnerNumber();
            ridingReport["insurancePolicyId"] = riding.GetPolicy().GetAgreementNumber();
            ridingReport["vin"] = riding.GetVIN();
            ridingReport["startTime"] = riding.GetStartTime().ToStringLocalUpToSeconds();
            ridingReport["endTime"] = riding.GetEndTime().ToStringLocalUpToSeconds();
            ridingReport["cost"]["amount"] = riding.GetFinalCost(withBaseCost);
            ridingReport["cost"]["currency"] = "RUB";
            postData[riding.GetKey()] = ridingReport;
            indexes[riding.GetKey()] = index;
        }

        NNeh::THttpRequest req;
        req.SetUri(Config.GetPutRidesPath());
        req.SetPostData(TBlob::FromString(postData.GetStringRobust())).SetRequestType("PUT").AddHeader("Content-Type", "application/json");
        req.AddHeader("Authorization", "Bearer " + AuthModule.GetAuthToken());

        auto reply = Impl->SendMessageSync(req, Now() + reqTimeout);

        NDrive::TEventLog::Log("IngosRequest", NJson::TMapBuilder
            ("http_code", reply.Code())
            ("reply", NJson::ToJson(NJson::JsonString(reply.Content())))
            ("key", "puth_rides")
            ("request", postData)
            ("is_base_cost_included", withBaseCost)
        );

        TUnistatSignalsCache::SignalAdd("ingos-put-reply_code", ::ToString(reply.Code()), 1);
        if (reply.Code() == 200) {
            NJson::TJsonValue replyJson;
            if (!ReadJsonFastTree(reply.Content(), &replyJson)) {
                return false;
            }

            for (auto&& putResult : replyJson.GetMap()) {
                auto it = indexes.find(putResult.first);
                if (it != indexes.end()) {
                    callback(putResult.second["status"].GetUInteger() == 204, ridings[it->second]);
                }
            }
            return true;
        }

        if (reply.Code() == 401) {
            AuthModule.RefreshToken(reqTimeout, true);
        }
        return false;
    }
}
