#include "client.h"

#include <drive/backend/logging/events.h>

#include <library/cpp/json/json_reader.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/unistat/cache.h>


namespace {
    NJson::TJsonValue BuildReport(const NDrive::TInsuranceInfo& insuranceData, bool withBaseCost) {
        NJson::TJsonValue report;
        report["DriverSurname"] = "";
        report["DriverFirstname"] = "";
        report["DriverPatronymic"] = "";

        report["VIN"] = insuranceData.GetVIN();
        report["StartTime"] = insuranceData.GetStartTime().ToStringLocalUpToSeconds();
        report["EndTime"] = insuranceData.GetEndTime().ToStringLocalUpToSeconds();
        report["Key"] = insuranceData.GetKey();
        report["Cost"] = insuranceData.GetFinalCost(withBaseCost);
        report["AgreementPartnerNum"] = insuranceData.GetPolicy().GetAgreementPartnerNumber();
        report["AgreementNum"] = insuranceData.GetPolicy().GetAgreementNumber();
        return report;
    }
}

void NDrive::TReninsClientConfig::Init(const TYandexConfig::Section* section, const TMap<TString, NSimpleMeta::TConfig>* requestPolicy) {
    TAsyncApiImpl::TConfig::Init(section, requestPolicy);
    ApiPath = section->GetDirectives().Value("ApiPath", ApiPath);
    AssertCorrectConfig(!!ApiPath, "no 'ApiPath' field");
    ListApiPath = section->GetDirectives().Value("ListApiPath", ListApiPath);
    AssertCorrectConfig(!!ListApiPath, "no 'ListApiPath' field");
}

void NDrive::TReninsClientConfig::ToString(IOutputStream& os) const {
    TAsyncApiImpl::TConfig::ToString(os);
    os << "ApiPath: " << ApiPath << Endl;
    os << "ListApiPath: " << ListApiPath << Endl;
}


NDrive::TReninsClient::TReninsClient(const TReninsClientConfig& config)
    : Config(config)
    , Impl(Config, "renins_api")
{
}

void NDrive::TReninsClient::PushDataPack(const TVector<NDrive::TInsuranceInfo>& segments, TDuration reqTimeout, bool withBaseCost, const std::function<void(bool, const TInsuranceInfo&)>& callback) const {
    TMap<TString, ui64> ids;
    NJson::TJsonValue postJson;
    NJson::TJsonValue& tripList = postJson.InsertValue("TripList", NJson::JSON_ARRAY);
    ui32 idx = 0;
    for (auto&& segment : segments) {
        NJson::TJsonValue report = BuildReport(segment, withBaseCost);
        tripList.AppendValue(report);
        ids[segment.GetKey()] = idx;
        ++idx;
    }

    TString postData = postJson.GetStringRobust();
    NNeh::THttpRequest req;
    req.SetUri(Config.GetListApiPath());
    req.SetPostData(TBlob::FromString(postData)).SetRequestType("POST").AddHeader("Content-Type", "application/json");

    auto reply = Impl->SendMessageSync(req, Now() + reqTimeout);

    NDrive::TEventLog::Log("ReninsRequest", NJson::TMapBuilder
        ("http_code", reply.Code())
        ("reply", reply.Content())
        ("key", "pack")
        ("request", postData)
        ("is_base_cost_included", withBaseCost)
    );

    TUnistatSignalsCache::SignalAdd("renins-export-reply_code", ::ToString(reply.Code()), 1);
    if (reply.Code() == 200) {
        NJson::TJsonValue replyJson;
        if (!ReadJsonFastTree(reply.Content(), &replyJson)) {
            return;
        }

        for (auto&& item : replyJson["Results"].GetArray()) {
            const TString& key = item["Key"].GetString();
            auto it = ids.find(key);
            if (it != ids.end()) {
                callback(true, segments[it->second]);
            }
        }
    }
}


void NDrive::TReninsClient::PushData(const TVector<NDrive::TInsuranceInfo>& segments, ui32 atOnce, TDuration reqTimeout, bool withBaseCost, const std::function<void(bool, const TInsuranceInfo&)>& callback) const {
    TVector<NNeh::THttpRequest> requests;
    for (auto&& segment : segments) {
        auto postJson = BuildReport(segment, withBaseCost);
        NNeh::THttpRequest req;
        req.SetUri(Config.GetApiPath());
        req.SetPostData(TBlob::FromString(postJson.GetStringRobust())).SetRequestType("POST").AddHeader("Content-Type", "application/json");
        requests.emplace_back(std::move(req));
        NDrive::TEventLog::Log("ReninsRequest", NJson::TMapBuilder
            ("key", segment.GetKey())
            ("request", std::move(postJson))
            ("is_base_cost_included", withBaseCost)
        );
    }

    auto replies = Impl->SendPack(requests, Now() + TDuration::Minutes(1), atOnce, reqTimeout);
    Y_ENSURE_BT(requests.size() == replies.size(), requests.size() << "/" << replies.size());
    ui64 idx = 0;
    for (auto&& reply : replies) {
        TUnistatSignalsCache::SignalAdd("renins-export-reply_code", ::ToString(reply.Code()), 1);
        NDrive::TEventLog::Log("ReninsRequest", NJson::TMapBuilder
            ("http_code", reply.Code())
            ("key", segments[idx].GetKey())
            ("reply", reply.Content())
            ("is_base_cost_included", withBaseCost)
        );
        callback(reply.Code() == 200, segments[idx]);
        ++idx;
    }
}
