#include "history.h"

TInsuranceNotification::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase) {
    NotificationId = GetFieldDecodeIndex("notification_id", decoderBase);
    Start = GetFieldDecodeIndex("start", decoderBase);
    Finish = GetFieldDecodeIndex("finish", decoderBase);
    UserId = GetFieldDecodeIndex("user_id", decoderBase);
    CarId = GetFieldDecodeIndex("object_id", decoderBase);
    SessionId = GetFieldDecodeIndex("session_id", decoderBase);
}

bool TInsuranceNotification::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, NotificationId);
    READ_DECODER_VALUE(decoder, values, UserId);
    READ_DECODER_VALUE(decoder, values, CarId);
    READ_DECODER_VALUE(decoder, values, SessionId);
    READ_DECODER_VALUE_INSTANT(decoder, values, Start);
    READ_DECODER_VALUE_INSTANT(decoder, values, Finish);
    return true;
}

bool TInsuranceNotification::DeserializeFromTableRecord(const NStorage::TTableRecord& record, const ITagsHistoryContext* /*context*/) {
    ui32 tsStart;
    if (!record.TryGet("start", tsStart)) {
        return false;
    }
    Start = TInstant::Seconds(tsStart);

    ui32 tsFinish;
    if (!record.TryGet("finish", tsFinish)) {
        return false;
    }
    Finish = TInstant::Seconds(tsFinish);

    if (!record.TryGet("notification_id", NotificationId)) {
        return false;
    }

    UserId = record.Get("user_id");
    CarId = record.Get("object_id");
    SessionId = record.Get("session_id");
    return true;
}

NStorage::TTableRecord TInsuranceNotification::SerializeToTableRecord() const {
    NStorage::TTableRecord record;
    record.Set("session_id", SessionId);
    record.Set("object_id", CarId);
    record.Set("user_id", UserId);
    record.Set("notification_id", NotificationId);
    record.Set("start", Start.Seconds());
    record.Set("finish", Finish.Seconds());
    return record;
}

template <>
NJson::TJsonValue NJson::ToJson(const TInsuranceNotification& object) {
    NJson::TJsonValue result;
    result["id"] = object.GetNotificationId();
    result["start"] = NJson::ToJson(object.GetStart());
    result["finish"] = NJson::ToJson(object.GetFinish());
    result["car_id"] = object.GetCarId();
    result["user_id"] = object.GetUserId();
    result["session_id"] = object.GetSessionId();
    return result;
}

TInsuranceHistoryManager::TOptionalNotifications TInsuranceHistoryManager::GetSession(const TString& sessionId, NDrive::TEntitySession& tx) const {
    auto events = GetEvents({}, {}, tx, TQueryOptions()
        .AddGenericCondition("session_id", sessionId)
    );
    if (!events) {
        return {};
    }
    TMap<ui64, TProcessedInsuranceNotification> result;
    for (auto&& ev : *events) {
        auto p = result.emplace(ev.GetNotificationId(), ev).first;
        if (p == result.end()) {
            tx.SetErrorInfo("TProcessedInsuranceNotification", "internal error: cannot emplace notification");
            return {};
        }
        if (ev.GetHistoryAction() == EObjectHistoryAction::Add) {
            p->second.SetCreated(ev.GetHistoryInstant());
        }
        if (ev.GetHistoryAction() == EObjectHistoryAction::Remove) {
            p->second.SetSent(ev.GetHistoryInstant());
        }
    }
    return MakeVector(NContainer::Values(result));
}
