#include "localization.h"

#include <rtline/library/unistat/cache.h>

namespace NLocalization {
    const auto CzeLocalizationId = ToString(ELocalization::Cze);
    const auto EngLocalizationId = ToString(ELocalization::Eng);
    const auto GerLocalizationId = ToString(ELocalization::Ger);
    const auto RusLocalizationId = ToString(ELocalization::Rus);

    TStringBuf GetLocalizationId(ELocalization locale) {
        switch (locale) {
        case ELocalization::Cze: return CzeLocalizationId;
        case ELocalization::Eng: return EngLocalizationId;
        case ELocalization::Ger: return GerLocalizationId;
        case ELocalization::Rus: return RusLocalizationId;
        }
    }

    NJson::TJsonValue TResource::TResourceLocalization::SerializeToJson() const {
        NJson::TJsonValue result = NJson::JSON_MAP;
        result.InsertValue("l", Localization);
        result.InsertValue("v", Value);
        return result;
    }

    bool TResource::TResourceLocalization::DeserializeFromJson(const NJson::TJsonValue& info) {
        return
            NJson::ParseField(info, "l", Localization) &&
            NJson::ParseField(info, "v", Value);
    }

    TMaybe<TString> TResource::GetLocalization(ELocalization locale) const {
        auto localizationId = GetLocalizationId(locale);
        for (auto&& i : Localizations) {
            if (localizationId == i.GetLocalization()) {
                return i.GetValue();
            }
        }
        return {};
    }

    bool TResource::DeserializeFromJson(const NJson::TJsonValue& info) {
        if (!DeserializeDataFromJson(info)) {
            return false;
        }
        if (!info["resource_id"].GetString(&Id) || !Id) {
            return false;
        }
        return NJson::ParseField(info, "revision", Revision, false);
    }

    bool TResource::DeserializeDataFromJson(const NJson::TJsonValue& info) {
        const NJson::TJsonValue::TArray* arr;
        if (!info["localizations"].GetArrayPointer(&arr)) {
            return false;
        }
        for (auto&& i : *arr) {
            TResourceLocalization rl;
            if (rl.DeserializeFromJson(i)) {
                Localizations.emplace_back(std::move(rl));
            } else {
                ERROR_LOG << "Cannot parse from TResourceLocalization from " << i << Endl;
            }
        }
        return true;
    }

    bool TResource::DeserializeFromTableRecord(const NStorage::TTableRecord& record) {
        Id = record.Get("resource_id");
        if (!Id) {
            return false;
        }
        NJson::TJsonValue jsonData;
        if (!NJson::ReadJsonFastTree(record.Get("resource_data"), &jsonData) || !DeserializeDataFromJson(jsonData)) {
            return false;
        }
        if (!TryFromString(record.Get("revision"), MutableRevision())) {
            return false;
        }
        return true;
    }

    NStorage::TTableRecord TResource::SerializeToTableRecord() const {
        NStorage::TTableRecord result;
        result.Set("resource_id", Id).Set("resource_data", SerializeDataToJson());
        result.Set("revision", Revision);
        return result;
    }

    NJson::TJsonValue TResource::SerializeDataToJson() const {
        NJson::TJsonValue result = NJson::JSON_MAP;
        NJson::TJsonValue& localizations = result.InsertValue("localizations", NJson::JSON_ARRAY);
        for (auto&& i : Localizations) {
            localizations.AppendValue(i.SerializeToJson());
        }
        return result;
    }

    NJson::TJsonValue TResource::SerializeToJson() const {
        NJson::TJsonValue result = SerializeDataToJson();
        result.InsertValue("resource_id", Id);
        result.InsertValue("revision", Revision);
        return result;
    }

    bool TResource::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
        READ_DECODER_VALUE(decoder, values, Id);
        if (!Id) {
            return false;
        }
        NJson::TJsonValue jsonMeta;
        READ_DECODER_VALUE_JSON(decoder, values, jsonMeta, Meta);
        READ_DECODER_VALUE_DEF(decoder, values, Revision, 0);
        return DeserializeDataFromJson(jsonMeta);
    }

    bool TResource::HasRevision() const {
        return Revision != 0;
    }

    TMaybe<ui64> TResource::OptionalRevision() const {
        return HasRevision() ? Revision : TMaybe<ui64>();
    }

    TResource::TResource(const TString& id)
        : Id(id)
    {
    }

    bool TLocalizationDB::UpsertObject(const TObject& object, const TString& userId, NDrive::TEntitySession& session) const {
        NStorage::TTableRecord newObject = object.SerializeToTableRecord();
        newObject.Set("updated_at", ModelingNow().Seconds());
        NStorage::TTableRecord uniqueObject;
        uniqueObject.Set("resource_id", object.GetId());
        NStorage::ITableAccessor::TPtr table = HistoryCacheDatabase->GetTable("localization");
        NStorage::TObjectRecordsSet<TObject> records;
        bool isUpdate = false;
        switch (table->UpsertWithRevision(newObject, session.GetTransaction(), uniqueObject, object.OptionalRevision(), "revision", &records)) {
            case NStorage::ITableAccessor::EUpdateWithRevisionResult::IncorrectRevision:
                session.SetErrorInfo("upsert_localization", "incorect_revision", EDriveSessionResult::InconsistencyUser);
                return false;
            case NStorage::ITableAccessor::EUpdateWithRevisionResult::Failed:
                session.SetErrorInfo("upsert_localization", "UpsertWithRevision failure", EDriveSessionResult::TransactionProblem);
                return false;
            case NStorage::ITableAccessor::EUpdateWithRevisionResult::Updated:
                isUpdate = true;
            case NStorage::ITableAccessor::EUpdateWithRevisionResult::Inserted:
                break;
        }
        return HistoryManager->AddHistory(records.GetObjects(), userId, isUpdate ? EObjectHistoryAction::UpdateData : EObjectHistoryAction::Add, session);
    }

    bool TLocalizationDB::RemoveObject(const TSet<TString>& ids, const TString& userId, NDrive::TEntitySession& session) const {
        NStorage::ITableAccessor::TPtr table = HistoryCacheDatabase->GetTable("localization");
        TString condition = "true";
        if (ids.size()) {
            condition = "resource_id IN (" + session->Quote(ids) + ")";
        } else {
            session.SetErrorInfo("localization", "remove", EDriveSessionResult::IncorrectRequest);
            return false;
        }

        TVector<TResource> resources = GetCachedObjectsVector(ids, Now());
        NStorage::IQueryResult::TPtr result = table->RemoveRow(condition, session.GetTransaction());
        if (!result || !result->IsSucceed()) {
            session.SetErrorInfo("localization", "remove", EDriveSessionResult::TransactionProblem);
            return false;
        }
        if (!HistoryManager->AddHistory(resources, userId, EObjectHistoryAction::Remove, session)) {
            return false;
        }
        return true;
    }

    TMaybe<TString> TLocalizationDB::GetLocalStringImpl(ELocalization locale, const TString& resourceId) const {
        auto resource = GetObject(resourceId);
        if (!resource) {
            return {};
        }
        return resource->GetLocalization(locale);
    }

    const IPropositionsManager<TResource>* TLocalizationDB::GetPropositions() const {
        return &Propositions;
    }

    TResource::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase) {
        Id = GetFieldDecodeIndex("resource_id", decoderBase);
        Meta = GetFieldDecodeIndex("resource_data", decoderBase);
        Revision = GetFieldDecodeIndex("revision", decoderBase);
    }

    TMaybe<TString> TLocalizationDBDefaultLocalePreffered::GetLocalStringImpl(ELocalization locale, const TString& resourceId) const {
        const auto resource = GetObject(resourceId);
        if (!resource) {
            return {};
        }

        const auto localString = resource->GetLocalization(locale);
        if (!localString && locale != DefaultLocale) {
            return resource->GetLocalization(DefaultLocale);
        }
        return localString;
    }
}
