#pragma once

#include "config.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/database/history/cache.h>
#include <drive/backend/database/history/db_entities.h>
#include <drive/backend/database/history/manager.h>
#include <drive/backend/database/transaction/tx.h>

#include <rtline/util/types/accessor.h>

namespace NLocalization {
    class TResource {
    public:
        using TId = TString;
        using TRevision = ui64;

        class TDecoder: public TBaseDecoder {
            R_FIELD(i32, Id, -1);
            R_FIELD(i32, Meta, -1);
            R_FIELD(i32, Revision, -1);

        public:
            TDecoder() = default;
            TDecoder(const TMap<TString, ui32>& decoderBase);
        };

    private:
        class TResourceLocalization {
            R_FIELD(TString, Localization);
            R_FIELD(TString, Value);

        public:
            TResourceLocalization() = default;
            TResourceLocalization(const TString& localizationId, const TString& value)
                : Localization(localizationId)
                , Value(value)
            {
            }

            NJson::TJsonValue SerializeToJson() const;

            bool DeserializeFromJson(const NJson::TJsonValue& info);
        };

        R_FIELD(TString, Id);
        R_FIELD(TVector<TResourceLocalization>, Localizations);
        R_FIELD(TRevision, Revision, 0);

    public:
        TResource() = default;
        explicit TResource(const TString& id);

        bool Parse(const NStorage::TTableRecord& record) {
            return DeserializeFromTableRecord(record);
        }

        TMaybe<TString> GetLocalization(ELocalization locale) const;

        bool DeserializeFromJson(const NJson::TJsonValue& info);
        NStorage::TTableRecord SerializeToTableRecord() const;
        NJson::TJsonValue SerializeDataToJson() const;
        NJson::TJsonValue SerializeToJson() const;

        bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);

        bool DeserializeFromTableRecord(const NStorage::TTableRecord& record);
        bool DeserializeDataFromJson(const NJson::TJsonValue& info);

        NJson::TJsonValue BuildJsonReport() const {
            return SerializeToJson();
        }

        static TString GetTableName() {
            return "localization";
        }

        bool HasRevision() const;
        TMaybe<ui64> OptionalRevision() const;
    };

    class TLocalizationHistoryManager: public TIndexedAbstractHistoryManager<TResource> {
    private:
        using TBase = TIndexedAbstractHistoryManager<TResource>;

    public:
        TLocalizationHistoryManager(const IHistoryContext& context, const THistoryConfig& hConfig)
            : TBase(context, "localization_history", hConfig)
        {
        }
    };

    class TLocalizationPropositions: public TPropositionsManager<TResource> {
    private:
        using TBase = TPropositionsManager<TResource>;

    public:
        TLocalizationPropositions(const IHistoryContext& context, const TPropositionsManagerConfig& config)
            : TBase(context, "localization_propositions", config)
        {
        }
    };

    class TLocalizationDB
        : public TDBCacheWithHistoryOwner<TLocalizationHistoryManager, TResource>
        , public IDBEntitiesWithPropositionsManager<TResource>
        , public ILocalization
    {
    private:
        using TBase = TDBCacheWithHistoryOwner<TLocalizationHistoryManager, TResource>;
        using TObject = TResource;

    private:
        TLocalizationPropositions Propositions;

    private:
        virtual bool DoRebuildCacheUnsafe() const override {
            TRecordsSet records;
            {
                TTransactionPtr transaction = HistoryCacheDatabase->CreateTransaction(true);
                auto tagsTable = HistoryCacheDatabase->GetTable("localization");

                TQueryResultPtr queryResult;

                queryResult = tagsTable->GetRows("", records, transaction);

                if (!queryResult->IsSucceed()) {
                    return false;
                }
            }

            for (auto&& i : records) {
                TResource resource;
                if (resource.DeserializeFromTableRecord(i)) {
                    Objects.emplace(resource.GetId(), resource);
                } else {
                    ERROR_LOG << "Cannot parse localization info: " << i.ToCSV(";") << Endl;
                }
            }
            return true;
        }

        virtual TStringBuf GetEventObjectId(const TObjectEvent<TResource>& ev) const override {
            return ev.GetId();
        }
        virtual void AcceptHistoryEventUnsafe(const TObjectEvent<TResource>& ev) const override {
            if (ev.GetHistoryAction() == EObjectHistoryAction::Remove) {
                Objects.erase(ev.GetId());
            } else {
                Objects[ev.GetId()] = ev;
            }
        }

    protected:
        virtual TMaybe<TString> GetLocalStringImpl(ELocalization locale, const TString& resourceId) const override;

    public:
        bool GetInfo(const TSet<TString>& ids, TVector<TResource>& result, const TInstant reqActuality) const {
            const auto action = [&result](const TResource& obj) {
                result.emplace_back(obj);
            };
            return TBase::ForObjectsList(action, reqActuality, ids.empty() ? nullptr : &ids);
        }

        TLocalizationDB(const IHistoryContext& context, const TLocalizationConfig& config)
            : TBase("localization", context, config.GetHistoryConfig())
            , Propositions(context, config.GetPropositionsConfig())
        {
            Y_ENSURE_BT(Start());
        }

        ~TLocalizationDB() {
            if (!Stop()) {
                ERROR_LOG << "cannot stop LocalizationDB" << Endl;
            }
        }

        bool UpsertObject(const TObject& object, const TString& userId, NDrive::TEntitySession& session) const override;
        bool RemoveObject(const TSet<TString>& ids, const TString& userId, NDrive::TEntitySession& session) const override;
        bool GetObjects(TMap<TString, TObject>& /*objects*/, const TInstant /*reqActuality*/ = TInstant::Zero()) const override {
            return false;
        }
        bool AddObjects(const TVector<TObject>& /*objects*/, const TString& /*userId*/, NDrive::TEntitySession& /*session*/, NStorage::TObjectRecordsSet<TObject>* /*containerExt*/ = nullptr) const override {
            return false;
        }
        bool ForceUpsertObject(const TObject& /*object*/, const TString& /*userId*/, NDrive::TEntitySession& /*session*/, NStorage::TObjectRecordsSet<TObject>* /*containerExt*/ = nullptr) const override {
            return false;
        }
        const IPropositionsManager<TResource>* GetPropositions() const override;
    };

    class TLocalizationDBDefaultLocalePreffered: public TLocalizationDB {
    public:
    TLocalizationDBDefaultLocalePreffered(const IHistoryContext& context, const TLocalizationConfig& config)
        : TLocalizationDB(context, config)
    {
    }

    protected:
        virtual TMaybe<TString> GetLocalStringImpl(ELocalization locale, const TString& resourceId) const override;
    };
}
