#include "events.h"

#include <drive/library/cpp/network/data/data.h>
#include <drive/library/cpp/searchserver/replier.h>

#include <library/cpp/charset/ci_string.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/writer/json.h>
#include <library/cpp/string_utils/base64/base64.h>
#include <library/cpp/svnversion/svnversion.h>

#include <rtline/library/json/adapters.h>

#include <util/datetime/base.h>
#include <util/digest/fnv.h>
#include <util/stream/str.h>
#include <util/stream/zlib.h>
#include <util/system/hostname.h>
#include <util/system/tls.h>

Y_POD_STATIC_THREAD(const IReplyContext*) CurrentReplyContext;
Y_STATIC_THREAD(TString) CurrentReqId;
Y_STATIC_THREAD(TString) CurrentSource;
Y_STATIC_THREAD(TString) CurrentUserId;

namespace {
    TString CompressString(TStringBuf data) {
        TString result;
        {
            TStringOutput so(result);
            TZLibCompress zo(&so, ZLib::ZLib);
            zo.Write(data.data(), data.size());
        }
        return Base64Encode(result);
    }
    TString CompressJson(const NJson::TJsonValue& data) {
        if (data.IsString()) {
            return CompressString(data.GetStringSafe());
        } else {
            return CompressString(data.GetStringRobust());
        }
    }

    void FillCommon(NJsonWriter::TBuf& r, TInstant timestamp = TInstant::Zero()) {
        TInstant now = timestamp ? timestamp : Now();
        r.WriteKey("unixtime").WriteULongLong(now.Seconds());
        r.WriteKey("evstamp").WriteULongLong(now.MicroSeconds());
        r.WriteKey("host").WriteString(HostName());
        r.WriteKey("user_id").WriteString(TlsRef(CurrentUserId));
        r.WriteKey("source").WriteString(TlsRef(CurrentSource));
    }
    void FillReqId(NJsonWriter::TBuf& r, TStringBuf reqid) {
        r.WriteKey("reqid").WriteString(reqid);
    }
    void FillReqId(NJsonWriter::TBuf& r, const IReplyContext& context) {
        auto reqid = NUtil::GetReqId(context.GetRequestData(), context.GetCgiParameters());
        if (reqid) {
            FillReqId(r, reqid);
        } else {
            auto generatedReqId = ToString(context.GetRequestId()) + '-' + HostName();
            FillReqId(r, generatedReqId);
        }
    }
    void FillRequestMeta(NJsonWriter::TBuf& r, const IReplyContext& context) {
        const auto& rd = context.GetRequestData();
        r.WriteKey("ip").WriteString(NUtil::GetClientIp(rd));
        r.WriteKey("raw_ip").WriteString(rd.RemoteAddr());
        r.WriteKey("query").WriteString(rd.Query());
        r.WriteKey("reqstamp").WriteULongLong(rd.RequestBeginTime());
        FillReqId(r, context);
    }
    void FillRequestData(NJsonWriter::TBuf& r, const IReplyContext& context, NJson::TJsonValue data) {
        const TServerRequestData& rd = context.GetRequestData();
        NJson::TJsonValue& headers = data["headers"];
        TStringBuf contentType;
        TStringBuf testBuckets;

        ci_equal_to equals;
        for (auto&& i : rd.HeadersIn()) {
            const TString& key = i.first;
            TString value = i.second;
            if (equals(key, TStringBuf("Authorization"))) {
                value = ToString(FnvHash<ui64>(value));
            }
            if (equals(key, TStringBuf("Cookie"))) {
                value = ToString(FnvHash<ui64>(value));
            }
            if (equals(key, TStringBuf("X-Ya-Service-Ticket"))) {
                value = ToString(FnvHash<ui64>(value));
            }
            if (equals(key, TStringBuf("X-Ya-User-Ticket"))) {
                value = ToString(FnvHash<ui64>(value));
            }
            if (equals(key, TStringBuf("Content-Type"))) {
                contentType = value;
            }
            if (equals(key, TStringBuf("X-Yandex-ExpBoxes"))) {
                testBuckets = value;
            }
            if (equals(key, TStringBuf("External-Auth-Token"))) {
                value = ToString(FnvHash<ui64>(value));
            }
            headers.InsertValue(key, value);
        }

        const TBlob& blob = context.GetBuf();
        if (!blob.Empty()) {
            TStringBuf post(blob.AsCharPtr(), blob.Size());
            if (contentType.find("image/") != TStringBuf::npos) {
                data["post"] = Base64Encode(post);
            } else {
                data["post"] = NJson::ToJson(NJson::JsonString(post));
            }
        }
        r.WriteKey("data").WriteJsonValue(&data);

        if (testBuckets) {
            r.WriteKey("test_buckets").WriteString(testBuckets);
        }
    }
}

NDrive::TEventLog::TEventLog(THolder<TLogBackend> backend)
    : TSearchLog()
{
    ResetBackend(std::move(backend));
}

NDrive::TEventLog::TState NDrive::TEventLog::CaptureState() {
    TState result;
    result.EventLogger = NThreading::CaptureEventLogger();
    if (auto context = TlsRef(CurrentReplyContext)) {
        result.ReqId = TString{NUtil::GetReqId(context->GetRequestData(), context->GetCgiParameters())};
    } else {
        result.ReqId = TlsRef(CurrentReqId);
    }
    result.Source = TlsRef(CurrentSource);
    result.UserId = TlsRef(CurrentUserId);
    return result;
}

TStringBuf NDrive::TEventLog::GetReqId() {
    if (auto context = TlsRef(CurrentReplyContext)) {
        return NUtil::GetReqId(context->GetRequestData(), context->GetCgiParameters());
    } else {
        return TlsRef(CurrentReqId);
    }
}

TString NDrive::TEventLog::GetSource() {
    return TlsRef(CurrentSource);
}

void NDrive::TEventLog::Log(ERequestEvent event, const IReplyContext& context, const NJson::TJsonValue& data) noexcept try {
    TString eventString = ToString(event);
    NJsonWriter::TBuf r;
    r.BeginObject();
    FillCommon(r);
    FillRequestMeta(r, context);
    FillRequestData(r, context, data);
    r.WriteKey("event").WriteString(eventString);
    r.EndObject();
    Log(eventString, r);
} catch (const std::exception& e) {
    ERROR_LOG << "EventLog::Log " << event << " triggered an exception: " << FormatExc(e) << Endl;
}

void NDrive::TEventLog::Log(EResponseEvent event, const IReplyContext& context, i32 code, const NJson::TJsonValue& data, TStringBuf serialized) noexcept try {
    TString eventString = ToString(event);
    TInstant deadline = context.GetRequestDeadline();
    TInstant start = context.GetRequestStartTime();
    TInstant finish = Now();
    TDuration duration = finish - start;

    NJsonWriter::TBuf r;
    r.BeginObject();
    FillCommon(r, finish);
    FillRequestMeta(r, context);
    r.WriteKey("event").WriteString(eventString);
    r.WriteKey("code").WriteLongLong(code);
    r.WriteKey("reply_time").WriteLongLong(duration.MicroSeconds());
    if (finish > deadline) {
        r.WriteKey("timeouted").WriteBool(true);
    }
    if ((data.IsArray() || data.IsMap()) && serialized) {
        TStringBuf stripped = StripString(serialized);
        if (event == CompressedResponse) {
            r.WriteKey("data").WriteString(CompressString(stripped));
        } else {
            r.WriteKey("data").UnsafeWriteValue(stripped);
        }
    } else {
        if (event == CompressedResponse) {
            r.WriteKey("data").WriteString(CompressJson(data));
        } else if (data.IsDefined()) {
            r.WriteKey("data").WriteJsonValue(&data);
        } else {
            r.WriteKey("data").WriteString({});
        }
    }
    r.EndObject();
    Log(eventString, r);
} catch (const std::exception& e) {
    ERROR_LOG << "EventLog::Log " << event << " triggered an exception: " << FormatExc(e) << Endl;
}

void NDrive::TEventLog::Log(const TString& event, const NJson::TJsonValue& data, TMaybe<i32> code) noexcept try {
    NJsonWriter::TBuf r;
    r.BeginObject();
    FillCommon(r);
    if (CurrentReplyContext) {
        FillRequestMeta(r, *CurrentReplyContext);
    } else if (const auto& reqid = TlsRef(CurrentReqId)) {
        FillReqId(r, reqid);
    }
    if (code) {
        r.WriteKey("code").WriteLongLong(*code);
    }
    r.WriteKey("event").WriteString(event);
    r.WriteKey("data").WriteJsonValue(&data);
    r.EndObject();
    Log(event, r);
} catch (const std::exception& e) {
    ERROR_LOG << "EventLog::Log " << event << " triggered an exception: " << FormatExc(e) << Endl;
}

void NDrive::TEventLog::Log(TStringBuf event, const NJsonWriter::TBuf& r) {
    if (TLoggerOperator<TEventLog>::Usage()) {
        TLoggerOperator<TEventLog>::Log() << r.Str() << Endl;
    } else {
        INFO_LOG << r.Str() << Endl;
    }
    auto size = r.Str().size();
    if (size > 32 * 1024 * 1024) {
        WARNING_LOG << "HugeEventLog: event=" << event << " reqid=" << TlsRef(CurrentReqId) << " source=" << TlsRef(CurrentSource) << " size=" << size << Endl;
    }
}

void NDrive::TEventLog::SetContext(const IReplyContext* value) {
    CurrentReplyContext = value;
}

void NDrive::TEventLog::SetSource(const TString& value) {
    CurrentSource = value;
}

void NDrive::TEventLog::SetUserId(const TString& value) {
    CurrentUserId = value;
}

void NDrive::TEventLog::ClearContext() {
    CurrentReplyContext = nullptr;
}

void NDrive::TEventLog::ClearSource() {
    TlsRef(CurrentSource).clear();
}

void NDrive::TEventLog::ClearUserId() {
    TlsRef(CurrentUserId).clear();
}

NDrive::TEventLog::TContextGuard::TContextGuard(const IReplyContext* value) {
    SetContext(value);
    if (value) {
        SetSource(TString{value->GetRequestData().ScriptName()});
    }
}

NDrive::TEventLog::TContextGuard::~TContextGuard() {
    if (CurrentReplyContext) {
        ClearSource();
    }
    ClearContext();
}

NDrive::TEventLog::TReqIdGuard::TReqIdGuard(const TString& value)
    : PreviousValue(TlsRef(CurrentReqId))
{
    TlsRef(CurrentReqId) = value;
}

NDrive::TEventLog::TReqIdGuard::~TReqIdGuard() {
    TlsRef(CurrentReqId) = PreviousValue;
}

NDrive::TEventLog::TSourceGuard::TSourceGuard(const TString& value)
    : PreviousValue(TlsRef(CurrentSource))
{
    SetSource(value);
}

NDrive::TEventLog::TSourceGuard::~TSourceGuard() {
    SetSource(PreviousValue);
}

TString PendingUserId = "__pending_user_id";

NDrive::TEventLog::TUserIdGuard::TUserIdGuard()
    : TUserIdGuard(PendingUserId)
{
}

NDrive::TEventLog::TUserIdGuard::TUserIdGuard(const TString& value)
    : PreviousUserId(TlsRef(CurrentUserId))
{
    SetUserId(value);
}

NDrive::TEventLog::TUserIdGuard::~TUserIdGuard() {
    SetUserId(PreviousUserId);
}

NDrive::TEventLog::TStateGuard::TStateGuard(const TState& value)
    : EventLoggerGuard(value.EventLogger)
    , ReqIdGuard(value.ReqId)
    , SourceGuard(value.Source)
    , UserIdGuard(value.UserId)
{
}

void NDrive::TEventLog::TUserIdGuard::Set(const TString& value) {
    Y_ASSERT(TlsRef(CurrentUserId) == PendingUserId);
    SetUserId(value);
}

NDrive::TEventLog::TTimeGuard::TTimeGuard(const TString& name)
    : Name(name)
    , Start(Now())
{
}

NDrive::TEventLog::TTimeGuard::~TTimeGuard() {
    auto finish = Now();
    auto duration = finish - Start;
    auto uncaughtExceptions = std::uncaught_exceptions();
    NDrive::TEventLog::Log(Name + "TimeGuard", NJson::TMapBuilder
        ("duration", NJson::ToJson(duration))
        ("finish", NJson::ToJson(finish))
        ("start", NJson::ToJson(Start))
        ("svn_revision", GetProgramSvnRevision())
        ("uncaught_exceptions", uncaughtExceptions)
    );
}
