#pragma once

#include "evlog.h"

#include <rtline/library/json/builder.h>

#include <kernel/searchlog/searchlog.h>

#include <library/cpp/json/json_value.h>

#include <util/generic/maybe.h>
#include <util/datetime/base.h>

class IReplyContext;

namespace NJsonWriter {
    class TBuf;
}

namespace NDrive {
    class TEventLog: public TSearchLog {
    public:
        enum ERequestEvent {
            Access,
            Authorization,
        };
        enum EResponseEvent {
            CompressedResponse,
            ResponseInfo,
            Response,
            Timeout,
        };

        struct TState {
            NThreading::TEventLoggerPtr EventLogger;
            TString ReqId;
            TString Source;
            TString UserId;
        };

        class TContextGuard {
        public:
            TContextGuard(const IReplyContext* value);
            ~TContextGuard();
        };

        class TReqIdGuard {
        public:
            TReqIdGuard(const TString& value);
            ~TReqIdGuard();

        private:
            TString PreviousValue;
        };

        class TSourceGuard {
        public:
            TSourceGuard(const TString& value);
            ~TSourceGuard();

        private:
            TString PreviousValue;
        };

        class TUserIdGuard {
        public:
            TUserIdGuard();
            TUserIdGuard(const TString& value);
            ~TUserIdGuard();

            static void Set(const TString& value);

        private:
            TString PreviousUserId;
        };

        class TStateGuard {
        public:
            TStateGuard(const TState& value);

        private:
            NThreading::TEventLoggerGuard EventLoggerGuard;
            TReqIdGuard ReqIdGuard;
            TSourceGuard SourceGuard;
            TUserIdGuard UserIdGuard;
        };

        class TTimeGuard {
        public:
            TTimeGuard(const TString& name);
            ~TTimeGuard();

        private:
            TString Name;
            TInstant Start;
        };

    public:
        using TSearchLog::TSearchLog;
        TEventLog(THolder<TLogBackend> backend);

        static TStateGuard Guard(const TState& state) {
            return { state };
        }
        static TTimeGuard Guard(const TString& name) {
            return { name };
        }

        static TState CaptureState();
        static TStringBuf GetReqId();
        static TString GetSource();

        static void Log(ERequestEvent event, const IReplyContext& context, const NJson::TJsonValue& data = {}) noexcept;
        static void Log(EResponseEvent event, const IReplyContext& context, i32 code, const NJson::TJsonValue& data, TStringBuf serialized = {}) noexcept;
        static void Log(const TString& event, const NJson::TJsonValue& data, TMaybe<i32> code = {}) noexcept;

    private:
        static void Log(TStringBuf event, const NJsonWriter::TBuf& r);
        static void SetContext(const IReplyContext* value);
        static void SetSource(const TString& value);
        static void SetUserId(const TString& value);
        static void ClearContext();
        static void ClearSource();
        static void ClearUserId();
    };
}
