#include "logging.h"

#include <drive/library/cpp/searchserver/replier.h>

TEnumSignal<HttpCodes, double> NDrive::TUnistatSignals::CodesMeter({ "frontend-reply-codes" }, false);
TEnumSignal<HttpCodes, double> NDrive::TUnistatSignals::CodesTimes({ "frontend-reply-times" }, NRTLineHistogramSignals::IntervalsRTLineReply);
TSignalByKey<TString, TReplyInfo> NDrive::TUnistatSignals::ServiceSignalsReply;
TSignalByKey<TString, double> NDrive::TUnistatSignals::ServiceSignalsAccess;
TSignalByKey<TString, double> NDrive::TUnistatSignals::ServiceSignalsTimeouted;

class TServiceResults: public IContextSignal<TReplyInfo> {
private:
    TEnumSignal<HttpCodes, double, true> SignalCounter;
    TEnumSignal<HttpCodes, double, true> SignalTimes;
    TUnistatSignal<double> Signal5xx;

public:
    TServiceResults(const TString& processorName)
        : SignalCounter({ "frontend-" + processorName + "-reply-count" }, false)
        , SignalTimes({ "frontend-" + processorName + "-reply-times" }, NRTLineHistogramSignals::IntervalsRTLineReply)
        , Signal5xx({ "frontend-" + processorName + "-reply-count-5xx" }, false)
    {
    }

    void Signal(const TReplyInfo& context) const override {
        auto code = context.GetCode();
        auto httpCode = static_cast<HttpCodes>(code);
        SignalCounter.Signal(httpCode, 1);
        SignalTimes.Signal(httpCode, context.GetDuration().MilliSeconds());
        if (IsServerError(httpCode)) {
            Signal5xx.Signal(1);
        }
    }
};

class TServiceAccess: public IContextSignal<double> {
private:
    TUnistatSignal<double> SignalCounter;

public:
    TServiceAccess(const TString& processorName)
        : SignalCounter({ "frontend-" + processorName + "-access-count" }, false)
    {
    }

    void Signal(const double& value) const override {
        SignalCounter.Signal(value);
    }
};

class TServiceTimeout: public IContextSignal<double> {
private:
    TUnistatSignal<double> SignalCounter;

public:
    TServiceTimeout(const TString& processorName)
        : SignalCounter({ "frontend-" + processorName + "-timeouted" }, false)
    {
    }

    void Signal(const double& value) const override {
        SignalCounter.Signal(value);
    }
};

void NDrive::TUnistatSignals::RegisterFallback() {
    ServiceSignalsReply.RegisterFallback(MakeHolder<TServiceResults>("incorrect_procesor"));
    ServiceSignalsAccess.RegisterFallback(MakeHolder<TServiceAccess>("incorrect_procesor"));
    ServiceSignalsTimeouted.RegisterFallback(MakeHolder<TServiceTimeout>("incorrect_processor"));
}

void NDrive::TUnistatSignals::RegisterSignals(const TString& processorName) {
    ServiceSignalsReply.RegisterSignal(processorName, MakeHolder<TServiceResults>(processorName));
    ServiceSignalsAccess.RegisterSignal(processorName, MakeHolder<TServiceAccess>(processorName));
    ServiceSignalsTimeouted.RegisterSignal(processorName, MakeHolder<TServiceTimeout>(processorName));
}

void NDrive::TUnistatSignals::OnAccess(const TString& processorName) {
    ServiceSignalsAccess.Signal(processorName, 1);
}

void NDrive::TUnistatSignals::OnReply(const TString& processorName, HttpCodes code, const IReplyContext& context) {
    const auto finish = Now();
    const auto start = context.GetRequestStartTime();
    const auto deadline = context.GetRequestDeadline();
    OnReply(processorName, code, start, finish, deadline);
}

void NDrive::TUnistatSignals::OnReply(const TString& processorName, HttpCodes code, TInstant start, TInstant finish, TInstant deadline) {
    const TDuration duration = finish - start;
    TReplyInfo info(code, duration);
    ServiceSignalsReply.Signal(processorName, info);
    CodesMeter.Signal(code, 1);
    CodesTimes.Signal(code, duration.MilliSeconds());
    if (finish > deadline) {
        ServiceSignalsTimeouted.Signal(processorName, 1);
    }
}

template<>
IOutputStream& operator<<<HttpCodes>(IOutputStream& os, const HttpCodes code) {
    os << ui32(code);
    return os;
}
