#include "entity.h"

#include <rtline/util/json_processing.h>

#include <library/cpp/json/json_reader.h>


const TString TMaintenanceInfo::InternalSource = "frontend";

bool TMaintenanceInfo::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, VIN);
    if (decoder.GetMileage() >= 0) {
        TStringBuf mileageStr = values[decoder.GetMileage()];
        if (!!mileageStr) {
            double mileage;
            if (TryFromString(mileageStr, mileage)) {
                Mileage = mileage;
            }
        }
    }
    READ_DECODER_VALUE_INSTANT(decoder, values, StartDate);
    if (decoder.GetReadyDate() >= 0) {
        TStringBuf readyDateStr = values[decoder.GetReadyDate()];
        if (!!readyDateStr) {
            ui32 readyDate;
            if (TryFromString(readyDateStr, readyDate)) {
                ReadyDate = TInstant::Seconds(readyDate);
            }
        }
    }
    READ_DECODER_VALUE(decoder, values, Source);
    if (decoder.GetIntermediate() >= 0) {
        READ_DECODER_VALUE_DEF(decoder, values, Intermediate, false);
    }
    return true;
}

bool TMaintenanceInfo::Parse(const NJson::TJsonValue& info) {
    if (info["LastTORun"].IsDefined()) {
        double mileage;
        JREAD_DOUBLE(info, "LastTORun", mileage);
        Mileage = mileage;
    }
    if (info.Has("ReadyDate") && info["ReadyDate"].IsDefined()) {
        TInstant readyDate;
        JREAD_INSTANT_ISOFORMAT_NULLABLE_OPT(info, "ReadyDate", readyDate);
        ReadyDate = readyDate;
    }
    JREAD_INSTANT_ISOFORMAT_NULLABLE_OPT(info, "StartDate", StartDate);
    JREAD_STRING(info, "VIN", VIN);
    Intermediate = false;
    return true;
}

NStorage::TTableRecord TMaintenanceInfo::SerializeToTableRecord() const {
    NStorage::TTableRecord result;
    result.Set("vin", VIN).Set("start_date", StartDate.Seconds());
    if (!!Mileage) {
        result.Set("mileage", *Mileage);
    } else {
        result.Set("mileage", "get_null()");
    }
    if (!!ReadyDate) {
        result.Set("date", ReadyDate->Seconds());
    } else {
        result.Set("date", "get_null()");
    }
    result.Set("source", Source);
    result.Set("is_intermediate", Intermediate);
    return result;
}

NJson::TJsonValue TMaintenanceInfo::SerializeToJson() const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    JWRITE(result, "vin", VIN);
    if (Mileage) {
        JWRITE(result, "mileage", *Mileage);
    }
    JWRITE(result, "start_date", StartDate.Seconds());
    if (ReadyDate) {
        JWRITE(result, "ready_date", ReadyDate->Seconds());
    }
    JWRITE(result, "source", Source);
    JWRITE(result, "is_intermediate", Intermediate);
    return result;
}

bool TMaintenanceInfo::DeserializeFromJson(const NJson::TJsonValue& info) {
    JREAD_STRING(info, "vin", VIN);
    JREAD_STRING_OPT(info, "source", Source);
    if (!Source) {
        Source = "undefined";
    }
    JREAD_INSTANT(info, "start_date", StartDate);
    JREAD_BOOL_OPT(info, "is_intermediate", Intermediate);
    TJsonProcessor::Read(info, "ready_date", ReadyDate);
    TJsonProcessor::Read(info, "mileage", Mileage);
    return true;
}

bool TMaintenanceInfo::IsEqual(const TMaintenanceInfo& info) const {
    return VIN == info.VIN
        && Mileage == info.Mileage
        && StartDate == info.StartDate
        && ReadyDate == info.ReadyDate
        && Intermediate == info.Intermediate;
}

bool TMaintenanceInfo::operator<(const TMaintenanceInfo& item) const {
    if (VIN == item.VIN) {
        if (StartDate == item.StartDate) {
            return Mileage.GetOrElse(0) > item.Mileage.GetOrElse(0);
        } else {
            return StartDate > item.StartDate;
        }
    } else {
        return VIN < item.VIN;
    }
}

TMaintenanceInfo::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase) {
    VIN = GetFieldDecodeIndex("vin", decoderBase);
    Mileage = GetFieldDecodeIndex("mileage", decoderBase);
    StartDate = GetFieldDecodeIndex("start_date", decoderBase);
    ReadyDate = GetFieldDecodeIndex("date", decoderBase);
    Source = GetFieldDecodeIndex("source", decoderBase);
    Intermediate = GetFieldDecodeIndex("is_intermediate", decoderBase);
}
