#include "client.h"

#include <util/datetime/base.h>

TMajorClient::TMajorClient(const TMajorClientConfig& config)
    : Config(config)
    , Logger("MajorClient")
    , Token("")
    , TokenDeadline(TInstant::Now() - TDuration::Hours(1))
{
}

TMajorClient::~TMajorClient() {
}

bool TMajorClient::GetOrUpdateToken(TString& token, TMessagesCollector& errors) const {
    TInstant deadline = TInstant::Zero();
    {
        TReadGuard rg(Mutex);
        deadline = TokenDeadline;
    }
    if (TInstant::Now() > deadline) {
        TWriteGuard wg(Mutex);
        NMajorClient::TTokenRequest request(Config.GetUserName(), Config.GetPassword(), Token, TokenDeadline, Logger, errors);
        if (!SendRequest(Config.GetTokenAPIPath(), request, false)) {
            ERROR_LOG << "Token is not defined : error in request" << Endl;
            return false;
        }
    }
    {
        TReadGuard rg(Mutex);
        token = Token;
    }
    return true;
}

bool TMajorClient::GetMaintenanceInfo(TVector<TMaintenanceInfo>& info, TMessagesCollector& errors) const {
    NMajorClient::TMaintenanceInfoRequest request(info, Logger, errors);
    if (!SendRequest(Config.GetMaintenanceInfoAPIPath(), request)) {
        return false;
    }
    std::sort(info.begin(), info.end());
    ui32 delta = 0;
    for (ui32 i = 1; i + delta < info.size();) {
        info[i] = info[i + delta];
        if (info[i - 1].GetVIN() == info[i].GetVIN()) {
            ++delta;
        } else {
            ++i;
        }
    }
    info.resize(info.size() - delta);
    return true;
}

bool TMajorClient::SendRequest(const TString& uri, NMajorClient::IMajorRequest& majorRequest, bool auth) const {
    NUtil::THttpRequest request(uri, majorRequest.CreateContent());
    request.SetIsHttps(Config.GetHttps());
    request.SetContentType("application/json");
    request.SetTimeout(Config.GetRequestTimeout().MilliSeconds());
    request.SetAttemptionsMax(Config.GetAttemptionsMax());
    if (auth) {
        TString token;
        if (!GetOrUpdateToken(token, majorRequest.MutableErrors())) {
            return false;
        }
        request.SetHeader("Authorization", token);
    }
    NUtil::THttpReply reply = request.Execute(Config.GetHost(), Config.GetPort());
    return majorRequest.ProcessReply(reply);
}

bool TMajorClient::GetAllQueryList(TVector<NMajorClient::TGetQueriesRequest::TQuery>& queryList, TMessagesCollector& errors) const {
    NMajorClient::TGetQueriesRequest request(queryList, Logger, errors);
    return SendRequest(Config.GetQueriesAPIPath(), request);
}

bool TMajorClient::GetQueryList(bool closed, TVector<NMajorClient::TGetQueriesRequest::TQuery>& queryList, TMessagesCollector& errors) const {
    NMajorClient::TGetQueriesRequest request(closed, queryList, Logger, errors);
    return SendRequest(Config.GetQueriesAPIPath(), request);
}

bool TMajorClient::GetTypeQueryList(TVector<NMajorClient::TTypeQueryRequest::TType>& result, TMessagesCollector& errors) const {
    NMajorClient::TTypeQueryRequest request(result, Logger, errors);
    return SendRequest(Config.GetTypeAPIPath(), request);
}

bool TMajorClient::CreateQuery(const NMajorClient::TCreateQueryRequest::TQuery& query, NMajorClient::TCreateQueryRequest::TResponse& response, TMessagesCollector& errors) const {
    NMajorClient::TCreateQueryRequest request(query, response, Config, Logger, errors);
    return SendRequest(Config.GetCreateAPIPath(), request);
}

bool TMajorClient::GetQueryInfo(const TString& id, NMajorClient::TQueryInfoRequest::TFullQueryInfo& parameterList, TMessagesCollector& errors) const {
    NMajorClient::TQueryInfoRequest request(id, parameterList, Logger, errors);
    return SendRequest(Config.GetInfoAPIPath(), request);
}

bool TMajorClient::CancelQuery(const TString& id, TMessagesCollector& errors) const {
    NMajorClient::TCancelRequest request(id, Logger, errors);
    return SendRequest(Config.GetCancelAPIPath(), request);
}

bool TMajorClient::GetOSAGO(const TString& vin, const bool isCurrent , NMajorClient::TOSAGORequest::TOSAGODocument& osago, TMessagesCollector& errors) const {
    NMajorClient::TOSAGORequest request(vin, isCurrent, osago, Logger, errors);
    return SendRequest(Config.GetOSAGOAPIPath(), request);
}

bool TMajorClient::CreateSTSRequest(const TString& vin, NMajorClient::TSTSRequest::TResponse& commonResponse, TMessagesCollector& errors) const {
    NMajorClient::TSTSRequest request(vin, commonResponse, Config, Logger, errors);
    return SendRequest(Config.GetCommonCreateAPIPath(), request);
}

bool TMajorClient::GetQueryHistoryInfo(const TString& id, NMajorClient::THistoryInfoRequest::TInfoWithHistory& history, TMessagesCollector& errors) const {
    NMajorClient::THistoryInfoRequest request(id, history, Logger, errors);
    return SendRequest(Config.GetCommonHistoryAPIPath(), request);
}

bool TMajorClient::GetStatus(const TString& id, TString& status, bool& isClosed, TMessagesCollector& errors) const {
    NMajorClient::THistoryInfoRequest::TInfoWithHistory history;
    NMajorClient::THistoryInfoRequest request(id, history, Logger, errors);
    if (!SendRequest(Config.GetCommonHistoryAPIPath(), request) || history.GetHistory().empty()) {
        return false;
    }
    status = history.GetCurrentState().GetStatus();
    isClosed = history.GetCurrentState().GetClosed();
    return true;
}

bool TMajorClient::GetFiles(const TString& id, const TSet<TString>& extensions, TVector<NMajorClient::TGetFileRequest::TMajorFile>& files, TMessagesCollector& errors) const {
    TVector<NMajorClient::TFileListRequest::TFileInfo> fileList;
    NMajorClient::TFileListRequest request(id, fileList, Logger, errors);
    if (!SendRequest(Config.GetFileListAPIPath(), request)) {
        return false;
    }

    TVector <NMajorClient::TGetFileRequest::TMajorFile> result;
    for (const auto& file : fileList) {
        if (!extensions.empty() && !extensions.contains(file.GetExtension())) {
            continue;
        }
        NMajorClient::TGetFileRequest::TMajorFile binaryFile;
        NMajorClient::TGetFileRequest request(file, binaryFile, Logger, errors);
        if (!SendRequest(Config.GetFileAPIPath(), request)) {
            return false;
        }
        result.emplace_back(binaryFile);
    }
    files = result;
    return true;
}

bool TMajorClient::GetTyreServiceInfo(NMajorClient::TTyreServiceInfoRequest::TTyreServiceInfo& info, const TInstant since, const TInstant until, TMessagesCollector& errors) const {
    NMajorClient::TTyreServiceInfoRequest request(info, since, until, Logger, errors);
    return SendRequest(Config.GetTyreServiceInfoAPIPath(), request);
}

bool TMajorClient::GetTyreStorageInfo(const TVector<NMajorClient::ETyreRegion>& regions, NMajorClient::ETyreStorageType storageType, NMajorClient::TTyreStorageInfoRequest::TInfo& info, TMessagesCollector& errors) const {
    NMajorClient::TTyreStorageInfoRequest request(regions, storageType, info, Logger, errors);
    return SendRequest(Config.GetTyreStorageInfoAPIPath(), request);
}

bool TMajorClient::GetReservedInfo(TVector<NMajorClient::TCarInfo>& info, TMessagesCollector& errors) const {
    NMajorClient::TReservedInfoRequest request(info, Logger, errors);
    return SendRequest(Config.GetReservedInfoAPIPath(), request);
}

bool TMajorClient::GetAllCarInfo(TVector<NMajorClient::TIssuedCarInfo>& info, TMessagesCollector& errors) const {
    NMajorClient::TIssuedCarInfoRequest request(info, Logger, errors);
    return SendRequest(Config.GetIssuedCarInfoAPIPath(), request);
}

bool TMajorClient::GetCarPenalties(const TString& vin, NMajorClient::TCarPenaltiesRequest::EPenaltyCheckPolicy policy, TVector<NMajorClient::TCarPenaltiesRequest::TCarPenaltyInfo>& penalties, TMessagesCollector& errors, const TMaybe<ui64> lastReceivedID) const {
    return GetCarPenalties(TVector<TString>({vin}), policy, penalties, errors, lastReceivedID);
}

bool TMajorClient::GetCarPenalties(const TVector<TString>& vins, NMajorClient::TCarPenaltiesRequest::EPenaltyCheckPolicy policy, TVector<NMajorClient::TCarPenaltiesRequest::TCarPenaltyInfo>& penalties, TMessagesCollector& errors, const TMaybe<ui64> lastReceivedID) const {
    NMajorClient::TCarPenaltiesRequest request(vins, policy, penalties, Logger, errors, lastReceivedID);
    return SendRequest(Config.GetCarPenaltiesAPIPath(), request);
}

bool TMajorClient::GetCarPenaltyDecree(const i64 externalId, NMajorClient::TCarPenaltyDecreeRequest::TCarPenaltyDecree& penaltyDecree, TMessagesCollector& errors) const {
    return GetCarPenaltyDecree(externalId, {}, penaltyDecree, errors);
}

bool TMajorClient::GetCarPenaltyDecree(const TString& rulingNumber, NMajorClient::TCarPenaltyDecreeRequest::TCarPenaltyDecree& penaltyDecree, TMessagesCollector& errors) const {
    return GetCarPenaltyDecree({}, rulingNumber, penaltyDecree, errors);
}

bool TMajorClient::GetCarPenaltyDecree(const TMaybe<i64>& externalId, const TMaybe<TString>& rulingNumber, NMajorClient::TCarPenaltyDecreeRequest::TCarPenaltyDecree& penaltyDecree, TMessagesCollector& errors) const {
    TVector<NMajorClient::TCarPenaltyDecreeRequest::TCarPenaltyDecree> penaltyDecrees;
    NMajorClient::TCarPenaltyDecreeRequest request(externalId, rulingNumber, penaltyDecrees, Logger, errors);
    if (!SendRequest(Config.GetCarPenaltyDecreeAPIPath(), request)) {
        return false;
    }
    if (penaltyDecrees.size() != 1) {
        errors.AddMessage(__LOCATION__, "Expected only 1 penalty decree but received " + ToString(penaltyDecrees.size()));
        return false;
    }
    penaltyDecree = penaltyDecrees.front();
    return true;
}

bool TMajorClient::GetLastTyreService(const TVector<TString>& vins, TVector<NMajorClient::TLastTyreServiceRequest::TInfo>& infos, TMessagesCollector& errors) const {
    NMajorClient::TLastTyreServiceRequest request(vins, infos, Logger, errors);
    return SendRequest(Config.GetLastTyreServiceAPIPath(), request);
}

bool TMajorClient::GetTyreFiles(const TString& vin, NMajorClient::ETyreFileStatus status, TVector<NMajorClient::TTyreFilesRequest::TFile>& files, TMessagesCollector& errors) const {
    NMajorClient::TTyreFilesRequest request(vin, status, files, Logger, errors);
    return SendRequest(Config.GetTyreFilesAPIPath(), request);
}
