#include "entities.h"

#include <drive/library/cpp/raw_text/datetime.h>

#include <rtline/util/json_processing.h>

#include <library/cpp/json/json_reader.h>

template <>
bool NJson::TryFromJson<NMajorClient::TTyreType>(const NJson::TJsonValue& value, NMajorClient::TTyreType& result) {
    return result.Parse(value);
}

template <>
bool NJson::TryFromJson<NMajorClient::TRimType>(const NJson::TJsonValue& value, NMajorClient::TRimType& result) {
    return result.Parse(value);
}

template <>
NJson::TJsonValue NJson::ToJson<NMajorClient::TTyreServiceInfoRequest::TRim>(const NMajorClient::TTyreServiceInfoRequest::TRim& object) {
    return object.ToJson();
}

template <>
NJson::TJsonValue NJson::ToJson<NMajorClient::TTyreServiceInfoRequest::TTyre>(const NMajorClient::TTyreServiceInfoRequest::TTyre& object) {
    return object.ToJson();
}

template <>
bool NJson::TryFromJson<NMajorClient::TTyreServiceInfoRequest::TRim>(const NJson::TJsonValue& value, NMajorClient::TTyreServiceInfoRequest::TRim& result) {
    return result.Parse(value);
}

template <>
bool NJson::TryFromJson<NMajorClient::TTyreServiceInfoRequest::TTyre>(const NJson::TJsonValue& value, NMajorClient::TTyreServiceInfoRequest::TTyre& result) {
    return result.Parse(value);
}

bool NMajorClient::TTyreServiceInfoRequest::TServiceQuery::Parse(const NJson::TJsonValue& json) {
    ui64 id;
    if (!NJson::ParseField(json["QueryID"], id, true)) {
        return false;
    }

    TInstant closeDate = TInstant::Zero();
    if (!NJson::ParseField(json["CloseDate"], closeDate)) {
        return false;
    }

    TInstant createDate = TInstant::Zero();
    if (!NJson::ParseField(json["CreateDate"], createDate, true)) {
        return false;
    }

    if (!NJson::ParseField(json["Tyres"], Tyres, true) || !NJson::ParseField(json["Rims"], Rims, true)) {
        return false;
    }

    for (auto& tyre : Tyres) {
        tyre.SetQueryId(id).SetCreateDate(createDate);
        if (closeDate != TInstant::Zero()) {
            tyre.SetCloseDate(closeDate);
        }
    }
    for (auto& rim : Rims) {
        rim.SetQueryId(id).SetCreateDate(createDate);
        if (closeDate != TInstant::Zero()) {
            rim.SetCloseDate(closeDate);
        }
    }
    return true;
}

template <>
bool NJson::TryFromJson<NMajorClient::TTyreServiceInfoRequest::TServiceQuery>(const NJson::TJsonValue& value, NMajorClient::TTyreServiceInfoRequest::TServiceQuery& result) {
    return result.Parse(value);
}

bool NMajorClient::TTyreServiceInfoRequest::TTyreServiceInfo::Parse(const NJson::TJsonValue& json) {
    NJson::TJsonValue result;
    if (!NJson::ReadJsonTree(json["Result"].GetStringRobust(), &result)) {
        return false;
    }
    return NJson::ParseField(result["Tyres"], TyreTypes, true) && NJson::ParseField(result["Rims"], RimTypes, true) && NJson::ParseField(result["Queries"], Queries, true);
}

template <>
NJson::TJsonValue NJson::ToJson<NMajorClient::TTyreStorageInfoRequest::TRim>(const NMajorClient::TTyreStorageInfoRequest::TRim& object) {
    return object.ToJson();
}

template <>
NJson::TJsonValue NJson::ToJson<NMajorClient::TTyreStorageInfoRequest::TTyre>(const NMajorClient::TTyreStorageInfoRequest::TTyre& object) {
    return object.ToJson();
}

template <>
bool NJson::TryFromJson<NMajorClient::TTyreStorageInfoRequest::TTyre>(const NJson::TJsonValue& value, NMajorClient::TTyreStorageInfoRequest::TTyre& result) {
    return result.Parse(value);
}

template <>
bool NJson::TryFromJson<NMajorClient::TTyreStorageInfoRequest::TRim>(const NJson::TJsonValue& value, NMajorClient::TTyreStorageInfoRequest::TRim& result) {
    return result.Parse(value);
}

bool NMajorClient::TTyreStorageInfoRequest::TInfo::Parse(const NJson::TJsonValue& json) {
    NJson::TJsonValue result;
    if (!NJson::ReadJsonTree(json["Result"].GetStringRobust(), &result)) {
        return false;
    }
    return NJson::ParseField(result["Tyres"], TyreTypes, true) && NJson::ParseField(result["Rims"], RimTypes, true) && NJson::ParseField(result["StorageTyre"], StorageTyre, true) && NJson::ParseField(result["StorageRim"], StorageRim, true);
}

NMajorClient::TCarPenaltiesRequest::TCarPenaltiesRequest(const TVector<TString>& vins, const EPenaltyCheckPolicy policy, TVector<TCarPenaltyInfo>& result, const TMajorLogger& logger, TMessagesCollector& errors, const TMaybe<ui64> lastReceivedID)
    : IMajorRequest(EMajorOperationType::GetCarPenalties, logger, errors, new TParseEntityList<TCarPenaltyInfo>(result, EMajorOperationType::GetCarPenalties, logger, errors))
    , Vins(vins)
    , Policy(policy)
    , LastReceivedID(lastReceivedID)
{
}

TString NMajorClient::TCarPenaltiesRequest::CreateContent() const {
    NJson::TJsonValue data(NJson::JSON_MAP);
    data["VINSet"] = JoinSeq(",", Vins);

    NJson::TJsonValue isActive;
    if (Policy == EPenaltyCheckPolicy::NotPaid) {
        isActive = true;
    } else if (Policy == EPenaltyCheckPolicy::Paid) {
        isActive = false;
    } else {
        isActive = NJson::JSON_NULL;
    }
    data["IsActive"] = isActive;
    if (LastReceivedID.Defined()) {
        data["LastReceivedID"] = *LastReceivedID;
    }

    return data.GetStringRobust();
}

bool NMajorClient::TCarPenaltiesRequest::ProcessSuccess(const NJson::TJsonValue& data) {
    if (data.IsNull()) {
        return true;
    }
    return TBase::ProcessSuccess(data);
}

const TString NMajorClient::TCarPenaltiesRequest::TCarPenaltyInfo::DefaultTimezoneName = "Europe/Moscow";

inline static bool ReadLocalTimeInstant(const NJson::TJsonValue& data, const TString& field, TInstant& result) {
    JREAD_INSTANT_ISOFORMAT_NULLABLE_OPT(data, field, result);
    if (!!result && !NUtil::ApplyTimeZone(result, NMajorClient::TCarPenaltiesRequest::TCarPenaltyInfo::DefaultTimezoneName, result)) {
        return false;
    }
    return true;
}

bool NMajorClient::TCarPenaltiesRequest::TCarPenaltyInfo::Parse(const NJson::TJsonValue& data) {
    JREAD_INT_OPT(data, "ID", Id);

    JREAD_STRING(data, "VIN", Vin);
    JREAD_STRING_OPT(data, "BrandName", BrandName);
    JREAD_STRING_OPT(data, "ModelName", ModelName);

    JREAD_STRING_OPT(data, "PenaltyIssuanceName", PenaltyIssuanceName);

    if (!ReadLocalTimeInstant(data, "CheckDate", CheckDate)) {  // parsed as a local timestamp
        return false;
    }

    JREAD_INSTANT_ISOFORMAT_NULLABLE_OPT(data, "DocDate", DocDate);  // parsed as a date
    JREAD_STRING_OPT(data, "DocNo", DocNo);

    if (!ReadLocalTimeInstant(data, "ViolationAt", ViolationAt)) {  // parsed as a local timestamp
        return false;
    }

    JREAD_DOUBLE_OPT(data, "SummFull", SumToPayWithoutDiscount);
    JREAD_INSTANT_ISOFORMAT_NULLABLE_OPT(data, "PayTo", PayUntil);  // parsed as a date

    JREAD_BOOL_OPT(data, "IsDiscount", HasDiscount);
    JREAD_DOUBLE_OPT(data, "Summ", SumToPay);
    JREAD_INSTANT_ISOFORMAT_NULLABLE_OPT(data, "DiscountTo", DiscountUntil);  // parsed as a date

    JREAD_BOOL_OPT(data, "IsCompens", HasCompensation);
    JREAD_COMMON_NULLABLE_OPT(data, "IsMajorPayed", IsMajorPayed, Boolean);  // however it's always null
    JREAD_COMMON_NULLABLE_OPT(data, "IsDecreeExists", IsDecreeExists, Boolean);
    JREAD_COMMON_NULLABLE_OPT(data, "IsWillBeBilled", WillBeIncludedToBill, Boolean);

    JREAD_STRING_NULLABLE_OPT(data, "Requisites", RequisitesRaw);
    return Requisites.ParseFromString(RequisitesRaw);
}

TInstant NMajorClient::TCarPenaltiesRequest::TCarPenaltyInfo::GetRulingDate() const {
    return (!!GetDocDate()) ? GetDocDate() : GetRequisites().GetDateField(TRequisites::ERequisiteFieldName::RulingDate);
}

TString NMajorClient::TCarPenaltiesRequest::TCarPenaltyInfo::GetRulingNumber() const {
    return (!!GetDocNo()) ? GetDocNo() : GetRequisites().GetField(TRequisites::ERequisiteFieldName::RulingNumber);
}

TInstant NMajorClient::TCarPenaltiesRequest::TCarPenaltyInfo::GetViolationTime() const {
    return (!!GetViolationAt()) ? GetViolationAt() : GetRequisites().GetDateTimeField(TRequisites::ERequisiteFieldName::ViolationTime);
}

bool NMajorClient::TCarPenaltiesRequest::TCarPenaltyInfo::TRequisites::ParseFromString(const TString& data) {
    for (TStringBuf line : StringSplitter(data).SplitBySet("\r\n").SkipEmpty()) {
        TVector<TString> values = StringSplitter(line).SplitByString(": ");
        if (values.size() == 2) {
            Data[values[0]] = values[1];
        }
    }
    return true;
}

TString NMajorClient::TCarPenaltiesRequest::TCarPenaltyInfo::TRequisites::GetField(ERequisiteFieldName name, const TString& defaultValue) const {
    return Data.Value(::ToString(name), defaultValue);
}

TInstant NMajorClient::TCarPenaltiesRequest::TCarPenaltyInfo::TRequisites::GetDateField(ERequisiteFieldName name, const TString& format, const TInstant& defaultValue) const {
    return NUtil::ParseFomattedLocalDatetime(GetField(name), format, defaultValue);
}

TInstant NMajorClient::TCarPenaltiesRequest::TCarPenaltyInfo::TRequisites::GetDateTimeField(ERequisiteFieldName name, const TString& format, const TString& tzName, const TInstant& defaultValue) const {
    return NUtil::ParseFomattedLocalDatetime(GetField(name), format, tzName, defaultValue);
}

bool NMajorClient::TCarPenaltyDecreeRequest::TCarPenaltyDecree::Parse(const NJson::TJsonValue& data) {
    if (!TJsonProcessor::Read(data, "FileName", FileName)) {
        return false;
    }
    if (!TJsonProcessor::Read(data, "FileExt", FileExtension)) {
        return false;
    }
    if (!data["File"].IsString()) {
        return false;
    }
    Content = Base64Decode(data["File"].GetString());
    return true;
}

NMajorClient::TCarPenaltyDecreeRequest::TCarPenaltyDecreeRequest(const TMaybe<i64>& externalId, const TMaybe<TString>& rulingNumber, TVector<TCarPenaltyDecree>& result, const TMajorLogger& logger, TMessagesCollector& errors)
    : IMajorRequest(EMajorOperationType::GetCarPenaltyDecree, logger, errors, new TParseEntityList<TCarPenaltyDecree>(result, EMajorOperationType::GetCarPenaltyDecree, logger, errors))
    , ExternalId(externalId)
    , RulingNumber(rulingNumber)
{
    Y_ENSURE_BT(ExternalId.Defined() || RulingNumber.Defined());
}

TString NMajorClient::TCarPenaltyDecreeRequest::CreateContent() const {
    NJson::TJsonValue data(NJson::JSON_MAP);

    if (!!ExternalId) {
        data["ID"] = *ExternalId;
    } else if (!!RulingNumber) {
        data["DocNo"] = *RulingNumber;
    } else {
        Y_UNREACHABLE();
    }

    return data.GetStringRobust();
}
