#pragma once
#include "config.h"
#include "logger.h"

#include <drive/backend/maintenance/entity.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_writer.h>
#include <library/cpp/string_utils/base64/base64.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/parse.h>
#include <rtline/util/network/http_request.h>
#include <rtline/util/types/accessor.h>

#include <util/datetime/base.h>
#include <util/generic/map.h>
#include <util/string/vector.h>

namespace NMajorClient {
    enum EWorkType {
        None = 0 /* "none" */,
        UpdateSTS = 1 /* "update_sts" */,
        Diagnostics = 266 /* "diagnostics" */,
        MechanicRepair = 267 /* "mechanic_repair" */,
        Maintenance = 268 /* "maintenance" */,
        BodyRepair = 271 /* "body_repair" */,
    };

    class IMajorParser {
    public:
        IMajorParser(EMajorOperationType type, const TMajorLogger& logger, TMessagesCollector& errors, const bool parsingErrorsAvailable = false)
            : Type(type)
            , Logger(logger)
            , Errors(errors)
            , ParsingErrorsAvailable(parsingErrorsAvailable)
        {}

        virtual bool Parse(const NJson::TJsonValue::TArray& data) = 0;

        virtual ~IMajorParser() {}

    protected:
        TMessagesCollector& MutableErrors() {
            return Errors;
        }

        template <class TEntity>
        bool ParseJsonArray(const NJson::TJsonValue::TArray& data, TVector<TEntity>& result) {
            result.clear();
            if (data.size() == 0) {
                return true;
            }
            if (data.size() != 1 || !data.front().IsArray()) {
                Logger.ProcessError(Type, "Incorrect json format", Errors);
                return false;
            }
            for (const auto& json : data.front().GetArray()) {
                TEntity entity;
                if (!entity.Parse(json)) {
                    Logger.ProcessError(Type, "Incorrect answer format " + json.GetStringRobust(), Errors);
                    if (!ParsingErrorsAvailable) {
                        return false;
                    }
                } else {
                    result.push_back(entity);
                }
            }
            return true;
        }

    private:
        EMajorOperationType Type;
        const TMajorLogger& Logger;
        TMessagesCollector& Errors;
        R_READONLY(bool, ParsingErrorsAvailable, false);
    };

    template <class TEntity>
    class TParseEntity : public IMajorParser {
    public:
        TParseEntity(TEntity& result, EMajorOperationType type, const TMajorLogger& logger, TMessagesCollector& errors)
            : IMajorParser(type, logger, errors)
            , Result(result)
        {}

        virtual bool Parse(const NJson::TJsonValue::TArray& data) override {
            TVector<TEntity> resultArray;
            if (!ParseJsonArray(data, resultArray)) {
                return false;
            }
            if (resultArray.size() != 1) {
                MutableErrors().AddMessage("parser", "Incorrect answer format");
                return false;
            }
            Result = resultArray.front();
            return true;
        }

    private:
        TEntity& Result;
    };

    template <class TEntity>
    class TParseEntityList : public IMajorParser {
    public:
        TParseEntityList(TVector<TEntity>& result, EMajorOperationType type, const TMajorLogger& logger, TMessagesCollector& errors, const bool parsingErrorsAvailable = false)
            : IMajorParser(type, logger, errors, parsingErrorsAvailable)
            , Result(result)
        {}

        virtual bool Parse(const NJson::TJsonValue::TArray& data) override {
            return ParseJsonArray(data, Result);
        }

    private:
        TVector<TEntity>& Result;
    };

    class IMajorRequest {
    public:
        IMajorRequest(EMajorOperationType type, const TMajorLogger& logger, TMessagesCollector& errors, TAtomicSharedPtr<IMajorParser> parser = nullptr)
            : Type(type)
            , Logger(logger)
            , Errors(errors)
            , Parser(parser)
        {
            Logger.ProcessStart(Type);
        }

        virtual ~IMajorRequest() {}

        virtual TString CreateContent() const = 0;

        bool ProcessReply(const NUtil::THttpReply& reply) {
            Logger.ProcessReply(Type, reply.Code());
            if (reply.IsSuccessReply()) {
                DEBUG_LOG << reply.Content() << Endl;
                NJson::TJsonValue json;
                if (!NJson::ReadJsonFastTree(reply.Content(), &json)) {
                    ProcessError("Incorrect reply format");
                    return false;
                }
                if (json.Has("error") && json["error"] != NJson::JSON_NULL) {
                    ProcessError("Internal error: " + json["error"]["message"].GetString() + " requestKey: " + json["requestKey"].GetStringRobust());
                    return false;
                }
                return ProcessSuccess(json["data"]);
            }
            ProcessError(reply);
            return false;
        }

        const TMessagesCollector& GetErrors() const {
            return Errors;
        }

        TMessagesCollector& MutableErrors() {
            return Errors;
        }

    private:
        virtual void ProcessError(const NUtil::THttpReply& reply) {
            ProcessError(ToString(reply.Code()) + " " + reply.ErrorMessage());
        }

    protected:
        virtual bool ProcessSuccess(const NJson::TJsonValue& data) {
            if (!data.IsArray()) {
                ProcessError("Incorrect json format");
                return false;
            }
            if (!!Parser) {
                return Parser->Parse(data.GetArraySafe());
            }
            return true;
        }

        void ProcessError(const TString& message) {
            Logger.ProcessError(Type, message, Errors);
        }

    private:
        EMajorOperationType Type;
        const TMajorLogger& Logger;
        TMessagesCollector& Errors;
        TAtomicSharedPtr<IMajorParser> Parser;
    };

    class IByIdMajorRequest : public IMajorRequest {
    public:
        IByIdMajorRequest(const TString& id, EMajorOperationType type, const TMajorLogger& logger, TMessagesCollector& errors, TAtomicSharedPtr<IMajorParser> parser = nullptr)
            : IMajorRequest(type, logger, errors, parser)
            , Id(id)
        {}

        virtual TString CreateContent() const override {
            NJson::TJsonValue json(NJson::JSON_MAP);
            json["QueryID"] = Id;
            return json.GetStringRobust();
        }

    private:
        TString Id;
    };

    class TCancelRequest : public IByIdMajorRequest {
    public:
        TCancelRequest(const TString& id, const TMajorLogger& logger, TMessagesCollector& errors)
            : IByIdMajorRequest(id, EMajorOperationType::CancelQuery, logger, errors)
        {}
    };

    class TOSAGORequest : public IMajorRequest {
    public:
        class TOSAGODocument {
            R_READONLY(TString, File);
            R_READONLY(TInstant, DateFrom, TInstant::Zero());
            R_READONLY(TInstant, DateTo, TInstant::Zero());
            R_READONLY(TString, Number);

        public:
            bool Parse(const NJson::TJsonValue& json) {
                File = Base64Decode(json["File"].GetStringRobust());
                Number = json["DocNo"].GetString();
                return TInstant::TryParseIso8601(json["DateFrom"].GetString(), DateFrom) && TInstant::TryParseIso8601(json["DateTo"].GetString(), DateTo);
            }
        };

    public:
        TOSAGORequest(const TString& vin, const bool isCurrent, TOSAGODocument& result, const TMajorLogger& logger, TMessagesCollector& errors)
            : IMajorRequest(EMajorOperationType::RequestOSAGO, logger, errors, new TParseEntity<TOSAGODocument>(result, EMajorOperationType::RequestOSAGO, logger, errors))
            , Vin(vin)
            , IsCurrent(isCurrent)
        {}

        virtual TString CreateContent() const override {
            NJson::TJsonValue json(NJson::JSON_MAP);
            json["Vin"] = Vin;
            json["IsCurrent"] = IsCurrent;
            return json.GetStringRobust();
        }

    private:
        TString Vin;
        bool IsCurrent;
    };

    class TMaintenanceInfoRequest: public IMajorRequest {
    public:
        TMaintenanceInfoRequest(TVector<::TMaintenanceInfo>& result, const TMajorLogger& logger, TMessagesCollector& errors)
            : IMajorRequest(EMajorOperationType::GetMaintenanceInfo, logger, errors, new TParseEntityList<::TMaintenanceInfo>(result, EMajorOperationType::GetMaintenanceInfo, logger, errors, true))
        {
        }

        virtual TString CreateContent() const override {
            return "";
        }
    };

    class TTyreType {
        R_READONLY(ui64, Id, 0);
        R_READONLY(TString, Brand);
        R_READONLY(TString, Model);
        R_READONLY(TString, Season);
        R_READONLY(TString, Size);

    public:
        bool Parse(const NJson::TJsonValue& json) {
            return NJson::ParseField(json["TyreID"], Id, true) && NJson::ParseField(json["Brand"], Brand, true)
                && NJson::ParseField(json["Model"], Model, true) && NJson::ParseField(json["Season"], Season, true) && NJson::ParseField(json["Size"], Size, true);
        }

        NJson::TJsonValue ToJson() const {
            return NJson::TMapBuilder("id", Id)("brand", Brand)("model", Model)("season", Season)("size", Size);
        }
    };

    class TRimType {
        R_READONLY(ui64, Id, 0);
        R_READONLY(TString, Type);
        R_READONLY(TString, Model);
        R_READONLY(ui8, Diameter, 0);

    public:
        bool Parse(const NJson::TJsonValue& json) {
            return NJson::ParseField(json["RimID"], Id, true) && NJson::ParseField(json["Type"], Type, true)
                && NJson::ParseField(json["Model"], Model, true) && NJson::ParseField(json["Diameter"], Diameter, true);
        }

        NJson::TJsonValue ToJson() const {
            return NJson::TMapBuilder("id", Id)("type", Type)("model", Model)("diameter", Diameter);
        }
    };

    class TTyreServiceInfoRequest : public IMajorRequest {
    public:
        class ITyreServiceEntity {
            R_FIELD(ui64, QueryId, 0);
            R_FIELD(TInstant, CreateDate, TInstant::Zero());
            R_OPTIONAL(TInstant, CloseDate);
            R_READONLY(TString, TagId);
            R_READONLY(TString, Status);
            R_READONLY(ui64, TypeId, 0);

        public:
            virtual ~ITyreServiceEntity() = default;
            bool Parse(const NJson::TJsonValue& json) {
                return NJson::ParseField(json["TagID"], TagId, true) && NJson::ParseField(json["Status"], Status, true) && DoParse(json);
            }

            NJson::TJsonValue ToJson() const {
                NJson::TJsonValue result(NJson::TMapBuilder("query_id", QueryId)("create_date", CreateDate.Seconds())("tag_id", TagId)("status", Status)("type_id", TypeId));
                if (CloseDate) {
                    result.InsertValue("close_date", CloseDate->Seconds());
                }
                return result;
            }

        private:
            virtual bool DoParse(const NJson::TJsonValue& json) = 0;
        };

        class TTyre : public ITyreServiceEntity {
            virtual bool DoParse(const NJson::TJsonValue& json) override {
                return NJson::ParseField(json["TyreID"], MutableProtectedTypeId(), true);
            }
        };

        using TTyres = TVector<TTyre>;

        class TRim : public ITyreServiceEntity {
            virtual bool DoParse(const NJson::TJsonValue& json) override {
                return NJson::ParseField(json["RimID"], MutableProtectedTypeId(), true);
            }
        };

        using TRims = TVector<TRim>;

        class TServiceQuery {
            R_READONLY(TTyres, Tyres);
            R_READONLY(TRims, Rims);

        public:
            bool Parse(const NJson::TJsonValue& json);
        };

        class TTyreServiceInfo {
            R_READONLY(TVector<TServiceQuery>, Queries);
            R_READONLY(TVector<TTyreType>, TyreTypes);
            R_READONLY(TVector<TRimType>, RimTypes);

        public:
            bool Parse(const NJson::TJsonValue& json);
        };

    public:
        TTyreServiceInfoRequest(TTyreServiceInfo& result, const TInstant since, const TInstant until, const TMajorLogger& logger, TMessagesCollector& errors)
            : IMajorRequest(EMajorOperationType::GetTyreServiceInfo, logger, errors, new TParseEntity<TTyreServiceInfo>(result, EMajorOperationType::GetTyreServiceInfo, logger, errors))
            , Since(since)
            , Until(until)
        {}

        virtual TString CreateContent() const override {
            NJson::TJsonValue json;
            if (Since != TInstant::Zero()) {
                json.InsertValue("DateFrom", Since.FormatLocalTime("%Y-%m-%d"));
            }
            if (Until != TInstant::Max()) {
                json.InsertValue("DateTo", Until.FormatLocalTime("%Y-%m-%d"));
            }
            return json.GetStringRobust();
        }

    private:
        const TInstant Since;
        const TInstant Until;
    };

    enum class ETyreRegion {
        UNDEFINED,
        KZN,
        MSK,
        SPB,
    };
    enum class ETyreStorageType {
        Service,
        Season,
    };

    class TTyreStorageInfoRequest : public IMajorRequest {
    public:
        class IStorageEntity {
            R_READONLY(ui64, Count, 0);
            R_READONLY(ETyreRegion, Region, ETyreRegion::UNDEFINED);
            R_READONLY(TString, Status);
            R_READONLY(ui64, Id, 0);

        public:
            virtual ~IStorageEntity() = default;
            bool Parse(const NJson::TJsonValue& json) {
                TString regionStr;
                return NJson::ParseField(json["Region"], regionStr, false) && TryFromString(regionStr, Region) && NJson::ParseField(json["Status"], Status, true) && NJson::ParseField(json["Count"], Count, true) && DoParse(json);
            }

            NJson::TJsonValue ToJson() const {
                NJson::TJsonValue result(NJson::TMapBuilder("region", ToString(Region))("status", Status)("count", Count)("id", Id));
                return result;
            }

        private:
            virtual bool DoParse(const NJson::TJsonValue& json) = 0;
        };

        class TTyre : public IStorageEntity {
            virtual bool DoParse(const NJson::TJsonValue& json) override {
                return NJson::ParseField(json["TyreID"], MutableProtectedId(), true);
            }
        };

        using TTyres = TVector<TTyre>;

        class TRim : public IStorageEntity {
            virtual bool DoParse(const NJson::TJsonValue& json) override {
                return NJson::ParseField(json["RimID"], MutableProtectedId(), true);
            }
        };

        using TRims = TVector<TRim>;

        class TInfo {
            R_READONLY(TVector<TTyre>, StorageTyre);
            R_READONLY(TVector<TRim>, StorageRim);
            R_READONLY(TVector<TTyreType>, TyreTypes);
            R_READONLY(TVector<TRimType>, RimTypes);

        public:
            bool Parse(const NJson::TJsonValue& json);
        };

    public:
        TTyreStorageInfoRequest(const TVector<ETyreRegion>& regions, ETyreStorageType storageType, TInfo& result, const TMajorLogger& logger, TMessagesCollector& errors)
            : IMajorRequest(EMajorOperationType::GetTyreStorageInfo, logger, errors, new TParseEntity<TInfo>(result, EMajorOperationType::GetTyreStorageInfo, logger, errors))
            , Regions(regions)
            , StorageType(storageType)
        {}

        virtual TString CreateContent() const override {
            NJson::TJsonValue json;
            TString regionsString;
            for (const auto& region : Regions) {
                if (!regionsString.empty()) {
                    regionsString += ",";
                }
                regionsString += ToString(region);
            }

            json.InsertValue("Region", regionsString);
            json.InsertValue("Category", ToString(StorageType));
            return json.GetStringRobust();
        }

    private:
        const TVector<ETyreRegion> Regions;
        const ETyreStorageType StorageType;
    };

    class TLastTyreServiceRequest : public IMajorRequest {
    public:
        class TInfo {
            R_READONLY(TString, Vin);
            R_READONLY(TInstant, Date, TInstant::Zero());

        public:
            bool Parse(const NJson::TJsonValue& json) {
                return NJson::ParseField(json, "VIN", Vin, true) && NJson::ParseField(json, "DoneDate", Date, false);
            }
        };

    public:
        TLastTyreServiceRequest(const TVector<TString>& vins, TVector<TInfo>& result, const TMajorLogger& logger, TMessagesCollector& errors)
            : IMajorRequest(EMajorOperationType::GetLastTyreServiceInfo, logger, errors, new TParseEntityList<TInfo>(result, EMajorOperationType::GetLastTyreServiceInfo, logger, errors))
            , Vins(vins)
        {}

        virtual TString CreateContent() const override {
            NJson::TJsonValue json;
            json.InsertValue("Vin", JoinSeq(",", Vins));
            return json.GetStringRobust();
        }

    private:
        const TVector<TString> Vins;
    };

    enum ETyreFileStatus {
        Undefined /* "undefined" */,
        New /* "new" */,
        Old /* "old" */,
    };

    class TTyreFilesRequest : public IMajorRequest {
    public:
        class TFile {
            R_READONLY(TString, Name);
            R_READONLY(TInstant, Date, TInstant::Zero());
            R_READONLY(ETyreFileStatus, Status, ETyreFileStatus::Undefined);
            R_READONLY(TString, FileBase64);

        public:
            bool Parse(const NJson::TJsonValue& json) {
                TString strStatus;
                return NJson::ParseField(json, "FileName", Name, true) && NJson::ParseField(json, "CreateDate", Date, true) && NJson::ParseField(json, "Status", strStatus, true) && TryFromString(strStatus, Status) && NJson::ParseField(json, "File", FileBase64, true);
            }

            NJson::TJsonValue ToJson() const {
                return NJson::TMapBuilder("name", Name)("timestamp", Date.Seconds())("status", ToString(Status))("file", FileBase64);
            }
        };

    public:
        TTyreFilesRequest(const TString& vin, const ETyreFileStatus status, TVector<TFile>& result, const TMajorLogger& logger, TMessagesCollector& errors)
            : IMajorRequest(EMajorOperationType::GetTyreFiles, logger, errors, new TParseEntityList<TFile>(result, EMajorOperationType::GetTyreFiles, logger, errors))
            , Vin(vin)
            , Status(status)
        {}

        virtual TString CreateContent() const override {
            NJson::TJsonValue json;
            if (Status != ETyreFileStatus::Undefined) {
                json.InsertValue("Status", ToString(Status));
            }
            json.InsertValue("Vin", Vin);
            return json.GetStringRobust();
        }

    private:
        const TString Vin;
        const ETyreFileStatus Status;
    };

    class TCarInfo {
    public:
        class TConfiguration {
            R_READONLY(TString, Brand);
            R_READONLY(TString, Model);
            R_READONLY(TString, IntegrationName);
            R_READONLY(TString, ModificationName);
            R_READONLY(TString, Color);
            R_READONLY(TString, EtalonColor);
            R_READONLY(TString, EngineTypeName);
            R_READONLY(TString, EngineTypeCode);

        public:
            bool Parse(const NJson::TJsonValue& json) {
                return ParseImpl(json, true);
            }

            bool SimpleParse(const NJson::TJsonValue& json) {
                return ParseImpl(json, false);
            }

            bool Like(const TConfiguration& configuration) const {
                if (Brand != configuration.GetBrand() || Model != configuration.GetModel()) {
                    return false;
                }
                return SimpleLike(configuration);
            }

            bool SimpleLike(const TConfiguration& configuration) const {
                if (IntegrationName && IntegrationName != configuration.GetIntegrationName()) {
                    return false;
                }
                if (ModificationName && ModificationName != configuration.GetModificationName()) {
                    return false;
                }
                if (Color && Color != configuration.GetColor()) {
                    return false;
                }
                if (EtalonColor && EtalonColor != configuration.GetEtalonColor()) {
                    return false;
                }
                if (EtalonColor && EtalonColor != configuration.GetEtalonColor()) {
                    return false;
                }
                if (EngineTypeCode && EngineTypeCode != configuration.GetEngineTypeCode()) {
                    return false;
                }
                if (EngineTypeName && EngineTypeName != configuration.GetEngineTypeName()) {
                    return false;
                }
                return true;
            }

            NJson::TJsonValue ToJson() const {
                return NJson::TMapBuilder("brand", Brand)("model", Model)("integration_name", IntegrationName)("modification_name", ModificationName)("color", Color)("etalon_color", EtalonColor)("engine_type_code", EngineTypeCode)("engine_type_name", EngineTypeName);
            }
        private:
            bool ParseImpl(const NJson::TJsonValue& json, bool required) {
                return NJson::ParseField(json["BrandName"], Brand, required)
                    && NJson::ParseField(json["ModelName"], Model, required)
                    && NJson::ParseField(json["IntegrationName"], IntegrationName)
                    && NJson::ParseField(json["ModificationName"], ModificationName, required)
                    && NJson::ParseField(json["ColorName"], Color)
                    && NJson::ParseField(json["EtalonColorName"], EtalonColor)
                    && NJson::ParseField(json["EngineTypeCode"], EngineTypeCode)
                    && NJson::ParseField(json["EngineTypeName"], EngineTypeName);
            }
        };

    private:
        R_READONLY(TString, Vin);
        R_READONLY(TInstant, DeliveryDate, TInstant::Zero());
        R_READONLY(TInstant, DateSale, TInstant::Zero());
        R_READONLY(TInstant, DeliveryDateEstimate, TInstant::Zero());
        R_READONLY(TString, Number);
        R_READONLY(TConfiguration, Configuration);

    public:
        bool Parse(const NJson::TJsonValue& json) {
            return NJson::ParseField(json["VIN"], Vin, true)
                && NJson::ParseField(json["RegNo"], Number)
                && NJson::ParseField(json["DeliveryDate"], DeliveryDate)
                && NJson::ParseField(json["DeliveryDateEstimate"], DeliveryDateEstimate)
                && NJson::ParseField(json["DateSale"], DateSale)
                && Configuration.Parse(json);
        }
    };

    class TIssuedCarInfo : public TCarInfo {
        R_READONLY(bool, IsIssued, true);

    public:
        bool Parse(const NJson::TJsonValue& json) {
            return NJson::ParseField(json["IsIssued"], IsIssued, true) && TCarInfo::Parse(json);
        }
    };

    class TReservedInfoRequest : public IMajorRequest {
    public:
        TReservedInfoRequest(TVector<TCarInfo>& result, const TMajorLogger& logger, TMessagesCollector& errors)
            : IMajorRequest(EMajorOperationType::GetReservedInfo, logger, errors, new TParseEntityList<TCarInfo>(result, EMajorOperationType::GetReservedInfo, logger, errors))
        {}

        virtual TString CreateContent() const override {
            return "{}";
        }
    };

    class TIssuedCarInfoRequest : public IMajorRequest {
    public:
        TIssuedCarInfoRequest(TVector<TIssuedCarInfo>& result, const TMajorLogger& logger, TMessagesCollector& errors)
            : IMajorRequest(EMajorOperationType::GetIssuedCarsInfo,logger, errors, new TParseEntityList<TIssuedCarInfo>(result, EMajorOperationType::GetIssuedCarsInfo, logger, errors))
        {}

        virtual TString CreateContent() const override {
            return "{}";
        }
    };

    class TQueryInfoRequest: public IByIdMajorRequest {
    public:
        class TInfo {
            R_READONLY(TString, Name);
            R_READONLY(TString, Code);
            R_READONLY(TString, Type);
            R_READONLY(TString, FullValue);
            R_READONLY(TString, Value);

        public:
            bool Parse(const NJson::TJsonValue& json) {
                Name = json["Name"].GetString();
                Code = json["Code"].GetString();
                Type = json["TypeOIDCode"].GetString();
                FullValue = json["ValueView"].GetString();
                Value = json["Value"].GetStringRobust();
                return true;
            }

            NJson::TJsonValue GetJsonReport() const {
                NJson::TJsonValue result;
                result["name"] = Name;
                result["value"] = FullValue;
                result["internal_value"] = Value;
                return result;
            }
        };

        class TFullQueryInfo {
            R_FIELD(TVector<TInfo>, Parameters, {});

        public:
            template <class TValue>
            bool TryGetValue(const TString& key, TValue& value) const {
                for (const auto& param : Parameters) {
                    if (param.GetName() == key) {
                        return TryFromString(param.GetValue(), value);
                    }
                }
                return false;
            }

            bool TryGetInstant(const TString& key, TInstant& value) const {
                for (const auto& param : Parameters) {
                    if (param.GetName() == key) {
                        return TInstant::TryParseIso8601(param.GetValue(), value);
                    }
                }
                return false;
            }

            NJson::TJsonValue GetJsonReport() const {
                NJson::TJsonValue result;
                for (const auto& param : Parameters) {
                    result.AppendValue(param.GetJsonReport());
                }
                return result;
            }
        };

    public:
        TQueryInfoRequest(const TString& id, TFullQueryInfo& result, const TMajorLogger& logger, TMessagesCollector& errors)
            : IByIdMajorRequest(id, EMajorOperationType::GetInfo, logger, errors, new TParseEntityList<TInfo>(result.MutableParameters(), EMajorOperationType::GetInfo, logger, errors))
        {}
    };

    class THistoryInfoRequest : public IByIdMajorRequest {
    public:
        class TAction {
            R_READONLY(TInstant, Date, TInstant::Zero());
            R_READONLY(TString, Status);
            R_READONLY(TString, Note);
            R_READONLY(bool, Closed, false);
            R_READONLY(TDuration, Duration, TDuration::Zero());

        public:
            bool Parse(const NJson::TJsonValue& json) {
                Status = json["StatusOID_Name"].GetString();
                Note = json["Note"].GetString();
                Closed = json["IsStatusClosed"].GetBoolean();
                Duration = TDuration::Minutes(json["MinutesStay"].GetUInteger());
                return TInstant::TryParseIso8601(json["Date"].GetString(), Date);
            }

            NJson::TJsonValue GetJsonReport() const {
                NJson::TJsonValue result;
                result["status"] = Status;
                result["note"] = Note;
                result["is_closed"] = Closed;
                result["timestamp"] = Date.Seconds();
                result["duration"] = Duration.Minutes();
                return result;
            }
        };

        class TInfoWithHistory {
            R_FIELD(TVector<TAction>, History, {});

        public:
            NJson::TJsonValue GetJsonReport() const {
                NJson::TJsonValue result;
                result.InsertValue("current_state", NJson::JSON_UNDEFINED);
                result.InsertValue("history", NJson::JSON_ARRAY);
                if (!History.empty()) {
                    result["current_state"] = GetCurrentState().GetJsonReport();
                }
                for (const auto& action : History) {
                    result["history"].AppendValue(action.GetJsonReport());
                }
                return result;
            }

            const TAction& GetCurrentState() const {
                return History.back();
            }
        };

    public:
        THistoryInfoRequest(const TString& id, TInfoWithHistory& result, const TMajorLogger& logger, TMessagesCollector& errors)
            : IByIdMajorRequest(id, EMajorOperationType::GetInfo, logger, errors, new TParseEntityList<TAction>(result.MutableHistory(), EMajorOperationType::GetInfo, logger, errors))
        {}
    };

    class ICommonCreateMajorRequest : public IMajorRequest {
    public:
        class TResponse {
            R_READONLY(TString, GlobalId);
            R_READONLY(TString, Id);

        public:
            bool Parse(const NJson::TJsonValue& json) {
                Id = json["ID"].GetStringRobust();
                GlobalId = json["OID"].GetStringRobust();
                return !!Id && !!GlobalId;
            }
        };

    public:
        ICommonCreateMajorRequest(TResponse& response, EMajorOperationType type, const TMajorLogger& logger, TMessagesCollector& errors)
            : IMajorRequest(type, logger, errors, new TParseEntity<TResponse>(response, type, logger, errors))
        {}
    };

    class TCreateQueryRequest : public ICommonCreateMajorRequest {
    public:
        class TQuery {
        public:
            TQuery(const TString& vin, const EWorkType type)
                : VIN(vin)
                , WorkType(type)
            {}

            NJson::TJsonValue SerializeToJson() const {
                NJson::TJsonValue json;
                json["TypeQueryOID"] = QueryId;
                json["VIN"] = VIN;
                json["WorkType"] = WorkType;
                json["ClientContactFIO"] = ContactFIO;
                json["ClientContactPhone"] = ContactPhone;
                if (Time != TInstant::Zero()) {
                    json["ServiceDate"] = Time.ToStringUpToSeconds();
                }
                json["RepairWork"] = Comment;
                json["CommonCarRun"] = CommonCarRun;
                if (!!Longitude && !!Latitude) {
                    json["Longitude"] = Longitude;
                    json["Latitude"] = Latitude;
                }
                json["QueryLocalID"] = LocalID;
                if (!InsuranceNumber.empty()) {
                    json["InsuranceNumber"] = InsuranceNumber;
                }
                return json;
            }

            TQuery& SetCoord(double longitude, double latitude) {
                Longitude = longitude;
                Latitude = latitude;
                return *this;
            }

        private:
            R_FIELD(ui64, QueryId, 0);
            const TString VIN;
            const EWorkType WorkType;
            R_FIELD(TString, ContactFIO);
            R_FIELD(TString, ContactPhone);
            R_FIELD(TInstant, Time, TInstant::Zero());
            R_FIELD(TString, Comment);
            R_FIELD(ui64, CommonCarRun, 0);
            R_FIELD(TString, LocalID);
            R_FIELD(TString, InsuranceNumber);
            double Longitude = 0;
            double Latitude = 0;
        };

    public:
        TCreateQueryRequest(const TQuery& query, ICommonCreateMajorRequest::TResponse& response, const TMajorClientConfig& config, const TMajorLogger& logger, TMessagesCollector& errors)
            : ICommonCreateMajorRequest(response, EMajorOperationType::CreateQuery, logger, errors)
            , Query(query)
        {
            if (!Query.GetContactFIO()) {
                Query.SetContactFIO(config.GetContactFIO());
            }
            if (!Query.GetContactPhone()) {
                Query.SetContactPhone(config.GetContactPhone());
            }
            if (!Query.GetQueryId()) {
                Query.SetQueryId(config.GetQueryType());
            }
        }

        virtual TString CreateContent() const override {
            return Query.SerializeToJson().GetStringRobust();
        }

    private:
        TQuery Query;
    };

    class TSTSRequest : public ICommonCreateMajorRequest {
    public:
        R_READONLY(TString, Vin);
        R_READONLY(ui64, TypeId, 0);

    public:
        TSTSRequest(const TString& vin, ICommonCreateMajorRequest::TResponse& response, const TMajorClientConfig& config, const TMajorLogger& logger, TMessagesCollector& errors)
            : ICommonCreateMajorRequest(response, EMajorOperationType::CreateQuery, logger, errors)
            , Vin(vin)
            , TypeId(config.GetSTSType())
        {}

        virtual TString CreateContent() const override {
            NJson::TJsonValue data;
            data.InsertValue("ParamOID.Value", Vin);
            data.InsertValue("ParamOID.Code", "MainStaticSIOCarInBaseOID");

            NJson::TJsonValue content;
            content["TypeQueryOID"] = TypeId;
            content["Data"][0]["Root"][0] = data;
            return content.GetStringRobust();
        }
    };

    class TTypeQueryRequest : public IMajorRequest {
    public:
        class TType {
            R_READONLY(ui64, Id, 0);
            R_READONLY(TString, Name);

        public:
            bool Parse(const NJson::TJsonValue& json) {
                Id = json["TypeQueryOID"].GetUInteger();
                Name = json["TypeQueryName"].GetString();
                return !!Id && !!Name;
            }
        };

    public:
        TTypeQueryRequest(TVector<TType>& result, const TMajorLogger& logger, TMessagesCollector& errors)
            : IMajorRequest(EMajorOperationType::GetTypeRequest, logger, errors, new TParseEntityList<TType>(result, EMajorOperationType::GetTypeRequest, logger, errors))
        {}

        virtual TString CreateContent() const override {
            return "";
        }
    };

    class TGetQueriesRequest : public IMajorRequest {
    public:
        class TQuery {
            R_READONLY(TString, Id);
            R_READONLY(TInstant, CreateTime, TInstant::Zero());
            R_READONLY(TString, State);
            R_READONLY(TString, StateDetails);
            R_READONLY(TString, TypeName);
            R_READONLY(TString, VIN);
            R_READONLY(TString, Brand);
            R_READONLY(TString, Model);
            R_READONLY(EWorkType, WorkType, EWorkType::None);
            R_READONLY(TString, ServiceName);

        public:
            bool Parse(const NJson::TJsonValue& json) {
                Id = json["ID"].GetStringRobust();
                State = json["StatusQueryOID_Name"].GetString();
                StateDetails = json["StatusDetail_Name"].GetString();
                TypeName = json["TypeQueryOID_Name"].GetString();
                CreateTime = TInstant::ParseIso8601(json["CreateDate"].GetString()) - TDuration::Hours(3);
                VIN = json["VIN"].GetString();
                Brand = json["Brand"].GetString();
                Model = json["Model"].GetString();
                WorkType = static_cast<EWorkType>(json["WorkTypeID"].GetUInteger());
                ServiceName = json["ServiceName"].GetString();
                return !!Id && !!State && !!TypeName;
            }

            NJson::TJsonValue GetJsonReport() const {
                NJson::TJsonValue result;
                result["id"] = Id;
                result["vin"] = VIN;
                result["type"] = TypeName;
                result["work"] = ToString(WorkType);
                result["state"] = State;
                result["state_details"] = StateDetails;
                result["create_time"] = CreateTime.Seconds();
                result["service_name"] = ServiceName;
                return result;
            }
        };

    public:
        TGetQueriesRequest(TVector<TQuery>& result, const TMajorLogger& logger, TMessagesCollector& errors)
            : IMajorRequest(EMajorOperationType::GetQueriesRequest, logger, errors, new TParseEntityList<TQuery>(result, EMajorOperationType::GetQueriesRequest, logger, errors))
            , Full(true)
            , Closed(false)
        {}

        TGetQueriesRequest(bool closed, TVector<TQuery>& result, const TMajorLogger& logger, TMessagesCollector& errors)
            : IMajorRequest(EMajorOperationType::GetQueriesRequest, logger, errors, new TParseEntityList<TQuery>(result, EMajorOperationType::GetQueriesRequest, logger, errors))
            , Full(false)
            , Closed(closed)
        {}

        virtual TString CreateContent() const override {
            if (Full) {
                return "{}";
            }
            NJson::TJsonValue json(NJson::JSON_MAP);
            json["IsNotClosed"] = !Closed;
            return json.GetStringRobust();
        }

    private:
        bool Full;
        bool Closed;
    };

    class TTokenRequest : public IMajorRequest {
    public:
        TTokenRequest(const TString& userName, const TString& password, TString& token, TInstant& tokenDeadline, const TMajorLogger& logger, TMessagesCollector& errors)
            : IMajorRequest(EMajorOperationType::UpdateToken, logger, errors)
            , UserName(userName)
            , Password(password)
            , Token(token)
            , TokenDeadline(tokenDeadline)
        {}

        virtual TString CreateContent() const override {
            NJson::TJsonValue json(NJson::JSON_MAP);
            json["UserName"] = UserName;
            json["Password"] = Password;
            return json.GetStringRobust();
        }

        virtual bool ProcessSuccess(const NJson::TJsonValue& data) override {
            NJson::TJsonValue::TMapType jsonMap;
            if (!data.GetMap(&jsonMap)) {
                ProcessError("Incorrect json format");
                return false;
            }
            if (!jsonMap.contains("access_token") || !jsonMap.contains("life_time_minutes")) {
                ProcessError("Internal error " + data.GetStringRobust());
                return false;
            }
            Token = "Bearer " + jsonMap["access_token"].GetString();
            TokenDeadline = TInstant::Now() + TDuration::Minutes(jsonMap["life_time_minutes"].GetUInteger());
            return true;
        }

    private:
        TString UserName;
        TString Password;
        TString& Token;
        TInstant& TokenDeadline;
    };

    class TFileListRequest : public IByIdMajorRequest {
    public:
        class TFileInfo : public ICommonCreateMajorRequest::TResponse {
            R_READONLY(TString, Extension);

        public:
            bool Parse(const NJson::TJsonValue& json) {
                Extension = json["FileExt"].GetStringRobust();
                return ICommonCreateMajorRequest::TResponse::Parse(json);
            }
        };

    public:
        TFileListRequest(const TString& id, TVector<TFileInfo>& result, const TMajorLogger& logger, TMessagesCollector& errors)
            : IByIdMajorRequest(id, EMajorOperationType::GetFileList, logger, errors, new TParseEntityList<TFileInfo>(result, EMajorOperationType::GetFileList, logger, errors))
        {}
    };

    class TGetFileRequest : public IMajorRequest {
    public:
        class TMajorFile {
            R_READONLY(TString, File);

        public:
            bool Parse(const NJson::TJsonValue& json) {
                File = Base64Decode(json["File"].GetStringRobust());
                return true;
            }
        };

    public:
        TGetFileRequest(const TFileListRequest::TFileInfo& fileInfo, TMajorFile& result, const TMajorLogger& logger, TMessagesCollector& errors)
            : IMajorRequest(EMajorOperationType::GetFile, logger, errors, new TParseEntity<TMajorFile>(result, EMajorOperationType::GetFile, logger, errors))
            , FileInfo(fileInfo)
        {}

        virtual TString CreateContent() const override {
            NJson::TJsonValue json(NJson::JSON_MAP);
            json["OID"] = FileInfo.GetGlobalId();
            json["FileID"] = FileInfo.GetId();
            return json.GetStringRobust();
        }

    private:
        TFileListRequest::TFileInfo FileInfo;
    };

    class TCarPenaltiesRequest : public IMajorRequest {
        using TBase = IMajorRequest;

    public:
        enum class EPenaltyCheckPolicy {
            All,
            Paid,
            NotPaid
        };

        class TCarPenaltyInfo {
        public:
            static const TString DefaultTimezoneName;  // "Europe/Moscow"

            class TRequisites {
            public:
                enum class ERequisiteFieldName {
                    Sts /* "Свидетельство о регистрации" */,
                    CarNumber /* "Гос.номер транспортного средства" */,
                    RulingTitle /* "Постановление" */,
                    RulingDate /* "Дата постановления" */,
                    RulingNumber /* "УИН (начисление в ГИС ГМП)" */,
                    ArticleKoap /* "Нарушение" */,
                    ViolationPlace /* "Место нарушения" */,
                    ViolationTime /* "Дата и время нарушения" */,
                    AmountToPay /* "Сумма штрафа, руб." */,
                    DiscountDate /* "Скидка 50%" */,  // format: активна до 31.08.2020
                    DiscountedAmountToPay /* "К оплате (с учетом скидки), руб." */,
                };

                bool ParseFromString(const TString& data);
                TString GetField(ERequisiteFieldName name, const TString& defaultValue = "") const;
                TInstant GetDateField(ERequisiteFieldName name, const TString& format = "%d.%m.%Y", const TInstant& defaultValue = TInstant::Zero()) const;
                TInstant GetDateTimeField(ERequisiteFieldName name, const TString& format = "%d.%m.%Y %H:%M:%S", const TString& tzName = DefaultTimezoneName, const TInstant& defaultValue = TInstant::Zero()) const;

            private:
                TMap<TString, TString> Data;
            };

            R_READONLY(i64, Id, 0);

            R_READONLY(TString, Vin);
            R_READONLY(TString, BrandName);
            R_READONLY(TString, ModelName);

            R_READONLY(TString, PenaltyIssuanceName);       // odps name

            R_READONLY(TInstant, CheckDate, TInstant::Zero());       // ts when Major obtained fine information
            R_READONLY(TInstant, DocDate, TInstant::Zero());
            R_READONLY(TString, DocNo);
            R_READONLY(TInstant, ViolationAt, TInstant::Zero());

            R_READONLY(double, SumToPayWithoutDiscount, 0.0);
            R_READONLY(TInstant, PayUntil, TInstant::Zero());

            R_READONLY(bool, HasDiscount, false);
            R_READONLY(double, SumToPay, 0.0);
            R_READONLY(TInstant, DiscountUntil, TInstant::Zero());

            R_READONLY(bool, HasCompensation, false);
            R_READONLY(bool, IsMajorPayed, false);
            R_READONLY(bool, IsDecreeExists, false);
            R_OPTIONAL(bool, WillBeIncludedToBill);

            R_READONLY(TString, RequisitesRaw);
            R_READONLY(TRequisites, Requisites);

        public:
            bool Parse(const NJson::TJsonValue& data);

            TInstant GetRulingDate() const;
            TString GetRulingNumber() const;
            TInstant GetViolationTime() const;
        };

    public:
        TCarPenaltiesRequest(const TVector<TString>& vins, const EPenaltyCheckPolicy policy, TVector<TCarPenaltyInfo>& result, const TMajorLogger& logger, TMessagesCollector& errors, const TMaybe<ui64> lastReceivedID = {});
        virtual TString CreateContent() const override;

    private:
        virtual bool ProcessSuccess(const NJson::TJsonValue& data) override;

        TVector<TString> Vins;
        const EPenaltyCheckPolicy Policy;
        const TMaybe<ui64> LastReceivedID;
    };

    class TCarPenaltyDecreeRequest : public IMajorRequest {
        using TBase = IMajorRequest;

    public:
        class TCarPenaltyDecree {
            R_READONLY(TString, FileName);
            R_READONLY(TString, FileExtension);
            R_READONLY(TString, Content);

        public:
            bool Parse(const NJson::TJsonValue& data);
        };

    public:
        TCarPenaltyDecreeRequest(const TMaybe<i64>& externalId, const TMaybe<TString>& rulingNumber, TVector<TCarPenaltyDecree>& result, const TMajorLogger& logger, TMessagesCollector& errors);
        virtual TString CreateContent() const override;

    private:
        const TMaybe<i64> ExternalId;
        const TMaybe<TString> RulingNumber;
    };
}
