#include <drive/backend/major/client.h>
#include <drive/backend/major/config.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/string/builder.h>

Y_UNIT_TEST_SUITE(MajorClient) {
    Y_UNIT_TEST(ParseCarPenaltyInfo) {
        using TCarPenaltyInfo = NMajorClient::TCarPenaltiesRequest::TCarPenaltyInfo;

        TString rawData = R"(
        {
            "ID": 123456,
            "VIN": "SALYA2BX1KA208965",
            "BrandName": "LandRover",
            "ModelName": "Range Rover Velar",
            "PenaltyIssuanceName": "ГИБДД",
            "CheckDate": "2019-10-10T00:00:00",
            "DocDate": "2019-10-07T00:00:00",
            "DocNo": "18810177191007263786",
            "ViolationAt": "2019-09-26T03:01:00",
            "Requisites": "Внимание, постановление получено от Почты России!\r\nПочта России не предоставляет информацию о скидке.\r\n--------------------------------------------------\r\n\r\nСвидетельство о регистрации: 9908555274\r\nГос.номер транспортного средства: м882нк799\r\nПостановление: ШТРАФ ПО АДМИНИСТРАТИВНОМУ ПРАВОНАРУШЕНИЮ ПОСТАНОВЛЕНИЕ №18810177191007263786\r\nДата постановления: 07.10.2019\r\nУИН (начисление в ГИС ГМП): 18810177191007263786\r\nНарушение: 12.9ч.2 - превышение установленной скорости движения транспортного средства на величину более 20, но не более 40 километров в час\r\nМесто нарушения: МОСКВА Г.   ЛЕНИНСКИЙ ПРОСП. Д.53\r\nДата и время нарушения: 26.09.2019 03:01:00\r\nПолучатель платежа: уфк по г. москве (управление гибдд гу мвд россии по г.москве л/сч. №04731440640)\r\nИНН: 7707089101\r\nКПП: 770731005\r\nРасчетный счет: 40101810045250010041\r\nБанк получателя: главное управление банка россии по центральному федеральному округу г. москвы (сокращенное наименование - гу банка россии по цфо)\r\nБИК: 044525000\r\nОКТМО(ОКАТО): 45379000\r\nКБК: 18811630020016000140\r\nСумма штрафа, руб.: 500.00\r\nСкидка 50%: активна до 28.10.2019\r\nК оплате (с учетом скидки), руб.: 250.00\r\n",
            "SummFull": 500.0,
            "PayTo": "2019-12-16T00:00:00",
            "IsDiscount": true,
            "Summ": 250.0,
            "DiscountTo": "2019-10-28T00:00:00",
            "IsMajorPayed": null,
            "IsCompens": false,
            "IsDecreeExists": true,
            "IsWillBeBilled": true
        })";

        NJson::TJsonValue data;
        UNIT_ASSERT(NJson::ReadJsonTree(rawData, &data));
        UNIT_ASSERT(data.IsDefined());

        TCarPenaltyInfo penalty;
        UNIT_ASSERT_C(penalty.Parse(data), "Unable to parse data: " + data.GetStringRobust());

        UNIT_ASSERT_VALUES_EQUAL(penalty.GetId(), 123456);
        UNIT_ASSERT_STRINGS_EQUAL(penalty.GetVin(), "SALYA2BX1KA208965");
        UNIT_ASSERT_STRINGS_EQUAL(penalty.GetPenaltyIssuanceName(), "ГИБДД");
        UNIT_ASSERT_STRINGS_EQUAL(penalty.GetCheckDate().ToStringUpToSeconds(), "2019-10-09T21:00:00Z");  // parsed as a local timestamp
        UNIT_ASSERT_STRINGS_EQUAL(penalty.GetDocDate().ToStringUpToSeconds(), "2019-10-07T00:00:00Z");  // parsed as a date
        UNIT_ASSERT_STRINGS_EQUAL(penalty.GetDocNo(), "18810177191007263786");
        UNIT_ASSERT_STRINGS_EQUAL(penalty.GetViolationAt().ToStringUpToSeconds(), "2019-09-26T00:01:00Z");  // parsed as a local timestamp
        UNIT_ASSERT_VALUES_EQUAL(penalty.GetSumToPayWithoutDiscount(), 500);
        UNIT_ASSERT_VALUES_EQUAL(penalty.GetSumToPay(), 250);
        UNIT_ASSERT_VALUES_EQUAL(penalty.GetHasDiscount(), true);
        UNIT_ASSERT_VALUES_EQUAL(penalty.GetIsMajorPayed(), false);
        UNIT_ASSERT_STRINGS_EQUAL(penalty.GetPayUntil().ToStringUpToSeconds(), "2019-12-16T00:00:00Z");  // parsed as a date
        UNIT_ASSERT_STRINGS_EQUAL(penalty.GetDiscountUntil().ToStringUpToSeconds(), "2019-10-28T00:00:00Z");  // parsed as a date
        UNIT_ASSERT_VALUES_EQUAL(penalty.GetIsDecreeExists(), true);
        UNIT_ASSERT_VALUES_EQUAL(penalty.HasWillBeIncludedToBill() ? penalty.GetWillBeIncludedToBillRef() : false, true);

        UNIT_ASSERT_STRINGS_EQUAL(
            penalty.GetRequisites().GetField(TCarPenaltyInfo::TRequisites::ERequisiteFieldName::ArticleKoap),
            "12.9ч.2 - превышение установленной скорости движения транспортного средства на величину более 20, но не более 40 километров в час"
        );

        UNIT_ASSERT_STRINGS_EQUAL(
            penalty.GetRequisites().GetField(TCarPenaltyInfo::TRequisites::ERequisiteFieldName::RulingDate),
            "07.10.2019"
        );
        UNIT_ASSERT_STRINGS_EQUAL(
            penalty.GetRequisites().GetDateField(TCarPenaltyInfo::TRequisites::ERequisiteFieldName::RulingDate).ToStringUpToSeconds(),
            "2019-10-07T00:00:00Z"
        );

        UNIT_ASSERT_STRINGS_EQUAL(
            penalty.GetRequisites().GetField(TCarPenaltyInfo::TRequisites::ERequisiteFieldName::ViolationTime),
            "26.09.2019 03:01:00"
        );
        UNIT_ASSERT_STRINGS_EQUAL(
            penalty.GetRequisites().GetDateTimeField(TCarPenaltyInfo::TRequisites::ERequisiteFieldName::ViolationTime).ToStringUpToSeconds(),
            "2019-09-26T00:01:00Z"
        );
    }
}
