#include "storage.h"

#include <drive/backend/logging/events.h>
#include <drive/backend/offers/ranking/model.h>

#include <rtline/library/storage/abstract.h>
#include <rtline/util/algorithm/ptr.h>

NDrive::TModelsStorage::TModelsStorage(TAtomicSharedPtr<NRTLine::IVersionedStorage> storage)
    : IAutoActualization("models_storage")
    , Storage(storage)
{
    CHECK_WITH_LOG(Storage);
    SetPeriod(TDuration::Minutes(1));
    Y_ENSURE_BT(Start());
}

NDrive::TModelsStorage::~TModelsStorage() {
    if (!Stop()) {
        ERROR_LOG << "cannot stop ModelsStorage" << Endl;
    }
}

NDrive::TOfferModelConstPtr NDrive::TModelsStorage::AddOfferModel(TOfferModelPtr model) {
    if (!model) {
        return nullptr;
    }

    TVersionedModel versionedModel;
    versionedModel.Model = model;

    const TString& data = model->Serialize<TString>();
    const TString& name = model->GetName();
    const TString& path = GetModelPath(name);
    if (!Checked(Storage)->SetValue(path, data, /*storeHistory=*/true, /*lock=*/true, &versionedModel.Version)) {
        return nullptr;
    }
    {
        TWriteGuard guard(Lock);
        OfferModelsCache[name] = std::move(versionedModel);
    }
    return model;
}

NDrive::TOfferModelConstPtr NDrive::TModelsStorage::GetOfferModel(TStringBuf name) const {
    NDrive::TEventLog::Log("TModelsStorage::GetOfferModel", NJson::TMapBuilder("name", name));
    {
        TReadGuard guard(Lock);
        auto p = OfferModelsCache.find(name);
        if (p != OfferModelsCache.end()) {
            return p->second.Model;
        }
    }
    auto versionedModel = TryCreateOfferModel(name);
    auto model = versionedModel.Model;
    if (versionedModel) {
        TWriteGuard guard(Lock);
        OfferModelsCache[model->GetName()] = std::move(versionedModel);
    }
    return model;
}

NDrive::TOptionalOfferMultiModel NDrive::TModelsStorage::GetOfferMultiModel(TConstArrayRef<TString> names) const {
    TVector<TOfferModelConstPtr> models;
    for (auto&& name : names) {
        auto&& model = GetOfferModel(name);
        if (!model) {
            ERROR_LOG << "model does not exist: " << name << Endl;
            return Nothing();
        }
        models.emplace_back(model);
    }
    return MakeMaybe<TOfferMultiModel>(models);
}

TVector<TString> NDrive::TModelsStorage::ListOfferModels() const {
    TVector<TString> result;
    if (!Checked(Storage)->GetNodes(GetModelRoot(), result, false)) {
        ERROR_LOG << "cannot list entries in " << GetModelRoot() << Endl;
    }
    return result;
}

bool NDrive::TModelsStorage::RemoveOfferModel(TStringBuf name) const {
    if (!Checked(Storage)->RemoveNode(GetModelPath(name))) {
        return false;
    }
    {
        TWriteGuard guard(Lock);
        auto p = OfferModelsCache.find(name);
        if (p != OfferModelsCache.end()) {
            OfferModelsCache.erase(p);
        }
    }
    return true;
}

bool NDrive::TModelsStorage::Refresh() {
    for (auto&& name : ListOfferModels()) {
        auto version = GetModelVersion(name);
        if (version > 0) {
            TReadGuard guard(Lock);
            auto p = OfferModelsCache.find(name);
            if (p != OfferModelsCache.end() && p->second.Version == version) {
                DEBUG_LOG << name << ": skip update by unchanged version " << version << Endl;
                continue;
            }
        }
        auto versionedModel = TryCreateOfferModel(name);
        if (versionedModel) {
            auto version = versionedModel.Version;
            TWriteGuard guard(Lock);
            OfferModelsCache[versionedModel.Model->GetName()] = std::move(versionedModel);
            INFO_LOG << name << ": updated with version " << version << Endl;
        }
    }
    return true;
}

TString NDrive::TModelsStorage::GetModelPath(TStringBuf name) const {
    return GetModelRoot() + '/' + name;
}

TString NDrive::TModelsStorage::GetModelRoot() const {
    return "models";
}

i64 NDrive::TModelsStorage::GetModelVersion(TStringBuf name) const {
    const TString& path = GetModelPath(name);
    i64 version = -1;
    if (!Checked(Storage)->GetVersion(path, version)) {
        ERROR_LOG << name << ": cannot get version of " << path << Endl;
    }
    return version;
}

NDrive::TModelsStorage::TVersionedModel NDrive::TModelsStorage::TryCreateOfferModel(TStringBuf name) const {
    if (!name) {
        return {};
    }
    const TString& path = GetModelPath(name);
    TVersionedModel result;
    if (!Checked(Storage)->GetVersion(path, result.Version)) {
        ERROR_LOG << name << ": cannot get version of " << path << Endl;
        return {};
    }

    TString data;
    if (!Checked(Storage)->GetValue(path, data, result.Version)) {
        ERROR_LOG << name << ": cannot get value of " << path << Endl;
        return {};
    }

    try {
        result.Model = IOfferModel::Construct(data);
        return result;
    } catch (const std::exception& e) {
        ERROR_LOG << name << ": an exception has occurred: " << FormatExc(e) << Endl;
        return {};
    }
}
