#pragma once

#include <drive/backend/abstract/base.h>
#include <drive/backend/abstract/notifier.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <library/cpp/cache/cache.h>

#include <rtline/library/json/parse.h>
#include <rtline/library/unistat/cache.h>
#include <rtline/util/types/accessor.h>

#include <util/generic/map.h>

template <typename EEventType>
class TNotifyHandler {
public:
    using EEvent = EEventType;

    using TResult = NDrive::INotifier::TResult;
    using TResultPtr = TResult::TPtr;

    virtual ~TNotifyHandler() = default;

    NDrive::INotifier::TPtr GetNotifier(const IServerBase& server) const {
        return server.GetNotifier(GetNotifierName());
    }

    TResultPtr Handle(const IServerBase& server) const {
        TString messageData;
        TMessagesCollector errors;
        if (!GetMessageData(messageData, errors)) {
            TString errorMessage = "Cannot obtain message data to notify: " + errors.GetStringReport();
            ERROR_LOG << errorMessage << Endl;
            return MakeAtomicShared<TResult>(errorMessage);
        }
        return Notify(server, messageData);
    }

    static NDrive::TScheme GetScheme(const IServerBase& server) {
        NDrive::TScheme scheme;
        scheme.Add<TFSVariants>("event_type", "Тип события").InitVariants<EEvent>().SetRequired(true);
        scheme.Add<TFSVariants>("notifier_name", "Имя нотификатора").SetVariants(server.GetNotifierNames()).SetRequired(true);
        scheme.Add<TFSString>("message_template", "Шаблон сообщения");
        return scheme;
    }

    bool DeserializeFromJson(const NJson::TJsonValue& data) {
        return (NJson::ParseField(data["event_type"], NJson::Stringify(EventType), /* required = */ true) &&
                NJson::ParseField(data["notifier_name"], NotifierName, /* required = */ true) &&
                NJson::ParseField(data["message_template"], MessageTemplate));
    }

    NJson::TJsonValue SerializeToJson() const {
        NJson::TJsonValue data(NJson::JSON_MAP);
        NJson::InsertField(data, "event_type", NJson::Stringify(EventType));
        NJson::InsertField(data, "notifier_name", NotifierName);
        NJson::InsertField(data, "message_template", MessageTemplate);
        return data;
    }

protected:
    TResultPtr Notify(const IServerBase& server, const TString& messageData) const {
        auto notifier = GetNotifier(server);
        if (!notifier) {
            return MakeAtomicShared<TResult>("No notifier configured: " + NotifierName);
        }

        NDrive::TNotifierContext context;
        context.SetServer(&server);

        NJson::TJsonValue messageJsonData;
        if (NJson::ReadJsonTree(messageData, &messageJsonData)) {
            return notifier->Notify(messageJsonData, context);
        }

        return notifier->Notify(NDrive::INotifier::TMessage(messageData), context);
    }

    bool GetMessageData(TString& messageData, TMessagesCollector& /* errors */) const {
        messageData = MessageTemplate;
        return true;
    }

private:
    R_FIELD(EEvent, EventType);
    R_FIELD(TString, NotifierName);
    R_FIELD(TString, MessageTemplate);
};

template <typename EEventType, typename TContextFetcher>
class TContextNotifyHandler: public TNotifyHandler<EEventType> {
    using TBase = TNotifyHandler<EEventType>;

public:
    using TContext = typename TContextFetcher::TContextType;

    using EEvent = typename TBase::EEvent;
    using TResult = typename TBase::TResult;
    using TResultPtr = typename TBase::TResultPtr;

    using TBase::GetEventType;
    using TBase::SetEventType;
    using TBase::GetNotifierName;
    using TBase::SetNotifierName;
    using TBase::GetMessageTemplate;
    using TBase::SetMessageTemplate;

    using TBase::GetNotifier;
    using TBase::Handle;
    using TBase::GetScheme;
    using TBase::DeserializeFromJson;
    using TBase::SerializeToJson;

    TResultPtr Handle(const TContext& context) const {
        TString messageData;
        TMessagesCollector errors;
        if (!GetMessageData(context, messageData, errors)) {
            TString errorMessage = "Cannot obtain message data to notify: " + errors.GetStringReport();
            ERROR_LOG << errorMessage << Endl;
            return MakeAtomicShared<TResult>(errorMessage);
        }
        return TBase::Notify(context.GetServer(), messageData);
    }

protected:
    bool GetMessageData(const TContext& context, TString& messageData, TMessagesCollector& errors) const {
        messageData = TContextFetcher::ProcessText(GetMessageTemplate(), context, errors);
        return !errors.HasMessages();
    }
};

template <typename TResultPtr>
class IResultCachePolicy {
public:
    virtual ~IResultCachePolicy() = default;

    virtual void Accept(TResultPtr result) = 0;
    virtual TResultPtr GetResult(const TString& transitId) const = 0;
};

class TNotifyResultPolicyBase: public IResultCachePolicy<NDrive::INotifier::TResult::TPtr> {
    using TBase = IResultCachePolicy<NDrive::INotifier::TResult::TPtr>;

public:
    using TResultPtr = NDrive::INotifier::TResult::TPtr;
    using TPtr = TAtomicSharedPtr<TNotifyResultPolicyBase>;
};

class TNotifyResultDropPolicy: public TNotifyResultPolicyBase {
public:
    virtual void Accept(TResultPtr result) override {
        Y_UNUSED(result);
    }

    virtual TResultPtr GetResult(const TString& transitId) const override {
        Y_UNUSED(transitId);
        return nullptr;
    }
};

class TNotifyResultCachePolicy: public TNotifyResultPolicyBase {
    using TBase = TNotifyResultPolicyBase;

public:
    static constexpr size_t DefaultCacheSize = 100;  // configure through scheme and external json

    TNotifyResultCachePolicy(size_t cacheSize = DefaultCacheSize)
        : TBase()
        , Cache(cacheSize)
    {
    }

    virtual void Accept(TResultPtr result) override {
        if (!!result && result->GetTransitId()) {
            Cache.Update(result->GetTransitId(), result);
        }
    }

    virtual TResultPtr GetResult(const TString& transitId) const override {
        if (!!transitId) {
            auto it = Cache.FindWithoutPromote(transitId);
            if (it != Cache.End()) {
                return *it;
            }
        }
        return nullptr;
    }

private:
    TLRUCache<TString, TResultPtr> Cache;
};

template <typename THandlerType>
class TNotifyHandlerCollectionBase: public NDrive::INotifyResultProvider {
    static_assert(std::is_same<typename THandlerType::TResultPtr, typename NDrive::INotifyResultProvider::TNotifyResultPtr>::value);

public:
    using THandler = THandlerType;
    using EEvent = typename THandler::EEvent;
    using TNotifyResultPtr = typename THandler::TResultPtr;

    using TResultCachePolicyPtr = TNotifyResultPolicyBase::TPtr;
    using TEventHandlerMapping = TMap<EEvent, THandler>;

    explicit TNotifyHandlerCollectionBase(const TString& signalType, TResultCachePolicyPtr resultCachePolicyPtr = MakeAtomicShared<TNotifyResultDropPolicy>())
        : SignalType(signalType)
        , ResultCachePolicyPtr(resultCachePolicyPtr)
    {
    }

    void AddHandler(THandler&& handler) {
        EventHandlerMapping.emplace(handler.GetEventType(), std::forward<THandler>(handler));
    }

    TNotifyResultPtr Handle(const EEvent event, const IServerBase& server, const double value = 1) const {
        AddSignal(event, value);

        TNotifyResultPtr result = nullptr;

        auto handlerPtr = EventHandlerMapping.FindPtr(event);
        if (!!handlerPtr) {
            result = handlerPtr->Handle(server);
            ResultCachePolicyPtr->Accept(result);
        }

        return result;
    }

    TNotifyResultPtr GetHandlingResult(const TString& transitId) const {
        return ResultCachePolicyPtr->GetResult(transitId);
    }

    TSet<TString> GetContextPlaceholders() const {
        return {};
    }

    static NDrive::TScheme GetScheme(const IServerBase& server) {
        NDrive::TScheme scheme;
        scheme.Add<TFSArray>("event_handlers", "Обработчики событий").SetElement(THandler::GetScheme(server));
        return scheme;
    }

    bool DeserializeFromJson(const NJson::TJsonValue& data) {
        for (const auto& handlerData : data["event_handlers"].GetArray()) {
            THandler handler;
            if (!handler.DeserializeFromJson(handlerData)) {
                return false;
            }
            EventHandlerMapping.emplace(handler.GetEventType(), std::move(handler));
        }
        return true;
    }

    NJson::TJsonValue SerializeToJson() const {
        NJson::TJsonValue result;

        auto& serializedHandlers = result.InsertValue("event_handlers", NJson::JSON_ARRAY);
        for (auto&& [_, handler] : EventHandlerMapping) {
            serializedHandlers.AppendValue(handler.SerializeToJson());
        }

        return result;
    }

protected:
    void AddSignal(const EEvent event, const double value = 1) const {
        TUnistatSignalsCache::SignalAdd(SignalType + ((!!SignalName) ? ("-" + SignalName) : ""), ::ToString(event), value);
    }

private:
    R_READONLY(TEventHandlerMapping, EventHandlerMapping);

    R_FIELD(TString, SignalType);
    R_FIELD(TString, SignalName);

    R_READONLY(TResultCachePolicyPtr, ResultCachePolicyPtr);
};

template <typename EEventType>
class TNotifyHandlerCollection: public TNotifyHandlerCollectionBase<TNotifyHandler<EEventType>> {
    using TBase = TNotifyHandlerCollectionBase<TNotifyHandler<EEventType>>;
    using TSelf = TNotifyHandlerCollection<EEventType>;

public:
    using TPtr = TAtomicSharedPtr<TSelf>;

    using THandler = typename TBase::THandler;
    using EEvent = typename TBase::EEvent;
    using TNotifyResultPtr = typename TBase::TNotifyResultPtr;

    using TBase::TBase;

    using TBase::AddHandler;

    using TBase::Handle;
    using TBase::GetHandlingResult;
    using TBase::GetContextPlaceholders;

    using TBase::GetScheme;
    using TBase::DeserializeFromJson;
    using TBase::SerializeToJson;

    using TBase::GetSignalType;
    using TBase::SetSignalType;
    using TBase::GetSignalName;
    using TBase::SetSignalName;
};

template <typename EEventType, typename TContextFetcher>
class TContextNotifyHandlerCollection: public TNotifyHandlerCollectionBase<TContextNotifyHandler<EEventType, TContextFetcher>> {
    using TBase = TNotifyHandlerCollectionBase<TContextNotifyHandler<EEventType, TContextFetcher>>;
    using TSelf = TContextNotifyHandlerCollection<EEventType, TContextFetcher>;

public:
    using TPtr = TAtomicSharedPtr<TSelf>;

    using TContext = typename TContextFetcher::TContextType;

    using THandler = typename TBase::THandler;
    using EEvent = typename TBase::EEvent;
    using TNotifyResultPtr = typename TBase::TNotifyResultPtr;

    using TBase::TBase;

    using TBase::AddHandler;

    using TBase::Handle;
    using TBase::GetHandlingResult;

    using TBase::GetScheme;
    using TBase::DeserializeFromJson;
    using TBase::SerializeToJson;

    using TBase::GetSignalType;
    using TBase::SetSignalType;
    using TBase::GetSignalName;
    using TBase::SetSignalName;

    TNotifyResultPtr Handle(const EEvent event, const TContext& context, const double value = 1) const {
        TBase::AddSignal(event, value);

        TNotifyResultPtr result = nullptr;

        auto handlerPtr = TBase::GetEventHandlerMapping().FindPtr(event);
        if (!!handlerPtr) {
            auto extendedContext = ConstructSignalContext(context, ::ToString(value));
            result = handlerPtr->Handle(extendedContext);
            TBase::GetResultCachePolicyPtr()->Accept(result);
        }

        return result;
    }

    TSet<TString> GetContextPlaceholders() const {
        auto placeholders = TContext::GetRegisteredFetchers();
        placeholders.emplace("signal_type");
        placeholders.emplace("signal_name");
        placeholders.emplace("signal_value");
        return placeholders;
    }

    TSet<TString> GetContextPlaceholders(const TContext& context) const {
        auto placeholders = GetContextPlaceholders();
        for (auto&& [name, _] : context.GetDynamiContext()) {
            placeholders.emplace(name);
        }
        return placeholders;
    }

private:
    TContext ConstructSignalContext(const TContext& originalContext, TString signalValue) const {
        TContext context(originalContext);

        auto& dynamicContext = context.MutableDynamicContext();
        dynamicContext.emplace("signal_type", GetSignalType());
        dynamicContext.emplace("signal_name", GetSignalName());
        dynamicContext.emplace("signal_value", signalValue);

        return context;
    }
};
