#include "mail.h"

#include <drive/backend/logging/events.h>

#include <library/cpp/mediator/global_notifications/system_status.h>
#include <library/cpp/string_utils/base64/base64.h>
#include <library/cpp/string_utils/quote/quote.h>

#include <rtline/library/async_proxy/async_delivery.h>
#include <rtline/library/json/builder.h>
#include <rtline/library/unistat/cache.h>
#include <rtline/util/network/neh.h>

#include <util/stream/file.h>
#include <util/string/builder.h>
#include <util/string/join.h>

TMailNotificationsConfig::TFactory::TRegistrator<TMailNotificationsConfig> TMailNotificationsConfig::Registrator("mail");

void TMailNotificationsConfig::DoInit(const TYandexConfig::Section* section) {
    AssertCorrectConfig(section->GetDirectives().GetValue("Host", Host), "Incorrect parameter 'Host' for mail notifications");
    Port = section->GetDirectives().Value<ui16>("Port", Port);
    IsHttps = section->GetDirectives().Value<bool>("IsHttps", IsHttps);
    AssertCorrectConfig(section->GetDirectives().GetValue("Account", Account), "Incorrect parameter 'Account' for mail notifications");

    TFsPath tokenPath = section->GetDirectives().Value<TFsPath>("TokenPath");
    if (tokenPath) {
        AssertCorrectConfig(tokenPath.Exists(), "Can't read token info from " + tokenPath.GetPath());
        AuthToken = Strip(TFileInput(tokenPath).ReadAll());
    } else {
        AuthToken = section->GetDirectives().Value<TString>("Token");
    }
    AssertCorrectConfig(!!AuthToken, "Auth token for sender.yandex-team.ru was not initialized.");

    OverridenRecipient = section->GetDirectives().Value<TString>("OverridenRecipient");

    MaxInFlight = section->GetDirectives().Value<ui32>("MaxInFlight", MaxInFlight);
    ResponseTimeout = section->GetDirectives().Value<TDuration>("ResponseTimeout", ResponseTimeout);
}

void TMailNotificationsConfig::DoToString(IOutputStream& os) const {
    os << "Host: " << Host << Endl;
    os << "Port: " << Port << Endl;
    os << "IsHttps: " << IsHttps << Endl;
    os << "Account: " << Account << Endl;
    os << "OverridenRecipient: " << OverridenRecipient << Endl;
    os << "MaxInFlight: " << MaxInFlight << Endl;
    os << "ResponseTimeout: " << ResponseTimeout << Endl;
}

NDrive::INotifier::TPtr TMailNotificationsConfig::Construct() const {
    return new TMailNotifier(*this);
}

bool TMailNotificationsConfig::DeserializeFromJson(const NJson::TJsonValue& info, TMessagesCollector& errors) {
    JREAD_STRING_OPT(info, "host", Host);
    JREAD_UINT_OPT(info, "port", Port);
    JREAD_BOOL_OPT(info, "is_https", IsHttps);
    JREAD_STRING_OPT(info, "auth_token", AuthToken);
    JREAD_STRING_OPT(info, "account", Account);
    JREAD_STRING_OPT(info, "overriden_recipient", OverridenRecipient);
    JREAD_UINT_OPT(info, "max_in_fly", MaxInFlight);
    JREAD_DURATION_OPT(info, "response_timeout", ResponseTimeout);
    return TBase::DeserializeFromJson(info, errors);
}

NJson::TJsonValue TMailNotificationsConfig::SerializeToJson() const {
    NJson::TJsonValue result = TBase::SerializeToJson();
    JWRITE(result, "host", Host);
    JWRITE(result, "port", Port);
    JWRITE(result, "is_https", IsHttps);
    JWRITE(result, "auth_token", AuthToken);
    JWRITE(result, "account", Account);
    JWRITE(result, "overriden_recipient", OverridenRecipient);
    JWRITE(result, "max_in_fly", MaxInFlight);
    JWRITE_DURATION(result, "response_timeout", ResponseTimeout);
    return result;
}

NDrive::TScheme TMailNotificationsConfig::GetScheme(const IServerBase& server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSString>("host").SetDefault(Host);
    result.Add<TFSNumeric>("port").SetDefault(Port);
    result.Add<TFSBoolean>("is_https").SetDefault(IsHttps);
    result.Add<TFSString>("auth_token").SetDefault(AuthToken);
    result.Add<TFSString>("account").SetDefault(Account);
    result.Add<TFSString>("overriden_recipient").SetDefault(OverridenRecipient);
    result.Add<TFSNumeric>("max_in_fly").SetDefault(MaxInFlight);
    result.Add<TFSDuration>("response_timeout").SetDefault(ResponseTimeout);
    return result;
}

TMailNotifier::TMailNotifier(const TMailNotificationsConfig& config)
    : NDrive::INotifier(config)
    , Config(config)
{
}

TMailNotifier::~TMailNotifier() {
    NDrive::INotifier::Stop();
}

void TMailNotifier::DoStart(const IServerBase* /*server*/) {
    CHECK_WITH_LOG(!Agent);
    AD = new TAsyncDelivery;
    AD->Start(1, 1);
    Agent.Reset(new NNeh::THttpClient(AD));

    NSimpleMeta::TConfig reaskConfig;
    reaskConfig.SetGlobalTimeout(Config.GetResponseTimeout()).SetMaxAttempts(1);
    Agent->RegisterSource("mail", Config.GetHost(), Config.GetPort(), reaskConfig, Config.GetIsHttps());
}

void TMailNotifier::DoStop() {
    if (AD) {
        AD->Stop();
    }
}

NDrive::INotifier::TResult::TPtr TMailNotifier::DoNotify(const TMessage& message, const TContext& context) const {
    CHECK_WITH_LOG(Agent);

    auto templateId = message.GetHeader();
    if (!templateId) {
        return new TResult("Missing template id");
    }

    NNeh::THttpRequest request;
    request.SetUri("/api/0/" + Config.GetAccount() + "/transactional/" + templateId + "/send");

    TStringBuilder postData;
    postData << "async=true";
    auto templateArgsSerializedJson = message.GetBody();
    if (templateArgsSerializedJson) {
        Quote(templateArgsSerializedJson);
        postData << "&args=" << templateArgsSerializedJson;
    }

    const auto* attachments = message.GetAdditionalInfo().GetValueByPath("attachments");
    if (attachments && attachments->IsDefined()) {
        auto serializedAttachments = attachments->GetStringRobust();
        Quote(serializedAttachments);
        postData << "&attachments=" << serializedAttachments;
    }

    request.SetContentType("application/x-www-form-urlencoded; charset=UTF-8");
    request.SetPostData(postData);

    TString encoded;
    Base64Encode(Config.GetAuthToken() + ":", encoded);
    request.AddHeader("Authorization", Sprintf("Basic %s", encoded.data()));

    TVector<TString> recipientsEmails;
    if (Config.GetOverridenRecipient()) {
        recipientsEmails.push_back(Config.GetOverridenRecipient());
    } else {
        const TRecipients& recipients = context.GetRecipients();
        for (auto&& userInfo : recipients) {
            recipientsEmails.push_back(userInfo.GetEmail());
        }
    }

    TMap<TString, NNeh::THttpRequest> requests;
    for (auto&& email : recipientsEmails) {
        Quote(email);
        request.SetCgiData("to_email=" + email);
        requests.emplace(email, request);
    }

    auto responses = Agent->SendPack(requests, TInstant::Max(), Config.GetMaxInFlight(),
                                     Config.GetResponseTimeout());
    NJson::TJsonValue errors = NJson::JSON_ARRAY;
    bool hasErrors = false;
    for (const auto& it : responses) {
        const TString& email = it.first;
        const NNeh::THttpReply& resp = it.second;
        TUnistatSignalsCache::SignalAdd("frontend-mail", "codes-" + ::ToString(resp.Code()), 1);
        if (resp.Code() != 200) {
            hasErrors = true;
            errors.AppendValue(NJson::TMapBuilder("request", request.GetRequest())
                ("response_code", resp.Code())
                ("response", resp.Content())
                ("response_error", resp.ErrorMessage()));
        } else {
            NDrive::TEventLog::Log(
                    "SendMail", NJson::TMapBuilder("email", email)("template_id", templateId));
        }
    }

    return new TResult(hasErrors ? "Failed request" : "", errors);
}
