#include "manager.h"

#include <library/cpp/tvmauth/client/facade.h>

NDrive::TScheme TNotifierContainer::GetScheme(const IServerBase& server) const {
    NDrive::TScheme result;
    result.Add<TFSNumeric>("revision", "Версия конфигурации").SetReadOnly(true);
    result.Add<TFSString>("name", "Идентификатор процесса");
    result.Add<TFSString>("display_name", "Человекочитаемое имя процесса");
    result.Add<TFSBoolean>("notifier_is_active", "Нотификатор активен").SetReadOnly(true);
    if (NotifierConfig) {
        result.Add<TFSStructure>("meta", "Настройки").SetStructure(NotifierConfig->GetScheme(server));
    }
    return result;
}

bool TNotifierContainer::DeserializeWithDecoderVerbose(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, TMessagesCollector& errors, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE_DEF(decoder, values, Revision, Max<ui64>());
    READ_DECODER_VALUE(decoder, values, Name);
    if (!Name && Revision != Max<ui64>()) {
        return false;
    }
    if (decoder.GetDisplayName() > -1) {
        READ_DECODER_VALUE(decoder, values, DisplayName);
    }
    TString typeName;
    READ_DECODER_VALUE_TEMP(decoder, values, typeName, Type);
    NotifierConfig = NDrive::INotifierConfig::TFactory::Construct(typeName);
    if (!NotifierConfig) {
        errors.AddMessage("NotifierContainer::DeserializeWithDecoderVerbose", "cannot construct config for " + typeName);
        return false;
    }
    NotifierConfig->SetTypeName(typeName);
    NotifierConfig->SetName(Name);

    NJson::TJsonValue jsonMeta;
    READ_DECODER_VALUE_JSON(decoder, values, jsonMeta, Meta);
    if (!NotifierConfig->DeserializeFromJson(jsonMeta, errors)) {
        return false;
    }

    Notifier = NotifierConfig->Construct();
    if (!Notifier) {
        errors.AddMessage("NotifierContainer::DeserializeWithDecoderVerbose", "cannot construct notifier");
        return false;
    }
    return true;
}

bool TNotifierContainer::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* hContext) {
    TMessagesCollector errors;
    return DeserializeWithDecoderVerbose(decoder, values, errors, hContext);
}

NJson::TJsonValue TNotifierContainer::GetReport() const {
    NJson::TJsonValue result;
    result.InsertValue("name", Name);
    result.InsertValue("display_name", GetDisplayName());
    result.InsertValue("notifier_is_active", Notifier && Notifier->IsActive());
    if (HasRevision()) {
        result.InsertValue("revision", Revision);
    }
    if (NotifierConfig) {
        result.InsertValue("type", NotifierConfig->GetTypeName());
        result.InsertValue("meta", NotifierConfig->SerializeToJson());
    }
    return result;
}

NJson::TJsonValue TNotifierContainer::GetUserReport(const IServerBase& server) const {
    NJson::TJsonValue result;
    result.InsertValue("name", Name);
    result.InsertValue("display_name", GetDisplayName());
    if (HasRevision()) {
        result.InsertValue("revision", Revision);
    }
    if (NotifierConfig) {
        result.InsertValue("type", NotifierConfig->GetTypeName());
        result.InsertValue("meta", NotifierConfig->GetUserReport(server));
    }
    return result;
}

NStorage::TTableRecord TNotifierContainer::SerializeToTableRecord() const {
    NStorage::TTableRecord result;
    result.Set("name", Name);
    result.Set("display_name", GetDisplayName());
    if (HasRevision()) {
        result.Set("revision", GetRevision());
    }
    if (NotifierConfig) {
        result.Set("type", NotifierConfig->GetTypeName());
        result.Set("meta", NotifierConfig->SerializeToJson());
    }
    return result;
}

void TNotifierContainer::Start(const IServerBase& server) {
    if (Notifier) {
        if (const auto& clientId = Notifier->GetSelfTvmId()) {
            Notifier->SetTvmClient(server.GetTvmClient(*clientId));
        }
        Notifier->Start(&server);
    }
}

TNotifierContainer TNotifierContainer::DeepCopy() const {
    TNotifierContainer result;
    if (!TBaseDecoder::DeserializeFromTableRecord<>(result, SerializeToTableRecord())) {
        ERROR_LOG << "Incorrect deep copy for notifier " << GetName() << Endl;
        Y_ASSERT(false);
    }
    return result;
}