#pragma once

#include <drive/backend/abstract/notifier.h>
#include <drive/backend/database/history/db_entities.h>

#include <util/generic/ptr.h>

class TNotifierContainer {
public:
    using TId = TString;

private:
    NDrive::INotifierConfig::TPtr NotifierConfig;
    NDrive::INotifier::TPtr Notifier;

    R_FIELD(TString, Name);
    R_FIELD(TString, DisplayName);
    R_READONLY(ui64, Revision, Max<ui64>());

public:
    TNotifierContainer() = default;
    TNotifierContainer(NDrive::INotifierConfig::TPtr config)
        : NotifierConfig(config)
        , Notifier(config->Construct())
    {
    }

    ~TNotifierContainer() = default;

    const TString& GetInternalId() const {
        return Name;
    }

    bool HasRevision() const {
        return Revision != Max<ui64>();
    }

    TMaybe<ui64> OptionalRevision() const {
        return HasRevision() ? Revision : TMaybe<ui64>();
    }

    TAtomicSharedPtr<NDrive::INotifier> GetNotifierPtr() const {
        return Notifier;
    }

    TAtomicSharedPtr<NDrive::INotifierConfig> GetNotifierConfigPtr() const {
        return NotifierConfig;
    }

    static TString GetPropositionsTableName() {
        return "notifiers_propositions";
    }

    const NDrive::INotifier* operator->() const {
        return Notifier.Get();
    }

    bool operator<(const TNotifierContainer& other) const {
        return GetName() < other.GetName();
    }

    bool operator!() const noexcept {
        return !NotifierConfig || !Notifier;
    }

    template<class T>
    TAtomicSharedPtr<T> GetAs() const noexcept {
        return std::dynamic_pointer_cast<T>(NotifierConfig);
    }

    class TDecoder: public TBaseDecoder {
    public:
        R_FIELD(i32, Revision, -1);
        R_FIELD(i32, Name, -1);
        R_FIELD(i32, DisplayName, -1);
        R_FIELD(i32, Type, -1);
        R_FIELD(i32, Meta, -1);

    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase, const bool strict = true)
            : TBaseDecoder(strict)
        {
            Revision = GetFieldDecodeIndex("revision", decoderBase);
            Name = GetFieldDecodeIndex("name", decoderBase);
            DisplayName = GetFieldDecodeIndex("display_name", decoderBase);
            Meta = GetFieldDecodeIndex("meta", decoderBase);
            Type = GetFieldDecodeIndex("type", decoderBase);
        }

        static bool NeedVerboseParsingErrorLogging() {
            return false;
        }
    };

    NDrive::TScheme GetScheme(const IServerBase& server) const;

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);
    bool DeserializeWithDecoderVerbose(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, TMessagesCollector& errors, const IHistoryContext* /*hContext*/);

    template<class TNotifierConfig = NDrive::INotifierConfig>
    bool UserPatch(const NJson::TJsonValue& data, TMessagesCollector& errors) {
        if (!GetInternalId()) {
            return false;
        }
        if (
            !NJson::ParseField(data["revision"], Revision) ||
            !NJson::ParseField(data["display_name"], DisplayName)
        ) {
            return false;
        }

        auto notifierConfigImpl = GetAs<TNotifierConfig>();
        if (!notifierConfigImpl) {
            return false;
        }
        return notifierConfigImpl->UserPatch(data["meta"], errors);
    }

    NJson::TJsonValue GetReport() const;
    NJson::TJsonValue GetUserReport(const IServerBase& server) const;

    NJson::TJsonValue BuildJsonReport() const {
        return GetReport();
    }

    NStorage::TTableRecord SerializeToTableRecord() const;

    static TString GetHistoryTableName() {
        return "notifiers_history";
    }

    static TString GetTableName() {
        return "notifiers";
    }

    void Start(const IServerBase& server);

    TNotifierContainer DeepCopy() const;
};

class TNotifiersManagerConfig: public TDBEntitiesManagerWithPropositionsConfig {
public:
    using TDBEntitiesManagerWithPropositionsConfig::TDBEntitiesManagerConfig;
};

class TNotifiersManager: public TDBEntitiesManagerWithPropositions<TNotifierContainer> {
    using TBase = TDBEntitiesManagerWithPropositions<TNotifierContainer>;

public:
    using TBase::TBase;

    TNotifiersManager(const IServerBase& server, const IHistoryContext& context, const TNotifiersManagerConfig& config)
        : TBase(context, config)
        , Server(server)
    {
    }

protected:
    virtual TNotifierContainer PrepareForUsage(const TNotifierContainer& evHistory) const override {
        TNotifierContainer result = evHistory.DeepCopy();
        result.Start(Server);
        return result;
    }

private:
    const IServerBase& Server;
};