#include "chat.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/database/transaction/tx.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/merge.h>

TChatRobotNotifierConfig::TFactory::TRegistrator<TChatRobotNotifierConfig> TChatRobotNotifierConfig::Registrator("native_chat");

NDrive::INotifier::TPtr TChatRobotNotifierConfig::Construct() const {
    return new TNativeChatNotifier(*this);
}

bool TChatRobotNotifierConfig::DeserializeFromJson(const NJson::TJsonValue& info, TMessagesCollector& errors) {
    if (!TBase::DeserializeFromJson(info, errors)) {
        return false;
    }
    if (GetName()) {
        SetChatId(GetName());
    } else {
        errors.AddMessage("ChatRobotNotifierConfig::DeserializeFromJson", "notifier name is empty");
        return false;
    }
    {
        TString botClassName;
        JREAD_STRING(info, "bot_class_name", botClassName);
        if (!TryFromString(botClassName, MutableBotClassName())) {
            return false;
        }
    }
    if (info.Has("chat_script")) {
        NJson::TJsonValue script = info["chat_script"];
        if (!MutableChatScript().Parse(script, errors)) {
            return false;
        }
        MutableChatScript().SetRawScript(script);
    } else {
        return false;
    }
    JREAD_STRING(info, "initial_step_id", MutableInitialStepId());
    JREAD_BOOL(info, "start_from_clean_chat", MutableStartFromCleanChat());
    JREAD_BOOL(info, "create_on_read", MutableCreateOnRead());
    JREAD_BOOL(info, "is_hidden", MutableIsHidden());
    JREAD_BOOL_OPT(info, "mark_all_read", MutableMarkAllRead());
    JREAD_BOOL_OPT(info, "persist_entered_states", MutablePersistVisitedStates());
    JREAD_BOOL(info, "is_static", MutableIsStatic());
    JREAD_STRING_OPT(info, "title", MutableTitle());
    JREAD_STRING_OPT(info, "icon", MutableIcon());
    if (info.Has("theme") && !MutableTheme().DeserializeFromJson(info["theme"], errors)) {
        errors.AddMessage("theme", "failed to deserialize");
        return false;
    }

    {
        bool isCommon = false;
        JREAD_BOOL_OPT(info, "is_common", isCommon);
        if (isCommon) {
            SetCommon(true);
            JREAD_INSTANT(info, "common_instant", MutableCommonInstant());
        }
    }

    {
        bool isPinned = false;
        JREAD_BOOL_OPT(info, "is_pinned", isPinned);
        SetPinned(isPinned);
    }

    if (info.Has("common_sharding_policy") && !MutableCommonShardingPolicy().DeserializeFromJson(info["common_sharding_policy"])) {
        return false;
    }

    if (info.Has("auditory_condition") && info["auditory_condition"].IsDefined()) {
        SetAuditoryConditionRaw(info["auditory_condition"]);
        MutableAuditoryCondition() = ICondition::DeserializeFromJson(info["auditory_condition"], &errors);
        if (!GetAuditoryCondition()) {
            return false;
        }
    }

    return
        NJson::ParseField(info, "locale", NJson::Stringify(MutableLocale())) &&
        NJson::ParseField(info, "robot_name", MutableRobotName()) &&
        NJson::ParseField(info, "robot_id", MutableRobotUserId());
}

NJson::TJsonValue TChatRobotNotifierConfig::SerializeToJson() const {
    NJson::TJsonValue result = TBase::SerializeToJson();
    result["chat_id"] = GetChatId();
    result["bot_class_name"] = ::ToString(GetBotClassName());
    result["chat_script"] = GetChatScript().DumpToJson();
    result["initial_step_id"] = GetInitialStepId();
    result["start_from_clean_chat"] = GetStartFromCleanChat();
    result["create_on_read"] = GetCreateOnRead();
    result["is_hidden"] = GetIsHidden();
    result["mark_all_read"] = GetMarkAllRead();
    result["persist_entered_states"] = GetPersistVisitedStates();
    result["is_static"] = GetIsStatic();
    result["theme"] = GetTheme().SerializeToJson();
    result["title"] = GetTitle();
    result["icon"] = GetIcon();
    result["is_common"] = IsCommon();
    if (IsCommon()) {
        JWRITE_INSTANT(result, "common_instant", GetCommonInstant());
    }
    JWRITE(result, "common_sharding_policy", GetCommonShardingPolicy().SerializeToJson());
    result["is_pinned"] = IsPinned();
    if (GetAuditoryConditionRaw().IsDefined()) {
        result["auditory_condition"] = GetAuditoryConditionRaw();
    }
    NJson::InsertField(result, "robot_name", GetRobotName());
    NJson::InsertField(result, "robot_id", GetRobotUserId());
    NJson::InsertField(result, "locale", NJson::Stringify(GetLocale()));
    return result;
}

NDrive::TScheme TChatRobotNotifierConfig::GetScheme(const IServerBase& server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSString>("chat_id", "ID чата").SetDeprecated(true);
    result.Add<TFSVariants>("bot_class_name", "Логика чата").InitVariants<NChatRobot::ELogic>();
    result.Add<TFSJson>("chat_script", "Сценарий чата");
    result.Add<TFSString>("title", "Заголовок чата");
    result.Add<TFSString>("icon", "Кастомная иконка для чата");
    result.Add<TFSStructure>("theme", "Тема").SetStructure<NDrive::TScheme>(TChatTheme::GetScheme(server));
    result.Add<TFSString>("initial_step_id", "Стартовое состояние чата").SetDefault("intro");
    result.Add<TFSBoolean>("start_from_clean_chat", "Не пытаться восстанавливать состояние чата").SetDefault(false);
    result.Add<TFSBoolean>("create_on_read", "Создавать диалог на чтении, если его не было").SetDefault(true);
    result.Add<TFSBoolean>("is_hidden", "Скрывать чат для пользователя").SetDefault(false);
    result.Add<TFSBoolean>("mark_all_read", "Делать вид, что сообщения пользователя читают").SetDefault(false);
    result.Add<TFSBoolean>("persist_entered_states", "Записывать все посещенные ноды (даже skip action) в историю").SetDefault(false);
    result.Add<TFSBoolean>("is_static", "Статичный чат (только чтение)").SetDefault(false);
    result.Add<TFSBoolean>("is_common", "Раздавать чат на всех").SetDefault(false);
    result.Add<TFSNumeric>("common_instant", "При раздаче чата на всех, в какой timestamp считать его отправленным");
    result.Add<TFSStructure>("common_sharding_policy", "Настройка шардирования для раздачи общих чатов").SetStructure<NDrive::TScheme>(TObjectSharding::GetScheme());
    result.Add<TFSBoolean>("is_pinned", "Пинить чат наверху").SetDefault(false);
    result.Add<TFSJson>("auditory_condition", "Аудитория для общих чатов");
    result.Add<TFSString>("robot_name", "Имя роботного пользователя");
    result.Add<TFSString>("robot_id", "Идентификатор роботного пользователя");
    result.Add<TFSVariants>("locale", "Локаль").InitVariants<ELocalization>();
    return result;
}

void TNativeChatMessage::DoSerializeToJson(NJson::TJsonValue& result) const {
    NJson::MergeJson(ChatMessage.SerializeToJson(), result);
    result["topic"] = Topic;
}

bool TNativeChatMessage::DoDeserializeFromJson(const NJson::TJsonValue& data) {
    return NJson::ParseField(data["topic"], Topic) && ChatMessage.DeserializeBasicsFromJson(data);
}

TNativeChatNotifier::TNativeChatNotifier(const TChatRobotNotifierConfig& config)
    : NDrive::INotifier(config)
    , Config(config)
{
}

TNativeChatNotifier::~TNativeChatNotifier() {
    NDrive::INotifier::Stop();
}

void TNativeChatNotifier::DoStart(const IServerBase* server) {
    auto impl = server->GetAsPtrSafe<NDrive::IServer>();
    if (impl) {
        EnsureInitialized(impl);
    }
}

void TNativeChatNotifier::DoStop() {
}

NJson::TJsonValue TNativeChatNotifier::GetDescriptor(const TString& /*userId*/, const TString& topic) const {
    NJson::TJsonValue result;
    result["id"] = Config.GetChatId() + (topic ? ("." + topic) : "");
    result["name"] = Config.GetTitle();
    if (Config.GetIcon()) {
        result["icon"] = Config.GetIcon();
    }
    NJson::InsertField(result, "styles", Config.GetTheme().GetStyles());
    return result;
}

bool TNativeChatNotifier::EnsureInitialized(const NDrive::IServer* server) const {
    if (!IsRobotInitialized) {
        TWriteGuard wg(MutexInit);
        if (!IsRobotInitialized && !server->RegisterChatRobot(Config)) {
            return false;
        }
        IsRobotInitialized = true;
    }
    return true;
}

TNativeChatNotifier::TMessage::TPtr TNativeChatNotifier::ConstructMessage(const NJson::TJsonValue& messageData) const {
    auto message = MakeAtomicShared<TNativeChatMessage>();
    if (!message->DeserializeFromJson(messageData)) {
        return nullptr;
    }
    return message;
}

NDrive::INotifier::TResult::TPtr TNativeChatNotifier::DoNotify(const TMessage& message, const TContext& context) const {
    auto server = context.GetServer()->GetAsPtrSafe<NDrive::IServer>();
    if (!server) {
        return new TResult("there is no server");
    }

    if (!EnsureInitialized(server)) {
        return new TResult("could not lazy initialize chat robot");
    }

    TReadGuard rg(MutexInit);
    auto chatMessage = message.GetAs<TNativeChatMessage>();
    if (!chatMessage) {
        return new TResult("message is not TNativeChatMessage");
    }

    auto bot = server->GetChatRobot(Config.GetChatId());

    auto session = bot->BuildChatEngineSession();
    for (auto&& recipient : context.GetRecipients()) {
        auto mutableMessage = chatMessage->GetChatMessage();
        if (!bot->SendArbitraryMessage(recipient.GetUserId(), chatMessage->GetTopic(), "robot-frontend-notify", mutableMessage, session)) {
            return new TResult("could not send messages");
        }
    }

    if (!session.Commit()) {
        return new TResult("could not commit messages session");
    }

    return nullptr;
}
