#include "push.h"

#include <drive/backend/logging/events.h>

#include <kernel/reqid/reqid.h>

#include <library/cpp/mediator/global_notifications/system_status.h>
#include <library/cpp/string_utils/quote/quote.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/unistat/cache.h>
#include <rtline/util/network/neh.h>
#include <rtline/util/network/neh_request.h>

#include <util/stream/file.h>

void TPushNotifier::DoStart(const IServerBase* /*server*/) {
    CHECK_WITH_LOG(!Agent);
    ReaskConfig.SetGlobalTimeout(TDuration::Seconds(10)).SetMaxAttempts(1);
    AD = new TAsyncDelivery;
    AD->Start(1, 8);
    Agent.Reset(new NNeh::THttpClient(AD));
    Agent->RegisterSource("push", Config.GetHost(), Config.GetPort(), ReaskConfig, Config.GetIsHttps());
}

NDrive::INotifier::TResult::TPtr TPushNotifier::DoNotify(const TMessage& message, const TContext& context) const {
    if (!context.GetRecipients()) {
        return new TResult("Empty recipients list");
    }

    const TRecipients& recipients = context.GetRecipients();

    bool hasErrors = false;
    for (size_t i = 0; i < recipients.size();) {
        const size_t endIndex = Min(recipients.size(), i + Config.GetPackSize());
        auto response = NotifyImpl(message, recipients.begin() + i,
                                   recipients.begin() + endIndex);
        hasErrors |= (response.Code() != 200);
        if (endIndex < recipients.size()) {
            Sleep(Config.GetPacksInterval());
        }
        i = endIndex;
    }
    return new TResult(hasErrors ? "Failed request" : "");
}

NUtil::THttpReply TPushNotifier::NotifyImpl(const TMessage& message,
                                            TRecipients::const_iterator recipientsBegin,
                                            TRecipients::const_iterator recipientsEnd) const {
    CHECK_WITH_LOG(!!Agent);
    TString reqid = ReqIdGenerate("DRIVE");
    NNeh::THttpRequest request;
    request.SetCgiData("token=" + Config.GetAuthToken() + "&event=" + Config.GetEventType() + "&reqid=" + reqid);

    NJson::TJsonValue jsonPost;
    NJson::TJsonValue& recipientsJson = jsonPost.InsertValue("recipients", NJson::JSON_ARRAY);
    NJson::TJsonValue userIds;
    for (auto r = recipientsBegin; r != recipientsEnd; ++r) {
        recipientsJson.AppendValue(r->GetUid());
        userIds.AppendValue(r->GetUserId());
    }
    NJson::TJsonValue& payload = jsonPost.InsertValue("payload", NJson::JSON_MAP);
    if (message.HasAdditionalInfo()) {
        payload.InsertValue("details", message.GetAdditionalInfo());
    }
    payload["body"] = message.GetBody();
    payload["sound"] = "default";
    payload["icon"] = "push";
    payload["title"] = message.GetTitle();

    NJson::TJsonValue& repack = jsonPost.InsertValue("repack", NJson::JSON_MAP);
    {
        NJson::TJsonValue alert;
        if (!message.GetTitle().empty()) {
            alert["title"] = message.GetTitle();
            alert["body"] = message.GetBody();
        } else {
            alert = message.GetBody();
        }
        NJson::TJsonValue& aps = repack.InsertValue("apns", NJson::JSON_MAP).InsertValue("aps", NJson::JSON_MAP);
        aps["sound"] = "default";
        aps["badge"] = 1;
        aps["alert"] = std::move(alert);
        if (Config.IsContentAvailable()) {
            aps["content-available"] = 1;
        }
        if (Config.IsMutableContent()) {
            aps["mutable-content"] = 1;
        }
    }

    {
        NJson::TJsonValue& fcm = repack.InsertValue("fcm", NJson::JSON_MAP);
        fcm["priority"] = "high";
        repack.InsertValue("hms", fcm);
    }

    if (!Config.GetAppNames().empty()) {
        NJson::TJsonValue& subscriptions = jsonPost.InsertValue("subscriptions", NJson::JSON_ARRAY);
        NJson::TJsonValue appsArray = NJson::JSON_ARRAY;
        for (auto&& element : Config.GetAppNames()) {
            appsArray.AppendValue(element);
        }
        NJson::TJsonValue appFilter;
        appFilter["app"] = std::move(appsArray);
        subscriptions.AppendValue(std::move(appFilter));
    }
    if (Config.IsContentAvailable() || Config.IsMutableContent()) {
        NJson::TJsonValue& yamp = payload.InsertValue("yamp", NJson::JSON_MAP);
        yamp["b"] = false;
        bool alwaysFillYampTitle = true;
        if (const auto& title = message.GetTitle(); title || alwaysFillYampTitle) {
            yamp["d"]["e"] = title;
        }
        if (const auto& body = message.GetBody()) {
            yamp["d"]["g"] = body;
        }
    }

    request.SetUri("/" + Config.GetUri()).SetRequestType("POST").SetPostData(jsonPost.GetStringRobust());
    NDrive::TEventLog::Log("PushRequest", NJson::TMapBuilder
        ("host", Config.GetHost())
        ("event_type", Config.GetEventType())
        ("name", message.GetName())
        ("reqid", reqid)
        ("data", jsonPost)
    );
    auto tgResult = Agent->SendMessageSync(request, Now() + TDuration::Seconds(10));
    auto response = tgResult.Serialize();
    auto transitId = tgResult.GetHeaders().FindHeader("TransitID");
    {
        response["reqid"] = reqid;
    }
    NDrive::TEventLog::Log("PushResponse", response);

    if (tgResult.Code() == 200) {
        TUnistatSignalsCache::SignalAdd("frontend-push", "success", 1);
        NDrive::TEventLog::Log("PushSent", NJson::TMapBuilder
            ("name", message.GetName())
            ("body", message.GetBody())
            ("title", message.GetTitle())
            ("tid", message.GetTransitId())
            ("reqid", reqid)
            ("transit_id", transitId ? NJson::ToJson(transitId->Value()) : NJson::JSON_NULL)
            ("user_ids", std::move(userIds))
        );
    } else {
        TUnistatSignalsCache::SignalAdd("frontend-push", "fail", 1);
    }
    TUnistatSignalsCache::SignalAdd("frontend-push", "codes-" + ::ToString(tgResult.Code()), 1);
    TUnistatSignalsCache::SignalAdd("frontend-push-" + NotifierName, "codes-" + ::ToString(tgResult.Code()), 1);
    return tgResult;
}

void TPushNotifier::SubscribeUser(const TString& appName, const TString& platform, const TString& passportUid, const TString& uuid, const TString& pushToken, THolder<NNeh::THttpAsyncReport::ICallback>&& callback) const {
    CHECK_WITH_LOG(!!Agent);
    NNeh::THttpRequest request;

    NJson::TJsonValue postData;
    postData["push_token"] = pushToken;

    request.SetUri("/" + Config.GetSubscriptionUri()).SetRequestType("POST").SetPostData(postData.GetStringRobust());
    request.SetCgiData("token=" + Config.GetSubscriptionToken() + "&service=" + Config.GetServiceName() + "&app_name=" + appName + "&platform=" + platform + "&user=" + passportUid + "&uuid=" + uuid + "&push_token=" + pushToken);

    Agent->Send(request, Now() + TDuration::Seconds(10), std::move(callback));
}

TPushNotifier::TPushNotifier(const TPushNotificationsConfig& config)
    : NDrive::INotifier(config)
    , Config(config)
{
}

TPushNotifier::~TPushNotifier() {
    NDrive::INotifier::Stop();
}

void TPushNotifier::DoStop() {
    if (!!AD) {
        AD->Stop();
    }
}

TPushNotificationsConfig::TFactory::TRegistrator<TPushNotificationsConfig> TPushNotificationsConfig::Registrator("push");

void TPushNotificationsConfig::DoInit(const TYandexConfig::Section* section) {
    Host = section->GetDirectives().Value("Host", Host);
    AssertCorrectConfig(!!Host, "Incorrect host in TPushNotificationsConfig");
    Port = section->GetDirectives().Value<ui16>("Port", Port);
    IsHttps = section->GetDirectives().Value<bool>("IsHttps", IsHttps);
    Uri = section->GetDirectives().Value("Uri", Uri);
    ContentAvailable = section->GetDirectives().Value("ContentAvailable", ContentAvailable);
    MutableContent = section->GetDirectives().Value("MutableContent", MutableContent);
    AssertCorrectConfig(!!Uri, "Incorrect uri in TPushNotificationsConfig");

    EventType = section->GetDirectives().Value("EventType", EventType);

    TokenPath = section->GetDirectives().Value("TokenPath", TokenPath);
    if (TokenPath.Exists()) {
        AssertCorrectConfig(TokenPath.Exists(), "Can't read token info from " + TokenPath.GetPath());
        AuthToken = Strip(TFileInput(TokenPath).ReadAll());
    } else {
        AuthToken = section->GetDirectives().Value<TString>("AuthToken");
    }
    AssertCorrectConfig(!!AuthToken, "no 'AuthToken'");

    SubscriptionTokenPath = section->GetDirectives().Value("SubscriptionTokenPath", SubscriptionTokenPath);
    if (SubscriptionTokenPath.Exists()) {
        SubscriptionToken = Strip(TFileInput(SubscriptionTokenPath).ReadAll());
    }

    PackSize = section->GetDirectives().Value("PackSize", PackSize);
    PacksInterval = section->GetDirectives().Value("PacksInterval", PacksInterval);
}

NDrive::INotifier::TPtr TPushNotificationsConfig::Construct() const {
    return new TPushNotifier(*this);
}

bool TPushNotificationsConfig::DeserializeFromJson(const NJson::TJsonValue& info, TMessagesCollector& errors) {
    JREAD_STRING_OPT(info, "host", Host);
    JREAD_UINT_OPT(info, "port", Port);
    JREAD_BOOL_OPT(info, "is_https", IsHttps);
    JREAD_BOOL_OPT(info, "is_content_available", ContentAvailable);
    JREAD_BOOL_OPT(info, "is_mutable_content", MutableContent);
    JREAD_STRING_OPT(info, "uri", Uri);
    JREAD_STRING_OPT(info, "auth_token", AuthToken);
    JREAD_STRING_OPT(info, "event_type", EventType);
    JREAD_STRING_OPT(info, "service_name", ServiceName);
    JREAD_STRING_OPT(info, "subscription_uri", SubscriptionUri);
    JREAD_STRING_OPT(info, "subscription_token", SubscriptionToken);
    JREAD_UINT_OPT(info, "pack_size", PackSize);
    JREAD_DURATION_OPT(info, "packs_interval", PacksInterval);
    if (!TJsonProcessor::ReadContainer(info, "app_names", AppNames)) {
        return false;
    }
    return TBase::DeserializeFromJson(info, errors);
}

NJson::TJsonValue TPushNotificationsConfig::SerializeToJson() const {
    NJson::TJsonValue result = TBase::SerializeToJson();
    JWRITE(result, "host", Host);
    JWRITE(result, "port", Port);
    JWRITE(result, "is_https", IsHttps);
    JWRITE(result, "uri", Uri);
    JWRITE(result, "auth_token", AuthToken);
    JWRITE(result, "event_type", EventType);
    JWRITE(result, "service_name", ServiceName);
    JWRITE(result, "subscription_uri", SubscriptionUri);
    JWRITE(result, "subscription_token", SubscriptionToken);
    JWRITE(result, "is_content_available", ContentAvailable);
    JWRITE(result, "is_mutable_content", MutableContent);
    JWRITE(result, "pack_size", PackSize);
    JWRITE_DURATION(result, "packs_interval", PacksInterval);
    TJsonProcessor::WriteContainerArray(result, "app_names", AppNames);
    return result;
}

NDrive::TScheme TPushNotificationsConfig::GetScheme(const IServerBase& server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSString>("host").SetDefault(Host);
    result.Add<TFSNumeric>("port").SetDefault(Port);
    result.Add<TFSBoolean>("is_https").SetDefault(IsHttps);
    result.Add<TFSBoolean>("is_content_available").SetDefault(ContentAvailable);
    result.Add<TFSBoolean>("is_mutable_content").SetDefault(MutableContent);
    result.Add<TFSString>("uri").SetDefault(Uri);
    result.Add<TFSString>("auth_token").SetDefault(AuthToken);
    result.Add<TFSString>("event_type").SetDefault(EventType);
    result.Add<TFSString>("service_name").SetDefault(ServiceName);
    result.Add<TFSString>("subscription_uri").SetDefault(SubscriptionUri);
    result.Add<TFSString>("subscription_token").SetDefault(SubscriptionToken);
    result.Add<TFSNumeric>("pack_size").SetDefault(PackSize);
    result.Add<TFSDuration>("packs_interval").SetDefault(PacksInterval);
    result.Add<TFSArray>("app_names").SetElement<TFSString>();
    return result;
}

void TPushNotificationsConfig::DoToString(IOutputStream& os) const {
    os << "EventType: " << EventType << Endl;
    os << "Host: " << Host << Endl;
    os << "Port: " << Port << Endl;
    os << "IsHttps: " << IsHttps << Endl;
    os << "Uri: " << Uri << Endl;
    os << "TokenPath: " << TokenPath << Endl;

    os << "ServiceName: " << ServiceName << Endl;
    os << "SubscriptionUri: " << SubscriptionUri << Endl;
    os << "SubscriptionTokenPath: " << SubscriptionTokenPath << Endl;
    os << "ContentAvailable: " << ContentAvailable << Endl;
    os << "MutableContent: " << MutableContent << Endl;

    os << "PackSize: " << PackSize << Endl;
    os << "PacksInterval: " << PacksInterval << Endl;

    os << "AppNames: " << JoinStrings(AppNames, ",") << Endl;
}
