#include "sms.h"

#include <drive/backend/logging/events.h>

#include <library/cpp/mediator/global_notifications/system_status.h>
#include <library/cpp/string_utils/base64/base64.h>
#include <library/cpp/string_utils/quote/quote.h>
#include <library/cpp/xml/sax/simple.h>

#include <rtline/library/async_proxy/async_delivery.h>
#include <rtline/library/json/builder.h>
#include <rtline/library/unistat/cache.h>
#include <rtline/util/network/neh.h>

#include <util/stream/file.h>
#include <util/string/cast.h>
#include <util/string/vector.h>

namespace {
    class TYaSmsResponse {
        R_FIELD(TString, SmsId);
        R_FIELD(TString, ErrorMessage);
        R_FIELD(TString, ErrorCode);

    public:
        bool IsSucceed() const {
            return !SmsId.empty();
        }
    };

    class TYaSmsResponseParser : public NXml::ISimpleSaxHandler {
    public:
        void OnStartElement(const TStringBuf& name, const TAttr* attrs, size_t count) override {
            if (name == "message-sent") {
                for (size_t i = 0; i < count; ++i) {
                    const TAttr& attr = attrs[i];
                    if ("id" == attr.Name) {
                        Result.SetSmsId(ToString(attr.Value));
                        break;
                    }
                }
            } else if (name == "error") {
                State = EState::ErrorMessage;
            } else if (name == "errorcode") {
                State = EState::ErrorCode;
            }
        }

        void OnEndElement(const TStringBuf& name) override {
            if (name == "error" || name == "errorcode") {
                State = EState::Unbound;
            }
        }

        void OnText(const TStringBuf& data) override {
            switch (State) {
                case EState::ErrorMessage:
                    Result.SetErrorMessage(ToString(data));
                    break;
                case EState::ErrorCode:
                    Result.SetErrorCode(ToString(data));
                    break;
                default:
                    break;
            }
        }

        THolder<TYaSmsResponse> GetResult() const {
            if (Result.GetSmsId() || Result.GetErrorMessage() || Result.GetErrorCode()) {
                return MakeHolder<TYaSmsResponse>(Result);
            } else {
                return nullptr;
            }
        }

    private:
        enum class EState : char {
            Unbound = 0,
            ErrorMessage,
            ErrorCode
        };
        TYaSmsResponse Result;
        EState State = EState::Unbound;
    };

    THolder<TYaSmsResponse> ParseResponse(const TString& content) {
        TYaSmsResponseParser parser;
        TStringInput si(content);
        NXml::Parse(si, &parser);
        return parser.GetResult();
    }
}  // namespace

TSMSNotificationsConfig::TFactory::TRegistrator<TSMSNotificationsConfig> TSMSNotificationsConfig::Registrator("sms");

void TSMSNotificationsConfig::DoInit(const TYandexConfig::Section* section) {
    const auto& directives = section->GetDirectives();
    DefaultRecipients = StringSplitter(
        directives.Value("DefaultRecipients", JoinStrings(DefaultRecipients, ","))
    ).SplitBySet(" ,").SkipEmpty();
    AssertCorrectConfig(section->GetDirectives().GetValue("Host", Host), "Incorrect parameter 'Host' for sms notifications");
    Port = section->GetDirectives().Value<ui16>("Port", Port);
    IsHttps = section->GetDirectives().Value<bool>("IsHttps", IsHttps);
    SelfTvmId = section->GetDirectives().Value<ui32>("SelfTvmId", SelfTvmId);
    DestinationTvmId = section->GetDirectives().Value<ui32>("DestinationTvmId", DestinationTvmId);
    AssertCorrectConfig(section->GetDirectives().GetValue("Sender", Sender),
                        "Incorrect parameter 'Sender' for sms notifications");
    AssertCorrectConfig(section->GetDirectives().GetValue("Route", Route),
                        "Incorrect parameter 'Route' for sms notifications");
    MaxInFlight = section->GetDirectives().Value<ui32>("MaxInFlight", MaxInFlight);
    ResponseTimeout = section->GetDirectives().Value<TDuration>("ResponseTimeout", ResponseTimeout);
}

void TSMSNotificationsConfig::DoToString(IOutputStream& os) const {
    os << "DefaultRecipients: " << JoinStrings(DefaultRecipients, ",") << Endl;
    os << "Host: " << Host << Endl;
    os << "Port: " << Port << Endl;
    os << "IsHttps: " << IsHttps << Endl;
    os << "SelfTvmId: " << SelfTvmId << Endl;
    os << "DestinationTvmId: " << DestinationTvmId << Endl;
    os << "Sender: " << Sender << Endl;
    os << "Route: " << Route << Endl;
    os << "MaxInFlight: " << MaxInFlight << Endl;
    os << "ResponseTimeout: " << ResponseTimeout << Endl;
}

NDrive::INotifier::TPtr TSMSNotificationsConfig::Construct() const {
    return new TSMSNotifier(*this);
}

bool TSMSNotificationsConfig::DeserializeFromJson(const NJson::TJsonValue& info, TMessagesCollector& errors) {
    JREAD_STRING_OPT(info, "host", Host);
    JREAD_UINT_OPT(info, "port", Port);
    JREAD_BOOL_OPT(info, "is_https", IsHttps);
    JREAD_STRING_OPT(info, "sender", Sender);
    JREAD_STRING_OPT(info, "route", Route);
    JREAD_STRING_OPT(info, "host", Host);
    JREAD_UINT_OPT(info, "self_tvm_id", SelfTvmId);
    JREAD_UINT_OPT(info, "destination_tvm_id", DestinationTvmId);
    JREAD_UINT_OPT(info, "max_in_fly", MaxInFlight);
    JREAD_DURATION_OPT(info, "response_timeout", ResponseTimeout);
    return TBase::DeserializeFromJson(info, errors)
        && NJson::ParseField(info["default_recipients"], DefaultRecipients, errors)
    ;
}

NJson::TJsonValue TSMSNotificationsConfig::SerializeToJson() const {
    NJson::TJsonValue result = TBase::SerializeToJson();
    NJson::InsertField(result, "default_recipients", DefaultRecipients);
    JWRITE(result, "host", Host);
    JWRITE(result, "port", Port);
    JWRITE(result, "is_https", IsHttps);
    JWRITE(result, "sender", Sender);
    JWRITE(result, "route", Route);
    JWRITE(result, "self_tvm_id", SelfTvmId);
    JWRITE(result, "destination_tvm_id", DestinationTvmId);
    JWRITE(result, "max_in_fly", MaxInFlight);
    JWRITE_DURATION(result, "response_timeout", ResponseTimeout);
    return result;
}

NDrive::TScheme TSMSNotificationsConfig::GetScheme(const IServerBase& server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSArray>("default_recipients").SetElement<TFSString>();
    result.Add<TFSString>("host").SetDefault(Host);
    result.Add<TFSNumeric>("port").SetDefault(Port);
    result.Add<TFSBoolean>("is_https").SetDefault(IsHttps);
    result.Add<TFSString>("sender").SetDefault(Sender);
    result.Add<TFSString>("route").SetDefault(Route);
    result.Add<TFSNumeric>("self_tvm_id").SetDefault(SelfTvmId);
    result.Add<TFSNumeric>("destination_tvm_id").SetDefault(DestinationTvmId);
    result.Add<TFSNumeric>("max_in_fly").SetDefault(MaxInFlight);
    result.Add<TFSDuration>("response_timeout").SetDefault(ResponseTimeout);
    return result;
}

TSMSNotifier::TSMSNotifier(const TSMSNotificationsConfig& config)
    : NDrive::INotifier(config)
    , Config(config)
{
}

TSMSNotifier::~TSMSNotifier() {
    NDrive::INotifier::Stop();
}

void TSMSNotifier::DoStart(const IServerBase* /*server*/) {
    CHECK_WITH_LOG(!Agent);
    AD = new TAsyncDelivery;
    AD->Start(1, 1);
    Agent.Reset(new NNeh::THttpClient(AD));

    NSimpleMeta::TConfig reaskConfig;
    reaskConfig.SetGlobalTimeout(Config.GetResponseTimeout()).SetMaxAttempts(1);
    Agent->RegisterSource("sms", Config.GetHost(), Config.GetPort(), reaskConfig, Config.GetIsHttps());
}

void TSMSNotifier::DoStop() {
    if (AD) {
        AD->Stop();
    }
}

NDrive::INotifier::TResult::TPtr TSMSNotifier::DoNotify(const TMessage& message,
                                                        const TContext& context) const {
    CHECK_WITH_LOG(Agent);
    if (GetSelfTvmId() && !Tvm) {
        ERROR_LOG << "Missing tvm client for: " << Config.GetName() << Endl;
    }

    NNeh::THttpRequest request;
    request.SetUri("/sendsms");
    const TString baseCgi = TStringBuilder() << "sender=" << Config.GetSender()
                                             << "&text=" << CGIEscapeRet(message.GetBody())
                                             << "&route=" << Config.GetRoute() << "&utf8=1";
    if (Tvm) {
        request.AddHeader("X-Ya-Service-Ticket",
                          Tvm->GetServiceTicketFor(Config.GetDestinationTvmId()));
    }

    TMap<TString, NNeh::THttpRequest> requests;
    auto recipients = context.GetRecipients();
    if (recipients.empty()) {
        for (auto&& phone : Config.GetDefaultRecipients()) {
            recipients.emplace_back().SetPhone(phone);
        }
    }
    for (size_t i = 0; i < recipients.size(); ++i) {
        const auto& r = recipients[i];
        TStringBuilder cgi = TStringBuilder() << baseCgi;
        if (r.GetPhone()) {
            cgi << "&phone=" << r.GetPhone();
        } else {
            cgi << "&uid=" << r.GetUid();
        }
        request.SetCgiData(cgi);
        requests.emplace(r.GetUid(), request);
    }

    auto responses = Agent->SendPack(requests, TInstant::Max(), Config.GetMaxInFlight(),
                                     Config.GetResponseTimeout());
    bool hasErrors = false;
    NJson::TJsonValue errors = NJson::JSON_ARRAY;
    for (const auto& it : responses) {
        const TString& passportUid = it.first;
        const NNeh::THttpReply& resp = it.second;
        THolder<TYaSmsResponse> respInfo;
        if (resp.Code() == 200 &&
                (respInfo = ParseResponse(resp.Content())) &&
                respInfo->IsSucceed()) {
            NDrive::TEventLog::Log("SendSms", NJson::TMapBuilder("passport_uid", passportUid)
                                                                   ("text", message.GetBody())
                                                                   ("sms_id", respInfo->GetSmsId()));
            TUnistatSignalsCache::SignalAdd("frontend-sms", "success", 1);
        } else {
            NJson::TJsonValue error = NJson::TMapBuilder
                ("request", request.GetRequest())
                ("response_code", resp.Code())
                ("response", resp.Content())
                ("response_error", resp.ErrorMessage())
            ;
            NDrive::TEventLog::Log("SendSmsError", error);
            errors.AppendValue(std::move(error));
            TUnistatSignalsCache::SignalAdd("frontend-sms", "fail", 1);
            hasErrors = true;
        }
    }

    return new TResult(hasErrors ? "Failed request" : "", errors);
}

TMaybe<ui32> TSMSNotifier::GetSelfTvmId() const {
    return Config.GetSelfTvmId();
}

void TSMSNotifier::SetTvmClient(TAtomicSharedPtr<NTvmAuth::TTvmClient> tvmClient) {
    Tvm = tvmClient;
}
