#include "notifier.h"

#include <drive/backend/logging/events.h>

#include <aws/sns/model/PublishRequest.h>
#include <aws/sns/model/PublishResult.h>

#include <rtline/library/json/builder.h>

template <>
NJson::TJsonValue NJson::ToJson(const Aws::SNS::Model::PublishRequest& object) {
    NJson::TJsonValue result;
    if (object.MessageHasBeenSet()) {
        result.InsertValue("message", object.GetMessage());
    }
    if (object.PhoneNumberHasBeenSet()) {
        result.InsertValue("phone_number", object.GetPhoneNumber());
    }
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const Aws::SNS::Model::PublishOutcome& object) {
    NJson::TJsonValue result;
    if (object.IsSuccess()) {
        const auto& r = object.GetResult();
        result.InsertValue("message_id", r.GetMessageId());
        result.InsertValue("seq_number", r.GetSequenceNumber());
        result.InsertValue("request_id", r.GetResponseMetadata().GetRequestId());
    } else {
        const auto& error = object.GetError();
        result.InsertValue("error_message", error.GetMessage());
        result.InsertValue("request_id", error.GetRequestId());
    }
    return result;
}

TSnsNotifierConfig::TSnsNotifierConfig()
    : AwsGuard(InitializeAws())
{
}

NDrive::INotifier::TPtr TSnsNotifierConfig::Construct() const {
    return MakeAtomicShared<TSnsNotifier>(*this);
}

void TSnsNotifierConfig::DoInit(const TYandexConfig::Section* section) {
    const auto& directives = section->GetDirectives();
    Region = directives.Value("Region", Region);
    AccessKeyId = directives.Value("AccessKeyId", AccessKeyId);
    AccessSecretKey = directives.Value("AccessSecretKey", AccessSecretKey);
}

void TSnsNotifierConfig::DoToString(IOutputStream& os) const {
    os << "Region: " << Region << Endl;
    os << "AccessKeyId: " << AccessKeyId << Endl;
}

TSnsNotifier::TSnsNotifier(const TSnsNotifierConfig& config)
    : TBase(config)
    , Credentials(config.GetAccessKeyId(), config.GetAccessSecretKey())
{
    ClientConfiguration.region = config.GetRegion();
    ClientConfiguration.verifySSL = false;
}

TSnsNotifier::TResult::TPtr TSnsNotifier::DoNotify(const TMessage& message, const TContext& context) const {
    auto result = MakeAtomicShared<TResult>(TString{});

    Aws::SNS::Model::PublishRequest request;
    request.SetMessage(message.GetBody());
    for (auto&& recipient : context.GetRecipients()) {
        request.SetPhoneNumber(recipient.GetPhone());
        NDrive::TEventLog::Log("SnsPublishRequest", NJson::ToJson(request));
        auto outcome = Yensured(Client)->Publish(request);
        NDrive::TEventLog::Log("SnsPublishOutcome", NJson::ToJson(outcome));
        if (!outcome.IsSuccess()) {
            result->SetErrorDetails("incomplete");
            result->MutableErrorDetails().AppendValue(NJson::TMapBuilder
                ("recipient", NJson::ToJson(recipient))
                ("error", outcome.GetError().GetMessage())
            );
        } else {
            result->SetTransitId(outcome.GetResult().GetMessageId());
        }
    }
    return result;
}

void TSnsNotifier::DoStart(const IServerBase* /*server*/) {
    if (!Client) {
        Client = std::make_unique<Aws::SNS::SNSClient>(Credentials, ClientConfiguration);
    }
}

void TSnsNotifier::DoStop() {
    Client.reset();
}

TSnsNotifierConfig::TFactory::TRegistrator<TSnsNotifierConfig> TSnsNotifierConfig::Registrator("sns");
