#include "startrek.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/logging/events.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/builder.h>
#include <rtline/library/json/cast.h>
#include <rtline/library/json/parse.h>

#include <util/string/cast.h>

TStartrekNotificationsConfig::TFactory::TRegistrator<TStartrekNotificationsConfig> TStartrekNotificationsConfig::Registrator("startrek");

TStartrekNotificationsConfig::TStartrekNotificationsConfig()
    : TBase()
{
    SetProtectedMessageLengthLimit(0);
}

bool TStartrekNotificationsConfig::DeserializeFromJson(const NJson::TJsonValue& info, TMessagesCollector& errors) {
    return NJson::ParseField(info["duplicate_check_policy"], NJson::Stringify(DuplicateCheckPolicy), errors) &&
           NJson::ParseField(info["duplicate_action_policy"], NJson::Stringify(DuplicateActionPolicy), errors) &&
           NJson::ParseField(info["comment_if_duplicate"], CommentIfDuplicate, errors) &&
           TBase::DeserializeFromJson(info, errors);
}

NJson::TJsonValue TStartrekNotificationsConfig::SerializeToJson() const {
    auto result = TBase::SerializeToJson();
    NJson::InsertField(result, "duplicate_check_policy", NJson::Stringify(DuplicateCheckPolicy));
    NJson::InsertField(result, "duplicate_action_policy", NJson::Stringify(DuplicateActionPolicy));
    NJson::InsertField(result, "comment_if_duplicate", CommentIfDuplicate);
    return result;
}

NDrive::TScheme TStartrekNotificationsConfig::GetScheme(const IServerBase& server) const {
    auto scheme = TBase::GetScheme(server);
    scheme.Add<TFSVariants>("duplicate_check_policy", "Способ поиска дубликатов").InitVariants<ETicketDuplicateCheckPolicy>();
    scheme.Add<TFSVariants>("duplicate_action_policy", "Действие в случае дубликата").InitVariants<ETicketDuplicateActionPolicy>();
    scheme.Add<TFSString>("comment_if_duplicate", "Комментарий при дублировании");
    return scheme;
}

void TStartrekNotificationsConfig::DoInit(const TYandexConfig::Section* section) {
    {
        TString duplicateCheckPolicyStr;
        duplicateCheckPolicyStr = section->GetDirectives().Value<TString>("DuplicateCheckPolicy", "");
        TryFromString(duplicateCheckPolicyStr, DuplicateCheckPolicy);
    }
    {
        TString duplicateActionPolicyStr;
        duplicateActionPolicyStr = section->GetDirectives().Value<TString>("DuplicateActionPolicy", "");
        TryFromString(duplicateActionPolicyStr, DuplicateActionPolicy);
    }
    CommentIfDuplicate = section->GetDirectives().Value("CommentIfDuplicate", CommentIfDuplicate);
}

void TStartrekNotificationsConfig::DoToString(IOutputStream& os) const {
    os << "DuplicateCheckPolicy: " << ::ToString(DuplicateCheckPolicy) << Endl;
    os << "DuplicateActionPolicy: " << ::ToString(DuplicateActionPolicy) << Endl;
    os << "CommentIfDuplicate: " << CommentIfDuplicate << Endl;
}

NDrive::INotifier::TPtr TStartrekNotificationsConfig::Construct() const {
    return MakeAtomicShared<TStartrekNotifier>(*this);
}

TStartrekNotifierResult::TStartrekNotifierResult(const TStartrekClient::TTicket& ticket)
    : TBase()
    , Ticket(ticket)
{
}

NJson::TJsonValue TStartrekNotifierResult::SerializeToJson() const {
    auto result = TBase::SerializeToJson();
    NJson::InsertField(result, "data", Ticket);
    return result;
}

void TStartrekMessage::DoSerializeToJson(NJson::TJsonValue& result) const {
    result.InsertValue("issue", GetIssue());
    result.InsertValue("comment", GetComment().SerializeToJson());
    result.InsertValue("update", GetAdditionalInfo());
    result.InsertValue("transition", GetTransition());
    result.InsertValue("attachments", NJson::ToJson(GetAttachments()));
}

bool TStartrekMessage::DoDeserializeFromJson(const NJson::TJsonValue& data) {
    NJson::TJsonValue json;
    {
        TString issue;
        if (!NJson::ParseField(data["issue"], issue)) {
            return false;
        }
        SetIssue(issue);
    }
    if (data["comment"].IsDefined()) {
        TComment comment;
        if (!comment.DeserializeFromJson(data["comment"])) {
            return false;
        }
        SetComment(comment);
    }
    if (data["update"].IsDefined()) {
        TTicket update;
        if (!update.DeserializeFromJson(data["update"])) {
            return false;
        }
        SetUpdate(update);
    }
    {
        TString transition;
        if (!NJson::ParseField(data["transition"], transition)) {
            return false;
        }
        SetTransition(transition);
    }
    {
        if (!NJson::ParseField(data["attachments"], Attachments)) {
            return false;
        }
    }
    return true;
}

TStartrekNotifier::TStartrekNotifier(const TStartrekNotificationsConfig& config)
    : NDrive::INotifier(config)
    , Config(config)
{
}

TStartrekNotifier::~TStartrekNotifier() {
    NDrive::INotifier::Stop();
}

TStartrekNotifier::TResults TStartrekNotifier::MultiLinesNotify(const TString& /*commonHeader*/, const TStartrekNotifier::TMessages& reportMessages, const TContext& context) const {
    TResults results;
    for (auto message : reportMessages) {
        if (message) {
            results.push_back(Notify(*message, context));
        }
    }
    return results;
}

void TStartrekNotifier::DoStart(const IServerBase* /*server*/) {
}

void TStartrekNotifier::DoStop() {
}

TStartrekNotifier::TMessage::TPtr TStartrekNotifier::ConstructMessage(const NJson::TJsonValue& messageData) const {
    auto message = MakeAtomicShared<TStartrekMessage>();
    if (!message->DeserializeFromJson(messageData)) {
        return nullptr;
    }
    return message;
}

NDrive::INotifier::TResult::TPtr TStartrekNotifier::DoNotify(const TMessage& message, const TContext& context) const {
    if (!context.GetServer() || !context.GetServer()->GetStartrekClient()) {
        return MakeAtomicShared<TStartrekResult>("No startrek client available");
    }

    auto startrekClientPtr = context.GetServer()->GetStartrekClient();

    const auto* startrekMessagePtr = message.GetAs<TStartrekMessage>();
    if (!startrekMessagePtr) {
        return MakeAtomicShared<TStartrekResult>("Message cannot be interpreted as startrek ticket update");
    }

    TStartrekAttachmentIds attachmentIds;
    TMessagesCollector errors;
    if (!UploadAttachments(startrekClientPtr, startrekMessagePtr, attachmentIds, errors)) {
        return MakeAtomicShared<TStartrekResult>("Error uploading attachments", errors.GetReport());
    }

    if (startrekMessagePtr->GetIssue()) {
        return ProcessExistingIssue(startrekClientPtr, startrekMessagePtr, attachmentIds);
    } else {
        return ProcessNewIssue(startrekClientPtr, startrekMessagePtr, attachmentIds);
    }
}

bool TStartrekNotifier::UploadAttachments(const TStartrekClient* startrekClientPtr, const TStartrekMessage* startrekMessagePtr, TStartrekAttachmentIds& attachmentIds, TMessagesCollector& errors) const {
    if (startrekMessagePtr->GetAttachments()) {
        for (auto&& [name, data]: startrekMessagePtr->GetAttachments()) {
            TStartrekClient::TAttachmentId attachmentId;
            NDrive::TEventLog::Log("StartrekUploadAttachments", NJson::TMapBuilder
                ("name", name)
                ("data_size", data.size())
            );
            bool success = startrekClientPtr->UploadAttachment(name, data, attachmentId, errors);
            NDrive::TEventLog::Log("StartrekUploadAttachmentsResult", NJson::TMapBuilder
                ("name", name)
                ("attachment_id", attachmentId)
                ("errors", errors.GetReport())
                ("success", success)
            );
            if (!success) {
                return false;
            }
            attachmentIds.push_back(std::move(attachmentId));
        }
    }
    return true;
}

NDrive::INotifier::TResult::TPtr TStartrekNotifier::ProcessExistingIssue(const TStartrekClient* startrekClientPtr, const TStartrekMessage* startrekMessagePtr, const TStartrekAttachmentIds& attachmentIds) const {
    if (startrekMessagePtr->GetComment()) {
        return AddIssueComment(startrekClientPtr, startrekMessagePtr->GetComment(), {startrekMessagePtr->GetIssue()}, attachmentIds);
    }

    TStartrekTicket update = startrekMessagePtr->GetUpdate();
    if (attachmentIds) {
        update.SetAdditionalContainerValue(TStartrekTicket::EContainerTicketField::AttachmentIds, attachmentIds, /* modifyExisting = */ true);
    }

    if (!update && !startrekMessagePtr->GetTransition()) {
        return MakeAtomicShared<TStartrekResult>("Either ticket update or transition must not be empty");
    }

    if (startrekMessagePtr->GetTransition()) {
        return ExecuteIssueTransition(startrekClientPtr, startrekMessagePtr, std::move(update));
    } else {
        return PatchExistingIssue(startrekClientPtr, startrekMessagePtr->GetIssue(), std::move(update));
    }
}

NDrive::INotifier::TResult::TPtr TStartrekNotifier::AddIssueComment(const TStartrekClient* startrekClientPtr, TStartrekComment comment, TVector<TString> issues, const TStartrekAttachmentIds& attachmentIds) const {
    for (auto&& attachmentId: attachmentIds) {
        comment.MutableAttachmentIds().push_back(attachmentId);
    }

    if (!comment) {
        return MakeAtomicShared<TStartrekResult>("Comment to add is empty");
    }

    TMessagesCollector errors;
    for (auto&& issue: issues) {
        NDrive::TEventLog::Log("StartrekAddIssueComment", NJson::TMapBuilder
            ("comment", comment.SerializeToJson())
            ("issue", issue)
        );
        bool success = startrekClientPtr->AddComment(issue, comment, errors);
        NDrive::TEventLog::Log("StartrekAddIssueCommentResult", NJson::TMapBuilder
            ("comment_id", comment.GetId())
            ("issue", issue)
            ("success", success)
            ("errors", errors.GetReport())
        );
        if (!success) {
            return MakeAtomicShared<TStartrekResult>("Error adding comment", errors.GetReport());
        }
    }

    return nullptr;
}

NDrive::INotifier::TResult::TPtr TStartrekNotifier::ExecuteIssueTransition(const TStartrekClient* startrekClientPtr, const TStartrekMessage* startrekMessagePtr, TStartrekTicket&& update) const {
    TMessagesCollector errors;
    NDrive::TEventLog::Log("StartrekExecuteIssureTransition", NJson::TMapBuilder
        ("issue", startrekMessagePtr->GetIssue())
        ("transition", startrekMessagePtr->GetTransition())
    );
    bool success = startrekClientPtr->ExecuteTransition(startrekMessagePtr->GetIssue(), startrekMessagePtr->GetTransition(), errors, update);
    NDrive::TEventLog::Log("StartrekExecuteIssureTransitionResult", NJson::TMapBuilder
        ("issue", startrekMessagePtr->GetIssue())
        ("errors", errors.GetReport())
        ("success", success)
    );
    if (!success) {
        return MakeAtomicShared<TStartrekResult>("Error executing transition", errors.GetReport());
    }
    return nullptr;
}

NDrive::INotifier::TResult::TPtr TStartrekNotifier::PatchExistingIssue(const TStartrekClient* startrekClientPtr, const TString& issue, TStartrekTicket&& update) const {
    TStartrekTicket result;
    TMessagesCollector errors;
    NDrive::TEventLog::Log("StartrekPatchExistingIssue", NJson::TMapBuilder
        ("issue", issue)
        ("update", update.SerializeToJson())
    );
    bool success = startrekClientPtr->PatchIssue(issue, update, result, errors);
    NDrive::TEventLog::Log("StartrekPatchExistingIssueResult", NJson::TMapBuilder
        ("issue", issue)
        ("errors", errors.GetReport())
        ("success", success)
    );
    if (!success) {
        return MakeAtomicShared<TStartrekResult>("Error updating issue", errors.GetReport());
    }
    return MakeAtomicShared<TStartrekResult>(result);
}

NDrive::INotifier::TResult::TPtr TStartrekNotifier::ProcessNewIssue(const TStartrekClient* startrekClientPtr, const TStartrekMessage* startrekMessagePtr, const TStartrekAttachmentIds& attachmentIds) const {
    if (startrekMessagePtr->GetTransition() || startrekMessagePtr->GetComment()) {
        return MakeAtomicShared<TStartrekResult>("Transition execution and commenting are supported for a created issue only");
    }

    TStartrekTicket content = startrekMessagePtr->GetUpdate();
    if (attachmentIds) {
        content.SetAdditionalContainerValue(TStartrekTicket::EContainerTicketField::AttachmentIds, attachmentIds, /* modifyExisting = */ true);
    }

    TString summary = content.GetSummary();
    TString queue = content.GetQueue();
    if (!summary || !queue) {
        return MakeAtomicShared<TStartrekResult>("Ticket must have summary and queue defined");
    }

    auto duplicateCheckPolicy = Config.GetDuplicateCheckPolicy();
    auto duplicates = CheckDuplicates(startrekClientPtr, summary, queue);
    if (!duplicates.Defined() &&
        (duplicateCheckPolicy == ETicketDuplicateCheckPolicy::SummaryRequired ||
         duplicateCheckPolicy == ETicketDuplicateCheckPolicy::SummaryAndQueueRequired)
    ) {
        return MakeAtomicShared<TStartrekResult>("Error checking issue duplicates");
    }

    const bool hasDuplicates = duplicates.Defined() && (!duplicates->empty());
    if (!hasDuplicates) {
        return CreateNewIssue(startrekClientPtr, std::move(content));
    }

    TResult::TPtr result;

    switch (Config.GetDuplicateActionPolicy()) {
    case ETicketDuplicateActionPolicy::CreateAndLink:
        {
            NJson::TJsonValue links(NJson::JSON_ARRAY);
            for (auto&& issue: *duplicates) {
                links.AppendValue(TStartrekLink(issue, TStartrekLink::ERelationshipType::Duplicates).SerializeToJson());
            }
            content.SetAdditionalValue(::ToString(TStartrekTicket::EContainerTicketField::Links), links);
        }
        result = CreateNewIssue(startrekClientPtr, std::move(content));
        break;

    case ETicketDuplicateActionPolicy::Create:
        result = CreateNewIssue(startrekClientPtr, std::move(content));
        break;

    case ETicketDuplicateActionPolicy::SkipAndComment:
        result = AddIssueComment(startrekClientPtr, TStartrekComment(Config.GetCommentIfDuplicate()), *duplicates, {});
        break;

    case ETicketDuplicateActionPolicy::Skip:
        break;
    }

    return result;
}

TMaybe<TVector<TString>> TStartrekNotifier::CheckDuplicates(const TStartrekClient* startrekClientPtr, const TString& summary, const TString& queue) const {
    auto duplicateCheckPolicy = Config.GetDuplicateCheckPolicy();
    if (duplicateCheckPolicy == ETicketDuplicateCheckPolicy::None) {
        return TVector<TString>();
    }

    TVector<TStartrekTicket> tickets;
    TMessagesCollector errors;
    if (!startrekClientPtr->SearchIssue(summary, tickets, errors)) {
        return {};
    }

    TVector<TString> duplicates;

    for (auto&& ticket: tickets) {
        switch (duplicateCheckPolicy) {
        case ETicketDuplicateCheckPolicy::Summary:
        case ETicketDuplicateCheckPolicy::SummaryRequired:
            if (summary == ticket.GetSummary()) {
                duplicates.push_back(ticket.GetKey());
            }
            break;
        case ETicketDuplicateCheckPolicy::SummaryAndQueue:
        case ETicketDuplicateCheckPolicy::SummaryAndQueueRequired:
            if (summary == ticket.GetSummary() && queue == ticket.GetQueue()) {
                duplicates.push_back(ticket.GetKey());
            }
            break;
        case ETicketDuplicateCheckPolicy::None:
            break;
        }
    }

    return duplicates;
}

NDrive::INotifier::TResult::TPtr TStartrekNotifier::CreateNewIssue(const TStartrekClient* startrekClientPtr, TStartrekTicket&& content) const {
    TStartrekTicket result;
    TMessagesCollector errors;
    NDrive::TEventLog::Log("StartrekCreateNewIssue", NJson::TMapBuilder
        ("content", content.SerializeToJson())
    );
    bool success = startrekClientPtr->CreateIssue(content, result, errors);
    NDrive::TEventLog::Log("StartrekCreateNewIssueResult", NJson::TMapBuilder
        ("result", result.SerializeToJson())
        ("errors", errors.GetReport())
        ("success", success)
    );
    if (!success) {
        return MakeAtomicShared<TStartrekResult>("Error creating issue", errors.GetReport());
    }
    return MakeAtomicShared<TStartrekResult>(result);
}

bool TStartrekNotifier::SendDocument(const TMessage& message, const TString& /* mimeType */, const TContext& context) const {
    // Custom message classes and corresponding processing to be implemented further
    TStartrekMessage startrekMessage;

    startrekMessage.SetName(message.GetName());
    startrekMessage.SetTransitId(message.GetTransitId());

    startrekMessage.SetIssue(message.GetHeader());
    startrekMessage.MutableAttachments().emplace(message.GetTitle(), message.GetBody());  // fine name and content
    if (message.GetAdditionalInfo().IsString()) {
        TString commentData = message.GetAdditionalInfo().GetString();  // comment data json string
        if (commentData) {
            // a new comment with the attachment will be added instead of an ordinary issue attachment
            startrekMessage.SetComment(commentData);
        }
    }

    auto result = Notify(startrekMessage, context);
    if (result && result->HasErrors()) {
        ERROR_LOG << "Error sending attachment: " << result->SerializeToJson().GetStringRobust() << Endl;
        return false;
    }

    return true;
}

bool TStartrekNotifier::SendPhoto(const TMessage& message, const TContext& context) const {
    return SendDocument(message, "image/jpeg", context);
}
