#pragma once

#include <drive/backend/abstract/notifier.h>

#include <drive/library/cpp/startrek/client.h>

#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/json/json_reader.h>

enum class ETicketDuplicateCheckPolicy {
    None /* "none" */,
    Summary /* "summary" */,
    SummaryRequired /* "summary_required" */,
    SummaryAndQueue /* "summary_and_queue" */,
    SummaryAndQueueRequired /* "summary_and_queue_required" */,
};

enum class ETicketDuplicateActionPolicy {
    Create /* "create" */,
    CreateAndLink /* "create_and_link" */,
    Skip /* "skip" */,
    SkipAndComment /* "skip_and_comment" */,
};

class TStartrekNotificationsConfig: public NDrive::INotifierConfig {
private:
    using TBase = NDrive::INotifierConfig;

public:
    TStartrekNotificationsConfig();

    virtual bool DeserializeFromJson(const NJson::TJsonValue& info, TMessagesCollector& errors) override;
    virtual NJson::TJsonValue SerializeToJson() const override;

    virtual NDrive::TScheme GetScheme(const IServerBase& server) const override;

    void DoInit(const TYandexConfig::Section* section) override;
    void DoToString(IOutputStream& os) const override;

    NDrive::INotifier::TPtr Construct() const override;

private:
    static TFactory::TRegistrator<TStartrekNotificationsConfig> Registrator;

    R_FIELD(ETicketDuplicateCheckPolicy, DuplicateCheckPolicy, ETicketDuplicateCheckPolicy::None);
    R_FIELD(ETicketDuplicateActionPolicy, DuplicateActionPolicy, ETicketDuplicateActionPolicy::Create);
    R_FIELD(TString, CommentIfDuplicate);
};

class TStartrekNotifierResult: public NDrive::TNotifierResult {
    using TBase = NDrive::TNotifierResult;

public:
    using TBase::TBase;

    explicit TStartrekNotifierResult(const TStartrekClient::TTicket& ticket);

    NJson::TJsonValue SerializeToJson() const;

private:
    R_FIELD(TStartrekClient::TTicket, Ticket);
};

class TStartrekMessage: public NDrive::INotifier::TMessage {
    using TBase = NDrive::INotifier::TMessage;

public:
    using TTicket = TStartrekClient::TTicket;
    using TComment = TStartrekClient::TComment;
    using TAttachments = TMap<TString, TString>;  // { name, content }

    explicit TStartrekMessage(const TString& issue = "", const TComment& comment = {}, const TTicket& update = {}, const TString& transition = "")
        : TBase()
    {
        SetIssue(issue);
        SetComment(comment);
        SetUpdate(update);
        SetTransition(transition);
    }

    const TString& GetIssue() const {
        return GetTitle();
    }

    TStartrekMessage& SetIssue(const TString& issue) {
        SetTitle(issue);
        return *this;
    }

    TTicket GetUpdate() const {
        return TTicket(GetAdditionalInfo());
    }

    TStartrekMessage& SetUpdate(const TTicket& update) {
        SetAdditionalInfo(update.SerializeToJson());
        return *this;
    }

    const TString& GetCommentString() const {
        return GetBody();
    }

    TComment GetComment() const {
        TComment comment;
        comment.DeserializeFromString(GetBody());
        return comment;
    }

    TStartrekMessage& SetComment(const TString& comment) {
        SetBody(comment);
        return *this;
    }

    TStartrekMessage& SetComment(const TComment& comment) {
        SetBody(comment.SerializeToJson().GetStringRobust());
        return *this;
    }

    const TString& GetTransition() const {
        return GetHeader();
    }

    TStartrekMessage& SetTransition(const TString& transition) {
        SetHeader(transition);
        return *this;
    }

    virtual void DoSerializeToJson(NJson::TJsonValue& result) const override;
    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& data) override;

private:
    R_FIELD(TAttachments, Attachments);
};

class TStartrekNotifier: public NDrive::INotifier {
private:
    using TBase = NDrive::INotifier;

public:
    using TStartrekMessage = TStartrekMessage;
    using TStartrekResult = TStartrekNotifierResult;
    using TStartrekAttachmentIds = TVector<TStartrekClient::TAttachmentId>;

    TStartrekNotifier(const TStartrekNotificationsConfig& config);
    virtual ~TStartrekNotifier() override;

    virtual TResults MultiLinesNotify(const TString& commonHeader, const TMessages& reportMessages, const TContext& context = Default<TContext>()) const override;

    virtual bool SendDocument(const TMessage& message, const TString& /* mimeType */, const TContext& context = Default<TContext>()) const override;
    virtual bool SendPhoto(const TMessage& message, const TContext& context = Default<TContext>()) const override;

    void DoStart(const IServerBase* /*server*/) override;
    void DoStop() override;

private:
    virtual TMessage::TPtr ConstructMessage(const NJson::TJsonValue& messageData) const override;
    virtual TResult::TPtr DoNotify(const TMessage& message, const TContext& context = Default<TContext>()) const override;

    bool UploadAttachments(const TStartrekClient* startrekClientPtr, const TStartrekMessage* startrekMessagePtr, TStartrekAttachmentIds& attachmentIds, TMessagesCollector& errors) const;

    TResult::TPtr ProcessExistingIssue(const TStartrekClient* startrekClientPtr, const TStartrekMessage* startrekMessagePtr, const TStartrekAttachmentIds& attachmentIds) const;
    TResult::TPtr AddIssueComment(const TStartrekClient* startrekClientPtr, TStartrekComment comment, TVector<TString> issues, const TStartrekAttachmentIds& attachmentIds) const;
    TResult::TPtr ExecuteIssueTransition(const TStartrekClient* startrekClientPtr, const TStartrekMessage* startrekMessagePtr, TStartrekTicket&& update) const;
    TResult::TPtr PatchExistingIssue(const TStartrekClient* startrekClientPtr, const TString& issue, TStartrekTicket&& update) const;

    TResult::TPtr ProcessNewIssue(const TStartrekClient* startrekClientPtr, const TStartrekMessage* startrekMessagePtr, const TStartrekAttachmentIds& attachmentIds) const;

    TMaybe<TVector<TString>> CheckDuplicates(const TStartrekClient* startrekClientPtr, const TString& summary, const TString& queue) const;
    TResult::TPtr CreateNewIssue(const TStartrekClient* startrekClientPtr, TStartrekTicket&& content) const;

private:
    const TStartrekNotificationsConfig Config;
};
