#include "call.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/logging/events.h>

#include <drive/library/cpp/support_ai/client.h>
#include <drive/library/cpp/threading/future.h>

#include <rtline/library/json/builder.h>

#include <library/cpp/mediator/global_notifications/system_status.h>


namespace {
    class TCallFutures {
    private:
        using TFuture = NThreading::TFuture<TSupportAICall>;
        using TFutures = TVector<TFuture>;
        TFutures Futures;
        TVector<TString> PhoneMap;

    public:
        void Emplace(const TString& phone, TFuture&& result) {
            PhoneMap.emplace_back(phone);
            Futures.emplace_back(std::move(result));
        }

        size_t GetFuturesSize() const {
            return Futures.size();
        }

        bool Empty() const {
            return PhoneMap.empty() && Futures.empty();
        }

        void Clear() {
            Futures.clear();
            PhoneMap.clear();
        }

        bool Process(const TDuration& timeout, TSupportAINotifierResult& result) {
            auto waiter = NThreading::WaitAll(Futures);
            if (!waiter.Wait(timeout)) {
                NDrive::TEventLog::Log("SupportAICallError", NJson::TMapBuilder
                    ("error", "Failed request by timeout")
                    ("phones", NJson::ToJson(PhoneMap))
                );
                result.SetErrorMessage("Failed request by timeout");
                return false;
            }
            for (ui32 i = 0; i < Futures.size(); ++i) {
                auto& future = Futures[i];
                if (!future.HasValue()) {
                    const TString errorInfo = NThreading::GetExceptionMessage(future);
                    NDrive::TEventLog::Log("SupportAICallError", NJson::TMapBuilder
                        ("error", "Error in call process: " + errorInfo)
                        ("phone", PhoneMap.size() > i ? PhoneMap[i] : "")
                    );
                    result.SetErrorMessage("Failed to make request");
                    result.SetErrorDetails(errorInfo);
                    continue;
                }
                result.MutableCallData().emplace_back(future.ExtractValue());
                NDrive::TEventLog::Log("SupportAICallSuccess", NJson::TMapBuilder
                    ("phone", PhoneMap.size() > i ? PhoneMap[i] : "")
                    ("call_id", result.MutableCallData().back().GetExternalId())
                );
            }
            Clear();
            return true;
        }
    };
}

NJson::TJsonValue TSupportAINotifierResult::SerializeToJson() const {
    NJson::TJsonValue json = TBase::SerializeToJson();
    if (!CallData.empty()) {
        auto& calls = json.InsertValue("calls", NJson::JSON_ARRAY);
        for (const auto& call : CallData) {
            calls.AppendValue(call.GetExternalId());
        }
    }
    return json;
}

void TSupportAINotificationsConfig::DoInit(const TYandexConfig::Section* section) {
    MaxInFlight = section->GetDirectives().Value<ui32>("MaxInFlight", MaxInFlight);
    OverridePhone = section->GetDirectives().Value<TString>("OverridePhone", OverridePhone);
    auto children = section->GetAllChildren();
    auto sectionIt = children.find("OverrideFeatures");
    if (sectionIt != children.end()) {
        for (auto&& callerIt : sectionIt->second->GetDirectives()) {
            OverrideFeatures.emplace(callerIt.first, callerIt.second);
        }
    }
}

void TSupportAINotificationsConfig::DoToString(IOutputStream& os) const {
    os << "MaxInFlight: " << MaxInFlight << Endl;
    if (!OverrideFeatures.empty()) {
        os << "<OverrideFeatures>" << Endl;
        for (auto&& [ key, value ] : OverrideFeatures) {
            os << key << ": " << value << Endl;
        }
        os << "</OverrideFeatures>" << Endl;
    }
    if (OverridePhone) {
        os << "OverridePhone: " << OverridePhone << Endl;
    }
}

NDrive::INotifier::TPtr TSupportAINotificationsConfig::Construct() const {
    return new TSupportAINotifier(*this);
}

bool TSupportAINotificationsConfig::DeserializeFromJson(const NJson::TJsonValue& info, TMessagesCollector& errors) {
    return TBase::DeserializeFromJson(info, errors)
        && NJson::ParseField(info["override_features"], NJson::KeyValue(OverrideFeatures))
        && NJson::ParseField(info["max_in_fly"], MaxInFlight)
        && NJson::ParseField(info["override_phone"], OverridePhone);
}

NJson::TJsonValue TSupportAINotificationsConfig::SerializeToJson() const {
    NJson::TJsonValue result = TBase::SerializeToJson();
    NJson::InsertField(result, "max_in_fly", MaxInFlight);
    NJson::InsertNonNull(result, "override_phone", OverridePhone);
    if (!OverrideFeatures.empty()) {
        NJson::InsertField(result, "override_features", NJson::KeyValue(OverrideFeatures));
    }
    return result;
}

NDrive::TScheme TSupportAINotificationsConfig::GetScheme(const IServerBase& server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSNumeric>("max_in_fly", "Максимальное количество параллельных звонков").SetDefault(MaxInFlight);
    result.Add<TFSString>("override_phone", "Переопределить телефон");
    auto& features = result.Add<TFSArray>("override_features", "Переопределить фичи звонка");
    NDrive::TScheme& fields = features.SetElement<NDrive::TScheme>();
    fields.Add<TFSString>("key", "Идентификатор фичи").SetRequired(true);
    fields.Add<TFSString>("value", "Значение").SetRequired(true);
    return result;
}


TSupportAINotificationsConfig::TFactory::TRegistrator<TSupportAINotificationsConfig> TSupportAINotificationsConfig::Registrator("support_ai");

TSupportAINotifier::TSupportAINotifier(const TSupportAINotificationsConfig& config)
    : NDrive::INotifier(config)
    , Config(config)
{
}

TSupportAINotifier::~TSupportAINotifier() {
    NDrive::INotifier::Stop();
}

void TSupportAINotifier::DoStart(const IServerBase* /*server*/) {
}

void TSupportAINotifier::DoStop() {
}

NDrive::INotifier::TResult::TPtr TSupportAINotifier::DoNotify(const TMessage& message, const TContext& context) const {
    if (!Yensured(context.GetServer())->GetSupportAIClient()) {
        NDrive::TEventLog::Log("SupportAICallError", NJson::TMapBuilder("error", "Suppirt AI client not configured"));
        return nullptr;
    }

    TMap<TString, TString> features;
    if (Config.GetOverrideFeatures().empty()) {
        for (auto&& [ key, value ] : message.GetAdditionalInfo().GetMap()) {
            features.emplace(key, value.GetStringRobust());
        }
    }
    const auto& callFeatures = features.empty() ? Config.GetOverrideFeatures() : features;

    TSet<TString> phones;
    if (Config.GetOverridePhone()) {
        phones.insert(Config.GetOverridePhone());
    } else {
        for (auto&& userInfo : context.GetRecipients()) {
            if (userInfo.GetPhone()) {
                phones.insert(userInfo.GetPhone());
            }
        }
    }

    auto resultPtr = MakeAtomicShared<TSupportAINotifierResult>();
    auto& result = *resultPtr;
    TCallFutures futures;
    for (auto&& phone : phones) {
        if (futures.GetFuturesSize() >= Config.GetMaxInFlight()) {
            if (!futures.Process(context.GetServer()->GetSupportAIClient()->GetConfig().GetRequestTimeout(), result)) {
                return resultPtr;
            }
            continue;
        }
        futures.Emplace(phone, context.GetServer()->GetSupportAIClient()->MakeCall(phone, callFeatures));
    }
    if (!futures.Empty() && futures.Process(context.GetServer()->GetSupportAIClient()->GetConfig().GetRequestTimeout(), result)) {
        NDrive::TEventLog::Log("SupportAICallSuccess", NJson::TMapBuilder
            ("phones", NJson::ToJson(phones))
            ("features", NJson::ToJson(callFeatures))
            ("calls_data", result.SerializeToJson())
        );
    }
    return resultPtr;
}
