#include "telegram.h"

#include <drive/backend/data/leasing/company.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/logging/events.h>

#include <library/cpp/mediator/global_notifications/system_status.h>
#include <library/cpp/string_utils/quote/quote.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/builder.h>
#include <rtline/library/json/merge.h>
#include <rtline/util/network/neh.h>
#include <rtline/util/network/neh_request.h>

#include <util/string/builder.h>

bool TReaskConfigOperator::DeserializeFromJson(NSimpleMeta::TConfig& config, const NJson::TJsonValue& info) {
    JREAD_INT_OPT(info, "max_attempts", config.MutableMaxAttempts());
    JREAD_DURATION_OPT(info, "global_timeout", config.MutableGlobalTimeout());
    JREAD_DURATION_OPT(info, "tasks_check_interval", config.MutableTasksCheckInterval());
    return true;
}
NJson::TJsonValue TReaskConfigOperator::SerializeToJson(const NSimpleMeta::TConfig& config) {
    NJson::TJsonValue result;
    JWRITE(result, "max_attempts", config.GetMaxAttempts());
    TJsonProcessor::WriteDurationString(result, "global_timeout", config.GetGlobalTimeout());
    TJsonProcessor::WriteDurationString(result, "tasks_check_interval", config.GetTasksCheckInterval());
    return result;
}
NDrive::TScheme TReaskConfigOperator::GetScheme() {
    NDrive::TScheme result;
    result.Add<TFSNumeric>("max_attempts").SetDefault(2);
    result.Add<TFSDuration>("global_timeout").SetDefault(TDuration::MilliSeconds(1000));
    result.Add<TFSDuration>("tasks_check_interval").SetDefault(TDuration::MilliSeconds(500));
    return result;
}

void ITelegramBotConfig::Init(const TYandexConfig::Section* section) {
    auto children = section->GetAllChildren();
    auto it = children.find("ReaskConfig");
    if (it == children.end()) {
        ReaskConfig.SetGlobalTimeout(TDuration::Seconds(14)).SetTasksCheckInterval(TDuration::Seconds(7)).SetMaxAttempts(2);
    } else {
        ReaskConfig.InitFromSection(it->second);
    }
    NotificationTimeout = TDuration::MilliSeconds(section->GetDirectives().Value("NotificationTimeout", NotificationTimeout.MilliSeconds()));
}

void ITelegramBotConfig::ToString(IOutputStream& os) const {
    os << "NotificationTimeout: " << NotificationTimeout.MilliSeconds() << Endl;
    os << "<ReaskPolicy>" << Endl;
    ReaskConfig.ToString(os);
    os << "</ReaskPolicy>" << Endl;
}

bool ITelegramBotConfig::DeserializeFromJson(const NJson::TJsonValue& info) {
    if (info.Has("reask_config") && !TReaskConfigOperator::DeserializeFromJson(ReaskConfig, info["reask_config"])) {
        return false;
    }
    if (!NJson::ParseField(info["notification_timeout"], NotificationTimeout)) {
        return false;
    }
    return true;
}

NJson::TJsonValue ITelegramBotConfig::SerializeToJson() const {
    NJson::TJsonValue result;
    result.InsertValue("notification_timeout", NJson::ToJson(NotificationTimeout));
    result.InsertValue("reask_config", TReaskConfigOperator::SerializeToJson(ReaskConfig));
    return result;
}

NDrive::TScheme ITelegramBotConfig::GetScheme(const IServerBase& /*server*/) const {
    NDrive::TScheme result;
    result.Add<TFSDuration>("notification_timeout");
    result.Add<TFSStructure>("reask_config").SetStructure(TReaskConfigOperator::GetScheme());
    return result;
}

void TTelegramBotConfig::Init(const TYandexConfig::Section* section) {
    TBase::Init(section);
    AssertCorrectConfig(section->GetDirectives().GetValue("BotId", BotId), "Incorrect parameter 'BotId' for telegram notification");
}

void TTelegramBotConfig::ToString(IOutputStream& os) const {
    os << "BotId: " << BotId << Endl;
    TBase::ToString(os);
}

bool TTelegramBotConfig::DeserializeFromJson(const NJson::TJsonValue& info) {
    JREAD_STRING_OPT(info, "bot_id", BotId);
    return TBase::DeserializeFromJson(info);
}

NJson::TJsonValue TTelegramBotConfig::SerializeToJson() const {
    NJson::TJsonValue result = TBase::SerializeToJson();
    result.InsertValue("bot_id", BotId);
    return result;
}

NDrive::TScheme TTelegramBotConfig::GetScheme(const IServerBase& server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSString>("bot_id");
    return result;
}

const TString& TTelegramBotConfig::GetBotId(const NDrive::INotifier::TContext& /*context*/, const TString& /*chatId*/) const {
    return BotId;
}


void TTelegramBotCompanyConfig::Init(const TYandexConfig::Section* section) {
    TBase::Init(section);
    AssertCorrectConfig(section->GetDirectives().GetValue("Owner", Owner), "Incorrect parameter 'BotId' for telegram notification");
}

void TTelegramBotCompanyConfig::ToString(IOutputStream& os) const {
    os << "Owner: " << Owner << Endl;
    TBase::ToString(os);
}

bool TTelegramBotCompanyConfig::DeserializeFromJson(const NJson::TJsonValue& info) {
    JREAD_STRING(info, "owner", Owner);
    return TBase::DeserializeFromJson(info);
}

NJson::TJsonValue TTelegramBotCompanyConfig::SerializeToJson() const {
    NJson::TJsonValue result = TBase::SerializeToJson();
    result.InsertValue("owner", Owner);
    return result;
}

NJson::TJsonValue TTelegramBotCompanyConfig::GetUserReport(const IServerBase& server, const TString& chatId) const {
    NJson::TJsonValue result;
    auto botId = GetBotId(NDrive::INotifier::TContext().SetServer(&server), chatId);
    result.InsertValue("bot_id", botId);
    return result;
}

NDrive::TScheme TTelegramBotCompanyConfig::GetScheme(const IServerBase& server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSString>("owner");
    return result;
}

const TString& TTelegramBotCompanyConfig::GetBotId(const NDrive::INotifier::TContext& context, const TString& chatId) const {
    const auto* server = Yensured(context.GetServer())->GetAs<NDrive::IServer>();
    const auto tdCompany = Yensured(server)->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(Owner);
    auto tdCompanyImpl = tdCompany->GetAs<const NDrivematics::TUserOrganizationAffiliationTag::TDescription>();
    if (!tdCompanyImpl) {
        return Default<TString>();
    }

    auto botId = tdCompanyImpl->GetTelegramNotifier().GetBotId(chatId);
    if (!botId || !*botId) {
        return Default<TString>();
    }
    BotId = *botId;
    return BotId;
}

void TTelegramChatConfig::Init(const TYandexConfig::Section* section) {
    AssertCorrectConfig(section->GetDirectives().GetValue("ChatId", ChatId), "Incorrect parameter 'ChatId' for telegram notification");
}

void TTelegramChatConfig::ToString(IOutputStream& os) const {
    os << "ChatId: " << ChatId << Endl;
}

bool TTelegramChatConfig::DeserializeFromJson(const NJson::TJsonValue& info) {
    JREAD_STRING_OPT(info, "chat_id", ChatId);
    return true;
}

NJson::TJsonValue TTelegramChatConfig::SerializeToJson() const {
    NJson::TJsonValue result;
    result.InsertValue("chat_id", ChatId);
    return result;
}

NDrive::TScheme TTelegramChatConfig::GetScheme(const IServerBase& /*server*/) const {
    NDrive::TScheme result;
    result.Add<TFSString>("chat_id");
    return result;
}

bool ITelegramNotificationsConfig::DeserializeFromJson(const NJson::TJsonValue& info, TMessagesCollector& /*errors*/) {
    if (!info.Has("bot") || !info.Has("chat")) {
        return false;
    }
    if (!GetBot()->DeserializeFromJson(info["bot"])) {
        return false;
    }
    if (!Chat.DeserializeFromJson(info["chat"])) {
        return false;
    }
    return true;
}

NJson::TJsonValue ITelegramNotificationsConfig::SerializeToJson() const {
    NJson::TJsonValue result;
    result.InsertValue("bot", GetBot()->SerializeToJson());
    result.InsertValue("chat", Chat.SerializeToJson());
    return result;
}

NJson::TJsonValue ITelegramNotificationsConfig::GetUserReport(const IServerBase& server) const {
    NJson::TJsonValue result;
    result.InsertValue("bot", GetBot()->GetUserReport(server, Chat.GetChatId()));
    result.InsertValue("chat", Chat.SerializeToJson());
    return result;
}

NDrive::TScheme ITelegramNotificationsConfig::GetScheme(const IServerBase& server) const {
    NDrive::TScheme result;
    result.Add<TFSStructure>("bot").SetStructure(GetBot()->GetScheme(server));
    result.Add<TFSStructure>("chat").SetStructure(Chat.GetScheme(server));
    return result;
}
void ITelegramNotificationsConfig::DoInit(const TYandexConfig::Section* section) {
    GetBot()->Init(section);
    Chat.Init(section);
}

void ITelegramNotificationsConfig::DoToString(IOutputStream& os) const {
    GetBot()->ToString(os);
    Chat.ToString(os);
}

NDrive::INotifier::TPtr TTelegramNotificationsConfig::Construct() const {
    return new TTelegramNotifier(*this);
}

bool TTelegramNotificationsConfig::DeserializeFromJson(const NJson::TJsonValue& info, TMessagesCollector& errors) {
    return
        ITelegramNotificationsConfig::DeserializeFromJson(info, errors) &&
        TBase::DeserializeFromJson(info, errors);
}

NJson::TJsonValue TTelegramNotificationsConfig::SerializeToJson() const {
    NJson::TJsonValue result = TBase::SerializeToJson();
    return NJson::MergeJson(ITelegramNotificationsConfig::SerializeToJson(), result);
}

NDrive::TScheme TTelegramNotificationsConfig::GetScheme(const IServerBase& server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.MergeScheme(ITelegramNotificationsConfig::GetScheme(server));
    return result;
}

bool TTelegramNotificationsConfig::UserPatch(const NJson::TJsonValue& data, TMessagesCollector& errors) {
    if (data["chat"]["chat_id"] != GetChat().GetChatId()) {
        errors.AddMessage("UserPatch", "chat_id cant be changed");
        return false;
    }
    return true;
}

void TTelegramNotificationsConfig::DoInit(const TYandexConfig::Section* section) {
    ITelegramNotificationsConfig::DoInit(section);
}

void TTelegramNotificationsConfig::DoToString(IOutputStream& os) const {
    ITelegramNotificationsConfig::DoToString(os);
}

NDrive::INotifier::TPtr TTelegramNotificationsCompanyConfig::Construct() const {
    return new TTelegramNotifier(*this);
}

bool TTelegramNotificationsCompanyConfig::DeserializeFromJson(const NJson::TJsonValue& info, TMessagesCollector& errors) {
    return
        ITelegramNotificationsConfig::DeserializeFromJson(info, errors) &&
        TBase::DeserializeFromJson(info, errors);
}

NJson::TJsonValue TTelegramNotificationsCompanyConfig::SerializeToJson() const {
    NJson::TJsonValue result = TBase::SerializeToJson();
    return NJson::MergeJson(ITelegramNotificationsConfig::SerializeToJson(), result);
}

NJson::TJsonValue TTelegramNotificationsCompanyConfig::GetUserReport(const IServerBase& server) const {
    NJson::TJsonValue result = TBase::GetUserReport(server);
    return NJson::MergeJson(ITelegramNotificationsConfig::GetUserReport(server), result);
}

NDrive::TScheme TTelegramNotificationsCompanyConfig::GetScheme(const IServerBase& server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.MergeScheme(ITelegramNotificationsConfig::GetScheme(server));
    return result;
}
void TTelegramNotificationsCompanyConfig::DoInit(const TYandexConfig::Section* section) {
    ITelegramNotificationsConfig::DoInit(section);
}

void TTelegramNotificationsCompanyConfig::DoToString(IOutputStream& os) const {
    ITelegramNotificationsConfig::DoToString(os);
}

class TTelegramRequestCallback: public NNeh::THttpAsyncReport::ICallback {
public:
    void OnResponse(const std::deque<NNeh::THttpAsyncReport>& reports) override {
        for (auto&& i : reports) {
            if (i.GetHttpCode() != 200) {
                DEBUG_LOG << i.GetHttpCode() << " / " << i.GetSourceInfo() << " / " << (i.HasReport() ? i.GetReportSafe() : " NO_REPORT") << Endl;
            }
        }
    }
};

void TTelegramNotifierImpl::Start() {
    CHECK_WITH_LOG(!Agent);
    AD = new TAsyncDelivery;
    AD->Start(1, 8);
    Agent.Reset(new NNeh::THttpClient(AD));
    Agent->RegisterSource("telegram", "api.telegram.org", 443, Config->GetReaskConfig(), true);
}

void TTelegramNotifierImpl::Stop() {
    if (!!AD) {
        AD->Stop();
    }
}

NDrive::INotifier::TResult::TPtr TTelegramNotifierImpl::Notify(const NDrive::INotifier::TMessage& message, const TString& chatId, const NDrive::TNotifierContext& context) const {
    if (!Agent) {
        return MakeAtomicShared<NDrive::INotifier::TResult>("HttpClient is not configured");
    }
    NNeh::THttpRequest request;
    TString report = message.GetHeader();
    if (!!report) {
        report += ": ";
    }
    report += message.GetBody();
    if (!report) {
        return MakeAtomicShared<NDrive::INotifier::TResult>("Overall text is empty");
    }
    auto botId = Config->GetBotId(context, chatId);
    if (!botId) {
        return MakeAtomicShared<NDrive::INotifier::TResult>("botId is not set");
    }
    request.SetCgiData("chat_id=" + chatId + "&parse_mode=HTML&text=" + CGIEscapeRet(report));
    request.SetUri("/" + botId + "/sendMessage");
    auto deadline = Now() + Config->GetNotificationTimeout();
    while (true) {
        auto response = Agent->SendMessageSync(request, Now() + Config->GetReaskConfig().GetGlobalTimeout());
        auto code = response.Code();
        if (code == 429) {
            auto content = NJson::ToJson(NJson::JsonString(response.Content()));
            auto retryAfter = content["parameters"]["retry_after"].GetUInteger();
            if (!retryAfter) {
                retryAfter = 1;
            }
            auto sleepDuration = TDuration::Seconds(retryAfter);
            if (Now() + sleepDuration < deadline) {
                NDrive::TEventLog::Log("TelegramNotifierRetry", NJson::TMapBuilder
                    ("code", code)
                    ("content", std::move(content))
                    ("retry_after", retryAfter)
                );
                Sleep(sleepDuration);
                continue;
            }
        }
        if (code != 200) {
            return MakeAtomicShared<NDrive::INotifier::TResult>("telegram request unsuccessful", NJson::TMapBuilder
                ("code", code)
                ("content", NJson::ToJson(NJson::JsonString(response.Content())))
                ("error_message", response.ErrorMessage())
            );
        }
        break;
    }
    return nullptr;
}

bool TTelegramNotifierImpl::SendDocument(const NDrive::INotifier::TMessage& message, const TString& mimeType, const TString& chatId, const NDrive::TNotifierContext& context) const {
    auto botId = Config->GetBotId(context, chatId);
    if (!botId) {
        return false;
    }
    NNeh::THttpRequest request;
    request.SetUri("/" + botId + "/sendDocument");
    return SendAttachment(request, message, EAttachmentType::Document, mimeType, chatId);
}

bool TTelegramNotifierImpl::SendPhoto(const NDrive::INotifier::TMessage& message, const TString& chatId, const NDrive::TNotifierContext& context) const {
    auto botId = Config->GetBotId(context, chatId);
    if (!botId) {
        return false;
    }
    NNeh::THttpRequest request;
    request.SetUri("/" + botId + "/sendPhoto");
    return SendAttachment(request, message, EAttachmentType::Photo, "image/jpeg", chatId);
}

bool TTelegramNotifierImpl::SendAttachment(const NNeh::THttpRequest& simpleRequest, const NDrive::INotifier::TMessage& message, EAttachmentType type, const TString& mimeType, const TString& chatId) const {
    if (!Agent) {
        ERROR_LOG << "Incorrect agent for notifier" << Endl;
        return false;
    }
    NNeh::THttpRequest request = simpleRequest;
    request.SetCgiData("chat_id=" + chatId + "&caption=" + message.GetHeader() + ":" + message.GetAdditionalInfo().GetStringRobust());
    request.SetRequestType("POST");

    TString boundary = "--Asrf456BGe4h";
    request.AddHeader("Content-Type", "multipart/form-data; boundary=" + boundary);

    TString data = TStringBuilder()
        << "--" << boundary << "\r\n"
        << "Content-Disposition: form-data; name=\"" + ToString(type) + "\"; filename=\"" + message.GetTitle() + "\"\r\n"
        << "Content-Type: " << mimeType << "\r\n"
        << "Content-Length: " << message.GetBody().size() << "\r\n\r\n"
        << message.GetBody() << "\r\n"
        << "--" << boundary << "--\r\n";

    request.SetPostData(data);

    auto tgResult = Agent->SendMessageSync(request, Now() + TDuration::Seconds(60));
    if (tgResult.Code() != 200) {
        DEBUG_LOG << tgResult.Code() << " / " << tgResult.Content() << " / " << tgResult.ErrorMessage() << Endl;
        return false;
    }
    return true;
}

TTelegramNotifierImpl::TTelegramNotifierImpl(const ITelegramBotConfig::TPtr config)
    : Config(config)
{
}

TTelegramNotifier::~TTelegramNotifier() {
    NDrive::INotifier::Stop();
}

TTelegramNotificationsConfig::TFactory::TRegistrator<TTelegramNotificationsConfig> TTelegramNotificationsConfig::Registrator("telegram");
TTelegramNotificationsCompanyConfig::TFactory::TRegistrator<TTelegramNotificationsCompanyConfig> TTelegramNotificationsCompanyConfig::Registrator("telegram_company");
