#pragma once

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/abstract/notifier.h>

#include <library/cpp/yconf/conf.h>

#include <rtline/library/async_proxy/async_delivery.h>
#include <rtline/library/metasearch/simple/config.h>
#include <rtline/util/network/neh.h>
#include <rtline/util/types/accessor.h>

#include <util/generic/ptr.h>
#include <util/stream/output.h>

namespace NNeh {
    class THttpClient;
}

class TAsyncDelivery;

class TReaskConfigOperator {
public:
    static bool DeserializeFromJson(NSimpleMeta::TConfig& config, const NJson::TJsonValue& info);
    static NJson::TJsonValue SerializeToJson(const NSimpleMeta::TConfig& config);
    static NDrive::TScheme GetScheme();
};

class ITelegramBotConfig {
public:
    using TPtr = TAtomicSharedPtr<ITelegramBotConfig>;

protected:
    R_READONLY(NSimpleMeta::TConfig, ReaskConfig);
    R_READONLY(TDuration, NotificationTimeout, TDuration::Seconds(1));

public:
    virtual void Init(const TYandexConfig::Section* section);
    virtual void ToString(IOutputStream& os) const;

    virtual const TString& GetBotId(const NDrive::INotifier::TContext& /*context*/, const TString& /*chatId*/) const = 0;

    virtual bool DeserializeFromJson(const NJson::TJsonValue& info);
    virtual NJson::TJsonValue SerializeToJson() const;
    virtual NDrive::TScheme GetScheme(const IServerBase& server) const;
    virtual NJson::TJsonValue GetUserReport(const IServerBase& /*server*/, const TString& /*chatId*/) const {
        return Default<NJson::TJsonValue>();
    };

    virtual ~ITelegramBotConfig() {}
};

class TTelegramBotConfig: public ITelegramBotConfig {
private:
    using TBase =  ITelegramBotConfig;

protected:
    TString BotId;
    R_READONLY(NSimpleMeta::TConfig, ReaskConfig);
    R_READONLY(TDuration, NotificationTimeout, TDuration::Seconds(1));

public:
    void Init(const TYandexConfig::Section* section) override;
    void ToString(IOutputStream& os) const override;

    const TString& GetBotId(const NDrive::INotifier::TContext& context = Default<NDrive::TNotifierContext>(), const TString& chatId = Default<TString>()) const override;

    bool DeserializeFromJson(const NJson::TJsonValue& info) override;
    NJson::TJsonValue SerializeToJson() const override;
    NDrive::TScheme GetScheme(const IServerBase& /*server*/) const override;
};

class TTelegramBotCompanyConfig: public ITelegramBotConfig {
private:
    using TBase =  ITelegramBotConfig;

protected:
    R_FIELD(TString, Owner);
    mutable TString BotId;

public:
    void Init(const TYandexConfig::Section* section) override;
    void ToString(IOutputStream& os) const override;

    const TString& GetBotId(const NDrive::INotifier::TContext& context, const TString& chatId) const override;

    bool DeserializeFromJson(const NJson::TJsonValue& info) override;
    NJson::TJsonValue SerializeToJson() const override;
    NDrive::TScheme GetScheme(const IServerBase& server) const override;
    NJson::TJsonValue GetUserReport(const IServerBase& server, const TString& chatId) const override;
};

class TTelegramChatConfig {
private:
    TString ChatId;

public:
    void Init(const TYandexConfig::Section* section);
    void ToString(IOutputStream& os) const;

    const TString& GetChatId() const {
        return ChatId;
    }

    bool DeserializeFromJson(const NJson::TJsonValue& info);
    NJson::TJsonValue SerializeToJson() const;
    NDrive::TScheme GetScheme(const IServerBase& /*server*/) const;
};

class ITelegramNotificationsConfig {
public:
    ITelegramBotConfig::TPtr Bot;
    TTelegramChatConfig Chat;

public:
    ITelegramNotificationsConfig(ITelegramBotConfig::TPtr bot):
        Bot(bot)
    {
    }

public:
    const ITelegramBotConfig::TPtr GetBot() const {
        return Yensured(Bot);
    }

    const TTelegramChatConfig& GetChat() const {
        return Chat;
    }

    virtual bool DeserializeFromJson(const NJson::TJsonValue& info, TMessagesCollector& errors);
    virtual NJson::TJsonValue SerializeToJson() const;
    virtual NJson::TJsonValue GetUserReport(const IServerBase& server) const;
    virtual NDrive::TScheme GetScheme(const IServerBase& server) const;

protected:
    virtual void DoInit(const TYandexConfig::Section* section);
    virtual void DoToString(IOutputStream& os) const;
};

class TTelegramNotificationsConfig: public NDrive::INotifierConfig, public ITelegramNotificationsConfig {
protected:
    using TBase = NDrive::INotifierConfig;

private:
    static TFactory::TRegistrator<TTelegramNotificationsConfig> Registrator;

public:
    TTelegramNotificationsConfig():
        TBase(),
        ITelegramNotificationsConfig(MakeAtomicShared<TTelegramBotConfig>())
    {
    }

public:
    virtual NDrive::INotifier::TPtr Construct() const override;

    virtual bool DeserializeFromJson(const NJson::TJsonValue& info, TMessagesCollector& errors) override;
    virtual NJson::TJsonValue SerializeToJson() const override;
    virtual NDrive::TScheme GetScheme(const IServerBase& server) const override;

    bool UserPatch(const NJson::TJsonValue& data, TMessagesCollector& errors) override;

protected:
    virtual void DoInit(const TYandexConfig::Section* section) override;
    virtual void DoToString(IOutputStream& os) const override;
};

class TTelegramNotificationsCompanyConfig: public NDrive::INotifierConfig, public ITelegramNotificationsConfig {
protected:
    using TBase = NDrive::INotifierConfig;

private:
    static TFactory::TRegistrator<TTelegramNotificationsCompanyConfig> Registrator;

public:
    TTelegramNotificationsCompanyConfig():
        TBase(),
        ITelegramNotificationsConfig(MakeAtomicShared<TTelegramBotCompanyConfig>())
    {
    }

public:
    virtual NDrive::INotifier::TPtr Construct() const override;

    virtual bool DeserializeFromJson(const NJson::TJsonValue& info, TMessagesCollector& errors) override;
    virtual NJson::TJsonValue SerializeToJson() const override;
    virtual NJson::TJsonValue GetUserReport(const IServerBase& server) const override;
    virtual NDrive::TScheme GetScheme(const IServerBase& server) const override;

protected:
    virtual void DoInit(const TYandexConfig::Section* section) override;
    virtual void DoToString(IOutputStream& os) const override;
};

enum class EAttachmentType {
    Document /* "document" */,
    Photo /* "photo" */,
};

class TTelegramNotifierImpl: public TNonCopyable {
protected:
    TAtomicSharedPtr<TAsyncDelivery> AD;
    THolder<NNeh::THttpClient> Agent;
    const ITelegramBotConfig::TPtr Config;

public:
    NDrive::INotifier::TResult::TPtr Notify(const NDrive::INotifier::TMessage& message, const TString& chatId, const NDrive::TNotifierContext& context = Default<NDrive::TNotifierContext>()) const;
    bool SendDocument(const NDrive::INotifier::TMessage& message, const TString& mimeType, const TString& chatId, const NDrive::TNotifierContext& context = Default<NDrive::TNotifierContext>()) const;
    bool SendPhoto(const NDrive::INotifier::TMessage& message, const TString& chatId, const NDrive::TNotifierContext& context = Default<NDrive::TNotifierContext>()) const;

    TTelegramNotifierImpl(const ITelegramBotConfig::TPtr config);

    void Start();
    void Stop();

private:
    bool SendAttachment(const NNeh::THttpRequest& simpleRequest, const NDrive::INotifier::TMessage& message, EAttachmentType type, const TString& mimeType, const TString& chatId) const;
};

class TTelegramNotifier: public TTelegramNotifierImpl, public NDrive::INotifier {
protected:
    const ITelegramNotificationsConfig Config;

public:
    TTelegramNotifier(const TTelegramNotificationsConfig& config)
        : TTelegramNotifierImpl(config.GetBot())
        , NDrive::INotifier(config)
        , Config(config)
    {}

    TTelegramNotifier(const TTelegramNotificationsCompanyConfig& config)
        : TTelegramNotifierImpl(config.GetBot())
        , NDrive::INotifier(config)
        , Config(config)
    {}

    virtual ~TTelegramNotifier();

    virtual void DoStart(const IServerBase* /*server*/) override {
        TTelegramNotifierImpl::Start();
    }

    virtual bool SendDocument(const TMessage& message, const TString& mimeType, const TContext& context = Default<TContext>()) const override {
        return TTelegramNotifierImpl::SendDocument(message, mimeType, Config.GetChat().GetChatId(), context);
    }

    virtual bool SendPhoto(const TMessage& message, const TContext& context = Default<TContext>()) const override {
        return TTelegramNotifierImpl::SendPhoto(message, Config.GetChat().GetChatId(), context);
    }

    virtual void DoStop() override {
        TTelegramNotifierImpl::Stop();
    }

private:
    virtual TResult::TPtr DoNotify(const TMessage& message, const TContext& context = Default<TContext>()) const override {
        return TTelegramNotifierImpl::Notify(message, Config.GetChat().GetChatId(), context);
    }
};
