#include "notifier.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/logging/events.h>

#include <drive/library/cpp/threading/future_cast.h>

#include <kernel/reqid/reqid.h>

TTaxiCommunicationsNotifier::TTaxiCommunicationsNotifier(const TTaxiCommunicationsNotifierConfig& config)
    : NDrive::INotifier(config)
    , Config(config)
{
}

TTaxiCommunicationsNotifier::~TTaxiCommunicationsNotifier() {
    NDrive::INotifier::Stop();
}

NDrive::INotifier::TResult::TPtr TTaxiCommunicationsNotifier::DoNotify(const TMessage& message, const TContext& context) const {
    NJson::TJsonValue data;
    NJson::TJsonValue& payload = data.InsertValue("payload", NJson::JSON_MAP);
    if (message.HasAdditionalInfo()) {
        payload.InsertValue("details", message.GetAdditionalInfo());
    }
    payload["extra"]["service"] = "drive";
    payload["body"] = message.GetBody();
    payload["sound"] = "default";
    payload["icon"] = "push";
    payload["title"] = message.GetTitle();

    NJson::TJsonValue& repack = data.InsertValue("repack", NJson::JSON_MAP);
    {
        NJson::TJsonValue alert;
        if (!message.GetTitle().empty()) {
            alert["title"] = message.GetTitle();
            alert["body"] = message.GetBody();
        } else {
            alert = message.GetBody();
        }
        NJson::TJsonValue& aps = repack.InsertValue("apns", NJson::JSON_MAP).InsertValue("aps", NJson::JSON_MAP);
        aps["sound"] = "default";
        aps["badge"] = 1;
        aps["alert"] = std::move(alert);
    }

    {
        NJson::TJsonValue& fcm = repack.InsertValue("fcm", NJson::JSON_MAP);
        fcm["priority"] = "high";
        repack.InsertValue("hms", fcm);
    }

    TDuration ttl;
    const TString& intent = message.GetName();

    TMap<TString, NThreading::TFuture<void>> results;
    for (auto&& userId : context.GetExternalRecipients()) {
        auto idempotencyToken = message.GetTransitId() + ':' + userId;
        idempotencyToken.resize(std::min<size_t>(idempotencyToken.size(), 64));
        auto reqid = ReqIdGenerate("DRIVE");
        NDrive::TEventLog::Log("TaxiCommunicationsRequest", NJson::TMapBuilder
            ("data", data)
            ("host", Config.Endpoint)
            ("indempotency_token", idempotencyToken)
            ("intent", intent)
            ("ttl", NJson::ToJson(ttl))
            ("reqid", reqid)
            ("user_id", userId)
        );
        auto dt = data;
        auto result = Yensured(Client)->Push(std::move(dt), userId, intent, idempotencyToken, ttl);
        results.emplace(std::move(reqid), std::move(result));
    }
    ui32 failed = 0;
    ui32 succeeded = 0;
    for (auto&& [reqid, result] : results) {
        result.Wait();
        NDrive::TEventLog::Log("TaxiCommunicationsResponse", NJson::TMapBuilder
            ("reqid", reqid)
            ("result", NJson::ToJson(result))
        );
        if (result.HasValue()) {
            succeeded++;
        } else {
            failed++;
        }
    }
    if (failed) {
        auto result = MakeHolder<TResult>(TStringBuilder() << failed << " pushes failed");
        result->MutableErrorDetails()["failed"] = failed;
        result->MutableErrorDetails()["succeeded"] = succeeded;
        return result;
    }
    return nullptr;
}

void TTaxiCommunicationsNotifier::DoStart(const IServerBase* server) {
    auto tvm = (Config.SelfClientId && server) ? server->GetTvmClient(Config.SelfClientId) : nullptr;
    if (!tvm && Config.SelfClientId) {
        ERROR_LOG << GetNotifierName() << ": cannot get TvmClient for " << Config.SelfClientId << Endl;
    }
    Client = MakeHolder<NDrive::TTaxiCommunicationsClient>(Config, tvm);
}

void TTaxiCommunicationsNotifier::DoStop() {
    Client.Destroy();
}

NDrive::INotifier::TPtr TTaxiCommunicationsNotifierConfig::Construct() const {
    return new TTaxiCommunicationsNotifier(*this);
}

void TTaxiCommunicationsNotifierConfig::DoInit(const TYandexConfig::Section* section) {
    const auto& directives = section->GetDirectives();
    Endpoint = directives.Value("Endpoint", Endpoint);
    SelfClientId = directives.Value("SelfClientId", SelfClientId);
    DestinationClientId = directives.Value("DestinationClientId", DestinationClientId);
}

void TTaxiCommunicationsNotifierConfig::DoToString(IOutputStream& os) const {
    os << "Endpoint: " << Endpoint << Endl;
    os << "SelfClientId: " << SelfClientId << Endl;
}

bool TTaxiCommunicationsNotifierConfig::DeserializeFromJson(const NJson::TJsonValue& info, TMessagesCollector& errors) {
    return
        TBase::DeserializeFromJson(info, errors) &&
        NJson::ParseField(info["endpoint"], Endpoint) &&
        NJson::ParseField(info["self_client_id"], SelfClientId);
}

NJson::TJsonValue TTaxiCommunicationsNotifierConfig::SerializeToJson() const {
    NJson::TJsonValue result = TBase::SerializeToJson();
    result["endpoint"] = Endpoint;
    result["self_client_id"] = SelfClientId;
    return result;
}

NDrive::TScheme TTaxiCommunicationsNotifierConfig::GetScheme(const IServerBase& server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSVariants>("endpoint").SetVariants({
        NDrive::TaxiCommunicationsProductionEndpoint,
        NDrive::TaxiCommunicationsTestingEndpoint,
    });
    result.Add<TFSNumeric>("self_client_id");
    return result;
}

TTaxiCommunicationsNotifierConfig::TFactory::TRegistrator<TTaxiCommunicationsNotifierConfig> TTaxiCommunicationsNotifierConfig::Registrator("ucommunications");
