#include <drive/backend/ut/library/helper.h>
#include <drive/backend/notifications/startrek/startrek.h>
#include <drive/backend/notifications/ut_large/helpers.h>

#include <kernel/daemon/config/daemon_config.h>
#include <library/cpp/testing/unittest/tests_data.h>
#include <library/cpp/testing/unittest/registar.h>
#include <util/generic/algorithm.h>
#include <util/system/env.h>

namespace {
    TStartrekNotificationsConfig GetStartrekNotificationsConfig() {
        TStringStream ss;
        ss << "NotificationType: startrek";
        return NDrive::INotifierConfig::BuildFromString<TStartrekNotificationsConfig>(ss.Str());
    }
}

Y_UNIT_TEST_SUITE(StartrekNotifications) {
    Y_UNIT_TEST(AddComment) {
        const TString issueName = "DRIVETEST-1";

        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        auto configStr = configGenerator.GetString();

        TServerConfigConstructorParams params(configStr.data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TNotifierHolder<TStartrekNotifier, TStartrekNotificationsConfig> notifier(GetStartrekNotificationsConfig());
        TStartrekNotifier::TStartrekMessage message(issueName);
        TString comment = "Test comment at "+ TInstant::Now().ToString();
        message.SetComment(comment);
        auto resp = notifier.SendTestMessage(message, TStartrekNotifier::TContext().SetServer(server.Get()));
        UNIT_ASSERT_C(!resp || !resp->HasErrors(), ((resp) ? resp->SerializeToJson().GetStringRobust() : "Unknown errors as response is empty"));
    }

    Y_UNIT_TEST(UpdateBody) {
        const TString issueName = "DRIVETEST-1";
        const size_t maxDescriptionLength = 4096;

        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        auto configStr = configGenerator.GetString();

        TServerConfigConstructorParams params(configStr.data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        const TStartrekClient* client = server->GetStartrekClient();
        UNIT_ASSERT_C(!!client, "client is not initialized properly");

        TNotifierHolder<TStartrekNotifier, TStartrekNotificationsConfig> notifier(GetStartrekNotificationsConfig());

        TStartrekTicket ticketInfo;
        TMessagesCollector errors;

        UNIT_ASSERT_C(client->GetIssueInfo(issueName, ticketInfo, errors), "cannot obtain issue info: " << errors.GetStringReport());

        TString description = ticketInfo.GetDescription();

        if (description.length() >= maxDescriptionLength) {
            size_t defaultOffset = description.length() - maxDescriptionLength;
            size_t offset = description.find('\n', defaultOffset);
            if (offset == TString::npos) {
                offset = defaultOffset;
            }
            description = description.substr(offset + 1);
        }

        description += "\nUpdate at "+ TInstant::Now().ToString();

        TStartrekTicket ticketUpdate;
        ticketUpdate.SetDescription(description);
        TStartrekNotifier::TStartrekMessage message(issueName);
        message.SetUpdate(ticketUpdate);
        auto resp = notifier.SendTestMessage(message, TStartrekNotifier::TContext().SetServer(server.Get()));
        UNIT_ASSERT_C(!resp || !resp->HasErrors(), ((resp) ? resp->SerializeToJson().GetStringRobust() : "Unknown errors as response is empty"));

        TStartrekTicket updatedTicketInfo;
        UNIT_ASSERT_C(client->GetIssueInfo(issueName, updatedTicketInfo, errors), "cannot obtain issue info: " << errors.GetStringReport());
        UNIT_ASSERT_STRINGS_EQUAL(description, updatedTicketInfo.GetDescription());
    }

    Y_UNIT_TEST(UpdateFields) {
        const TString issueName = "DRIVETEST-11";
        const TString specificTagName = "dk_tag";
        const TString fieldName = "storyPoints";

        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        auto configStr = configGenerator.GetString();

        TServerConfigConstructorParams params(configStr.data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        const TStartrekClient* client = server->GetStartrekClient();
        UNIT_ASSERT_C(!!client, "client is not initialized properly");

        TNotifierHolder<TStartrekNotifier, TStartrekNotificationsConfig> notifier(GetStartrekNotificationsConfig());

        TStartrekNotifier::TStartrekMessage message(issueName);
        bool hasSpecificTag;
        i64 expectedStoryPoints;
        {
            TStartrekTicket ticketUpdate;

            TStartrekTicket ticketInfo;
            TMessagesCollector errors;
            UNIT_ASSERT_C(client->GetIssueInfo(issueName, ticketInfo, errors), "cannot obtain issue info: " << errors.GetStringReport());
            {
                expectedStoryPoints = ticketInfo.GetAdditionalValue<i64>(fieldName).GetOrElse(0) + 1;
                ticketUpdate.SetAdditionalValue(fieldName, expectedStoryPoints);
            }
            {
                TSet<TString> existingTags;
                Y_UNUSED(ticketInfo.GetAdditionalValue(::ToString(TStartrekTicket::EContainerTicketField::Tags), existingTags));

                hasSpecificTag = AnyOf(existingTags, [&specificTagName](const auto& tag) { return tag == specificTagName; });

                if (!hasSpecificTag) {
                    ticketUpdate.AddAdditionalContainerValue(TStartrekTicket::EContainerTicketField::Tags, TVector<TString>{specificTagName});
                } else {
                    ticketUpdate.RemoveAdditionalContainerValue(TStartrekTicket::EContainerTicketField::Tags, TVector<TString>{specificTagName});
                }
            }

            message.SetUpdate(ticketUpdate);
        }

        auto resp = notifier.SendTestMessage(message, TStartrekNotifier::TContext().SetServer(server.Get()));
        UNIT_ASSERT_C(!resp || !resp->HasErrors(), ((resp) ? resp->SerializeToJson().GetStringRobust() : "Unknown errors as response is empty"));

        {
            TStartrekTicket updatedTicketInfo;
            TMessagesCollector errors;
            UNIT_ASSERT_C(client->GetIssueInfo(issueName, updatedTicketInfo, errors), "cannot obtain issue info: " << errors.GetStringReport());

            UNIT_ASSERT_VALUES_EQUAL(expectedStoryPoints, updatedTicketInfo.GetAdditionalValue<i64>(fieldName).GetOrElse(0));

            bool hasUpdatedSpecificTag;
            {
                TSet<TString> updatedTags;
                Y_UNUSED(updatedTicketInfo.GetAdditionalValue(::ToString(TStartrekTicket::EContainerTicketField::Tags), updatedTags));

                hasUpdatedSpecificTag = AnyOf(updatedTags, [&specificTagName](const auto& tag) { return tag == specificTagName; });
            }
            UNIT_ASSERT_VALUES_UNEQUAL(hasSpecificTag, hasUpdatedSpecificTag);
        }
    }

    Y_UNIT_TEST(UpdateStatus) {
        const TString issueName = "DRIVETEST-1";

        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        auto configStr = configGenerator.GetString();

        TServerConfigConstructorParams params(configStr.data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        const TStartrekClient* client = server->GetStartrekClient();
        UNIT_ASSERT_C(!!client, "client is not initialized properly");

        TNotifierHolder<TStartrekNotifier, TStartrekNotificationsConfig> notifier(GetStartrekNotificationsConfig());

        TStartrekTicket ticketInfo;
        TMessagesCollector errors;
        UNIT_ASSERT_C(client->GetIssueInfo(issueName, ticketInfo, errors), "cannot obtain issue info: " << errors.GetStringReport());

        TString status;
        UNIT_ASSERT(ticketInfo.GetAdditionalValue(::ToString(TStartrekTicket::ETicketField::StatusKey), status));

        TString targetStatus, targetTransition;

        if (status == "open") {
            targetStatus = "inProgress";
            targetTransition = "start_progress";
        } else if (status == "inProgress") {
            targetStatus = "open";
            targetTransition = "stop_progress";
        } else {
            UNIT_FAIL("cannot change ticket " << issueName << " status: status " << status << " is unknown");
        }

        TStartrekNotifier::TStartrekMessage message(issueName);
        message.SetTransition(targetTransition);
        auto resp = notifier.SendTestMessage(message, TStartrekNotifier::TContext().SetServer(server.Get()));
        UNIT_ASSERT_C(!resp || !resp->HasErrors(), ((resp) ? resp->SerializeToJson().GetStringRobust() : "Unknown errors as response is empty"));

        TStartrekTicket updatedTicketInfo;
        UNIT_ASSERT_C(client->GetIssueInfo(issueName, updatedTicketInfo, errors), "cannot obtain issue info: " << errors.GetStringReport());
        UNIT_ASSERT_STRINGS_EQUAL(targetStatus, updatedTicketInfo.GetAdditionalValue<TString>(::ToString(TStartrekTicket::ETicketField::StatusKey)).GetOrElse(""));
    }

    Y_UNIT_TEST(UploadAttachment) {
        const TString issueName = "DRIVETEST-5";

        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        auto configStr = configGenerator.GetString();

        TServerConfigConstructorParams params(configStr.data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        const TStartrekClient* client = server->GetStartrekClient();
        UNIT_ASSERT_C(!!client, "client is not initialized properly");

        TNotifierHolder<TStartrekNotifier, TStartrekNotificationsConfig> notifier(GetStartrekNotificationsConfig());
        TMessagesCollector errors;

        ui32 existingAttachmentsCount = 0;
        {
            TVector<TStartrekAttachment> attachments;
            UNIT_ASSERT_C(client->GetIssueAttachments(issueName, attachments, errors), "cannot obtain issue attachments: " << errors.GetStringReport());
            existingAttachmentsCount = attachments.size();
        }

        TStartrekNotifier::TStartrekMessage message(issueName);

        // emulate plain text file
        const TString fileName = "some_text_file_" + ::ToString(Now().MicroSeconds()) + ".txt";
        const TString fileContent = "some file context";
        message.SetAttachments({{fileName, fileContent}});

        auto resp = notifier.SendTestMessage(message, TStartrekNotifier::TContext().SetServer(server.Get()));
        UNIT_ASSERT_C(!resp || !resp->HasErrors(), ((resp) ? resp->SerializeToJson().GetStringRobust() : "Unknown errors as response is empty"));

        ui32 updatedAttachmentsCount = 0;
        {
            TVector<TStartrekAttachment> attachments;
            UNIT_ASSERT_C(client->GetIssueAttachments(issueName, attachments, errors), "cannot obtain issue attachments: " << errors.GetStringReport());
            updatedAttachmentsCount = attachments.size();
        }

        UNIT_ASSERT_VALUES_EQUAL(existingAttachmentsCount + 1, updatedAttachmentsCount);
    }

    Y_UNIT_TEST(UploadAttachmentGeneralizedDeprecated) {
        // the mechanism below should not be extended or broadly used
        //   and it has been implemented to generalize the current implementation only

        const bool addAttachmentToComment = false;

        const TString issueName = "DRIVETEST-11";

        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        auto configStr = configGenerator.GetString();

        TServerConfigConstructorParams params(configStr.data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        const TStartrekClient* client = server->GetStartrekClient();
        UNIT_ASSERT_C(!!client, "client is not initialized properly");

        TNotifierHolder<TStartrekNotifier, TStartrekNotificationsConfig> notifier(GetStartrekNotificationsConfig());
        TMessagesCollector errors;

        ui32 existingAttachmentsCount = 0;
        {
            TVector<TStartrekAttachment> attachments;
            UNIT_ASSERT_C(client->GetIssueAttachments(issueName, attachments, errors), "cannot obtain issue attachments: " << errors.GetStringReport());
            existingAttachmentsCount = attachments.size();
        }

        // emulate plain text file
        const TString fileName = "some_text_file_" + ::ToString(Now().MicroSeconds()) + ".txt";
        const TString fileContent = "some file context";

        TStartrekNotifier::TStartrekMessage message;
        message.SetHeader(issueName);
        message.SetTitle(fileName);
        message.SetBody(fileContent);

        if (addAttachmentToComment) {
            message.SetAdditionalInfo("New comment with attachment at " + ::ToString(Now().MicroSeconds()));
        }

        bool resp = notifier.GetNotifier().SendDocument(message, "", TStartrekNotifier::TContext().SetServer(server.Get()));
        UNIT_ASSERT_C(resp, "Errors sending document");

        if (!addAttachmentToComment) {
            ui32 updatedAttachmentsCount = 0;
            {
                TVector<TStartrekAttachment> attachments;
                UNIT_ASSERT_C(client->GetIssueAttachments(issueName, attachments, errors), "cannot obtain issue attachments: " << errors.GetStringReport());
                updatedAttachmentsCount = attachments.size();
            }

            UNIT_ASSERT_VALUES_EQUAL(existingAttachmentsCount + 1, updatedAttachmentsCount);
        }
    }
}
