#pragma once
#include <drive/backend/abstract/notifier.h>

#include <mapreduce/yt/interface/common.h>

class TYtNotifierConfig: public NDrive::INotifierConfig {
private:
    using TBase = NDrive::INotifierConfig;
    static TFactory::TRegistrator<TYtNotifierConfig> Registrator;

    R_FIELD(TString, Cluster);
    R_FIELD(TString, TablePath);
    R_FIELD(TString, TableName);
    R_FIELD(bool, Append, false);

public:
    virtual NDrive::INotifier::TPtr Construct() const override;

    virtual bool DeserializeFromJson(const NJson::TJsonValue& info, TMessagesCollector& errors) override {
        return
            NJson::ParseField(info, "cluster", Cluster, true, errors) &&
            NJson::ParseField(info, "table_path", TablePath, true, errors) &&
            NJson::ParseField(info, "table_name", TableName, false, errors) &&
            NJson::ParseField(info, "append", Append, false, errors) &&
            TBase::DeserializeFromJson(info, errors);
    }

    virtual NJson::TJsonValue SerializeToJson() const override {
        NJson::TJsonValue result = TBase::SerializeToJson();
        NJson::InsertField(result, "cluster", Cluster);
        NJson::InsertField(result, "table_path", TablePath);
        NJson::InsertField(result, "table_name", TableName);
        NJson::InsertField(result, "append", Append);
        return result;
    }

    virtual NDrive::TScheme GetScheme(const IServerBase& server) const override {
        NDrive::TScheme result = TBase::GetScheme(server);
        result.Add<TFSString>("cluster", "Кластер на YT").SetRequired(true);
        result.Add<TFSString>("table_path", "Папка с таблицами на YT").SetRequired(true);
        result.Add<TFSString>("table_name", "Имя таблицы (создавать новые при пустой)");
        result.Add<TFSBoolean>("append", "Дописывать данные");
        return result;
    }

private:
    virtual void DoInit(const TYandexConfig::Section* section) override {
        Cluster = section->GetDirectives().Value("Cluster", Cluster);
        TablePath = section->GetDirectives().Value("TablePath", TablePath);
        TableName = section->GetDirectives().Value("TableName", TableName);
        Append = section->GetDirectives().Value("Append", Append);
    }

    virtual void DoToString(IOutputStream& os) const override {
        os << "Cluster: " << Cluster << Endl;
        os << "TablePath: " << TablePath << Endl;
        os << "TableName: " << TableName << Endl;
        os << "Append: " << Append << Endl;
    }
};

class TYtNotifier: public NDrive::INotifier, public TNonCopyable {
private:
    const TYtNotifierConfig Config;

public:
    TYtNotifier(const TYtNotifierConfig& config)
        : NDrive::INotifier(config)
        , Config(config)
    {}

    virtual void DoStart(const IServerBase* server) override;

    virtual void DoStop() override;

    virtual TResult::TPtr DoNotify(const TMessage& message, const TContext& context) const override;
    TResults MultiLinesNotify(const TString& commonHeader, const TMessages& reportMessages, const TContext& context = Default<TContext>()) const override;

private:
    NYT::TRichYPath ConstructRichPath() const;
};
