#include "abstract.h"

#include <drive/backend/offers/context.h>
#include <drive/backend/offers/actions/long_term.h>
#include <drive/backend/offers/offers/standart.h>

#include <drive/backend/areas/areas.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/data/delegation.h>
#include <drive/backend/data/dedicated_fleet.h>
#include <drive/backend/database/config.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/roles/manager.h>

#include <rtline/library/json/proto/adapter.h>

#include <kernel/daemon/common/time_guard.h>

void TCommonOfferActionTraits::FillScheme(NDrive::TScheme& result, const NDrive::IServer* server) const {
    auto tagsInPoint = server->GetDriveAPI()->GetAreasDB()->GetAreaTags(TInstant::Zero());
    result.Add<TFSVariants>("tags_in_point", "Geo теги объекта", 100000).SetVariants(std::move(tagsInPoint)).MulVariantsLeft({"", "!"}).SetMultiSelect(true);
    result.Add<TFSNumeric>("devices_in_area_count_filter", "Минимальное количество машин в геозоне активации offer", 100000).SetMin(0).SetMax(1000000).SetDefault(0);
}

bool TCommonOfferActionTraits::FromJson(const NJson::TJsonValue& value) {
    if (!TJsonProcessor::ReadContainer(value, "tags_in_point", DeviceTagsInPoint)) {
        return false;
    }
    JREAD_INT_OPT(value, "devices_in_area_count_filter", DevicesInAreaCountFilter);
    return true;
}

void TCommonOfferActionTraits::ToJson(NJson::TJsonValue& value) const {
    if (DeviceTagsInPoint.size()) {
        TJsonProcessor::WriteContainerArray(value, "tags_in_point", DeviceTagsInPoint);
    }
    if (DevicesInAreaCountFilter) {
        JWRITE(value, "devices_in_area_count_filter", DevicesInAreaCountFilter);
    }
}

TString TCommonOfferActionTraits::CalcCacheId() const {
    return JoinSeq("+", DeviceTagsInPoint) + ':' + ToString(DevicesInAreaCountFilter);
}

EOfferCorrectorResult TCommonOfferActionTraits::CheckGeoConditionsImpl(const TOffersBuildingContext& context) const {
    auto g = context.BuildEventGuard("CheckGeoConditionsImpl");
    if (GetDeviceTagsInPoint().size()) {
        if (context.GetUseLocationTags() && !TBaseAreaTagsFilter::FilterWeak(context.GetLocationTags(), DeviceTagsInPoint)) {
            return EOfferCorrectorResult::Unimplemented;
        }
        if (DeviceTagsInPoint.size() && DevicesInAreaCountFilter) {
            auto guard = context.BuildEventGuard("DevicesInAreaCountFilter");
            auto snapshots = context.GetServer()->GetSnapshotsManager().GetSnapshots();
            TSet<TString> ids;
            for (auto&& i : snapshots.ById()) {
                if (TBaseAreaTagsFilter::FilterWeak(i.second.GetTagsInPoint(), DeviceTagsInPoint)) {
                    ids.emplace(i.first);
                }
            }
            TVector<TTaggedDevice> objects;
            if (!context.GetServer()->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetCustomObjectsFromCache(objects, ids)) {
                return EOfferCorrectorResult::BuildingProblems;
            }
            ui32 count = 0;
            for (auto&& i : objects) {
                bool busy = false;
                for (auto&& t : i.GetTags()) {
                    if (!!t && !!t->GetPerformer()) {
                        busy = true;
                        break;
                    }
                }
                if (!busy) {
                    ++count;
                }
            }
            if (count < DevicesInAreaCountFilter) {
                return EOfferCorrectorResult::Unimplemented;
            }
        }
    }
    return EOfferCorrectorResult::Success;
}

EOfferCorrectorResult TCommonOfferActionTraits::CheckGeoConditions(const TOffersBuildingContext& context, const TString& cacheId) const {
    TMaybe<EOfferCorrectorResult> result = context.GetGeoConditionsCheckResult(cacheId);
    if (!result) {
        result = CheckGeoConditionsImpl(context);
        context.SetGeoConditionsCheckResult(cacheId, *result);
    }
    return *result;
}

EOfferCorrectorResult TCommonOfferActionTraits::CheckVisibleForUserGeoConditions(const TOffersBuildingContext& context) const {
    if (DeviceTagsInPoint.size()) {
        if (!TBaseAreaTagsFilter::FilterWeak(context.GetUserLocationTags(), DeviceTagsInPoint)) {
            return EOfferCorrectorResult::Unimplemented;
        }
    }
    return EOfferCorrectorResult::Success;
}

bool ICommonOfferBuilderAction::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) {
    if (!TBase::DeserializeSpecialsFromJson(jsonValue)) {
        return false;
    }
    if (!TCommonOfferActionTraits::FromJson(jsonValue)) {
        return false;
    }
    if (jsonValue.Has("activity_interval")) {
        if (!ActivityInterval.DeserializeFromJson(jsonValue["activity_interval"])) {
            return false;
        }
    }
    JREAD_BOOL_OPT(jsonValue, "is_publish", IsPublish);
    JREAD_BOOL_OPT(jsonValue, "hidden", Hidden);
    if (!TJsonProcessor::ReadContainer(jsonValue, "switch_offer_tags", SwitchOfferTagsList)) {
        return false;
    }
    JREAD_INT_OPT(jsonValue, "list_priority", ListPriority);
    JREAD_INT_OPT(jsonValue, "payment_discretization", PaymentDiscretization);
    JREAD_STRING_OPT(jsonValue, "group_name", GroupName);

    const NJson::TJsonValue::TArray* localizationsArray;
    if (jsonValue["localizations"].GetArrayPointer(&localizationsArray)) {
        for (auto&& i : *localizationsArray) {
            TString key;
            TString value;
            if (!i["key"].GetString(&key) || !i["value"].GetString(&value)) {
                ERROR_LOG << "Incorrect localizations format: " << jsonValue << Endl;
                continue;
            }
            Localizations[key] = value;
        }
    }

    const NJson::TJsonValue& acounts = jsonValue["chargable_accounts"];
    if (!acounts.IsDefined() || !acounts.IsArray()) {
        return false;
    }
    for (auto&& i : acounts.GetArray()) {
        ChargableAccounts.insert(i.GetStringRobust());
    }

    const NJson::TJsonValue& visual = jsonValue["visual"];
    if (visual.IsDefined()) {
        if (!Visual.ConstructInPlace().DeserializeFromJson(visual)) {
            return false;
        }
    }

    JREAD_BOOL_OPT(jsonValue, "transferable", Transferable);

    return
        NJson::ParseField(jsonValue["origins"], Origins) &&
        NJson::ParseField(jsonValue["target_user_tags"], TargetUserTags);
}

NJson::TJsonValue ICommonOfferBuilderAction::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    TCommonOfferActionTraits::ToJson(result);

    if (!ActivityInterval.Empty()) {
        JWRITE(result, "activity_interval", ActivityInterval.SerializeToJson());
    }
    if (Localizations.size()) {
        NJson::TJsonValue& jsonValue = result["localizations"];
        for (auto&& i : Localizations) {
            NJson::TJsonValue& local = jsonValue.AppendValue(NJson::JSON_MAP);
            local.InsertValue("key", i.first);
            local.InsertValue("value", i.second);
        }
    }

    JWRITE(result, "is_publish", IsPublish);
    JWRITE(result, "hidden", Hidden);
    JWRITE(result, "list_priority", ListPriority);
    JWRITE(result, "group_name", GroupName);
    TJsonProcessor::WriteContainerArray(result, "switch_offer_tags", SwitchOfferTagsList);

    NJson::TJsonValue& accounts = result.InsertValue("chargable_accounts", NJson::JSON_ARRAY);
    for (auto&& account : ChargableAccounts) {
        accounts.AppendValue(account);
    }
    if (Visual) {
        result["visual"] = Visual->SerializeToJson();
    }

    NJson::InsertNonNull(result, "payment_discretization", NonDefault(PaymentDiscretization, DefaultPaymentDiscretization));
    NJson::InsertNonNull(result, "target_user_tags", TargetUserTags);
    NJson::InsertField(result, "transferable", Transferable);
    NJson::InsertNonNull(result, "origins", Origins);
    return result;
}

NJson::TJsonValue ICommonOfferBuilderAction::GetPublicReport(ELocalization locale, const ILocalization& localization) const {
    NJson::TJsonValue result = TBase::GetPublicReport(locale, localization);
    if (GroupName) {
        result["title"] = localization.ApplyResources(GroupName, locale);
    }
    NJson::TJsonValue& image = result["image"].SetType(NJson::JSON_MAP);
    if (Visual) {
        JWRITE(image, "visual", Visual->SerializeToJson());
    }
    return result;
}

bool ICommonOfferBuilderAction::CheckDeviceTagsPerformer(const TTaggedObject& taggedObject, const TUserPermissions& permissions, bool checkEmptyPerformer) const {
    return CheckDeviceTagsPerformerDefault(taggedObject, permissions, checkEmptyPerformer);
}

bool ICommonOfferBuilderAction::CheckDeviceTagsPerformerDefault(const TTaggedObject& taggedObject, const TUserPermissions& permissions, bool checkEmptyPerformer) {
    auto chargableTags = taggedObject.GetTagsByClass<TChargableTag>();
    bool found = false;
    for (auto&& tag : chargableTags) {
        if (!tag) {
            continue;
        }
        const auto& performer = tag->GetPerformer();
        if ((checkEmptyPerformer && performer.empty())|| performer == permissions.GetUserId()) {
            found = true;
            break;
        }
    }
    return found;
}

EOfferCorrectorResult ICommonOfferBuilderAction::CheckOfferConditions(const TOffersBuildingContext& context, const TUserPermissions& permissions) const {
    auto gCheckOfferConditions = context.BuildEventGuard("CheckOfferConditions");
    if (context.HasCarId()) {
        auto td = context.GetTaggedCar();
        if (!td) {
            return EOfferCorrectorResult::Problems;
        }

        if (OrganizationId) {
            TSet<ui64> fleetTagOrganizations;
            const auto fleetTags = td->GetTagsByClass<TSpecialFleetTag>();
            for (const auto& dbFleetTag : fleetTags) {
                if (const auto fleetTagImpl = dbFleetTag.GetTagAs<TSpecialFleetTag>()) {
                    fleetTagOrganizations.emplace(fleetTagImpl->GetParentId());
                }
            }

            if (!fleetTagOrganizations.empty() && !fleetTagOrganizations.contains(*OrganizationId)) {
                return EOfferCorrectorResult::Unimplemented;
            }
        }

        auto gCheckVisibility = context.BuildEventGuard("CheckVisibility");
        auto requireVisibility = context.GetSetting<bool>("offer_builder.default.require_visibility").GetOrElse(context.IsVisibilityRequired());
        TUserPermissions::TUnvisibilityInfoSet unvisibilityDetails = 0;
        if (requireVisibility && permissions.GetVisibility(*td, NEntityTagsManager::EEntityType::Car, &unvisibilityDetails) == TUserPermissions::EVisibility::NoVisible) {
            if (!context.HasCarWaitingDuration() && (!td->GetFirstTagByClass<IDelegationTag>() || !!td->GetFirstTagByClass<IDelegationTag>()->GetPerformer())) {
                return EOfferCorrectorResult::Unimplemented;
            }
            if (unvisibilityDetails & TUserPermissions::EUnvisibilityInfo::Unavailable) {
                return EOfferCorrectorResult::Unimplemented;
            }
        }

        auto gCheckGeoConditions = context.BuildEventGuard("CheckGeoConditions");
        auto cacheId = CalcCacheId();
        auto geoCorrection = CheckGeoConditions(context, cacheId);
        if (geoCorrection != EOfferCorrectorResult::Success) {
            return geoCorrection;
        }
    }

    auto gCheckIntervals = context.BuildEventGuard("CheckIntervals");
    if (!ActivityInterval.Empty() && !ActivityInterval.IsActualNow(Now())) {
        return EOfferCorrectorResult::Unimplemented;
    }
    if (!Origins.empty() && !Origins.contains(context.GetOrigin())) {
        return EOfferCorrectorResult::Unimplemented;
    }
    auto gDoCheck = context.BuildEventGuard("DoCheck");
    return DoCheckOfferConditions(context, permissions);
}

TVector<IOfferReport::TPtr> ICommonOfferBuilderAction::BuildOffers(const TOffersBuildingContext& context, NDrive::TInfoEntitySession& session, bool useCorrectors) const {
    TVector<IOfferReport::TPtr> result;
    auto permissions = context.GetUserPermissions();
    R_ENSURE(permissions, HTTP_INTERNAL_SERVER_ERROR, "UserPermissions are missing", session);
    auto correctors = useCorrectors ? permissions->GetOfferCorrections() : TUserActions();
    auto server = context.GetServer();
    auto buildingResult = BuildOffers(*permissions, correctors, result, context, server, session);
    switch (buildingResult) {
        case EOfferCorrectorResult::Success:
        case EOfferCorrectorResult::RejectOffer:
        case EOfferCorrectorResult::Unimplemented:
            break;
        case EOfferCorrectorResult::BuildingProblems:
        case EOfferCorrectorResult::Problems:
            R_ENSURE(buildingResult == EOfferCorrectorResult::Success, {}, "unexpected BuildOffers result: " << buildingResult, session);
    }
    return result;
}

EOfferCorrectorResult ICommonOfferBuilderAction::BuildOffers(const TUserPermissions& permissions, const TUserActions& correctorsPtr, TVector<IOfferReport::TPtr>& offers, const TOffersBuildingContext& context, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const {
    offers.clear();
    TTimeGuard tgBuild("IOfferBuilderAction::BuildOffers");
    if (!context.HasUserHistoryContext()) {
        return EOfferCorrectorResult::Problems;
    }
    const auto& uhc = context.GetUserHistoryContextRef();

    {
        auto g = context.BuildEventGuard("check_condition");
        EOfferCorrectorResult res = CheckOfferConditions(context, permissions);
        if (res != EOfferCorrectorResult::Success) {
            return res;
        }
    }

    TSet<TString> accountIds;
    TString defaultCard;
    TString selectedCharge;
    {
        auto g = context.BuildEventGuard("account_name");
        accountIds = uhc.GetRequestAccountIds();
        defaultCard = uhc.GetRequestCreditCard();
        selectedCharge = uhc.GetSelectedCharge();
    }

    const TSet<TString> chargableAccounts = uhc.GetAvailableAccounts(GetGrouppingTags(), GetChargableAccounts());
    if (server->GetDriveAPI()->HasBillingManager()) {
        if (uhc.GetFilterAccounts() && !uhc.CheckRequestAccountId(GetGrouppingTags(), GetChargableAccounts())) {
            return EOfferCorrectorResult::Unimplemented;
        }
    }
    TVector<NDrive::NBilling::IBillingAccount::TPtr> userAccounts;
    {
        auto g = context.BuildEventGuard("get_user_accounts");
        userAccounts = uhc.GetUserAccounts();
    }

    TMaybe<TTaggedObject> td = context.GetTaggedCar();

    TVector<const IOfferCorrectorAction*> correctors;
    correctors.reserve(correctorsPtr.size());
    for (auto&& corrPtr : correctorsPtr) {
        const IOfferCorrectorAction* corrector = dynamic_cast<const IOfferCorrectorAction*>(corrPtr.Get());
        if (corrector) {
            if (!corrector->GetOfferTagsFilter().IsEmpty() && !corrector->GetOfferTagsFilter().IsMatching(GetGrouppingTags())) {
                continue;
            }
            if (corrector->GetGrouppingTags().size() || corrector->GetChargableAccounts().size()) {
                if (!uhc.CheckRequestAccountId(corrector->GetGrouppingTags(), corrector->GetChargableAccounts())) {
                    continue;
                }
            }
            correctors.emplace_back(corrector);
        }
    }
    std::sort(correctors.begin(), correctors.end(), [](const IOfferCorrectorAction* left, const IOfferCorrectorAction* right) {
        if (!left) {
            return true;
        }
        if (!right) {
            return false;
        }
        return std::tie(left->GetPriority(), left->GetName()) > std::tie(right->GetPriority(), right->GetName());
    });

    const EOfferCorrectorResult resBuilding = DoBuildOffers(permissions, offers, context, server, session);
    if (resBuilding == EOfferCorrectorResult::Success) {
        for (auto it = offers.begin(); it != offers.end();) {
            auto offerReport = it->Get();
            if (!offerReport || !offerReport->GetOfferPtrAs<ICommonOffer>()) {
                it = offers.erase(it);
                continue;
            }
            auto gOfferCorrection = context.BuildEventGuard("offer_correction_start");
            auto offer = offerReport->GetOfferPtrAs<ICommonOffer>();
            auto vehicleOffer = offerReport->GetOfferPtrAs<IOffer>();

            if (auto longtermOffer = offerReport->GetOfferAs<TLongTermOffer>();
                longtermOffer &&
                longtermOffer->GetGroupOfferId() &&
                !longtermOffer->IsGroupOffer()) {
                ++it;
                continue;
            }

            if (vehicleOffer) {
                if (context.HasOriginalRidingStart()) {
                    vehicleOffer->SetOriginalRidingStart(context.GetOriginalRidingStartRef());
                } else if (context.GetStartPosition()) {
                    vehicleOffer->SetOriginalRidingStart(*context.GetStartPosition());
                }
                if (context.HasAvailableStartArea()) {
                    vehicleOffer->SetAvailableStartArea(context.GetAvailableStartAreaUnsafe());
                }
                if (!vehicleOffer->GetMarker()) {
                    vehicleOffer->SetMarker(context.GetMarker());
                }
                if (!vehicleOffer->GetSwitchable()) {
                    vehicleOffer->SetSwitchable(SwitchOfferTagsList.size());
                }
                if (!vehicleOffer->IsSwitcher()) {
                    vehicleOffer->SetSwitcher(context.IsSwitching());
                }
                vehicleOffer->SetTransferable(GetTransferable());
                if (context.HasCarId()) {
                    if (context.GetCarIdUnsafe() != vehicleOffer->GetObjectId() || td->GetId() != vehicleOffer->GetObjectId()) {
                        return EOfferCorrectorResult::Problems;
                    }
                }
                if (!vehicleOffer->GetOrigin()) {
                    vehicleOffer->SetOrigin(context.GetOrigin());
                }
                if (context.HasCarWaitingDuration()) {
                    vehicleOffer->SetCarWaitingDuration(TDuration::Seconds(context.GetCarWaitingDurationUnsafe()));
                    offer->SetDeadline(offer->GetTimestamp() + TDuration::Seconds(context.GetCarWaitingDurationUnsafe() + server->GetSettings().GetValueDef("futures.waiting_gap_seconds", 300)));
                }
            }

            if (!offer->GetDeadline()) {
                offer->SetDeadline(offer->GetTimestamp() + server->GetDriveAPI()->GetConfig().GetOfferLivetime());
            }
            if (offer->GetName().empty()) {
                offer->SetName(GetDescription());
            }
            if (!offer->GetExternalUserId()) {
                offer->SetExternalUserId(context.GetExternalUserId());
            }
            offer->SetLocale(uhc.GetLocale());
            offer->SetIsPlusUser(context.GetIsPlusUser());
            offer->MutableTargetUserTags().insert(TargetUserTags.begin(), TargetUserTags.end());
            if (!offerReport->HasListPriority()) {
                offerReport->SetListPriority(GetListPriority());
            }
            offer->SetGroupName(GetGroupName());
            offer->SetBehaviourConstructorId(GetName());

            if (offerReport->GetPriceOfferConstructor()) {
                offer->SetPriceConstructorId(offerReport->GetPriceOfferConstructor()->GetName());
            } else {
                offer->SetPriceConstructorId(GetName());
            }
            offer->SetHidden(IsHidden());
            TVector<TString> accounts;
            for (auto&& acc : userAccounts) {
                if (acc->IsActive() && chargableAccounts.contains(acc->GetUniqueName())) {
                    accounts.emplace_back(acc->GetUniqueName());
                }
            }
            offer->SetChargableAccounts(accounts);
            if (selectedCharge) {
                offer->SetSelectedCharge(selectedCharge);
            }
            if (!!defaultCard) {
                offer->SetSelectedCreditCard(defaultCard);
            }

            if (uhc.HasYandexAccountExternalBalance()) {
                offer->SetYandexAccountBalance(uhc.GetYandexAccountExternalBalanceRef());
            }

            bool remove = false;

            offer->SetHiddenCashback(permissions.GetSetting<bool>(server->GetSettings(), "offer." + offer->GetTypeName() + ".hidden_cashback").GetOrElse(false));

            bool usedPromoCode = false;
            for (auto&& action : correctors) {
                if (!action) {
                    continue;
                }
                if (action->GetChargableAccounts().size()) {
                    bool notFound = false;
                    auto it = action->GetChargableAccounts().begin();
                    for (const auto& id : accountIds) {
                        if (!AdvanceSimple(it, action->GetChargableAccounts().end(), id)) {
                            notFound = true;
                            break;
                        }
                    }
                    if (notFound) {
                        continue;
                    }
                }
                if (offer->GetDisabledCorrectors().contains(action->GetName())) {
                    continue;
                }
                if (action->GetSourceContext().IsPromoCode() && server->GetSettings().GetValueDef("promo.use_one_promo_discount", false) && usedPromoCode) {
                    continue;
                }

                auto gCorrector = context.BuildEventGuard("corrector_" + action->GetName());
                const auto& tags = td ? td->GetTags() : Default<TVector<TDBTag>>();
                const EOfferCorrectorResult correctionResult = action->ApplyForOffer(offerReport, tags, context, permissions.GetUserId(), server, session);
                if (correctionResult == EOfferCorrectorResult::Problems) {
                    return correctionResult;
                }
                if (correctionResult == EOfferCorrectorResult::RejectOffer) {
                    remove = true;
                    break;
                }
                if (correctionResult == EOfferCorrectorResult::Success) {
                    if (action->GetSourceContext().IsPromoCode()) {
                        usedPromoCode = true;
                    }
                    offer->AddCorrector(action->GetName());
                }
            }
            offerReport->SetLocalizations(GetLocalizations());
            if (vehicleOffer) {
                auto gAreasInfo = context.BuildEventGuard("init_areas_info");
                vehicleOffer->InitAreaInfos(server, context);
            }

            if (GetSourceContext().IsLimitedPolicy()) {
                offer->MutableProfitableTags().emplace(GetSourceContext().GetTagId());
                if (GetSourceContext().GetUntil() != TInstant::Max()) {
                    offer->SetOfferValidUntil(GetSourceContext().GetUntil());
                }
                if (GetSourceContext().GetSince() != TInstant::Zero()) {
                    offer->SetOfferValidSince(GetSourceContext().GetSince());
                }
            }

            if (offer->IsHidden()) {
                remove = true;
            }
            if (!remove) {
                ++it;
            } else {
                it = offers.erase(it);
            }
        }
    }
    NDrive::LogCreatedOffers(offers);
    return resBuilding;
}

EOfferCorrectorResult ICommonOfferBuilderAction::BuildOffersClean(const TUserPermissions& permissions, TVector<IOfferReport::TPtr>& offers, const TOffersBuildingContext& context, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const {
    return DoBuildOffers(permissions, offers, context, server, session);
}

NDrive::TScheme ICommonOfferBuilderAction::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    {
        auto gTab = result.StartTabGuard("activation");
        result.Add<TFSJson>("activity_interval", "Диапазон активности");
        TCommonOfferActionTraits::FillScheme(result, server);
        result.Add<TFSBoolean>("transferable", "Разрешить передачу руля").SetDefault(true);
        result.Add<TFSArray>("origins", "Фильтр источников запросов (taxi, maas, ...)").SetElement<TFSString>();
    }
    {
        auto gTab = result.StartTabGuard("experiment");
        result.Add<TFSBoolean>("hidden", "Предложение по умолчанию скрыто (для экспериментальных предложений)").SetDefault(false);
    }
    {
        auto gTab = result.StartTabGuard("client_appearance");
        result.Add<TFSVariants>("switch_offer_tags", "Список тегов тарифов для переключения").SetVariants(server->GetDriveAPI()->GetRolesManager()->GetActionsDB().GetOfferBuilderTags<ICommonOfferBuilderAction>()).SetMultiSelect(true);
        result.Add<TFSString>("group_name", "Название группы предложений");
        result.Add<TFSNumeric>("list_priority", "Приоритет в списке").SetDefault(0);
        result.Add<TFSBoolean>("is_publish", "Публичное предложение").SetDefault(true);
        NDrive::TScheme visual = TOfferVisual::GetScheme();
        result.Add<TFSStructure>("visual", "Appearance settings").SetStructure(visual);
        result.Add<TFSVariants>("target_user_tags", "Теги, которые нужно навесить на пользователя при бронировании оффера").SetMultiSelect(true).SetReference("user_tags");
        {
            NDrive::TScheme& schemeClientVersion = result.Add<TFSArray>("localizations", "Ключи для локализации", 100000).SetElement<NDrive::TScheme>();
            schemeClientVersion.Add<TFSString>("key", "Ключ текста");
            schemeClientVersion.Add<TFSString>("value", "Текст (параметризуемый для локализации)");
        }
    }
    {
        auto gTab = result.StartTabGuard("billing");
        result.Add<TFSNumeric>("payment_discretization", "Дискретизация для задач биллинга (копейки)").SetDefault(DefaultPaymentDiscretization);
        result.Add<TFSVariants>("chargable_accounts", "Возможна оплата аккаунтами")
            .SetVariants(server->GetDriveAPI()->GetBillingManager().GetKnownAccounts()).SetMultiSelect(true)
            .SetRequired(true);
    }
    return result;
}

bool IOfferBuilderAction::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) {
    if (!TBase::DeserializeSpecialsFromJson(jsonValue)) {
        return false;
    }

    JREAD_BOOL_OPT(jsonValue, "available_for_scanner", AvailableForScanner);
    JREAD_BOOL_OPT(jsonValue, "futures_available", FuturesAvailable);
    JREAD_STRING_OPT(jsonValue, "promo_icon", PromoIcon);
    JREAD_STRING_OPT(jsonValue, "profit_description", ProfitDescription);
    JREAD_STRING_OPT(jsonValue, "promo_description", PromoDescription);
    JREAD_STRING_OPT(jsonValue, "promo_detailed_description", PromoDetailedDescription);

    return true;
}

NJson::TJsonValue IOfferBuilderAction::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();

    JWRITE(result, "available_for_scanner", AvailableForScanner);
    JWRITE(result, "futures_available", FuturesAvailable);
    JWRITE_DEF(result, "promo_icon", PromoIcon, "");
    JWRITE_DEF(result, "profit_description", ProfitDescription, "");
    JWRITE_DEF(result, "promo_description", PromoDescription, "");
    JWRITE_DEF(result, "promo_detailed_description", PromoDetailedDescription, "");

    return result;
}

NDrive::TScheme IOfferBuilderAction::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    {
        auto gTab = result.StartTabGuard("activation");
        result.Add<TFSBoolean>("futures_available", "Строить фьючерс").SetDefault(true);
    }
    {
        auto gTab = result.StartTabGuard("client_appearance");
        result.Add<TFSBoolean>("available_for_scanner", "Доступно для радара").SetDefault(true);
        result.Add<TFSString>("promo_icon", "Иконка для вкладки Блага");
        result.Add<TFSString>("profit_description", "Короткое описание скидочного предложения");
        result.Add<TFSText>("promo_description", "Описание тарифа на вкладке Блага");
        result.Add<TFSText>("promo_detailed_description", "Подробное описание тарифа на вкладке Блага");
    }
    return result;
}


NJson::TJsonValue IOfferBuilderAction::GetPublicReport(ELocalization locale, const ILocalization& localization) const {
    NJson::TJsonValue result = TBase::GetPublicReport(locale, localization);
    NJson::TJsonValue& image = result["image"].SetType(NJson::JSON_MAP);
    JWRITE_DEF(image, "icon", PromoIcon, "");
    JWRITE_DEF(image, "small_icon", PromoIcon, "");
    if (ProfitDescription) {
        result["discount_value"]["profit_description"] = localization.ApplyResources(ProfitDescription, locale);
    }
    NJson::InsertField(result, "full_description", localization.ApplyResources(PromoDescription, locale));
    NJson::InsertNonNull(result, "detailed_description", localization.ApplyResources(PromoDetailedDescription, locale));
    return result;
}

TVector<TString> IOfferBuilderAction::GetNames(const NDrive::IServer* server) {
    if (!server || !server->GetDriveAPI() || !server->GetDriveAPI()->GetRolesManager()) {
        return {};
    }

    return server->GetDriveAPI()->GetRolesManager()->GetActionsDB().GetActionNamesWithType<IOfferBuilderAction>(/*reportDeprecated=*/false);
}

bool IOfferCorrectorAction::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) {
    if (!TBase::DeserializeSpecialsFromJson(jsonValue)) {
        return false;
    }
    if (!TCommonOfferActionTraits::FromJson(jsonValue)) {
        return false;
    }

    if (jsonValue["offer_tags_filter"].IsString() && !OfferTagsFilter.DeserializeFromString(jsonValue["offer_tags_filter"].GetString())) {
        return false;
    }
    if (jsonValue.Has("host_filter") && !HostFilter.DeserializeFromJson(jsonValue["host_filter"])) {
        return false;
    }
    return
        NJson::ParseField(jsonValue["chargable_accounts"], ChargableAccounts) &&
        NJson::ParseField(jsonValue["priority"], Priority) &&
        NJson::ParseField(jsonValue["reset_surge_types"], ResetSurgeTypes) &&
        NJson::ParseField(jsonValue["insurance_descriptions"], InsuranceDescriptions);
}

NJson::TJsonValue IOfferCorrectorAction::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    TCommonOfferActionTraits::ToJson(result);
    NJson::InsertField(result, "priority", Priority);
    NJson::InsertField(result, "offer_tags_filter", OfferTagsFilter.ToString());
    NJson::InsertField(result, "reset_surge_types", ResetSurgeTypes);
    NJson::InsertField(result, "chargable_accounts", ChargableAccounts);
    TJsonProcessor::WriteSerializable(result, "host_filter", HostFilter);
    NJson::InsertField(result, "insurance_descriptions", InsuranceDescriptions);
    return result;
}

namespace {
    TVector<TString> GetInsuranceDescriptionVariants(const NDrive::IServer* server) {
        if (!server || !server->GetDriveAPI() || !server->GetDriveAPI()->GetRolesManager()) {
            return {};
        }
        return server->GetDriveAPI()->GetRolesManager()->GetActionsDB().GetActionNamesWithType<TAdditionalPricedFeatureAction>(/*reportDeprecated=*/false);
    }
}

NDrive::TScheme IOfferCorrectorAction::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    TCommonOfferActionTraits::FillScheme(result, server);
    result.Add<TFSVariants>("chargable_accounts", "Возможна при оплате аккаунтами").SetMultiSelect(true).SetReference("accounts");
    result.Add<TFSString>("offer_tags_filter", "фильтр тегов оффера для применения корректировки", 99990);
    result.Add<TFSNumeric>("priority", "Corrector priority");
    result.Add<TFSStructure>("host_filter", "фильтр хостов для применения корректировки", 100001).SetStructure(HostFilter.GetScheme(*server, ""));
    result.Add<TFSBoolean>("reset_surge_types", "Сбрасывать SurgeTypes после применения");
    {
        result.Add<TFSVariants>("insurance_descriptions", "разрешенные описания страховки")
            .SetMultiSelect(true)
            .SetVariants(GetInsuranceDescriptionVariants(server));
    }
    return result;
}

EOfferCorrectorResult IOfferCorrectorAction::ApplyForOffer(IOfferReport* offer, const TVector<TDBTag>& tags, const TOffersBuildingContext& context, const TString& userId, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const {
    if (offer && GetDeviceTagsInPoint().size()) {
        auto cacheId = CalcCacheId();
        auto geoCorrection = CheckGeoConditions(context, cacheId);
        if (geoCorrection != EOfferCorrectorResult::Success) {
            return geoCorrection;
        }
    }
    if (!HostFilter.CheckHost(*server)) {
        return EOfferCorrectorResult::Unimplemented;
    }
    if (InsuranceDescriptions) {
        auto action = context.GetInsuranceDescription();
        if (!action || !InsuranceDescriptions.contains(action->GetName())) {
            return EOfferCorrectorResult::Unimplemented;
        }
    }

    EOfferCorrectorResult result = DoApplyForOffer(offer, tags, context, userId, server, session);

    if (result == EOfferCorrectorResult::Success && offer && !!offer->GetOffer() && GetSourceContext().IsLimitedPolicy()) {
        offer->GetOffer()->MutableProfitableTags().emplace(GetSourceContext().GetTagId());
    }
    if (result == EOfferCorrectorResult::Success && ResetSurgeTypes) {
        context.MutableSurgeTypes().clear();
    }
    if (result != EOfferCorrectorResult::Success && result != EOfferCorrectorResult::Unimplemented) {
        TUnistatSignalsCache::SignalAdd("correctors-" + GetName(), "building-" + ::ToString(result), 1);
    }
    offer->RecalcPrices(server);
    return result;
}

TVector<TString> IOfferCorrectorAction::GetNames(const NDrive::IServer* server) {
    if (!server || !server->GetDriveAPI() || !server->GetDriveAPI()->GetRolesManager()) {
        return {};
    }
    return server->GetDriveAPI()->GetRolesManager()->GetActionsDB().GetActionNamesWithType<IOfferCorrectorAction>(/*reportDeprecated=*/false);
}

namespace NDrive {

    bool IDeliveryAreaFilter::CheckDeliveryLocation(const TGeoCoord& location, const IServer& server) const {
        auto filter = GetDeliveryAreaFilter();
        if (!filter) {
            return true;
        }
        auto tags = Yensured(server.GetDriveAPI())->GetTagsInPoint(location);
        return filter->FilterWeak(tags);
    }

    TString GetOfferAgreement(const TString& agreement, const TOffersBuildingContext& context, const TString& insuranceType) {
        bool isCorporate = false;
        for (const auto& account : context.GetUserHistoryContextRef().GetRequestAccountIds()) {
            if (!ICommonOffer::NonCorporateMethods.contains(account)) {
                isCorporate = true;
                break;
            }
        }
        auto result = TStringBuilder();
        if (agreement) {
            result << agreement;
        } else if (isCorporate) {
            result << CorpAgreementId;
        } else {
            result << DefaultAgreementId;
        }
        if (insuranceType != "standart") {
            result << '_' << insuranceType << "_insurance";
        }
        if (context.HasDelegationCarTag()) {
            result << "_transferred";
        }
        return result;
    }

    void LogCreatedOffers(const TVector<IOfferReport::TPtr>& offers) {
        for (auto&& offer : offers) {
            auto commonOffer = offer ? offer->GetOffer() : nullptr;
            if (!commonOffer) {
                continue;
            }
            auto serializedOffer = commonOffer->SerializeToProto();
            NDrive::TEventLog::Log("OfferCreated", NJson::ToJson(NJson::Proto(serializedOffer)));
        }
    }

}
