#include "additional_service.h"

#include "long_term.h"

#include <drive/backend/offers/actions/abstract.h>
#include <drive/backend/offers/offers/additional_service.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <rtline/library/json/field.h>

namespace NDrive {

    TDailySchedule::TDailySchedule(TDuration beginDelay, TDuration endDelay, TDuration dayTime, TMaybe<TInstant> nowTime)
        : BeginDelay(beginDelay)
        , EndDelay(endDelay)
        , DayTime(dayTime)
        , NowTime(nowTime)
    {
    }

    bool TDailySchedule::Contains(TInstant time) const { 
        return time != TInstant::Zero() && time == GetNext(time - TDuration::Seconds(1));
    }

    TInstant TDailySchedule::GetNext(TInstant time) const {
        auto now = NowTime.GetOrElse(Now());
        auto nextTime = std::max(time, now + BeginDelay);
        nextTime = TInstant::Days(nextTime.Days()) + DayTime;
        // Check that invariant is not broken.
        if (nextTime < now + BeginDelay || nextTime <= time) {
            nextTime += TDuration::Days(1);
        }
        if (nextTime < now + BeginDelay || nextTime >= now + EndDelay) {
            return TInstant::Zero();
        }
        return nextTime;
    }

    NJson::TJsonValue TDailySchedule::SerializeToJson() const {
        NJson::TJsonValue json;
        NJson::InsertField(json, "begin_delay", NJson::Hr(BeginDelay));
        NJson::InsertField(json, "end_delay", NJson::Hr(EndDelay));
        NJson::InsertField(json, "day_time", NJson::Hr(DayTime));
        return json;
    }

    bool TDailySchedule::DeserializeFromJson(const NJson::TJsonValue& json) {
        return NJson::ParseField(json["begin_delay"], BeginDelay) &&
            NJson::ParseField(json["end_delay"], EndDelay) &&
            NJson::ParseField(json["day_time"], DayTime);
    }
    
}

NJson::TJsonValue TAdditionalServiceOfferReport::BuildJsonReport(ELocalization locale, NDriveSession::TReportTraits traits, const NDrive::IServer& server, const TUserPermissions& permissions) const {
    auto report = TBase::BuildJsonReport(locale, traits, server, permissions);
    auto offer = Yensured(GetOffer());
    bool hasAvailableTimes = false;
    if (AvailableSchedule) {
        NJson::TJsonValue availableTimes = NJson::JSON_ARRAY;
        TInstant time = AvailableSchedule->GetNext(Now());
        int limit = server.GetSettings().GetValue<int>("offers.additional_service_offer.daily_schedule_limit").GetOrElse(365);
        for (int i = 0; i < limit && time != TInstant::Zero(); ++i) {
            NJson::TJsonValue availableTime;
            availableTime.InsertValue("ts", time.Seconds());
            availableTimes.AppendValue(std::move(availableTime));
            time = AvailableSchedule->GetNext(time);
            hasAvailableTimes = true;
        }
        report.InsertValue("available_times", std::move(availableTimes));
    }
    auto localization = server.GetLocalization();
    auto getLocalString = [offer, locale, localization](const TString& text) {
        return localization ? offer->FormDescriptionElement(localization->GetLocalString(locale, text), locale, localization) : text;
    };
    auto action = server.GetDriveDatabase().GetUserPermissionManager().GetAction(offer->GetBehaviourConstructorId());
    auto serviceOfferBuilder = action ? action->GetAs<TAdditionalServiceOfferBuilder>() : nullptr;
    if (serviceOfferBuilder) {
        NJson::TJsonValue orderFields = NJson::JSON_ARRAY;
        if (hasAvailableTimes) {
            NJson::TJsonValue scheduled;
            scheduled["type"] = "scheduled_at";
            scheduled["title"] = getLocalString(serviceOfferBuilder->GetScheduledAtTitle());
            scheduled["subtitle"] = getLocalString(serviceOfferBuilder->GetScheduledAtSubtitle());
            orderFields.AppendValue(std::move(scheduled));
        }
        {
            NJson::TJsonValue delivery;
            delivery["type"] = "delivery_location";
            delivery["title"] = getLocalString(serviceOfferBuilder->GetDeliveryLocationTitle());
            delivery["same_location"] = getLocalString(serviceOfferBuilder->GetSameLocationTitle());
            orderFields.AppendValue(std::move(delivery));
        }
        {
            NJson::TJsonValue details;
            details["type"] = "details";
            details["title"] = getLocalString(serviceOfferBuilder->GetDetailsTitle());
            details["subtitle"] = getLocalString(serviceOfferBuilder->GetDetailsSubtitle());
            orderFields.AppendValue(std::move(details));
        }
        report.InsertValue("order_fields", std::move(orderFields));
    }
    return report;
}

bool TAdditionalServiceOfferBuilder::DeserializeSpecialsFromJson(const NJson::TJsonValue& value) {
    return
        TBase::DeserializeSpecialsFromJson(value) &&
        CarTagsFilter.DeserializeFromString(value["car_tags_filter"].GetString()) &&
        NJson::ParseField(value["price"], Price) &&
        NJson::ParseField(value["title"], Title) &&
        NJson::ParseField(value["subtitle"], Subtitle) &&
        NJson::ParseField(value["order_button_text"], OrderButtonText) &&
        NJson::ParseField(value["icon"], Icon) &&
        NJson::ParseField(value["scheduled_at_title"], ScheduledAtTitle) &&
        NJson::ParseField(value["scheduled_at_subtitle"], ScheduledAtSubtitle) &&
        NJson::ParseField(value["delivery_location_title"], DeliveryLocationTitle) &&
        NJson::ParseField(value["same_location_title"], SameLocationTitle) &&
        NJson::ParseField(value["details_title"], DetailsTitle) &&
        NJson::ParseField(value["details_subtitle"], DetailsSubtitle) &&
        NJson::ParseField(value["delivery_area_tags_filter"], DeliveryAreaTagsFilter) &&
        NJson::ParseField(value["device_tags"], DeviceTags) &&
        NJson::ParseField(value["cancel_start_user_tags"], CancelStartUserTags) &&
        DailySchedule.DeserializeFromJson(value["daily_schedule"]);
}

NJson::TJsonValue TAdditionalServiceOfferBuilder::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    result["car_tags_filter"] = CarTagsFilter.ToString();
    result["price"] = Price;
    result["title"] = Title;
    result["subtitle"] = Subtitle;
    result["order_button_text"] = OrderButtonText;
    result["icon"] = Icon;
    result["scheduled_at_title"] = ScheduledAtTitle;
    result["scheduled_at_subtitle"] = ScheduledAtSubtitle;
    result["delivery_location_title"] = DeliveryLocationTitle;
    result["same_location_title"] = SameLocationTitle;
    result["details_title"] = DetailsTitle;
    result["details_subtitle"] = DetailsSubtitle;
    result["delivery_area_tags_filter"] = DeliveryAreaTagsFilter;
    result["device_tags"] = NJson::ToJson(DeviceTags);
    result["cancel_start_user_tags"] = NJson::ToJson(CancelStartUserTags);
    result["daily_schedule"] = DailySchedule.SerializeToJson();
    return result;
}

NDrive::TScheme TAdditionalServiceOfferBuilder::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    NDrive::TScheme dailySchedule;
    dailySchedule.Add<TFSDuration>("begin_delay", "Доступен через");
    dailySchedule.Add<TFSDuration>("end_delay", "Перестает быть доступен через");
    dailySchedule.Add<TFSDuration>("day_time", "Время с начала дня");
    result.Add<TFSStructure>("daily_schedule", "Расписание (по дням)").SetStructure<NDrive::TScheme>(dailySchedule);
    result.Add<TFSNumeric>("price", "Цена (копейки)");
    result.Add<TFSString>("car_tags_filter", "Фильтр тегов машины в аренде");
    result.Add<TFSString>("title", "Заголовок");
    result.Add<TFSString>("subtitle", "Подзаголовок");
    result.Add<TFSString>("order_button_text", "Текст на кнопке бронирования");
    result.Add<TFSString>("icon", "Иконка");
    result.Add<TFSString>("scheduled_at_title", "Заголовок для запланированного времени");
    result.Add<TFSString>("scheduled_at_subtitle", "Подзаголовок для запланированного времени");
    result.Add<TFSString>("delivery_location_title", "Заголовок места доставки");
    result.Add<TFSString>("same_location_title", "Подзаголовок для вернуть в то же место");
    result.Add<TFSString>("details_title", "Заголовок деталей");
    result.Add<TFSString>("details_subtitle", "Подзаголовок деталей");
    result.Add<TFSString>("delivery_area_tags_filter", "Delivery area tags filter (comma separated)");
    result.Add<TFSVariants>("device_tags", "Device tags to be added on servicing start").SetReference("device_tags").SetMultiSelect(true);
    result.Add<TFSVariants>("cancel_start_user_tags", "User tags to be added on cancel start").SetReference("user_tags").SetMultiSelect(true);
    return result;
}

EOfferCorrectorResult TAdditionalServiceOfferBuilder::DoBuildOffers(
        const TUserPermissions& /*permissions*/,
        TVector<IOfferReport::TPtr>& offers,
        const TOffersBuildingContext& context,
        const NDrive::IServer* server,
        NDrive::TInfoEntitySession& tx
) const {
    auto offer = MakeAtomicShared<TAdditionalServiceOffer>();
    auto variables = context.GetVariables();
    offer->SetPrice(GetPrice());
    offer->SetSessionId(context.GetBillingSessionId());
    auto scheduledAt = NJson::FromJson<TMaybe<TInstant>>(variables["scheduled_at"]).GetOrElse(DailySchedule.GetNext(Now()));
    scheduledAt = DailySchedule.GetNext(scheduledAt - TDuration::Seconds(1));
    if (scheduledAt == TInstant::Zero()) {
        return EOfferCorrectorResult::Unimplemented;
    }
    offer->SetScheduledAt(scheduledAt);
    offer->SetUserId(context.GetUserHistoryContextUnsafe().GetUserId());
    auto deliveryLocation = NJson::FromJson<TMaybe<TGeoCoord>>(variables["delivery_location"]);
    if (deliveryLocation) {
        if (!CheckDeliveryLocation(*deliveryLocation, *Yensured(server))) {
            tx.SetCode(HTTP_BAD_REQUEST);
            tx.SetErrorInfo(
                "TAdditionalServiceOfferBuilder::DoBuildOffers",
                TStringBuilder() << "cannot deliver to " << deliveryLocation,
                NDrive::MakeError("bad_delivery_location")
            );
            return EOfferCorrectorResult::BuildingProblems;
        }
        offer->SetSameLocation(false);
        offer->SetDeliveryLocation(deliveryLocation);
    }
    auto deliveryLocationName = NJson::FromJson<TMaybe<TString>>(variables["delivery_location_name"]);
    if (deliveryLocationName) {
        offer->SetDeliveryLocationName(*deliveryLocationName);
    }
    if (context.HasUserHistoryContext()) {
        auto userDestination = context.GetUserHistoryContextRef().GetUserDestinationPtr();
        auto userFinishArea = userDestination ? userDestination->GetFinishArea() : nullptr;
        if (userFinishArea) {
            offer->SetDeliveryArea(userFinishArea->OptionalFinishArea());
        }
    }
    auto billingSession = context.GetBillingSession();
    auto availableSchedule = GetAvailableSchedule(billingSession ? billingSession->GetCurrentOffer() : nullptr);
    auto report = MakeAtomicShared<TAdditionalServiceOfferReport>(offer, nullptr);
    report->SetAvailableSchedule(availableSchedule);
    offers.push_back(report);
    return EOfferCorrectorResult::Success;
}

EOfferCorrectorResult TAdditionalServiceOfferBuilder::DoCheckOfferConditions(const TOffersBuildingContext& context, const TUserPermissions& permissions) const {
    Y_UNUSED(context);
    Y_UNUSED(permissions);
    if (!CarTagsFilter.IsEmpty()) {
        auto td = context.GetTaggedCar();
        if (!td) {
            return EOfferCorrectorResult::Unimplemented;
        }
        if (!CarTagsFilter.IsMatching(td->GetTags())) {
            return EOfferCorrectorResult::Unimplemented;
        }
    }
    return EOfferCorrectorResult::Success;
}

TMaybe<TBaseAreaTagsFilter> TAdditionalServiceOfferBuilder::GetDeliveryAreaFilter() const {
    if (DeliveryAreaTagsFilter) {
        return TBaseAreaTagsFilter::BuildFromString(DeliveryAreaTagsFilter);
    }
    return {};
}

NDrive::ISchedule::TPtr TAdditionalServiceOfferBuilder::GetAvailableSchedule(IOffer::TPtr parentOffer) const {
    auto schedule = MakeAtomicShared<NDrive::TDailySchedule>(DailySchedule);
    schedule->NowTime = Now();
    if (parentOffer) {
        if (auto longTermOffer = std::dynamic_pointer_cast<TLongTermOffer>(parentOffer)) {
            if (*schedule->NowTime >= longTermOffer->GetUntil()) {
                schedule->EndDelay = TDuration::Zero();
            } else {
                schedule->EndDelay = std::min(schedule->EndDelay, longTermOffer->GetUntil() - *schedule->NowTime);
            }
        }
    }
    return schedule;
}

TUserAction::TFactory::TRegistrator<TAdditionalServiceOfferBuilder> TAdditionalServiceOfferBuilder::Registrator(TAdditionalServiceOfferBuilder::GetTypeName());
