#include "checkers.h"

#include "correctors.h"

#include <drive/backend/billing/manager.h>
#include <drive/backend/database/drive_api.h>

TBINChecker::TFactory::TRegistrator<TBINChecker> TBINChecker::Registrator(TDiscountOfferCorrector::GetTypeStatic());

IDynamicActionChecker::ECheckResult TBINChecker::CheckAction(const TString& userId, const TContext& context, const TUserAction& action, const NDrive::IServer& server) const {
    const TDiscountOfferCorrector* corrector = dynamic_cast<const TDiscountOfferCorrector*>(&action);
    if (!corrector) {
        return IDynamicActionChecker::ECheckResult::EOk;
    }

    if (corrector->GetOffers().size()) {
        bool accept = false;
        for (auto&& offer : context.GetOfferBuilders()) {
            if (corrector->GetOffers().contains(offer->GetName())) {
                accept = true;
                break;
            }
        }
        if (!accept) {
            return IDynamicActionChecker::ECheckResult::EServiceFails;
        }
    }
    return CheckBINs(userId, action, server) ? IDynamicActionChecker::ECheckResult::EOk : IDynamicActionChecker::ECheckResult::EUserFails;
}

bool TBINChecker::CheckBINs(const TString& userId, const TUserAction& action, const NDrive::IServer& server) const {
    const TDiscountOfferCorrector* corrector = dynamic_cast<const TDiscountOfferCorrector*>(&action);
    if (!corrector || !corrector->GetCheckBINs()) {
        return true;
    }

    if (!server.GetDriveAPI()->HasBillingManager()) {
        return false;
    }

    const TBillingManager& billingMananger = server.GetDriveAPI()->GetBillingManager();
    TMaybe<TString> defaultCard;
    TUserPermissions::TConstPtr permissions;
    if (Context && Context->HasUserHistoryContext()) {
        defaultCard = Context->GetUserHistoryContextUnsafe().GetRequestCreditCard();
        permissions = Context->GetUserHistoryContextUnsafe().GetUserPermissions();
    } else {
        defaultCard = billingMananger.GetDefaultCreditCard(userId, Now() - TDuration::Seconds(10));
    }

    if (!defaultCard || defaultCard->empty()) {
        return false;
    }
    if (!permissions) {
        permissions = server.GetDriveAPI()->GetUserPermissions(userId, TUserPermissionsFeatures());
    }
    if (!permissions) {
        return false;
    }

    auto methods = server.GetDriveAPI()->GetUserPaymentMethodsSync(*permissions, server, true);
    auto paymentMethods = TBillingManager::GetUserPaymentCards(methods.Get(), false);
    if (!paymentMethods) {
        return false;
    }
    for (const auto& method : *paymentMethods) {
        if (method.Check(*defaultCard)) {
            return corrector->GetBINs().contains(method.GetBIN());
        }
    }
    return false;
}
