#include "correctors.h"

#include "abstract.h"
#include "checkers.h"
#include "fix_point.h"
#include "pack.h"
#include "standart.h"
#include "standard_with_discount_area.h"

#include <drive/backend/offers/context.h>
#include <drive/backend/offers/discount.h>
#include <drive/backend/offers/notifications.h>
#include <drive/backend/offers/ranking/calcer.h>
#include <drive/backend/offers/ranking/model.h>

#include <drive/backend/areas/areas.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/models/storage.h>
#include <drive/backend/roles/manager.h>

#include <drive/library/cpp/saturn/client.h>
#include <drive/library/cpp/scheme/scheme.h>
#include <drive/library/cpp/threading/future.h>
#include <drive/library/cpp/threading/future_cast.h>

#include <rtline/util/json_processing.h>
#include <rtline/util/algorithm/container.h>

#include <util/digest/fnv.h>

EOfferCorrectorResult TDestinationPredictor::DoApplyForOffer(IOfferReport* offerReport, const TVector<TDBTag>& tags, const TOffersBuildingContext& context, const TString& userId, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const {
    Y_UNUSED(tags);
    Y_UNUSED(userId);
    auto modelsStorage = server->GetModelsStorage();
    auto model = modelsStorage ? modelsStorage->GetOfferModel(Model) : nullptr;
    if (!model) {
        session.AddErrorMessage(GetName(), TStringBuilder() << "model " << Model << " is missing");
        return EOfferCorrectorResult::Unimplemented;
    }

    auto pricesContext = offerReport ? offerReport->GetOfferAs<TMarketPricesContext>() : nullptr;
    if (!pricesContext) {
        session.AddErrorMessage(GetName(), "cannot cast offer to MarketPricesContext");
        return EOfferCorrectorResult::Unimplemented;
    }

    NDrive::TOfferFeatures* featuresPtr = nullptr;
    NDrive::TOfferFeatures dryRunFeatures;
    if (DryRun) {
        dryRunFeatures = pricesContext->GetFeatures();
        featuresPtr = &dryRunFeatures;
    } else {
        featuresPtr = &pricesContext->MutableFeatures();
    }
    auto& features = *Yensured(featuresPtr);
    if (features.Floats[NDriveOfferFactors::FI_DESTINATION_SCORE] > 0 && !AlwaysPredict) {
        session.AddErrorMessage(GetName(), "no need to predict destination");
        return EOfferCorrectorResult::Unimplemented;
    }

    auto destinationSuggest = context.MutableUserDestinationSuggest();
    if (!destinationSuggest) {
        session.AddErrorMessage(GetName(), "UserDestinationSuggest is missing");
        return EOfferCorrectorResult::Unimplemented;
    }

    auto deadline = context.GetUserHistoryContextUnsafe().GetDeadline();
    auto& elements = destinationSuggest->Elements;
    auto best = elements.begin();
    for (auto i = elements.begin(); i != elements.end(); ++i) {
        const auto asyncGeoFeatures = NThreading::Initialize(i->GeoFeatures);
        const auto asyncGeobaseFeatures = NThreading::Initialize(i->GeobaseFeatures);
        const auto asyncUserGeoFeatures = NThreading::Initialize(i->UserGeoFeatures);
        const auto asyncUserGeobaseFeatures = NThreading::Initialize(i->UserGeobaseFeatures);
        const auto asyncUserDoubleGeoFeatures = NThreading::Initialize(i->UserDoubleGeoFeatures);
        const auto asyncUserDoubleGeobaseFeatures = NThreading::Initialize(i->UserDoubleGeobaseFeatures);
        const auto asyncRoute = NThreading::Initialize(i->Route);
        const auto name = TStringBuilder() << "DestinationSuggest:" << i->Latitude << ' ' << i->Longitude;
        {
            const auto guard = context.BuildEventGuard("WaitGeoFeatures:" + name);
            if (!asyncGeoFeatures.Wait(deadline)) {
                // do nothing
            }
        }
        {
            const auto guard = context.BuildEventGuard("WaitGeobaseFeatures:" + name);
            if (!asyncGeobaseFeatures.Wait(deadline)) {
                // do nothing
            }
        }
        {
            const auto guard = context.BuildEventGuard("WaitUserGeoFeatures:" + name);
            if (!asyncUserGeoFeatures.Wait(deadline)) {
                // do nothing
            }
        }
        {
            const auto guard = context.BuildEventGuard("WaitUserGeobaseFeatures:" + name);
            if (!asyncUserGeobaseFeatures.Wait(deadline)) {
                // do nothing
            }
        }
        {
            const auto guard = context.BuildEventGuard("WaitUserDoubleGeoFeatures:" + name);
            if (!asyncUserDoubleGeoFeatures.Wait(deadline)) {
                // do nothing
            }
        }
        {
            const auto guard = context.BuildEventGuard("WaitUserDoubleGeobaseFeatures:" + name);
            if (!asyncUserDoubleGeobaseFeatures.Wait(deadline)) {
                // do nothing
            }
        }
        {
            const auto guard = context.BuildEventGuard("WaitRoute:" + name);
            if (!asyncRoute.Wait(deadline)) {
                // do nothing
            }
        }

        const auto coordinate = TGeoCoord(i->Longitude, i->Latitude);
        const auto geoFeatures = asyncGeoFeatures.HasValue() ? asyncGeoFeatures.GetValue().Get() : nullptr;
        const auto geobaseFeatures = asyncGeobaseFeatures.HasValue() ? asyncGeobaseFeatures.GetValue().Get() : nullptr;
        const auto userGeoFeatures = asyncUserGeoFeatures.HasValue() ? asyncUserGeoFeatures.GetValue().Get() : nullptr;
        const auto userGeobaseFeatures = asyncUserGeobaseFeatures.HasValue() ? asyncUserGeobaseFeatures.GetValue().Get() : nullptr;
        const auto userDoubleGeoFeatures = asyncUserDoubleGeoFeatures.HasValue() ? asyncUserDoubleGeoFeatures.GetValue().Get() : nullptr;
        const auto userDoubleGeobaseFeatures = asyncUserDoubleGeobaseFeatures.HasValue() ? asyncUserDoubleGeobaseFeatures.GetValue().Get() : nullptr;
        const auto route = asyncRoute.HasValue() ? asyncRoute.GetValue().Get() : nullptr;
        double distance = route ? route->Length : -2;
        double duration = route ? route->Time : -2;
        auto geobaseId = i->GeobaseId.GetOrElse(0);
        NDrive::CalcDestinationFeatures(features, coordinate, geoFeatures, userGeoFeatures, distance, duration);
        NDrive::CalcDestinationFeatures(features, geobaseId, geobaseFeatures, userGeobaseFeatures);
        NDrive::CalcDestinationFeatures(features, userDoubleGeoFeatures);
        NDrive::CalcDestinationFeatures(features, geobaseId, userDoubleGeobaseFeatures);
        i->Score = model->Calc(features);
        if (best->Score < i->Score) {
            best = i;
        }
    }
    NJson::TJsonValue destinationPredictorEvent = NJson::TMapBuilder
        ("offer_id", offerReport && offerReport->GetOffer() ? offerReport->GetOffer()->GetOfferId() : TString{})
        ("best", NJson::PointerToJson(best != elements.end() ? &*best : nullptr))
        ("model", model->GetName())
    ;
    if (VerboseLogging) {
        destinationPredictorEvent["elements"] = NJson::ToJson(elements);
        destinationPredictorEvent["source_geo_features"] = NJson::PointerToJson(context.GetGeoFeatures());
        destinationPredictorEvent["source_geobase_features"] = NJson::PointerToJson(context.GetGeobaseFeatures());
        destinationPredictorEvent["source_user_geo_features"] = NJson::PointerToJson(context.GetUserGeoFeatures());
        destinationPredictorEvent["source_user_geobase_features"] = NJson::PointerToJson(context.GetUserGeobaseFeatures());
        destinationPredictorEvent["user_features"] = NJson::PointerToJson(context.GetUserFeatures());
        destinationPredictorEvent["features"] = NJson::ToJson(features);
    }
    NDrive::TEventLog::Log("DestinationPredictor", destinationPredictorEvent);
    if (best != elements.end() && !best->Route.Initialized() && CalcBestRoute) {
        auto start = context.GetStartPosition();
        auto finish = MakeMaybe<TGeoCoord>(best->Longitude, best->Latitude);
        if (start && finish) {
            auto guard = context.BuildEventGuard("CalcBestRoute");
            best->Route = context.BuildRoute(*start, *finish, "ptAuto");
            if (!best->Route.Initialized() || !best->Route.Wait(deadline) || !best->Route.HasValue()) {
                NDrive::TEventLog::Log("CalcBestRouteError", NJson::ToJson(best->Route));
            }
        }
    }
    if (best != elements.end()) {
        const auto asyncGeoFeatures = NThreading::Initialize(best->GeoFeatures);
        const auto asyncGeobaseFeatures = NThreading::Initialize(best->GeobaseFeatures);
        const auto asyncUserGeoFeatures = NThreading::Initialize(best->UserGeoFeatures);
        const auto asyncUserGeobaseFeatures = NThreading::Initialize(best->UserGeobaseFeatures);
        const auto asyncUserDoubleGeoFeatures = NThreading::Initialize(best->UserDoubleGeoFeatures);
        const auto asyncUserDoubleGeobaseFeatures = NThreading::Initialize(best->UserDoubleGeobaseFeatures);
        const auto asyncRoute = NThreading::Initialize(best->Route);
        const auto coordinate = TGeoCoord(best->Longitude, best->Latitude);
        const auto geoFeatures = asyncGeoFeatures.HasValue() ? asyncGeoFeatures.GetValue().Get() : nullptr;
        const auto geobaseFeatures = asyncGeobaseFeatures.HasValue() ? asyncGeobaseFeatures.GetValue().Get() : nullptr;
        const auto userGeoFeatures = asyncUserGeoFeatures.HasValue() ? asyncUserGeoFeatures.GetValue().Get() : nullptr;
        const auto userGeobaseFeatures = asyncUserGeobaseFeatures.HasValue() ? asyncUserGeobaseFeatures.GetValue().Get() : nullptr;
        const auto userDoubleGeoFeatures = asyncUserDoubleGeoFeatures.HasValue() ? asyncUserDoubleGeoFeatures.GetValue().Get() : nullptr;
        const auto userDoubleGeobaseFeatures = asyncUserDoubleGeobaseFeatures.HasValue() ? asyncUserDoubleGeobaseFeatures.GetValue().Get() : nullptr;
        const auto route = asyncRoute.HasValue() ? asyncRoute.GetValue().Get() : nullptr;
        double distance = route ? route->Length : -2;
        double duration = route ? route->Time : -2;
        auto geobaseId = best->GeobaseId.GetOrElse(0);
        NDrive::CalcDestinationFeatures(features, coordinate, geoFeatures, userGeoFeatures, distance, duration);
        NDrive::CalcDestinationFeatures(features, geobaseId, geobaseFeatures, userGeobaseFeatures);
        NDrive::CalcDestinationFeatures(features, userDoubleGeoFeatures);
        NDrive::CalcDestinationFeatures(features, geobaseId, userDoubleGeobaseFeatures);
        const auto score = best->Score ? Sigmoid(*best->Score) : 0;
        NDrive::CalcBestDestinationFeatures(features, score);
    } else {
        return EOfferCorrectorResult::Unimplemented;
    }

    return EOfferCorrectorResult::Success;
}

bool TDestinationPredictor::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DeserializeSpecialsFromJson(jsonInfo)) {
        return false;
    }
    return
        NJson::ParseField(jsonInfo["always_predict"], AlwaysPredict) &&
        NJson::ParseField(jsonInfo["calc_best_route"], CalcBestRoute) &&
        NJson::ParseField(jsonInfo["dry_run"], DryRun) &&
        NJson::ParseField(jsonInfo["verbose_logging"], VerboseLogging) &&
        NJson::ParseField(jsonInfo["model"], Model);
}

NJson::TJsonValue TDestinationPredictor::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    result["always_predict"] = AlwaysPredict;
    result["calc_best_route"] = CalcBestRoute;
    result["dry_run"] = DryRun;
    result["verbose_logging"] = VerboseLogging;
    result["model"] = NJson::ToJson(Model);
    return result;
}

NDrive::TScheme TDestinationPredictor::DoGetScheme(const NDrive::IServer* server) const {
    auto modelsStorage = server ? server->GetModelsStorage() : nullptr;
    auto models = modelsStorage ? modelsStorage->ListOfferModels() : TVector<TString>();

    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSBoolean>("always_predict").SetDefault(AlwaysPredict);
    result.Add<TFSBoolean>("calc_best_route").SetDefault(CalcBestRoute);
    result.Add<TFSBoolean>("dry_run").SetDefault(DryRun);
    result.Add<TFSBoolean>("verbose_logging").SetDefault(VerboseLogging);
    result.Add<TFSVariants>("model", "Predition model").SetVariants(models);
    return result;
}

EOfferCorrectorResult TDisableCorrector::DoApplyForOffer(IOfferReport* offerReport, const TVector<TDBTag>& tags, const TOffersBuildingContext& context, const TString& userId, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const {
    EOfferCorrectorResult base = TBase::DoApplyForOffer(offerReport, tags, context, userId, server, session);
    if (base != EOfferCorrectorResult::Success) {
        return base;
    }
    if (!offerReport || !offerReport->GetOffer()) {
        return EOfferCorrectorResult::Problems;
    }
    offerReport->GetOffer()->MutableDisabledCorrectors().insert(Correctors.begin(), Correctors.end());
    return EOfferCorrectorResult::Success;
}

bool TDisableCorrector::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DeserializeSpecialsFromJson(jsonInfo)) {
        return false;
    }
    return
        NJson::ParseField(jsonInfo["correctors"], Correctors);
}

NJson::TJsonValue TDisableCorrector::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    result["correctors"] = NJson::ToJson(Correctors);
    return result;
}

NDrive::TScheme TDisableCorrector::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    const auto& actionsDb = server->GetDriveAPI()->GetRolesManager()->GetActionsDB();
    TSet<TString> actions;
    for (auto&& action : actionsDb.GetActionsWithType<TDisableCorrector>()) {
        actions.insert(action.GetName());
    }
    for (auto&& action : actionsDb.GetActionsWithType<TDiscountOfferCorrector>()) {
        actions.insert(action.GetName());
    }
    for (auto&& action : actionsDb.GetActionsWithType<TPriceModelOfferCorrector>()) {
        actions.insert(action.GetName());
    }
    for (auto&& action : actionsDb.GetActionsWithType<TVisibilityOfferCorrector>()) {
        actions.insert(action.GetName());
    }
    result.Add<TFSVariants>("correctors", "Correctors to disable").SetVariants(actions).SetMultiSelect(true);
    return result;
}

bool TDiscountOfferCorrector::THowToGetDescription::FromJson(const NJson::TJsonValue& json) {
    JREAD_BOOL(json, "flag", IsActive);
    JREAD_STRING_OPT(json, "title", Title);
    JREAD_STRING_OPT(json, "description", Description);
    JREAD_STRING_OPT(json, "detailed_description", DetailedDescription);
    JREAD_STRING_OPT(json, "more_button", MoreButton);
    JREAD_STRING_OPT(json, "logo", Logo);
    JREAD_STRING_OPT(json, "background_image", BGImage);
    return true;
}

NJson::TJsonValue TDiscountOfferCorrector::THowToGetDescription::ToJson() const {
    NJson::TJsonValue json;
    JWRITE(json, "flag", IsActive);
    JWRITE_DEF(json, "title", Title, "");
    JWRITE_DEF(json, "description", Description, "");
    JWRITE_DEF(json, "detailed_description", DetailedDescription, "");
    JWRITE_DEF(json, "more_button", MoreButton, "");
    JWRITE_DEF(json, "logo", Logo, "");
    JWRITE_DEF(json, "background_image", BGImage, "");
    return json;
}

NJson::TJsonValue TDiscountOfferCorrector::THowToGetDescription::GetPublicReport(ELocalization locale, const ILocalization& localization) const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "title", localization.ApplyResources(Title, locale));
    NJson::InsertNonNull(result, "full_description", localization.ApplyResources(Description, locale));
    NJson::InsertNonNull(result, "detailed_description", localization.ApplyResources(DetailedDescription, locale));
    NJson::InsertNonNull(result, "more_button", localization.ApplyResources(MoreButton, locale));

    auto& image = result["image"].SetType(NJson::JSON_MAP);
    NJson::InsertNonNull(image, "logo", Logo);
    NJson::InsertNonNull(image, "background_image", BGImage);
    return result;
}

NDrive::TScheme TDiscountOfferCorrector::THowToGetDescription::GetScheme() const {
    NDrive::TScheme scheme;
    scheme.Add<TFSBoolean>("flag", "Показывать блок \"Как получить\"").SetDefault(false);
    scheme.Add<TFSString>("title", "Заголовок блока \"Как получить\"");
    scheme.Add<TFSText>("description", "Описание блока \"Как получить\"");
    scheme.Add<TFSText>("detailed_description", "Полное описание блока \"Как получить\"");
    scheme.Add<TFSString>("more_button", "Кнопка полного описания блока \"Как получить\"");
    scheme.Add<TFSString>("logo", "Лого блока \"Как получить\"");
    scheme.Add<TFSString>("background_image", "Фон блока \"Как получить\"");
    return scheme;
}

bool TDiscountOfferCorrector::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DeserializeSpecialsFromJson(jsonInfo)) {
        return false;
    }
    JREAD_STRING_OPT(jsonInfo, "description", DiscountDescription);
    JREAD_DOUBLE_OPT(jsonInfo, "discount", Discount);
    JREAD_STRING_OPT(jsonInfo, "profit_description", ProfitDescription);
    JREAD_STRING_OPT(jsonInfo, "profit_color", ProfitColor);
    JREAD_BOOL_OPT(jsonInfo, "profit_visible", ProfitVisible);

    if (jsonInfo.Has("how_to_get") && !HowToGetBlock.FromJson(jsonInfo["how_to_get"])) {
        return false;
    }

    JREAD_BOOL_OPT(jsonInfo, "check_bins", CheckBINs);
    JREAD_BOOL_OPT(jsonInfo, "visible", Visible);
    JREAD_STRING_OPT(jsonInfo, "icon", Icon);
    JREAD_STRING_OPT(jsonInfo, "small_icon", SmallIcon);
    JREAD_STRING_OPT(jsonInfo, "tag_name", TagName);
    if (TagName.StartsWith("$")) {
        Matcher.Compile(TagName.substr(1));
        if (!Matcher.IsCompiled()) {
            return false;
        }
    }
    if (Discount > 0) {
        Discount = Min(1.0, Max(0.0, Discount));
    }
    if (jsonInfo.Has("tags_details")) {
        if (!jsonInfo["tags_details"].IsArray()) {
            return false;
        }
        for (auto&& i : jsonInfo["tags_details"].GetArraySafe()) {
            TDiscount::TDiscountDetails details;
            if (!details.DeserializeFromJson(i)) {
                return false;
            }
            DiscountByTagsPerforming.emplace(details.GetTagName(), details);
        }
    }

    if (jsonInfo.Has("bins_list")) {
        for (auto&& bin : jsonInfo["bins_list"].GetArray()) {
            if (!bin.IsInteger()) {
                return false;
            }
            BINs.insert(bin.GetUInteger());
        }
    }

    if (jsonInfo.Has("discount_timetable")) {
        TPriceByTimeConfig config;
        if (config.DeserializeFromJson(jsonInfo["discount_timetable"])) {
            DiscountTimetable = config;
        } else {
            return false;
        }
    }

    return true;
}

NJson::TJsonValue TDiscountOfferCorrector::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    JWRITE(result, "description", DiscountDescription);
    JWRITE_DEF(result, "discount", Discount, 0);
    JWRITE_DEF(result, "profit_description", ProfitDescription, "");
    JWRITE_DEF(result, "profit_color", ProfitColor, "");
    JWRITE_DEF(result, "profit_visible", ProfitVisible, false);
    JWRITE(result, "how_to_get", HowToGetBlock.ToJson());

    JWRITE(result, "icon", Icon);
    JWRITE(result, "small_icon", SmallIcon);
    JWRITE(result, "tag_name", TagName);
    JWRITE(result, "visible", Visible);
    if (DiscountByTagsPerforming.size()) {
        NJson::TJsonValue discountDetailsJson(NJson::JSON_ARRAY);
        for (auto&& i : DiscountByTagsPerforming) {
            discountDetailsJson.AppendValue(i.second.SerializeToJson());
        }
        result.InsertValue("tags_details", discountDetailsJson);
    }
    result["check_bins"] = CheckBINs;
    NJson::TJsonValue jsonBINs(NJson::JSON_ARRAY);
    for (auto&& bin : BINs) {
        jsonBINs.AppendValue(bin);
    }
    result.InsertValue("bins_list", std::move(jsonBINs));
    if (DiscountTimetable) {
        result.InsertValue("discount_timetable", DiscountTimetable->SerializeToJson());
    }

    return result;
}

NJson::TJsonValue TDiscountOfferCorrector::GetPublicReport(ELocalization locale, const ILocalization& localization) const {
    NJson::TJsonValue result = TBase::GetPublicReport(locale, localization);
    NJson::TJsonValue& discountValue = result["discount_value"].SetType(NJson::JSON_MAP);
    TString description = localization.ApplyResources(ProfitDescription, locale);
    if (!description) {
        description = "–" + localization.FormatPrice(locale, Discount * 10000) + "%";
    }
    JWRITE_DEF(discountValue, "discount", Discount, 0);
    JWRITE(discountValue, "profit_description", description);

    JWRITE(result, "full_description", localization.ApplyResources(DiscountDescription, locale));

    NJson::TJsonValue& image = result["image"].SetType(NJson::JSON_MAP);
    JWRITE_DEF(image, "icon", Icon, "");
    JWRITE_DEF(image, "small_icon", SmallIcon, "");
    JWRITE_DEF(image, "profit_color", ProfitColor, "");
    return result;
}

NDrive::TScheme TDiscountOfferCorrector::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSString>("tag_name", "Идентификатор активирующего тега");
    result.Add<TFSString>("small_icon", "Новая иконка скидки");
    result.Add<TFSString>("icon", "Иконка скидки");
    result.Add<TFSString>("description", "Описание скидки");
    result.Add<TFSNumeric>("discount", "Размер скидки (доля от цены)").SetPrecision(2).SetMin(-1).SetMax(1);
    result.Add<TFSBoolean>("profit_visible", "Отображать скидку на вкладке Блага");
    result.Add<TFSString>("profit_description", "Краткое описание значения скидки");
    result.Add<TFSString>("profit_color", "Цвет карточки скидки").SetVisual(TFSString::EVisualType::Color);
    result.Add<TFSStructure>("how_to_get", "Как получить скидку").SetStructure(HowToGetBlock.GetScheme());
    result.Add<TFSJson>("discount_timetable", "Расписание размера скидки в процентах");
    result.Add<TFSBoolean>("visible", "Отображение в карточке");

    NDrive::TScheme& detailScheme = result.Add<TFSArray>("tags_details", "Детализация по сегментам").SetElement<NDrive::TScheme>();

    detailScheme.Add<TFSString>("tag_name", "Имя тега сегмента");
    detailScheme.Add<TFSNumeric>("discount", "Размер скидки (доля от цены)").SetPrecision(2).SetMin(-1).SetMax(1);
    detailScheme.Add<TFSNumeric>("additional_duration", "Дополнительное бесплатное время в данной секции");

    detailScheme.Add<TFSVariants>("tags_in_point", "Пространственные теги активации").SetMultiSelect(true).SetVariants(server->GetDriveAPI()->GetAreasDB()->GetAreaTags(TDuration::Minutes(1))).MulVariantsLeft({"", "!"});
    detailScheme.Add<TFSJson>("free_timetable", "Расписание не тарифицируемых периодов");

    result.Add<TFSBoolean>("check_bins", "Включить скидки по бинам");
    result.Add<TFSJson>("bins_list", "Список бинов для скидки");

    return result;
}

EOfferCorrectorResult TDiscountOfferCorrector::DoApplyForOffer(IOfferReport* offerReport, const TVector<TDBTag>& tags, const TOffersBuildingContext& context, const TString& userId, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const {
    EOfferCorrectorResult base = TBase::DoApplyForOffer(offerReport, tags, context, userId, server, session);
    if (base != EOfferCorrectorResult::Success) {
        return base;
    }
    if (!offerReport || !offerReport->GetOffer()) {
        return EOfferCorrectorResult::Unimplemented;
    }
    IOfferWithDiscounts* dOffer = dynamic_cast<IOfferWithDiscounts*>(offerReport->GetOffer().Get());
    if (!dOffer) {
        return EOfferCorrectorResult::Unimplemented;
    }
    bool match = !TagName;
    if (!match) {
        for (auto&& i : tags) {
            if (TagName.StartsWith("$")) {
                if (Matcher.Match(i->GetName().data())) {
                    match = true;
                    break;
                }
            } else {
                if (i->GetName() == TagName) {
                    match = true;
                    break;
                }
            }
        }
    }
    if (!match) {
        return EOfferCorrectorResult::Unimplemented;
    }

    if (CheckBINs) {
        TBINChecker BINChecker(&context);
        if (!BINChecker.CheckBINs(userId, *this, *server)) {
            return EOfferCorrectorResult::Unimplemented;
        }
    }
    dOffer->AddDiscount(CreateDiscount(dOffer->GetTimestamp()));
    return EOfferCorrectorResult::Success;
}

TDiscount TDiscountOfferCorrector::CreateDiscount(TInstant timestamp) const {
    TDiscount discount;
    if (DiscountTimetable) {
        double value = 0.01 * DiscountTimetable->GetBasePrice(timestamp);
        discount.SetDiscount(value);
    } else {
        discount.SetDiscount(Discount);
    }
    discount.SetDetails(DiscountByTagsPerforming);
    discount.SetIdentifier(GetName());
    discount.SetVisible(Visible);
    discount.SetPromoCode(GetSourceContext().IsPromoCode());
    return discount;
}

EOfferCorrectorResult TInsuranceOfferCorrector::DoApplyForOffer(IOfferReport* offerReport, const TVector<TDBTag>& tags, const TOffersBuildingContext& context, const TString& userId, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const {
    EOfferCorrectorResult base = TBase::DoApplyForOffer(offerReport, tags, context, userId, server, session);
    if (base != EOfferCorrectorResult::Success) {
        return base;
    }

    auto standartOffer = offerReport ? offerReport->GetOfferAs<TStandartOffer>() : nullptr;
    if (!standartOffer) {
        return EOfferCorrectorResult::Unimplemented;
    }

    auto permissions = context.HasUserHistoryContext() ? context.GetUserHistoryContextRef().GetUserPermissions() : nullptr;
    auto option = permissions ? permissions->GetOptionSafe(InsuranceTypeId) : nullptr;
    auto optionValues = option ? option->GetValues() : TVector<TString>();
    bool hasInsuranceType = std::find(optionValues.begin(), optionValues.end(), InsuranceType) != optionValues.end();

    if (standartOffer->GetInsuranceType() != InsuranceType) {
        auto poc = offerReport->GetPriceOfferConstructor();
        auto pocPriceComponent = poc ? poc->GetEquilibrium().GetOptionalPriceComponent(InsuranceTypeId + ":" + InsuranceType) : nullptr;
        auto insurancePriceComponent = InsurancePriceComponent ? InsurancePriceComponent.Get() : pocPriceComponent;
        if (insurancePriceComponent && hasInsuranceType) {
            offerReport->MutableInsurancePrices()[InsuranceType] = insurancePriceComponent->GetMinuteRidingVolume();
        }
        return EOfferCorrectorResult::Unimplemented;
    }

    bool modified = false;
    auto packOffer = dynamic_cast<TPackOffer*>(standartOffer);
    if (packOffer) {
        auto duration = packOffer->GetDuration();
        if (PackDurationLowerLimit && *PackDurationLowerLimit > duration) {
            return EOfferCorrectorResult::Unimplemented;
        }
        if (PackDurationUpperLimit && *PackDurationUpperLimit <= duration) {
            return EOfferCorrectorResult::Unimplemented;
        }
        if (PackPriceMultiplier) {
            auto original = packOffer->GetPackPrice();
            auto multiplied = ICommonOffer::RoundPrice(*PackPriceMultiplier * original, 100);
            packOffer->SetPackPrice(multiplied);
            packOffer->SetPackInsurancePrice(static_cast<i32>(multiplied) - original);
            modified = true;
        }
        if (PackPriceMultiplier && ApplyToOverPrices) {
            auto originalOverrun = packOffer->GetOverrunKm();
            auto originalOvertimeParking = packOffer->GetOvertimeParking();
            auto originalOvertimeRiding = packOffer->GetOvertimeRiding();
            auto multipliedOverrun = ICommonOffer::RoundPrice(*PackPriceMultiplier * originalOverrun);
            auto multipliedOvertimeParking = ICommonOffer::RoundPrice(*PackPriceMultiplier * originalOvertimeParking);
            auto multipliedOvertimeRiding = ICommonOffer::RoundPrice(*PackPriceMultiplier * originalOvertimeRiding);
            packOffer->SetPackInsuranceOverrunPrice(static_cast<i32>(multipliedOverrun) - originalOverrun);
            if (packOffer->HasOverrunKm()) {
                packOffer->SetOverrunKm(multipliedOverrun);
            } else {
                packOffer->MutableKm().SetPrice(multipliedOverrun, GetName());
            }
            if (packOffer->HasOvertimeParking()) {
                packOffer->SetOvertimeParking(multipliedOvertimeParking);
            } else {
                packOffer->MutableParking().SetPrice(multipliedOvertimeParking, GetName());
            }
            if (packOffer->HasOvertimeRiding()) {
                packOffer->SetOvertimeRiding(multipliedOvertimeRiding);
            } else {
                packOffer->MutableRiding().SetPrice(multipliedOvertimeRiding, GetName());
            }
            modified = true;
        }
    }
    if (standartOffer) {
        auto poc = offerReport->GetPriceOfferConstructor();
        auto pocPriceComponent = poc ? poc->GetEquilibrium().GetOptionalPriceComponent(InsuranceTypeId + ":" + standartOffer->GetInsuranceType()) : nullptr;
        auto insurancePriceComponent = InsurancePriceComponent ? InsurancePriceComponent.Get() : pocPriceComponent;
        if (insurancePriceComponent && option && hasInsuranceType) {
            i32 before = standartOffer->GetRiding().GetPrice();
            insurancePriceComponent->ApplyForContext(/*useKm=*/static_cast<bool>(standartOffer->GetKm()), *standartOffer);
            i32 after = standartOffer->GetRiding().GetPrice();
            standartOffer->SetInsuranceCost(after - before);
            modified = true;
        }
    }

    auto fixPointOffer = dynamic_cast<TFixPointOffer*>(standartOffer);
    if (fixPointOffer) {
        i32 before = fixPointOffer->GetPackPrice();
        i32 after = fixPointOffer->CalcPackPrice(server);
        if (after > before) {
            fixPointOffer->SetPackInsurancePrice(after - before);
            modified = true;
        }
    }

    if (modified) {
        return EOfferCorrectorResult::Success;
    } else {
        return EOfferCorrectorResult::Unimplemented;
    }
}

bool TInsuranceOfferCorrector::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DeserializeSpecialsFromJson(jsonInfo)) {
        return false;
    }
    return
        NJson::ParseField(jsonInfo["apply_to_over_prices"], ApplyToOverPrices) &&
        NJson::ParseField(jsonInfo["pack_duration_lower_limit"], PackDurationLowerLimit) &&
        NJson::ParseField(jsonInfo["pack_duration_upper_limit"], PackDurationUpperLimit) &&
        NJson::ParseField(jsonInfo["pack_price_multiplier"], PackPriceMultiplier) &&
        NJson::ParseField(jsonInfo["insurance_price_component"], InsurancePriceComponent) &&
        NJson::ParseField(jsonInfo["insurance_type"], InsuranceType, true);
}

NJson::TJsonValue TInsuranceOfferCorrector::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    NJson::InsertField(result, "apply_to_over_prices", ApplyToOverPrices);
    NJson::InsertField(result, "insurance_type", InsuranceType);
    NJson::InsertNonNull(result, "insurance_price_component", InsurancePriceComponent);
    NJson::InsertNonNull(result, "pack_duration_lower_limit", PackDurationLowerLimit);
    NJson::InsertNonNull(result, "pack_duration_upper_limit", PackDurationUpperLimit);
    NJson::InsertNonNull(result, "pack_price_multiplier", PackPriceMultiplier);
    return result;
}

NDrive::TScheme TInsuranceOfferCorrector::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSBoolean>("apply_to_over_prices").SetDefault(ApplyToOverPrices);
    result.Add<TFSStructure>("insurance_price_component").SetStructure(TOptionalPriceComponent::GetScheme(server));
    result.Add<TFSVariants>("insurance_type").SetVariants({
        "full",
        "standart",
    });
    result.Add<TFSDuration>("pack_duration_lower_limit");
    result.Add<TFSDuration>("pack_duration_upper_limit");
    result.Add<TFSNumeric>("pack_price_multiplier");
    return result;
}

void TPriceModelOfferCorrector::LogModelResult(const IOfferReport* offerReport, const NDrive::IOfferModel& model, NDrive::TOfferFeatures& features) const {
    auto result = model.Calc(features);
    NDrive::TEventLog::Log("OfferCorrectorPriceModelResult", NJson::TMapBuilder
        ("offer_id", offerReport->GetOffer()->GetOfferId())
        ("corrector", GetName())
        ("model", model.GetName())
        ("result", result)
    );
}

EOfferCorrectorResult TPriceModelOfferCorrector::DoApplyForOffer(IOfferReport* offerReport, const TVector<TDBTag>& tags, const TOffersBuildingContext& context, const TString& userId, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const {
    EOfferCorrectorResult base = TBase::DoApplyForOffer(offerReport, tags, context, userId, server, session);
    if (base != EOfferCorrectorResult::Success) {
        return base;
    }
    if (!offerReport || !offerReport->GetOffer()) {
        session.SetErrorInfo("PriceModelOfferCorrector::DoApplyForOffer", "null offer", EDriveSessionResult::InconsistencyOffer);
        return EOfferCorrectorResult::Problems;
    }
    if (!server) {
        session.SetErrorInfo("PriceModelOfferCorrector::DoApplyForOffer", "null server", EDriveSessionResult::InconsistencySystem);
        return EOfferCorrectorResult::Problems;
    }
    auto models = server->GetModelsStorage();
    if (!models) {
        session.SetErrorInfo("PriceModelOfferCorrector::DoApplyForOffer", "null ModelsStorage", EDriveSessionResult::InconsistencySystem);
        return EOfferCorrectorResult::Problems;
    }
    TFullPricesContext* fpContext = dynamic_cast<TFullPricesContext*>(offerReport->GetOffer().Get());
    if (!fpContext) {
        return EOfferCorrectorResult::Unimplemented;
    }
    // This values should be updated after applying all models.
    auto scoringValue = fpContext->MutableFeatures().Floats2[NDriveOfferFactors2::SCORING_MODEL_VALUE];
    if (ScoringFeatureModel) {
        auto model = models->GetOfferModel(ScoringFeatureModel);
        if (!model) {
            session.SetErrorInfo("PriceModelOfferCorrector::DoApplyForOffer", TStringBuilder() << "ScoringFeatureModel " << ScoringFeatureModel << " is missing", EDriveSessionResult::InconsistencySystem);
            return EOfferCorrectorResult::Problems;
        }
        if (DryRun) {
            LogModelResult(offerReport, *model, fpContext->MutableFeatures());
        } else {
            scoringValue = model->Calc(fpContext->MutableFeatures());
        }
    }
    auto percentValue = fpContext->MutableFeatures().Floats2[NDriveOfferFactors2::PERCENT_MODEL_VALUE];
    if (PercentFeatureModel) {
        auto model = models->GetOfferModel(PercentFeatureModel);
        if (!model) {
            session.SetErrorInfo("PriceModelOfferCorrector::DoApplyForOffer", TStringBuilder() << "PercentFeatureModel " << PercentFeatureModel << " is missing", EDriveSessionResult::InconsistencySystem);
            return EOfferCorrectorResult::Problems;
        }
        if (DryRun) {
            LogModelResult(offerReport, *model, fpContext->MutableFeatures());
        } else {
            percentValue = model->Calc(fpContext->MutableFeatures());
        }
    }
    if (auto standart = offerReport->GetOfferAs<TStandartOffer>()) {
        if (CashbackPercentModel) {
            auto model = models->GetOfferModel(CashbackPercentModel);
            if (!model) {
                session.SetErrorInfo(
                    "PriceModelOfferCorrector::DoApplyForOffer",
                    TStringBuilder() << "CashbackPercentModel " << CashbackPercentModel << " is missing",
                    EDriveSessionResult::InconsistencySystem
                );
                return EOfferCorrectorResult::Problems;
            }
            if (DryRun) {
                LogModelResult(offerReport, *model, fpContext->MutableFeatures());
            } else {
                standart->ApplyCashbackPercentModel(*model);
            }
        }
        if (InsurancePriceModel && standart->GetInsuranceType() == FullInsuranceType) {
            auto model = models->GetOfferModel(InsurancePriceModel);
            if (!model) {
                session.SetErrorInfo("PriceModelOfferCorrector::DoApplyForOffer", TStringBuilder() << "InsurancePriceModel " << InsurancePriceModel << " is missing", EDriveSessionResult::InconsistencySystem);
                return EOfferCorrectorResult::Problems;
            }
            if (DryRun) {
                LogModelResult(offerReport, *model, fpContext->MutableFeatures());
            } else {
                standart->ApplyInsuranceCostModel(*model);
            }
        }
        if (DepositAmountModel) {
            auto model = models->GetOfferModel(DepositAmountModel);
            if (!model) {
                session.SetErrorInfo(
                    "PriceModelOfferCorrector::DoApplyForOffer",
                    TStringBuilder() << "DepositAmountModel " << DepositAmountModel << " is missing",
                    EDriveSessionResult::InconsistencySystem
                );
                return EOfferCorrectorResult::Problems;
            }
            if (DryRun) {
                LogModelResult(offerReport, *model, fpContext->MutableFeatures());
            } else {
                standart->ApplyDepositAmountModel(*model);
            }
        }
    }
    auto pack = offerReport->GetOfferAs<TPackOffer>();
    if (pack) {
        if (PackPriceModel) {
            auto model = models->GetOfferModel(PackPriceModel);
            if (!model) {
                session.SetErrorInfo("PriceModelOfferCorrector::DoApplyForOffer", TStringBuilder() << "PackPriceModel " << PackPriceModel << " is missing", EDriveSessionResult::InconsistencySystem);
                return EOfferCorrectorResult::Problems;
            }
            if (DryRun) {
                LogModelResult(offerReport, *model, fpContext->MutableFeatures());
            } else {
                pack->ApplyPackPriceModel(*model);
            }
        }
        if (OverrunPriceModel) {
            auto model = models->GetOfferModel(OverrunPriceModel);
            if (!model) {
                session.SetErrorInfo("PriceModelOfferCorrector::DoApplyForOffer", TStringBuilder() << "OverrunPriceModel " << OverrunPriceModel << " is missing", EDriveSessionResult::InconsistencySystem);
                return EOfferCorrectorResult::Problems;
            }
            if (DryRun) {
                LogModelResult(offerReport, *model, fpContext->MutableFeatures());
            } else {
                pack->ApplyOverrunKmModel(*model);
            }
        }
        if (DurationModel) {
            auto model = models->GetOfferModel(DurationModel);
            if (!model) {
                session.SetErrorInfo("PriceModelOfferCorrector::DoApplyForOffer", TStringBuilder() << "DurationModel " << DurationModel << " is missing", EDriveSessionResult::InconsistencySystem);
                return EOfferCorrectorResult::Problems;
            }
            if (DryRun) {
                LogModelResult(offerReport, *model, fpContext->MutableFeatures());
            } else {
                pack->ApplyDurationModel(*model);
            }
        }
        if (MileageLimitModel) {
            auto model = models->GetOfferModel(MileageLimitModel);
            if (!model) {
                session.SetErrorInfo("PriceModelOfferCorrector::DoApplyForOffer", TStringBuilder() << "MileageLimitModel " << MileageLimitModel << " is missing", EDriveSessionResult::InconsistencySystem);
                return EOfferCorrectorResult::Problems;
            }
            if (DryRun) {
                LogModelResult(offerReport, *model, fpContext->MutableFeatures());
            } else {
                pack->ApplyMileageLimitModel(*model);
            }
        }
        if (InsurancePackPriceModel && pack->GetInsuranceType() == FullInsuranceType) {
            auto model = models->GetOfferModel(InsurancePackPriceModel);
            if (!model) {
                session.SetErrorInfo("PriceModelOfferCorrector::DoApplyForOffer", TStringBuilder() << "InsurancePackPriceModel " << InsurancePackPriceModel << " is missing", EDriveSessionResult::InconsistencySystem);
                return EOfferCorrectorResult::Problems;
            }
            if (DryRun) {
                LogModelResult(offerReport, *model, fpContext->MutableFeatures());
            } else {
                pack->ApplyPackInsurancePriceModel(*model);
            }
        }
    }
    auto fixPoint = offerReport->GetOfferAs<TFixPointOffer>();
    if (fixPoint) {
        if (RouteDurationModel) {
            auto model = models->GetOfferModel(RouteDurationModel);
            if (!model) {
                session.SetErrorInfo(
                    "PriceModelOfferCorrector::DoApplyForOffer",
                    TStringBuilder() << "RouteDurationModel " << RouteDurationModel << " is missing",
                    EDriveSessionResult::InconsistencySystem
                );
                return EOfferCorrectorResult::Problems;
            }
            if (DryRun) {
                LogModelResult(offerReport, *model, fpContext->MutableFeatures());
            } else {
                fixPoint->ApplyRouteDurationModel(*model);
            }
        }
        if (FixPointAcceptancePriceModel && context.GetEnableAcceptanceCost()) {
            auto model = models->GetOfferModel(FixPointAcceptancePriceModel);
            if (!model) {
                session.SetErrorInfo(
                    "PriceModelOfferCorrector::DoApplyForOffer",
                    TStringBuilder() << "FixPointAcceptancePriceModel " << FixPointAcceptancePriceModel << " is missing",
                    EDriveSessionResult::InconsistencySystem
                );
                return EOfferCorrectorResult::Problems;
            }
            if (DryRun) {
                LogModelResult(offerReport, *model, fpContext->MutableFeatures());
            } else {
                fixPoint->ApplyAcceptancePriceModel(*model);
            }
        }
    }
    if (Model) {
        auto model = models->GetOfferModel(Model);
        if (!model) {
            session.SetErrorInfo("PriceModelOfferCorrector::DoApplyForOffer", TStringBuilder() << "model " << Model << " is missing", EDriveSessionResult::InconsistencySystem);
            return EOfferCorrectorResult::Problems;
        }
        if (DryRun) {
            LogModelResult(offerReport, *model, fpContext->MutableFeatures());
        } else {
            offerReport->ApplyRidingPriceModel(*model);
        }
    }
    if (AcceptanceModel && context.GetEnableAcceptanceCost()) {
        auto model = models->GetOfferModel(AcceptanceModel);
        if (!model) {
            session.SetErrorInfo("PriceModelOfferCorrector::DoApplyForOffer", TStringBuilder() << "AcceptanceModel " << AcceptanceModel << " is missing", EDriveSessionResult::InconsistencySystem);
            return EOfferCorrectorResult::Problems;
        }
        if (DryRun) {
            LogModelResult(offerReport, *model, fpContext->MutableFeatures());
        } else {
            auto price = 100 * model->Calc(fpContext->MutableFeatures());
            fpContext->MutableAcceptance().SetPrice(price, AcceptanceModel);
        }
    }
    if (ParkingPriceModel) {
        auto model = models->GetOfferModel(ParkingPriceModel);
        if (!model) {
            session.SetErrorInfo("PriceModelOfferCorrector::DoApplyForOffer", TStringBuilder() << "ParkingPriceModel " << ParkingPriceModel << " is missing", EDriveSessionResult::InconsistencySystem);
            return EOfferCorrectorResult::Problems;
        }
        if (DryRun) {
            LogModelResult(offerReport, *model, fpContext->MutableFeatures());
        } else {
            offerReport->ApplyParkingPriceModel(*model);
        }
    }
    if (auto standardWithDiscountAreaOffer = offerReport->GetOfferAs<TStandardWithDiscountAreaOffer>()) {
        bool needRebuild = false;
        if (StandardWithDiscountAreaDiscountModel) {
            auto model = models->GetOfferModel(StandardWithDiscountAreaDiscountModel);
            if (!model) {
                session.SetErrorInfo(
                    "PriceModelOfferCorrector::DoApplyForOffer",
                    TStringBuilder() << "StandardWithDiscountAreaDiscountModel " << StandardWithDiscountAreaDiscountModel << " is missing",
                    EDriveSessionResult::InconsistencySystem
                );
                return EOfferCorrectorResult::Problems;
            }
            if (DryRun) {
                LogModelResult(offerReport, *model, fpContext->MutableFeatures());
            } else {
                standardWithDiscountAreaOffer->ApplyDiscountModel(*model);
                needRebuild = true;
            }
        }
        if (needRebuild) {
            standardWithDiscountAreaOffer->RebuildDiscountedOffer();
        }
    }
    if (!DryRun) {
        fpContext->MutableFeatures().Floats2[NDriveOfferFactors2::SCORING_MODEL_VALUE] = scoringValue;
        fpContext->MutableFeatures().Floats2[NDriveOfferFactors2::PERCENT_MODEL_VALUE] = percentValue;
        if (Notification) {
            auto notification = MakeHolder<NDrive::TPriceModelOfferNotification>(Notification, percentValue);
            context.MutableNotifications().AddNotification(std::move(notification));
        }
    }
    if (DryRun) {
        return EOfferCorrectorResult::Unimplemented;
    }
    return EOfferCorrectorResult::Success;
}

bool TPriceModelOfferCorrector::IsDryRun() const {
    return DryRun;
}

bool TPriceModelOfferCorrector::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DeserializeSpecialsFromJson(jsonInfo)) {
        return false;
    }
    return
        NJson::ParseField(jsonInfo["model"], Model) &&
        NJson::ParseField(jsonInfo["acceptance_model"], AcceptanceModel) &&
        NJson::ParseField(jsonInfo["parking_price_model"], ParkingPriceModel) &&
        NJson::ParseField(jsonInfo["pack_price_model"], PackPriceModel) &&
        NJson::ParseField(jsonInfo["overrun_price_model"], OverrunPriceModel) &&
        NJson::ParseField(jsonInfo["duration_model"], DurationModel) &&
        NJson::ParseField(jsonInfo["mileage_limit_model"], MileageLimitModel) &&
        NJson::ParseField(jsonInfo["route_duration_model"], RouteDurationModel) &&
        NJson::ParseField(jsonInfo["cashback_percent_model"], CashbackPercentModel) &&
        NJson::ParseField(jsonInfo["scoring_feature_model"], ScoringFeatureModel) &&
        NJson::ParseField(jsonInfo["percent_feature_model"], PercentFeatureModel) &&
        NJson::ParseField(jsonInfo["fix_point_acceptance_price_model"], FixPointAcceptancePriceModel) &&
        NJson::ParseField(jsonInfo["insurance_price_model"], InsurancePriceModel) &&
        NJson::ParseField(jsonInfo["insurance_pack_price_model"], InsurancePackPriceModel) &&
        NJson::ParseField(jsonInfo["deposit_amount_model"], DepositAmountModel) &&
        NJson::ParseField(jsonInfo["notification"], Notification) &&
        NJson::ParseField(jsonInfo["standard_with_discount_area_discount_model"], StandardWithDiscountAreaDiscountModel) &&
        NJson::ParseField(jsonInfo["dry_run"], DryRun);
}

NJson::TJsonValue TPriceModelOfferCorrector::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    NJson::InsertNonNull(result, "model", Model);
    NJson::InsertNonNull(result, "acceptance_model", AcceptanceModel);
    NJson::InsertNonNull(result, "parking_price_model", ParkingPriceModel);
    NJson::InsertField(result, "pack_price_model", PackPriceModel);
    NJson::InsertField(result, "overrun_price_model", OverrunPriceModel);
    NJson::InsertField(result, "duration_model", DurationModel);
    NJson::InsertField(result, "mileage_limit_model", MileageLimitModel);
    NJson::InsertField(result, "route_duration_model", RouteDurationModel);
    NJson::InsertField(result, "cashback_percent_model", CashbackPercentModel);
    NJson::InsertField(result, "scoring_feature_model", ScoringFeatureModel);
    NJson::InsertField(result, "percent_feature_model", PercentFeatureModel);
    NJson::InsertField(result, "fix_point_acceptance_price_model", FixPointAcceptancePriceModel);
    NJson::InsertField(result, "insurance_price_model", InsurancePriceModel);
    NJson::InsertField(result, "insurance_pack_price_model", InsurancePackPriceModel);
    NJson::InsertField(result, "deposit_amount_model", DepositAmountModel);
    NJson::InsertField(result, "notification", Notification);
    NJson::InsertField(result, "standard_with_discount_area_discount_model", StandardWithDiscountAreaDiscountModel);
    NJson::InsertField(result, "dry_run", DryRun);
    return result;
}

NDrive::TScheme TPriceModelOfferCorrector::DoGetScheme(const NDrive::IServer* server) const {
    auto models = server && server->GetModelsStorage() ? server->GetModelsStorage()->ListOfferModels() : TVector<TString>();
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("model", "Price model to use").SetVariants(models);
    scheme.Add<TFSVariants>("acceptance_model", "Acceptance cost model to use").SetVariants(models);
    scheme.Add<TFSVariants>("parking_price_model", "Parking price model to use").SetVariants(models);
    scheme.Add<TFSVariants>("pack_price_model", "Pack price model to use").SetVariants(models);
    scheme.Add<TFSVariants>("overrun_price_model", "Overrun price model to use").SetVariants(models);
    scheme.Add<TFSVariants>("duration_model", "Duration model to use").SetVariants(models);
    scheme.Add<TFSVariants>("mileage_limit_model", "Mileage limit model to use").SetVariants(models);
    scheme.Add<TFSVariants>("route_duration_model", "Route duration model to use").SetVariants(models);
    scheme.Add<TFSVariants>("cashback_percent_model", "Cashback percent model to use").SetVariants(models);
    scheme.Add<TFSVariants>("scoring_feature_model", "User scoring model to use").SetVariants(models);
    scheme.Add<TFSVariants>("percent_feature_model", "Change percent model to use").SetVariants(models);
    scheme.Add<TFSVariants>("fix_point_acceptance_price_model", "Acceptance price model for FixPoint offer to use").SetVariants(models);
    scheme.Add<TFSVariants>("insurance_price_model", "Insurance price model to use").SetVariants(models);
    scheme.Add<TFSVariants>("insurance_pack_price_model", "Insurance pack price model to use").SetVariants(models);
    scheme.Add<TFSVariants>("deposit_amount_model", "Deposit amount model to use").SetVariants(models);
    scheme.Add<TFSString>("notification", "Name of notification for client");
    scheme.Add<TFSBoolean>("dry_run", "Flag to enable dry-run mode").SetDefault(false);
    scheme.Add<TFSVariants>("standard_with_discount_area_discount_model", "Discount model for StandardWithDiscountArea offer to use").SetVariants(models);
    return scheme;
}

EOfferCorrectorResult TSaturnModelOfferCorrector::DoApplyForOffer(
    IOfferReport* offerReport,
    const TVector<TDBTag>& tags,
    const TOffersBuildingContext& context,
    const TString& userId,
    const NDrive::IServer* server,
    NDrive::TInfoEntitySession& session
) const {
    if (!context.GetSetting<bool>("offers.saturn_model_offer_corrector.enabled").GetOrElse(true)) {
        return EOfferCorrectorResult::Unimplemented;
    }
    auto client = server->GetSaturnClient();
    if (!client) {
        session.SetErrorInfo(
            "TSaturnModelOfferCorrector::DoApplyForOffer",
            TStringBuilder() << "SaturnClient is missing",
            EDriveSessionResult::InconsistencySystem
        );
        return EOfferCorrectorResult::Problems;
    }
    NDrive::TSaturnClient::TDebtScoringOptions options;
    TFullPricesContext* fpContext = dynamic_cast<TFullPricesContext*>(offerReport->GetOffer().Get());
    if (!fpContext) {
        return EOfferCorrectorResult::Unimplemented;
    }
    if (!BuildSaturnOptions(options, offerReport, context, server, fpContext->MutableFeatures(), session)) {
        return EOfferCorrectorResult::Problems;
    }
    if (!options.PassportUid) {
        return EOfferCorrectorResult::Unimplemented;
    }
    auto scoringFuture = client->GetDebtScoring(options);
    if (!scoringFuture.Wait(context.GetUserHistoryContextUnsafe().GetDeadline())) {
        NDrive::TEventLog::Log(
            "SaturnModelOfferCorrectorError",
            NJson::TMapBuilder
                ("event", "scoring_future_wait_timeout")
                ("offer_id", offerReport->GetOffer()->GetOfferId())
                ("corrector", GetName())
                ("options", NJson::ToJson(options))
        );
        return EOfferCorrectorResult::Unimplemented;
    }
    if (scoringFuture.HasException()) {
        NDrive::TEventLog::Log(
            "SaturnModelOfferCorrectorError",
            NJson::TMapBuilder
                ("event", "scoring_future_exception")
                ("offer_id", offerReport->GetOffer()->GetOfferId())
                ("corrector", GetName())
                ("options", NJson::ToJson(options))
                ("message", NThreading::GetExceptionMessage(scoringFuture))
        );
        return EOfferCorrectorResult::Unimplemented;
    }
    if (!scoringFuture.HasValue()) {
        NDrive::TEventLog::Log(
            "SaturnModelOfferCorrectorError",
            NJson::TMapBuilder
                ("event", "scoring_future_no_value")
                ("offer_id", offerReport->GetOffer()->GetOfferId())
                ("corrector", GetName())
                ("options", NJson::ToJson(options))
        );
        return EOfferCorrectorResult::Unimplemented;
    }
    auto scoring = scoringFuture.GetValue();
    auto& features = fpContext->MutableFeatures();
    // Save original values.
    const auto originalScore = features.Floats2[NDriveOfferFactors2::SATURN_OFFER_SCORE];
    const auto originalScoreRaw = features.Floats2[NDriveOfferFactors2::SATURN_OFFER_SCORE_RAW];
    // Update features.
    features.Floats2[NDriveOfferFactors2::SATURN_OFFER_SCORE] = scoring.Score;
    features.Floats2[NDriveOfferFactors2::SATURN_OFFER_SCORE_RAW] = scoring.ScoreRaw;
    // Apply models.
    auto result = TBase::DoApplyForOffer(offerReport, tags, context, userId, server, session);
    // Rollback features for dry-run mode.
    if (IsDryRun()) {
        features.Floats2[NDriveOfferFactors2::SATURN_OFFER_SCORE] = originalScore;
        features.Floats2[NDriveOfferFactors2::SATURN_OFFER_SCORE_RAW] = originalScoreRaw;
        NDrive::TEventLog::Log("SaturnModelOfferCorrectorResult", NJson::TMapBuilder
            ("offer_id", offerReport->GetOffer()->GetOfferId())
            ("corrector", GetName())
            ("score", scoring.Score)
            ("score_raw", scoring.ScoreRaw)
        );
    }
    return result;
}

bool TSaturnModelOfferCorrector::BuildSaturnOptions(
    NDrive::TSaturnClient::TDebtScoringOptions& options,
    const IOfferReport* offerReport,
    const TOffersBuildingContext& context,
    const NDrive::IServer* server,
    NDrive::TOfferFeatures& features,
    NDrive::TInfoEntitySession& session
) const {
    const auto& passportUid = context.GetUserHistoryContextUnsafe().GetPassportUid();
    if (!passportUid) {
        options.PassportUid = 0;
        // We should return empty request to identify unauthorized user.
        return true;
    }
    if (!TryFromString(passportUid, options.PassportUid)) {
        session.SetErrorInfo(
            "TSaturnModelOfferCorrector::DoApplyForOffer",
            TStringBuilder() << "Unable to parse PassportUID: " << passportUid,
            EDriveSessionResult::InconsistencySystem
        );
        return false;
    }
    options.ReqId = context.GetReqId();
    options.Service = context.GetSetting<TString>("offers.saturn_model_offer_corrector.service").GetOrElse("drive");
    options.FormulaId = SaturnFormulaId;
    auto models = server->GetModelsStorage();
    if (!models) {
        session.SetErrorInfo(
            "TSaturnModelOfferCorrector::DoApplyForOffer",
            "null ModelsStorage",
            EDriveSessionResult::InconsistencySystem
        );
        return false;
    }
    if (SaturnAmountModel) {
        auto model = models->GetOfferModel(SaturnAmountModel);
        if (!model) {
            session.SetErrorInfo(
                "TSaturnModelOfferCorrector::DoApplyForOffer",
                TStringBuilder() << "SaturnAmountModel " << SaturnAmountModel << " is missing",
                EDriveSessionResult::InconsistencySystem
            );
            return false;
        }
        if (auto amount = model->Calc(features); amount >= 0) {
            options.Amount = amount;
        }
    } else if (auto pack = offerReport->GetOfferAs<TPackOffer>()) {
        const auto packPrice = features.Floats2[NDriveOfferFactors2::OFFER_PACK_PRICE];
        options.Amount = packPrice;
    } else if (auto standart = offerReport->GetOfferAs<TStandartOffer>()) {
        const auto score = features.Floats2[NDriveOfferFactors2::FI_DESTINATION_SCORE];
        const auto duration = features.Floats2[NDriveOfferFactors2::FI_DESTINATION_DURATION] / 60.0;
        const auto ridingPrice = features.Floats2[NDriveOfferFactors2::FI_PRICE];
        const auto threshold = context.GetSetting<double>("offers.saturn_model_offer_corrector.default_threshold").GetOrElse(0.9);
        if (duration > 0 && ridingPrice > 0 && score >= threshold) {
            options.Amount = duration * ridingPrice / 60.0;
        }
    }
    return true;
}

TString TSaturnModelOfferCorrector::GetTypeName() {
    return "saturn_model_offer_corrector";
}

TString TSaturnModelOfferCorrector::GetType() const {
    return GetTypeName();
}

bool TSaturnModelOfferCorrector::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonInfo) {
    return
        TBase::DeserializeSpecialsFromJson(jsonInfo) &&
        NJson::ParseField(jsonInfo["saturn_amount_model"], SaturnAmountModel) &&
        NJson::ParseField(jsonInfo["saturn_formula_id"], SaturnFormulaId);
}

NJson::TJsonValue TSaturnModelOfferCorrector::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    NJson::InsertField(result, "saturn_amount_model", SaturnAmountModel);
    NJson::InsertField(result, "saturn_formula_id", SaturnFormulaId);
    return result;
}

NDrive::TScheme TSaturnModelOfferCorrector::DoGetScheme(const NDrive::IServer* server) const {
    auto models = server && server->GetModelsStorage() ? server->GetModelsStorage()->ListOfferModels() : TVector<TString>();
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("saturn_amount_model", "Model for calculating saturn amount").SetVariants(models);
    scheme.Add<TFSString>("saturn_formula_id", "ID of saturn formula model");
    return scheme;
}

EOfferCorrectorResult TSelectiveOfferCorrector::DoApplyForOffer(IOfferReport* offerReport, const TVector<TDBTag>& tags, const TOffersBuildingContext& context, const TString& userId, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const {
    Y_UNUSED(userId);
    if (!offerReport || !offerReport->GetOffer()) {
        session.SetErrorInfo("SelectiveOfferCorrector::DoApplyForOffer", "null offer", EDriveSessionResult::InconsistencyOffer);
        return EOfferCorrectorResult::Problems;
    }

    auto offer = offerReport->GetOffer().Get();

    if (Probability < 1) {
        const auto reqidHash = FnvHash<ui32>(context.GetReqId());
        if (Probability < 1.0 * reqidHash / Max<ui32>()) {
            return EOfferCorrectorResult::Unimplemented;
        }
    }
    if (!Offers.empty()) {
        if (!Offers.contains(offer->GetBehaviourConstructorId())) {
            return EOfferCorrectorResult::Unimplemented;
        }
    }
    if (!PriceConstructors.empty()) {
        if (!!offer->GetPriceConstructorId()) {
            if (!PriceConstructors.contains(offer->GetPriceConstructorId())) {
                return EOfferCorrectorResult::Unimplemented;
            }
        } else {
            if (!PriceConstructors.contains("__ABSENT__")) {
                return EOfferCorrectorResult::Unimplemented;
            }
        }
    }
    if (!Geobuckets.empty()) {
        const size_t index = (offer->GetTimestamp().Seconds() / GeobucketDuration.Seconds()) % Geobuckets.size();
        const TString& geobucket = Geobuckets[index];
        if (!server) {
            session.SetErrorInfo("SelectiveOfferCorrector::DoApplyForOffer", "null offer", EDriveSessionResult::InconsistencySystem);
            return EOfferCorrectorResult::Problems;
        }
        const NDrive::TLocationTags& locationTags = context.GetLocationTags();
        if (!locationTags.contains(geobucket)) {
            return EOfferCorrectorResult::Unimplemented;
        }
    }
    if (!Buckets.empty()) {
        const ui32 bucketDuration = BucketDuration.Seconds();
        const ui32 bucket = offer->GetTimestamp().Seconds() / bucketDuration % TotalBuckets;
        const ui32 epoch = offer->GetTimestamp().Seconds() / (bucketDuration * TotalBuckets);
        auto buckets = &Buckets;
        if (!ComplementaryBuckets.empty()) {
            auto hash = FnvHash<ui32>(&epoch, sizeof(epoch));
            {
                hash ^= FnvHash<ui32>(BucketSalt);
            }
            for (auto&& tag : context.GetLocationTags()) {
                if (BucketHashLocationTagPrefix && !tag.StartsWith(BucketHashLocationTagPrefix)) {
                    continue;
                }
                hash ^= FnvHash<ui32>(tag);
            }
            if (hash & 1) {
                buckets = &ComplementaryBuckets;
            }
        }
        if (!buckets->contains(bucket)) {
            return EOfferCorrectorResult::Unimplemented;
        }
    }
    if (MaxPreviousOffersCount) {
        if (!context.HasUserHistoryContext()) {
            session.SetErrorInfo("SelectiveOfferCorrector::DoApplyForOffer", "incorrect user history context", EDriveSessionResult::InconsistencyOffer);
            return EOfferCorrectorResult::Problems;
        }
        const auto& asyncPreviousOffersCount = context.GetPreviousOffersCount();
        if (!asyncPreviousOffersCount.Initialized() || !asyncPreviousOffersCount.Wait(context.GetUserHistoryContextUnsafe().GetDeadline())) {
            session.AddErrorMessage("SelectiveOfferCorrector::DoApplyForOffer", "PreviousOffersCount is missing or timeouted: " + ToString(context.GetUserHistoryContextUnsafe().GetDeadline()));
            return EOfferCorrectorResult::Unimplemented;
        }
        if (asyncPreviousOffersCount.HasException()) {
            session.AddErrorMessage("SelectiveOfferCorrector::DoApplyForOffer", NThreading::GetExceptionMessage(asyncPreviousOffersCount));
            return EOfferCorrectorResult::Unimplemented;
        }
        if (!asyncPreviousOffersCount.HasValue()) {
            session.AddErrorMessage("SelectiveOfferCorrector::DoApplyForOffer", "PreviousOffersCount value is not set");
            return EOfferCorrectorResult::Unimplemented;
        }
        const ui64 previousOffersCount = asyncPreviousOffersCount.GetValue();
        if (previousOffersCount > MaxPreviousOffersCount) {
            session.AddErrorMessage(
                "SelectiveOfferCorrector::DoApplyForOffer",
                TStringBuilder() << "MaxPreviousOffersCount reached for " << GetName() << ": " << previousOffersCount << " > " << MaxPreviousOffersCount
            );
            return EOfferCorrectorResult::Unimplemented;
        }
    }
    if (!SourceGeobaseIds.empty()) {
        auto standartOffer = offerReport->GetOfferAs<TStandartOffer>();
        auto geobaseId = standartOffer
            ? FromStringWithDefault<TGeobaseId>(standartOffer->GetFeatures().Categories2[NDriveOfferCatFactors2::FI_GEOBASE_ID_A], 0)
            : 0;
        if (!SourceGeobaseIds.contains(geobaseId)) {
            return EOfferCorrectorResult::Unimplemented;
        }
    }
    if (!DestinationGeobaseIds.empty()) {
        auto standartOffer = offerReport->GetOfferAs<TStandartOffer>();
        auto geobaseId = standartOffer
            ? FromStringWithDefault<TGeobaseId>(standartOffer->GetFeatures().Categories2[NDriveOfferCatFactors2::FI_GEOBASE_ID_B], 0)
            : 0;
        if (!DestinationGeobaseIds.contains(geobaseId)) {
            return EOfferCorrectorResult::Unimplemented;
        }
    }
    if (TagsFilter) {
        if (!TagsFilter.IsMatching(tags)) {
            return EOfferCorrectorResult::Unimplemented;
        }
    }
    return EOfferCorrectorResult::Success;
}

bool TSelectiveOfferCorrector::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DeserializeSpecialsFromJson(jsonInfo)) {
        return false;
    }
    return
        NJson::ParseField(jsonInfo["bucket_count"], TotalBuckets) &&
        NJson::ParseField(jsonInfo["bucket_duration"], BucketDuration) &&
        NJson::ParseField(jsonInfo["bucket_hltp"], BucketHashLocationTagPrefix) &&
        NJson::ParseField(jsonInfo["bucket_salt"], BucketSalt) &&
        NJson::ParseField(jsonInfo["buckets"], Buckets) &&
        NJson::ParseField(jsonInfo["complementary_buckets"], ComplementaryBuckets) &&
        NJson::ParseField(jsonInfo["geobuckets"], Geobuckets) &&
        NJson::ParseField(jsonInfo["offers"], Offers) &&
        NJson::ParseField(jsonInfo["max_previous_offers_count"], MaxPreviousOffersCount) &&
        NJson::ParseField(jsonInfo["source_geobase_ids"], SourceGeobaseIds) &&
        NJson::ParseField(jsonInfo["destination_geobase_ids"], DestinationGeobaseIds) &&
        NJson::ParseField(jsonInfo["tags_filter"], TagsFilter) &&
        NJson::ParseField(jsonInfo["price_constructors"], PriceConstructors) &&
        NJson::ParseField(jsonInfo["probability"], Probability);
}

NJson::TJsonValue TSelectiveOfferCorrector::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    result["bucket_count"] = TotalBuckets;
    result["bucket_duration"] = NJson::ToJson(BucketDuration);
    result["buckets"] = NJson::ToJson(Buckets);
    result["complementary_buckets"] = NJson::ToJson(ComplementaryBuckets);
    result["geobuckets"] = NJson::ToJson(Geobuckets);
    result["offers"] = NJson::ToJson(Offers);
    result["price_constructors"] = NJson::ToJson(PriceConstructors);
    result["source_geobase_ids"] = NJson::ToJson(SourceGeobaseIds);
    result["destination_geobase_ids"] = NJson::ToJson(DestinationGeobaseIds);
    if (BucketHashLocationTagPrefix) {
        result["bucket_hltp"] = BucketHashLocationTagPrefix;
    }
    if (BucketSalt) {
        result["bucket_salt"] = BucketSalt;
    }
    if (MaxPreviousOffersCount) {
        result["max_previous_offers_count"] = MaxPreviousOffersCount;
    }
    if (Probability < 1) {
        result["probability"] = Probability;
    }
    if (TagsFilter) {
        result["tags_filter"] = NJson::ToJson(TagsFilter);
    }
    return result;
}

NDrive::TScheme TSelectiveOfferCorrector::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSNumeric>("bucket_count", "Time bucket count").SetDefault(TotalBuckets);
    scheme.Add<TFSDuration>("bucket_duration", "Time bucket duration").SetDefault(BucketDuration);
    scheme.Add<TFSString>("bucket_hltp", "Time bucket hash location tag prefix");
    scheme.Add<TFSString>("bucket_salt", "Time bucket salt");
    scheme.Add<TFSVariants>("buckets", "Time buckets to use").SetVariants(xrange(DefaultTotalBuckets)).SetMultiSelect(true);
    scheme.Add<TFSVariants>("complementary_buckets", "Complementary time buckets to use").SetVariants(xrange(DefaultTotalBuckets)).SetMultiSelect(true);
    scheme.Add<TFSVariants>("geobuckets", "Geobuckets to use").SetVariants(server && server->GetDriveAPI() ? server->GetDriveAPI()->GetAreasDB()->GetAreaTags(TInstant::Zero(), "geobucket") : NDrive::TLocationTags()).SetMultiSelect(true);
    scheme.Add<TFSVariants>("offers", "Offers to apply to").SetVariants(IOfferBuilderAction::GetNames(server)).SetMultiSelect(true);
    scheme.Add<TFSVariants>("price_constructors", "Допустимые конструкторы цен").SetVariants(TPriceOfferConstructor::GetNames(server)).AddVariant("__ABSENT__").SetMultiSelect(true);
    scheme.Add<TFSNumeric>("max_previous_offers_count", "Do not apply corrector after X consequential clicks");
    scheme.Add<TFSNumeric>("probability", "Probability to apply corrector").SetDefault(1);
    scheme.Add<TFSJson>("source_geobase_ids");
    scheme.Add<TFSJson>("destination_geobase_ids");
    scheme.Add<TFSString>("tags_filter", "Фильтр активации корректора по тегам машины в стандартном формате");
    return scheme;
}

EOfferCorrectorResult TVisibilityOfferCorrector::DoApplyForOffer(IOfferReport* offerReport, const TVector<TDBTag>& tags, const TOffersBuildingContext& context, const TString& userId, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const {
    EOfferCorrectorResult base = TBase::DoApplyForOffer(offerReport, tags, context, userId, server, session);
    if (base != EOfferCorrectorResult::Success) {
        return base;
    }
    if (!offerReport || !offerReport->GetOffer()) {
        session.SetErrorInfo("VisibilityOfferCorrector::DoApplyForOffer", "null offer", EDriveSessionResult::InconsistencyOffer);
        return EOfferCorrectorResult::Problems;
    }

    auto offer = offerReport->GetOffer().Get();
    bool canBeHidden = Hidden.contains(offer->GetPriceConstructorId()) || Hidden.contains(offer->GetBehaviourConstructorId());
    bool canBeRevealed = Revealed.contains(offer->GetPriceConstructorId()) || Revealed.contains(offer->GetBehaviourConstructorId());

    auto score = TMaybe<double>();
    if (ScoreModel) {
        auto eg = context.BuildEventGuard("ScoreModel");
        auto modelsStorage = server->GetModelsStorage();
        auto model = modelsStorage ? modelsStorage->GetOfferModel(ScoreModel) : nullptr;
        if (!model) {
            session.AddErrorMessage(GetName(), TStringBuilder() << "model " << ScoreModel << " is missing");
            return EOfferCorrectorResult::Unimplemented;
        }

        auto pricesContext = offerReport ? offerReport->GetOfferAs<TMarketPricesContext>() : nullptr;
        if (!pricesContext) {
            session.AddErrorMessage(GetName(), "cannot cast offer to MarketPricesContext");
            return EOfferCorrectorResult::Unimplemented;
        }

        score = model->Calc(pricesContext->MutableFeatures());
        if (eg) {
            eg->AddEvent(NJson::TMapBuilder
                ("event", "CalcVisibilityScore")
                ("score", NJson::ToJson(score))
            );
        }
        if (score < ScoreThreshold) {
            if (canBeHidden || canBeRevealed) {
                NDrive::TEventLog::Log("LowVisibilityScore", NJson::TMapBuilder
                    ("corrector", GetName())
                    ("offer_id", offer->GetOfferId())
                    ("score", NJson::ToJson(score))
                );
            }
            return EOfferCorrectorResult::Unimplemented;
        }
    }

    if (canBeRevealed) {
        NDrive::TEventLog::Log("OfferRevealed", NJson::TMapBuilder
            ("corrector", GetName())
            ("offer_id", offer->GetOfferId())
            ("score", NJson::ToJson(score))
        );
        offer->SetHidden(false);
        return EOfferCorrectorResult::Success;
    }
    if (canBeHidden) {
        NDrive::TEventLog::Log("OfferHidden", NJson::TMapBuilder
            ("corrector", GetName())
            ("offer_id", offer->GetOfferId())
            ("score", NJson::ToJson(score))
        );
        offer->SetHidden(true);
        return EOfferCorrectorResult::Success;
    }
    return EOfferCorrectorResult::Unimplemented;
}

bool TVisibilityOfferCorrector::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DeserializeSpecialsFromJson(jsonInfo)) {
        return false;
    }
    return NJson::TryFieldsFromJson(jsonInfo, GetFields());
}

NJson::TJsonValue TVisibilityOfferCorrector::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    NJson::FieldsToJson(result, GetFields());
    return result;
}

NDrive::TScheme TVisibilityOfferCorrector::DoGetScheme(const NDrive::IServer* server) const {
    auto models = server && server->GetModelsStorage() ? server->GetModelsStorage()->ListOfferModels() : TVector<TString>();

    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Remove("offers");
    scheme.Add<TFSVariants>("hidden", "Offers to be hidden").SetVariants(IOfferBuilderAction::GetNames(server)).SetMultiSelect(true);
    scheme.Add<TFSVariants>("revealed", "Offers to be revealed").SetVariants(IOfferBuilderAction::GetNames(server)).SetMultiSelect(true);
    scheme.Add<TFSVariants>("score_model", "Score model").SetVariants(models);
    scheme.Add<TFSNumeric>("score_threshold", "Score threshold").SetPrecision(3);
    return scheme;
}

bool TCashbackOfferCorrector::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DeserializeSpecialsFromJson(jsonInfo)) {
        return false;
    }
    Cashback.SetId(GetName());
    return NJson::ParseField(jsonInfo, Cashback, true);
}

NJson::TJsonValue TCashbackOfferCorrector::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    auto cashbackJson = NJson::ToJson(Cashback);
    return NJson::MergeJson(cashbackJson, result);
}

NDrive::TScheme TCashbackOfferCorrector::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    TCashbackInfo::AddScheme(result, server);
    return result;
}

EOfferCorrectorResult TCashbackOfferCorrector::DoApplyForOffer(IOfferReport* offerReport, const TVector<TDBTag>& tags, const TOffersBuildingContext& context, const TString& userId, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const {
    EOfferCorrectorResult base = TBase::DoApplyForOffer(offerReport, tags, context, userId, server, session);
    if (base != EOfferCorrectorResult::Success) {
        return base;
    }
    if (!offerReport) {
        return EOfferCorrectorResult::Unimplemented;
    }
    IOfferWithCashback* cOffer = dynamic_cast<IOfferWithCashback*>(offerReport->GetOffer().Get());
    if (!cOffer) {
        return EOfferCorrectorResult::Unimplemented;
    }
    cOffer->AddCashback(Cashback);
    return EOfferCorrectorResult::Success;
}

TUserAction::TFactory::TRegistrator<TDestinationPredictor> TDestinationPredictor::Registrator(TDestinationPredictor::GetTypeName());
TUserAction::TFactory::TRegistrator<TDisableCorrector> TDisableCorrector::Registrator(TDisableCorrector::GetTypeName());
TUserAction::TFactory::TRegistrator<TDiscountOfferCorrector> TDiscountOfferCorrector::Registrator(TDiscountOfferCorrector::GetTypeStatic());
TUserAction::TFactory::TRegistrator<TInsuranceOfferCorrector> TInsuranceOfferCorrector::Registrator(TInsuranceOfferCorrector::GetTypeName());
TUserAction::TFactory::TRegistrator<TPriceModelOfferCorrector> TPriceModelOfferCorrector::Registrator(TPriceModelOfferCorrector::GetTypeName());
TUserAction::TFactory::TRegistrator<TSaturnModelOfferCorrector> TSaturnModelOfferCorrector::Registrator(TSaturnModelOfferCorrector::GetTypeName());
TUserAction::TFactory::TRegistrator<TVisibilityOfferCorrector> TVisibilityOfferCorrector::Registrator(TVisibilityOfferCorrector::GetTypeName());
TUserAction::TFactory::TRegistrator<TCashbackOfferCorrector> TCashbackOfferCorrector::Registrator(TCashbackOfferCorrector::GetTypeName());
