#pragma once

#include "abstract.h"

#include <drive/backend/offers/discount.h>
#include <drive/backend/offers/price/price.h>
#include <drive/backend/roles/roles.h>
#include <drive/library/cpp/saturn/client.h>

#include <library/cpp/regex/pcre/regexp.h>

#include <rtline/util/types/accessor.h>
#include <rtline/util/types/field.h>

class IOffer;
class TDiscount;

using TGeobaseId = ui64;

class TSelectiveOfferCorrector: public IOfferCorrectorAction {
public:
    static constexpr ui32 DefaultTotalBuckets = 60;

private:
    using TBase = IOfferCorrectorAction;

public:
    using TBase::TBase;

    virtual EOfferCorrectorResult DoApplyForOffer(IOfferReport* offer, const TVector<TDBTag>& tags, const TOffersBuildingContext& context, const TString& userId, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const override;

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonInfo) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

private:
    TTagsFilter TagsFilter;

    TSet<ui32> Buckets;
    TSet<ui32> ComplementaryBuckets;
    TString BucketHashLocationTagPrefix;
    TString BucketSalt;
    TDuration BucketDuration = TDuration::Minutes(1);
    ui32 TotalBuckets = DefaultTotalBuckets;

    TVector<TString> Geobuckets;
    TDuration GeobucketDuration = TDuration::Minutes(10);

    TSet<TGeobaseId> SourceGeobaseIds;
    TSet<TGeobaseId> DestinationGeobaseIds;

    ui32 MaxPreviousOffersCount = 0;
    float Probability = 1;
    R_READONLY(TSet<TString>, Offers);
    R_READONLY(TSet<TString>, PriceConstructors);
};

class TDestinationPredictor : public IOfferCorrectorAction {
private:
    using TBase = IOfferCorrectorAction;

public:
    using TBase::TBase;

    virtual EOfferCorrectorResult DoApplyForOffer(IOfferReport* offer, const TVector<TDBTag>& tags, const TOffersBuildingContext& context, const TString& userId, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const override;

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonInfo) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

    static TString GetTypeName() {
        return "destination_predictor";
    }
    virtual TString GetType() const override {
        return GetTypeName();
    }

private:
    TString Model;
    bool AlwaysPredict = false;
    bool CalcBestRoute = false;
    bool DryRun = false;
    bool VerboseLogging = false;

private:
    static TFactory::TRegistrator<TDestinationPredictor> Registrator;
};

class TDisableCorrector : public TSelectiveOfferCorrector {
private:
    using TBase = TSelectiveOfferCorrector;

private:
    static TFactory::TRegistrator<TDisableCorrector> Registrator;

public:
    using TBase::TBase;

    virtual EOfferCorrectorResult DoApplyForOffer(IOfferReport* offer, const TVector<TDBTag>& tags, const TOffersBuildingContext& context, const TString& userId, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const override;

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonInfo) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

    static TString GetTypeName() {
        return "disable_corrector";
    }
    virtual TString GetType() const override {
        return GetTypeName();
    }

private:
    TVector<TString> Correctors;
};

class TDiscountOfferCorrector: public TSelectiveOfferCorrector {
private:
    using TBase = TSelectiveOfferCorrector;

private:
    static TFactory::TRegistrator<TDiscountOfferCorrector> Registrator;

public:
    class THowToGetDescription {
        R_READONLY(bool, IsActive, false);
        R_READONLY(TString, Title);
        R_READONLY(TString, Description);
        R_READONLY(TString, MoreButton);
        R_READONLY(TString, DetailedDescription);
        R_READONLY(TString, Logo);
        R_READONLY(TString, BGImage);
    public:
        bool FromJson(const NJson::TJsonValue& json);
        NJson::TJsonValue ToJson() const;
        NJson::TJsonValue GetPublicReport(ELocalization locale, const ILocalization& localization) const;
        NDrive::TScheme GetScheme() const;
    };

private:
    TString TagName;
    TMaybe<TPriceByTimeConfig> DiscountTimetable;
    double Discount = 0;
    TMap<TString, TDiscount::TDiscountDetails> DiscountByTagsPerforming;
    TRegExMatch Matcher;
    R_FIELD(bool, Visible, false);
    R_READONLY(TString, ProfitDescription);
    R_READONLY(TString, ProfitColor);
    R_READONLY(THowToGetDescription, HowToGetBlock);
    R_READONLY(TString, DiscountDescription);
    R_READONLY(TString, Icon);
    R_READONLY(TString, SmallIcon);

    bool ProfitVisible = false;
    R_READONLY(TSet<ui32>, BINs);
    R_READONLY(bool, CheckBINs, false);

public:
    using TBase::TBase;

    bool AddDiscount(const TDiscount::TDiscountDetails& discount) {
        return DiscountByTagsPerforming.emplace(discount.GetTagName(), discount).second;
    }

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonInfo) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;
    virtual NJson::TJsonValue GetPublicReport(ELocalization locale, const ILocalization& localization) const override;

    bool CheckVisibleBoons(const TOffersBuildingContext& context) const {
        if (!ProfitVisible) {
            return false;
        }
        return CheckVisibleForUserGeoConditions(context) == EOfferCorrectorResult::Success;
    }

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

    static TString GetTypeStatic() {
        return "offer_corrector_discounts";
    }

    static TString GetTypeName() {
        return GetTypeStatic();
    }

    virtual TString GetType() const override {
        return GetTypeStatic();
    }

    virtual EOfferCorrectorResult DoApplyForOffer(IOfferReport* offer, const TVector<TDBTag>& tags, const TOffersBuildingContext& context, const TString& userId, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const override;

private:
    TDiscount CreateDiscount(TInstant timestamp) const;
};

class TInsuranceOfferCorrector: public TSelectiveOfferCorrector {
private:
    using TBase = TSelectiveOfferCorrector;

public:
    using TBase::TBase;

    virtual EOfferCorrectorResult DoApplyForOffer(IOfferReport* offer, const TVector<TDBTag>& tags, const TOffersBuildingContext& context, const TString& userId, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const override;

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonInfo) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

    static TString GetTypeName() {
        return "offer_corrector_insurance";
    }
    virtual TString GetType() const override {
        return GetTypeName();
    }

private:
    TString InsuranceType;
    TMaybe<TOptionalPriceComponent> InsurancePriceComponent;
    TMaybe<TDuration> PackDurationLowerLimit;
    TMaybe<TDuration> PackDurationUpperLimit;
    TMaybe<double> PackPriceMultiplier;
    bool ApplyToOverPrices = false;

private:
    static TFactory::TRegistrator<TInsuranceOfferCorrector> Registrator;
};

class TPriceModelOfferCorrector: public TSelectiveOfferCorrector {
private:
    using TBase = TSelectiveOfferCorrector;

public:
    using TBase::TBase;

    virtual EOfferCorrectorResult DoApplyForOffer(IOfferReport* offer, const TVector<TDBTag>& tags, const TOffersBuildingContext& context, const TString& userId, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const override;

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonInfo) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

    static TString GetTypeName() {
        return "offer_corrector_price_model";
    }
    virtual TString GetType() const override {
        return GetTypeName();
    }

private:
    TString Model;
    TString AcceptanceModel;
    TString ParkingPriceModel;
    TString PackPriceModel;
    TString OverrunPriceModel;
    TString DurationModel;
    TString MileageLimitModel;
    TString RouteDurationModel;
    TString CashbackPercentModel;
    TString ScoringFeatureModel;
    TString PercentFeatureModel;
    TString FixPointAcceptancePriceModel;
    TString InsurancePriceModel;
    TString InsurancePackPriceModel;
    TString DepositAmountModel;
    TString Notification;
    TString StandardWithDiscountAreaDiscountModel;
    bool DryRun = false;

protected:
    bool IsDryRun() const;

private:
    void LogModelResult(const IOfferReport* offerReport, const NDrive::IOfferModel& model, NDrive::TOfferFeatures& features) const;

private:
    static TFactory::TRegistrator<TPriceModelOfferCorrector> Registrator;
};

class TSaturnModelOfferCorrector: public TPriceModelOfferCorrector {
public:
    using TBase = TPriceModelOfferCorrector;

    EOfferCorrectorResult DoApplyForOffer(
        IOfferReport* offerReport,
        const TVector<TDBTag>& tags,
        const TOffersBuildingContext& context,
        const TString& userId,
        const NDrive::IServer* server,
        NDrive::TInfoEntitySession& session
    ) const override;

    static TString GetTypeName();
    TString GetType() const override;

    bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonInfo) override;
    NJson::TJsonValue SerializeSpecialsToJson() const override;
    NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

private:
    TString SaturnAmountModel;
    TString SaturnFormulaId;

    bool BuildSaturnOptions(
        NDrive::TSaturnClient::TDebtScoringOptions& options,
        const IOfferReport* offerReport,
        const TOffersBuildingContext& context,
        const NDrive::IServer* server,
        NDrive::TOfferFeatures& features,
        NDrive::TInfoEntitySession& session
    ) const;

private:
    static TFactory::TRegistrator<TSaturnModelOfferCorrector> Registrator;
};

class TVisibilityOfferCorrector: public TSelectiveOfferCorrector {
private:
    using TBase = TSelectiveOfferCorrector;

public:
    using TBase::TBase;

    virtual EOfferCorrectorResult DoApplyForOffer(IOfferReport* offer, const TVector<TDBTag>& tags, const TOffersBuildingContext& context, const TString& userId, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const override;

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonInfo) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

    static TString GetTypeName() {
        return "offer_corrector_visibility";
    }
    virtual TString GetType() const override {
        return GetTypeName();
    }

private:
    TSet<TString> Hidden;
    TSet<TString> Revealed;

    TString ScoreModel;
    double ScoreThreshold = 0;

    DECLARE_FIELDS(
        Field(Hidden, "hidden"),
        Field(Revealed, "revealed"),
        Field(ScoreModel, "score_model"),
        Field(ScoreThreshold, "score_threshold")
    );

private:
    static TFactory::TRegistrator<TVisibilityOfferCorrector> Registrator;
};

class TCashbackOfferCorrector: public TSelectiveOfferCorrector {
private:
    using TBase = TSelectiveOfferCorrector;

private:
    static TFactory::TRegistrator<TCashbackOfferCorrector> Registrator;

private:
    TCashbackInfo Cashback;

public:
    using TBase::TBase;

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonInfo) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

    static TString GetTypeName() {
        return "offer_cashback_corrector";
    }

    virtual TString GetType() const override {
        return GetTypeName();
    }

    virtual EOfferCorrectorResult DoApplyForOffer(IOfferReport* offer, const TVector<TDBTag>& tags, const TOffersBuildingContext& context, const TString& userId, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const override;
};
