#include "dedicated_fleet.h"

#include <drive/backend/offers/actions/abstract.h>
#include <drive/backend/offers/offers/additional_service.h>

#include <drive/library/cpp/scheme/scheme.h>


DECLARE_FIELDS_JSON_SERIALIZER(TDedicatedFleetUnitOfferBuilder);
DECLARE_FIELDS_JSON_SERIALIZER(TDedicatedFleetOfferBuilder);


EOfferCorrectorResult TDedicatedFleetOfferBuilder::SetTimeInfo(TDedicatedFleetOffer& offer, const TOffersBuildingContext& context, const NDrive::IServer& /*server*/) const {
    auto now = context.GetRequestStartTime();

    auto variables = context.GetVariables();

    if (!TimeIntervalData.HasConstants()) {
        return EOfferCorrectorResult::Unimplemented;
    }

    const auto& timeConstants = TimeIntervalData.GetConstantsRef();

    auto closestBeginSinceNow = now + timeConstants.GetPrepareDuration();
    closestBeginSinceNow = TInstant::Days(closestBeginSinceNow.Days() + 1);
    const auto farthestBeginSinceNow = closestBeginSinceNow + timeConstants.GetMaxDuration();

    auto since = FromJsonWithDefault<TInstant>(variables["since"], closestBeginSinceNow);
    since = std::max(closestBeginSinceNow, since);
    since = std::min(farthestBeginSinceNow, since);

    const auto closesEndSinceBegin = since + timeConstants.GetMinDuration();
    const auto farthestEndSinceBegin = since + timeConstants.GetMaxDuration();
    auto until = FromJsonWithDefault<TInstant>(variables["until"], closesEndSinceBegin);
    until = std::max(until, closesEndSinceBegin);
    until = std::min(until, farthestEndSinceBegin);

    auto duration = until - since;

    offer.SetTimeIntervalValues(TimeIntervalData);
    auto& offerData = offer.MutableTimeIntervalValues();
    offerData.SetBeginTime(since);
    offerData.SetEndTime(until);
    offerData.SetClosestBeginInstant(closestBeginSinceNow);
    offerData.SetFarthestBeginInstant(farthestBeginSinceNow);
    offerData.SetClosestEndInstant(closesEndSinceBegin);
    offerData.SetFarthestEndInstant(farthestEndSinceBegin);
    offerData.SetDuration(duration);

    return EOfferCorrectorResult::Success;
}

EOfferCorrectorResult TDedicatedFleetOfferBuilder::SetFleetInfo(TDedicatedFleetOffer& offer, const TOffersBuildingContext& context, const NDrive::IServer& /*server*/) const {
    auto variables = context.GetVariables();

    if (!CommonFleetData.HasConstants()) {
        return EOfferCorrectorResult::Unimplemented;
    }

    //TODO calculate max fleet size using optional car tags

    const auto& commonFleetConstants = CommonFleetData.GetConstantsRef();
    const auto& minSize = commonFleetConstants.GetMinSize();
    const auto& maxSize = commonFleetConstants.GetMaxSize();
    const auto& defaultSize = commonFleetConstants.GetDefaultSize();

    if (minSize > maxSize || defaultSize > maxSize || defaultSize < minSize) {
        return EOfferCorrectorResult::Unimplemented;
    }

    auto count = FromJsonWithDefault<ui64>(variables["count"], defaultSize);
    count = std::max(count, minSize);
    count = std::min(count, maxSize);

    offer.SetFleetInfoValues(CommonFleetData);
    auto& offerData = offer.MutableFleetInfoValues();
    offerData.SetMaxSize(maxSize);
    offerData.SetMinSize(minSize);
    offerData.SetSize(count);

    return EOfferCorrectorResult::Success;
}

bool TDedicatedFleetOfferBuilder::DeserializeSpecialsFromJson(const NJson::TJsonValue& value) {
    if (value.Has("available_tags_filter")) {
        const auto& tagsFilter = value["available_tags_filter"];
        if (!tagsFilter.IsString() || !AvailableTagsFilter.DeserializeFromString(tagsFilter.GetString())) {
            return false;
        }
    }

    return NJson::TryFromJson(value, *this)
        && TBase::DeserializeSpecialsFromJson(value);
}

NJson::TJsonValue TDedicatedFleetOfferBuilder::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    NJson::FieldsToJson(result, GetFields());
    result.InsertValue("available_tags_filter", AvailableTagsFilter.ToString());
    return result;
}

NDrive::TScheme TDedicatedFleetOfferBuilder::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);

    auto offerHolderTags = server
        ? server->GetDriveDatabase().GetTagsManager().GetTagsMeta().GetRegisteredTagNames({TDedicatedFleetOfferHolderTag::Type()})
        : TSet<TString>();

    auto unitBuildersActions = server
        ? ::MakeSet(server->GetDriveAPI()->GetRolesManager()->GetActionsDB().GetActionNamesWithType<TDedicatedFleetUnitOfferBuilder>(/*reportDeprecated=*/false))
        : TSet<TString>();

    result.Add<TFSVariants>("offer_holder_tag", "Offer holder tag").SetVariants(offerHolderTags);
    result.Add<TFSVariants>("unit_offer_builder", "Fleet unit offer builder action").SetVariants(unitBuildersActions);
    result.Add<TFSStructure>("time_interval_data", "Настройка времени владения").SetStructure(TimeIntervalData.GetScheme(server));
    result.Add<TFSStructure>("common_fleet_data", "Настройки общих параметров владения парком").SetStructure(CommonFleetData.GetScheme(server));
    result.Add<TFSString>("available_tags_filter", "Теги для подсчета доступных машин");
    result.Add<TFSNumeric>("cancellation_on_prepare_cost", "Стоимость отмены на этапе подготовки(копейки)");

    result.Add<TFSBoolean>("need_defer", "Поместить коммуникацию в отложенные").SetDefault(NeedDeferCommunication);
    result.Add<TFSDuration>("defer_threshold", "Максимальный интервал от активации до старта аренды").SetDefault(DeferThreshold);

    return result;
}

EOfferCorrectorResult TDedicatedFleetOfferBuilder::DoBuildOffers(
    const TUserPermissions& permissions,
    TVector<IOfferReport::TPtr>& offers,
    const TOffersBuildingContext& context,
    const NDrive::IServer* server,
    NDrive::TInfoEntitySession& session
) const {
    auto offer = MakeAtomicShared<TDedicatedFleetOffer>();
    auto variables = context.GetVariables();

    if (auto result = SetTimeInfo(*offer, context, *server); result != EOfferCorrectorResult::Success) {
        return result;
    }

    if (auto result = SetFleetInfo(*offer, context, *server); result !=  EOfferCorrectorResult::Success) {
        return result;
    }

    auto builders = permissions.GetOfferBuilders();
    TDedicatedFleetUnitOfferBuilder::TConstPtr action(nullptr);
    for (const auto& builder : builders) {
        action = std::dynamic_pointer_cast<const TDedicatedFleetUnitOfferBuilder>(builder);
        if (action && action->GetIsPublish() && UnitOfferBuilder == action->GetName()) {
            break;
        }

        action.Reset();
    }

    if (!action) {
        return EOfferCorrectorResult::Unimplemented;
    }

    auto unitBuilderAction = action->Clone();
    if (!unitBuilderAction) {
        return EOfferCorrectorResult::Unimplemented;
    }

    if (auto result = SetOptions(unitBuilderAction.Get(), variables["options"]); result != EOfferCorrectorResult::Success) {
        return result;
    }

    auto& unitOffers = offer->MutableUnitsOffers();
    for (ui64 i = 0; i < offer->GetFleetInfoValues().GetSizeRef(); ++i) {
        auto resultCode = BuildUnitOffer(dynamic_cast<ICommonOfferBuilderAction*>(unitBuilderAction.Get()), unitOffers, permissions, context, server, session);
        if (resultCode != EOfferCorrectorResult::Success) {
            return resultCode;
        }
    }

    if (!unitOffers.empty()) {
        offer->SetBaseUnitOffer(unitOffers.begin()->second);
    }

    offer->SetUserId(context.GetUserHistoryContextUnsafe().GetUserId());
    offer->SetAccountId(variables["account_id"].GetInteger());
    offer->SetTargetHolderTag(OfferHolderTag);
    offer->SetNeedDeferCommunication(NeedDeferCommunication);
    offer->SetDeferThreshold(DeferThreshold);
    offer->SetCancellationCostOnPreparing(CancellationOnPrepareCost);

    //auto tariffPlan = GetTariffPlanDescription(tariffPlanId);

    auto report = MakeAtomicShared<TDedicatedFleetOfferReport>(offer, nullptr);
    offers.push_back(report);

    return EOfferCorrectorResult::Success;
}

EOfferCorrectorResult TDedicatedFleetOfferBuilder::BuildUnitOffer(ICommonOfferBuilderAction* action, TDedicatedFleetOffer::TUnitsOffersMap& reports,
    const TUserPermissions& permissions, const TOffersBuildingContext& context, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const {
    if (!action) {
        return EOfferCorrectorResult::Unimplemented;
    }

    TVector<IOfferReport::TPtr> buildedOffers;
    auto resultCode = action->BuildOffers(permissions, permissions.GetOfferCorrections(), buildedOffers, context, server, session);
    if (resultCode != EOfferCorrectorResult::Success) {
        return resultCode;
    }

    if (buildedOffers.empty()) {
        return EOfferCorrectorResult::BuildingProblems;
    }

    const auto& unitOffer = buildedOffers.front();
    if (auto uOffer = unitOffer->GetOfferAs<TDedicatedFleetUnitOffer>()) {
        reports.emplace(uOffer->GetOfferId(), unitOffer);
    } else {
        return EOfferCorrectorResult::BuildingProblems;
    }

    return EOfferCorrectorResult::Success;
}

EOfferCorrectorResult TDedicatedFleetOfferBuilder::SetOptions(TUserAction* action, const NJson::TJsonValue& optionsMap) const {
    if (!action) {
        return EOfferCorrectorResult::Unimplemented;
    }

    auto builder = dynamic_cast<TDedicatedFleetUnitOfferBuilder*>(action);
    if (!builder) {
        return EOfferCorrectorResult::Unimplemented;
    }

    auto& options = builder->MutableOptions();
    auto&& valuesMap = optionsMap.GetMap();
    for (auto& option : options) {
        if (!option.HasConstants()) {
            return EOfferCorrectorResult::Unimplemented;
        }
        const auto& constants = option.GetConstantsRef();
        if (auto it = valuesMap.find(constants.GetId()); it != valuesMap.end() && it->second.IsBoolean()) {
            option.SetValue(it->second.GetBoolean());
        }
    }

    return EOfferCorrectorResult::Success;
}

EOfferCorrectorResult TDedicatedFleetOfferBuilder::DoCheckOfferConditions(const TOffersBuildingContext& context, const TUserPermissions& permissions) const {
    Y_UNUSED(context);
    Y_UNUSED(permissions);
    return EOfferCorrectorResult::Success;
}

TUserAction::TFactory::TRegistrator<TDedicatedFleetOfferBuilder> TDedicatedFleetOfferBuilder::Registrator(TDedicatedFleetOfferBuilder::GetTypeName());


bool TDedicatedFleetUnitOfferBuilder::DeserializeSpecialsFromJson(const NJson::TJsonValue& value) {
    return NJson::TryFromJson(value, *this)
        && TBase::DeserializeSpecialsFromJson(value);
}

NJson::TJsonValue TDedicatedFleetUnitOfferBuilder::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    NJson::FieldsToJson(result, GetFields());
    return result;
}

NDrive::TScheme TDedicatedFleetUnitOfferBuilder::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSNumeric>("delta_cost", "Цена минуты владения(копейки)");

    {
        NDrive::TScheme optionScheme = TOption().GetScheme(server);
        result.Add<TFSArray>(TString(OptionsNameId), "Доступные опции на этапе оформления заявки").SetElement(std::move(optionScheme));
    }
    {
        auto&& tagsMeta = server->GetDriveDatabase().GetTagsManager().GetTagsMeta();
        auto tags = tagsMeta.GetRegisteredTags(NEntityTagsManager::EEntityType::Account, {TServiceTagRecord::TypeName});
        result.Add<TFSVariants>(TString(ServiceTagsToPerformId), "Сервисные теги, которые буду навешены на машину на этапе подготовки").SetVariants(NContainer::Keys(tags));
        result.Add<TFSVariants>(TString(ServiceTagsToCheckId), "Сервисные теги, которые должны быть на машине для ввода её в парк").SetVariants(NContainer::Keys(tags));
    }

    return result;
}

EOfferCorrectorResult TDedicatedFleetUnitOfferBuilder::DoBuildOffers(
    const TUserPermissions& /*permissions*/,
    TVector<IOfferReport::TPtr>& offers,
    const TOffersBuildingContext& /*context*/,
    const NDrive::IServer* /*server*/,
    NDrive::TInfoEntitySession& /*session*/
) const {
    auto offer = MakeAtomicShared<TDedicatedFleetUnitOffer>();
    offer->SetUnitOptions(Options);
    offer->SetServiceTagsToPerform(ServiceTagsToPerform);
    offer->SetServiceTagsToCheck(ServiceTagsToCheck);
    offer->SetDeltaDurationCost(DeltaCost);

    auto report = MakeAtomicShared<TDedicatedFleetUnitOfferReport>(offer, nullptr);
    offers.push_back(report);
    return EOfferCorrectorResult::Success;
}

EOfferCorrectorResult TDedicatedFleetUnitOfferBuilder::DoCheckOfferConditions(const TOffersBuildingContext& context, const TUserPermissions& permissions) const {
    Y_UNUSED(context);
    Y_UNUSED(permissions);
    return EOfferCorrectorResult::Success;
}

TUserAction::TFactory::TRegistrator<TDedicatedFleetUnitOfferBuilder> TDedicatedFleetUnitOfferBuilder::Registrator(TDedicatedFleetUnitOfferBuilder::GetTypeName());
