#include "distributing_block.h"
#include "flexipack.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/context_fetcher/json.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/database/drive/landing.h>
#include <drive/backend/distributing_block_storage/common.h>
#include <drive/backend/offers/offers/standart.h>
#include <drive/backend/offers/user_context.h>
#include <rtline/library/json/parse.h>

const TString TDistributingBlockAction::AllowDistributingBlockFlag = "allow_distributing_block";
const TString TDistributingBlockAction::EnableShowsRestrictionSetting = "offers.distributing_block.enable_shows_restriction";
const TString TDistributingBlockAction::AllowDistributingBlockInOffers = "offers.distributing_block.allow_in_offers";
const TString TDistributingBlockAction::AllowDistributingBlockInMainScreen = "offers.distributing_block.allow_in_main_screen";

bool TDistributingBlockAction::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) {
    return TBase::DeserializeSpecialsFromJson(jsonValue)
        && TCommonOfferActionTraits::FromJson(jsonValue)
        && NJson::ParseField(jsonValue["description"], Description)
        && NJson::ParseField(jsonValue["title"], Title, true)
        && NJson::ParseField(jsonValue["image_url"], ImageUrl, true)
        && NJson::ParseField(jsonValue["priority"], Priority, true)
        && NJson::ParseField(jsonValue["show_times"], ShowTimes)
        && NJson::ParseField(jsonValue["place_report_traits"], NJson::BitMask<EPlaceReportTraits>(PlaceReportTraits))
        && NJson::ParseField(jsonValue["time_restriction"], TimeRestriction)
        && NJson::ParseField(jsonValue["close_after_click_if_deeplink"], CloseAfterClickIfDeeplink)
        && NJson::ParseField(jsonValue["enable_periodical_shows_restrictions"], EnablePeriodicalShowsRestrictions)
        && NJson::ParseField(jsonValue["deadline"], NJson::Seconds(Deadline))
        && TimeRestriction.Compile()
        && DeserializeContextFromJson(jsonValue["context"]);
}

NJson::TJsonValue TDistributingBlockAction::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();;
    TCommonOfferActionTraits::ToJson(result);
    if (Description) {
        result["description"] = Description;
    }
    result["title"] = Title;
    result["image_url"] = ImageUrl;
    result["priority"] = Priority;
    result["show_times"] = NJson::ToJson(ShowTimes);
    result["deadline"] = NJson::ToJson(NJson::Seconds(Deadline));
    result["time_restriction"] = NJson::ToJson(TimeRestriction);
    result["place_report_traits"] = NJson::ToJson(NJson::BitMask<EPlaceReportTraits>(PlaceReportTraits));
    result["close_after_click_if_deeplink"] = CloseAfterClickIfDeeplink;
    result["enable_periodical_shows_restrictions"] = EnablePeriodicalShowsRestrictions;
    result["context"] = SerializeContextToJson();
    return result;
}

NDrive::TScheme TDistributingBlockAction::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    TCommonOfferActionTraits::FillScheme(scheme, server);
    scheme.Add<TFSString>("title", "Заголовок").SetRequired(true);
    scheme.Add<TFSString>("description", "Описание");
    scheme.Add<TFSString>("image_url", "Ссылка на картинку для иконки").SetRequired(true);
    scheme.Add<TFSNumeric>("priority").SetRequired(true);
    scheme.Add<TFSNumeric>("show_times", "Количество раз, которое показываем пользователю");
    scheme.Add<TFSNumeric>("deadline", "Показывать до даты (UTC)").SetVisual(TFSNumeric::EVisualType::DateTime);
    scheme.Add<TFSStructure>("time_restriction", "Время активности").SetStructure(TTimeRestriction::GetScheme());
    scheme.Add<TFSVariants>("place_report_traits", "Места показа дистроблока").SetMultiSelect(true).InitVariants<EPlaceReportTraits>();
    scheme.Add<TFSBoolean>("close_after_click_if_deeplink", "Закрывать сразу после клика, если дистроблок без целевого действия").SetDefault(false);
    scheme.Add<TFSBoolean>("enable_periodical_shows_restrictions", "Использовать периодические ограничения по показам").SetDefault(false);
    scheme.Add<TFSStructure>("context", "Контекст").SetStructure(DoGetContextScheme(server));
    return scheme;
}

NJson::TJsonValue TDistributingBlockAction::BuildJsonReport(const NDrive::IServer& server, const TOffersBuildingContext& context, const TVector<IOfferReport::TPtr>& offers, EPlaceReportTraits placeTrait) const {
    NJson::TJsonValue result;
    BuildCommon(server, result, offers);
    result["context"] = BuildContext(server, context, offers, placeTrait);
    return result;
}


bool TDistributingBlockAction::CheckPeriodicalConstraints(const TVector<TDistributingBlockEvent::TPtr>& events, const TOffersBuildingContext& context) const {
    if (!EnablePeriodicalShowsRestrictions) {
        return true;
    }
    ui32 showsPeriod = context.GetUserHistoryContextRef().GetSetting<ui32>("offers.distributing_block.shows_period").GetOrElse(0);
    ui32 nonShowsPeriod = context.GetUserHistoryContextRef().GetSetting<ui32>("offers.distributing_block.non_shows_period").GetOrElse(0);
    if (showsPeriod + nonShowsPeriod == 0) {
        ALERT_LOG << "Incorrect periods for distributing block shows" << Endl;
        return true;
    }
    int showsCount = 0;
    for (const auto& event : events) {
        showsCount += event->GetShowsCount();
    }
    auto remainder = showsCount % (showsPeriod + nonShowsPeriod);
    return remainder < showsPeriod;
}

bool TDistributingBlockAction::CheckAbsoluteConstraints(const TVector<TDistributingBlockEvent::TPtr>& events, const TOffersBuildingContext& /* context */) const {
    if (!ShowTimes) {
        return true;
    }
    ui32 count = 0;
    for (auto&& event : events) {
        TStringBuf actionType = event->GetType();
        if (auto pos = actionType.find(":"); pos != TStringBuf::npos) {
            actionType = actionType.Head(pos);
        }
        if (GetName() == actionType) {
            count += event->GetShowsCount();
        }
    }
    return count < *ShowTimes;
}

bool TDistributingBlockAction::CheckShowsRestriction(const TOffersBuildingContext& context) const {
    if (!context.GetUserHistoryContextRef().GetNeedDistributingBlockShowsRestriction()) {
        return true;
    }
    auto maybeEvents = context.GetUserHistoryContextRef().GetDistributingBlockEvents();
    if (!maybeEvents) {
        return true;
    }
    const auto& events = *maybeEvents;
    return CheckPeriodicalConstraints(events, context) && CheckAbsoluteConstraints(events, context);
}

bool TDistributingBlockAction::IsRelevant(const TOffersBuildingContext& context, const TVector<IOfferReport::TPtr>& /* offers */, EPlaceReportTraits placeTrait) const {
    auto cacheId = TCommonOfferActionTraits::CalcCacheId();
    auto geoCorrection = CheckGeoConditions(context, cacheId);
    auto now = Now();
    return (PlaceReportTraits & placeTrait) && TimeRestriction.IsActualNow(now) && geoCorrection == EOfferCorrectorResult::Success
        && (!Deadline || now < Deadline)
        && context.HasUserHistoryContext() && context.GetUserHistoryContextRef().GetUserPermissions()
        && CheckShowsRestriction(context) && context.GetUserHistoryContextRef().GetAllowDistributingBlock()
        && IsCompatibleWith(placeTrait);
}

bool TDistributingBlockAction::IsCompatibleWith(EPlaceReportTraits placeTrait) const {
    return GetCompatibilityTraitsImpl() & placeTrait;
}

void TDistributingBlockAction::BuildCommon(const NDrive::IServer& /* server */, NJson::TJsonValue& result, const TVector<IOfferReport::TPtr>& /* offers */) const {
    if (Description) {
        result["description"] = Description;
    }
    result["title"] = Title;
    result["image_url"] = ImageUrl;
    result["show_times"] = NJson::ToJson(ShowTimes);
}

TString TInsuranceDistributingBlockAction::GetTypeName() {
    return "distributing_block_insurance";
}

TString TInsuranceDistributingBlockAction::GetType() const {
    return GetTypeName();
}

bool TInsuranceDistributingBlockAction::IsRelevant(const TOffersBuildingContext& context, const TVector<IOfferReport::TPtr>& offers, EPlaceReportTraits placeTrait) const {
    if (!TBase::IsRelevant(context, offers, placeTrait)) {
        return false;
    }
    auto contextInsuranceTypes = context.GetInsuranceTypes();
    auto insuranceTypeOption = context.GetUserHistoryContextRef().GetUserPermissions()->GetOptionSafe("insurance_type");
    auto insuranceTypes = insuranceTypeOption ? insuranceTypeOption->GetValues() : TVector<TString>();

    if (placeTrait == EPlaceReportTraits::ReportInMainScreen) {
        return find(insuranceTypes.begin(), insuranceTypes.end(), FullInsuranceType) != insuranceTypes.end()
            && insuranceTypeOption->GetValueOrDefault(context.GetRequestStartTime()) == StandartInsuranceType;
    }
    return find(insuranceTypes.begin(), insuranceTypes.end(), FullInsuranceType) != insuranceTypes.end()
        && (!contextInsuranceTypes || find(contextInsuranceTypes->begin(), contextInsuranceTypes->end(), FullInsuranceType) != contextInsuranceTypes->end())
        && !offers.empty() && offers[0]->GetOfferAs<TStandartOffer>()
        && offers[0]->GetOfferAs<TStandartOffer>()->GetInsuranceType() == StandartInsuranceType;
}

TDistributingBlockAction::TPlaceReportTraits TInsuranceDistributingBlockAction::GetCompatibilityTraitsImpl() const {
    return ReportInOffers | ReportInMainScreen;
}

NDrive::TScheme TInsuranceDistributingBlockAction::DoGetContextScheme(const NDrive::IServer* /* server */) const {
    NDrive::TScheme scheme;
    scheme.Add<TFSBoolean>("action_inside", "Целевое действие внутри дистроблока").SetDefault(false);
    scheme.Add<TFSString>("fallback_description", "Описание, если не удалось узнать поминутную цену страховки");
    scheme.Add<TFSString>("fallback_title", "Заголовок, если не удалось узнать поминутную цену страховки");
    return scheme;
}

bool TInsuranceDistributingBlockAction::DeserializeContextFromJson(const NJson::TJsonValue& jsonValue) {
    return NJson::ParseField(jsonValue["action_inside"], ActionInside)
        && NJson::ParseField(jsonValue["fallback_description"], FallbackDescription)
        && NJson::ParseField(jsonValue["fallback_title"], FallbackTitle);
}

NJson::TJsonValue TInsuranceDistributingBlockAction::SerializeContextToJson() const {
    return NJson::TMapBuilder
        ("action_inside", ActionInside)
        ("fallback_description", FallbackDescription)
        ("fallback_title", FallbackTitle);
}

NJson::TJsonValue TInsuranceDistributingBlockAction::BuildContext(const NDrive::IServer& server, const TOffersBuildingContext& context, const TVector<IOfferReport::TPtr>& /* offers */, EPlaceReportTraits placeTrait) const {
    const auto& settings = server.GetSettings();
    NJson::TJsonValue result;
    result["type"] = GetName();
    result["deeplink"] = NJson::ToJson(settings.GetValue<TString>("offers.distributing_block.insurance_deeplink"));
    if (ActionInside) {
        switch (placeTrait) {
            case EPlaceReportTraits::ReportInOffers: {
                result["url"] = "/api/yandex/offers/create";

                auto insuranceTypeOption = context.GetUserHistoryContextRef().GetUserPermissions()->GetOptionSafe("insurance_type");
                auto insuranceTypes = insuranceTypeOption ? insuranceTypeOption->GetValues() : TVector<TString>();
                auto it = std::remove(insuranceTypes.begin(), insuranceTypes.end(), "full");
                insuranceTypes.erase(it, insuranceTypes.end());
                result["body_params"] = NJson::TMapBuilder("set_global_full_insurance", true)
                                                            ("insurance_type", "full")
                                                            (AllowDistributingBlockFlag, false)
                                                            ("complementary_insurance_types", NJson::ToJson(insuranceTypes));
                break;
            }
            case EPlaceReportTraits::ReportInMainScreen: {
                result["url"] = "/api/yandex/user/settings";
                auto userAppSettingsTagName = server.GetSettings().GetValueDef<TString>(NDrive::UserSettingsTagNameSetting, "user_app_settings");
                result["query_params"] = NJson::TMapBuilder("tag_name", userAppSettingsTagName);
                result["body_params"] = NJson::TMapBuilder("id", "insurance_type")
                                                          ("value", "full");
                break;
            }
        }
    } else {
        if (GetCloseAfterClickIfDeeplink()) {
            result["close_after_click"] = true;
        }
    }
    return result;
}

void TInsuranceDistributingBlockAction::BuildCommon(const NDrive::IServer& server, NJson::TJsonValue& result, const TVector<IOfferReport::TPtr>& offers) const {
    ui32 insurancePrice = 0;
    for (const auto& offer : offers) {
        if (offer->GetOffer()->GetTypeName() == TStandartOffer::GetTypeNameStatic()) {
            auto it = offer->GetInsurancePrices().find(FullInsuranceType);
            if (it != offer->GetInsurancePrices().end()) {
                insurancePrice = it->second;
                break;
            }
        }
    }
    if (insurancePrice || !FallbackDescription) {
        auto description = GetDescription();
        SubstGlobal(description, "_InsurancePrice_", server.GetLocalization()->FormatPrice(ELocalization::Rus, insurancePrice));
        if (description) {
            result["description"] = std::move(description);
        }
    } else {
        result["description"] = FallbackDescription;
    }

    if (insurancePrice || !FallbackTitle) {
        TString title = GetTitle();
        SubstGlobal(title, "_InsurancePrice_", server.GetLocalization()->FormatPrice(ELocalization::Rus, insurancePrice));
        result["title"] = std::move(title);
    } else {
        result["title"] = FallbackTitle;
    }
    result["image_url"] = GetImageUrl();
    result["show_times"] = NJson::ToJson(GetShowTimes());
}

TString TPlusDistributingBlockAction::GetTypeName() {
    return "distributing_block_plus";
}

TString TPlusDistributingBlockAction::GetType() const {
    return GetTypeName();
}

bool TPlusDistributingBlockAction::IsRelevant(const TOffersBuildingContext& context, const TVector<IOfferReport::TPtr>& offers, EPlaceReportTraits placeTrait) const {
    if (!TBase::IsRelevant(context, offers, placeTrait)) {
        return false;
    }
    BuildState(context);
    const auto& paymentMethods = context.GetUserHistoryContextRef().GetRequestAccountIds();
    bool plusUserCondition = State.GetIsPlusUser()
                            && State.GetPlusBalance() >= BalanceThreshold
                            && !paymentMethods.contains(ToString(NDrive::NBilling::EAccount::YAccount))
                            && paymentMethods.contains(ToString(NDrive::NBilling::EAccount::Trust));
    bool nonPlusUserCondition = !State.GetIsPlusUser();
    return (plusUserCondition || nonPlusUserCondition);
}

TDistributingBlockAction::TPlaceReportTraits TPlusDistributingBlockAction::GetCompatibilityTraitsImpl() const {
    return ReportInOffers;
}

void TPlusDistributingBlockAction::BuildState(const TOffersBuildingContext& context) const {
    auto eg = context.BuildEventGuard(GetTypeName()+":build_state");
    auto yPaymentMethod = context.GetUserHistoryContextRef().GetYandexPaymentMethod();
    if (context.GetUserHistoryContextRef().GetUserPermissions()->GetUserFeatures().GetIsPlusUser()) {
        State.SetIsPlusUser(true);
        State.SetPlusBalance(0);
        if (yPaymentMethod && yPaymentMethod->GetBalance() >= 0) {
            State.SetPlusBalance(static_cast<ui64>(yPaymentMethod->GetBalance()));
        }
    } else {
        State.SetIsPlusUser(false);
    }
}

NDrive::TScheme TPlusDistributingBlockAction::DoGetContextScheme(const NDrive::IServer* /* server */) const {
    NDrive::TScheme scheme;
    scheme.Add<TFSNumeric>("balance_threshold", "Порог, с которого предлагать потратить баллы (копейки)").SetRequired(true);
    scheme.Add<TFSString>("non_plus_user_title", "Заголовок, когда у пользователя нет плюса").SetRequired(true);
    scheme.Add<TFSString>("plus_user_title", "Заголовок, когда у пользователя достаточно баллов плюса").SetRequired(true);
    scheme.Add<TFSString>("non_plus_user_description", "Описание, когда у пользователя нет плюса").SetRequired(true);
    scheme.Add<TFSString>("plus_user_description", "Описание, когда у пользователя достаточно баллов плюса").SetRequired(true);
    scheme.Add<TFSBoolean>("action_inside", "Целевое действие внутри дистроблока").SetDefault(false);
    return scheme;
}

bool TPlusDistributingBlockAction::DeserializeContextFromJson(const NJson::TJsonValue& jsonValue) {
    return NJson::ParseField(jsonValue["balance_threshold"], BalanceThreshold, true)
        && NJson::ParseField(jsonValue["non_plus_user_title"], NonPlusUserTitle, true)
        && NJson::ParseField(jsonValue["plus_user_title"], PlusUserTitle, true)
        && NJson::ParseField(jsonValue["non_plus_user_description"], NonPlusUserDescription, true)
        && NJson::ParseField(jsonValue["plus_user_description"], PlusUserDescription, true)
        && NJson::ParseField(jsonValue["action_inside"], ActionInside);
}

NJson::TJsonValue TPlusDistributingBlockAction::SerializeContextToJson() const {
    NJson::TJsonValue result;
    result["balance_threshold"] = BalanceThreshold;
    result["non_plus_user_title"] = NonPlusUserTitle;
    result["plus_user_title"] = PlusUserTitle;
    result["non_plus_user_description"] = NonPlusUserDescription;
    result["plus_user_description"] = PlusUserDescription;
    result["action_inside"] = ActionInside;
    return result;
}

NJson::TJsonValue TPlusDistributingBlockAction::BuildContext(const NDrive::IServer& server, const TOffersBuildingContext& context, const TVector<IOfferReport::TPtr>& /* offers */, EPlaceReportTraits /* placeTrait */) const {
    const auto& settings = server.GetSettings();
    NJson::TJsonValue result;
    if (State.GetIsPlusUser()) {
        result["type"] = GetName() + ":spend_plus";
        if (ActionInside) {
            result["url"] = "/api/yandex/offers/create";
            NJson::TJsonValue paymentMethods;
            paymentMethods.AppendValue(NJson::TMapBuilder("account_id", ToString(NDrive::NBilling::EAccount::YAccount)));
            auto& cardPaymentMethod = paymentMethods.AppendValue(NJson::TMapBuilder("account_id", ToString(NDrive::NBilling::EAccount::Trust)));
            auto card = context.GetUserHistoryContextRef().GetRequestCreditCard();
            if (card) {
                cardPaymentMethod["card"] = std::move(card);
            }
            result["body_params"] = NJson::TMapBuilder("payment_methods", std::move(paymentMethods))
                                                        (AllowDistributingBlockFlag, false);
        } else {
            result["deeplink"] = NJson::ToJson(settings.GetValue<TString>("offers.distributing_block.spend_plus_deeplink"));
        }
    } else {
        result["type"] = GetName() + ":buy_plus";
        result["deeplink"] = NJson::ToJson(settings.GetValue<TString>("offers.distributing_block.plus_home_deeplink"));
    }
    if (!ActionInside) {
        if (GetCloseAfterClickIfDeeplink()) {
            result["close_after_click"] = true;
        }
    }
    return result;
}

void TPlusDistributingBlockAction::BuildCommon(const NDrive::IServer& /* server */, NJson::TJsonValue& result, const TVector<IOfferReport::TPtr>& /* offers */) const {
    if (State.GetIsPlusUser()) {
        TString description = PlusUserDescription;
        SubstGlobal(description, "_PlusBalance_", ToString(State.GetPlusBalance() / 100));
        if (description) {
            result["description"] = std::move(description);
        }

        TString title = PlusUserTitle;
        SubstGlobal(title, "_PlusBalance_", ToString(State.GetPlusBalance() / 100));
        result["title"] = std::move(title);
    } else {
        if (NonPlusUserDescription) {
            result["description"] = NonPlusUserDescription;
        }
        result["title"] = NonPlusUserTitle;
    }
    result["image_url"] = GetImageUrl();
    result["show_times"] = NJson::ToJson(GetShowTimes());
}

TString TFlexiblePackDistributingBlockAction::GetTypeName() {
    return "distributing_block_flexible_pack";
}

TString TFlexiblePackDistributingBlockAction::GetType() const {
    return GetTypeName();
}

bool TFlexiblePackDistributingBlockAction::IsRelevant(const TOffersBuildingContext& context, const TVector<IOfferReport::TPtr>& offers, EPlaceReportTraits placeTrait) const {
    if (!TBase::IsRelevant(context, offers, placeTrait)) {
        return false;
    }
    bool hasFlexiblePackOffer = std::count_if(offers.begin(), offers.end(), [](const IOfferReport::TPtr offer) {
        return offer->GetOfferAs<TFlexiblePackOffer>();
    });
    return hasFlexiblePackOffer;
}

TDistributingBlockAction::TPlaceReportTraits TFlexiblePackDistributingBlockAction::GetCompatibilityTraitsImpl() const {
    return ReportInOffers;
}

NDrive::TScheme TFlexiblePackDistributingBlockAction::DoGetContextScheme(const NDrive::IServer* /* server */) const {
    return {};
}

bool TFlexiblePackDistributingBlockAction::DeserializeContextFromJson(const NJson::TJsonValue& /* jsonValue */) {
    return true;
}

NJson::TJsonValue TFlexiblePackDistributingBlockAction::SerializeContextToJson() const {
    return NJson::JSON_UNDEFINED;
}

NJson::TJsonValue TFlexiblePackDistributingBlockAction::BuildContext(const NDrive::IServer& server, const TOffersBuildingContext& /* context */, const TVector<IOfferReport::TPtr>& offers, EPlaceReportTraits /* placeTrait */) const {
    const auto& settings = server.GetSettings();
    NJson::TJsonValue result;
    result["type"] = GetName();
    TString deeplink = settings.GetValue<TString>("offers.distributing_block.flexible_pack_offer_deeplink_template").GetOrElse("");
    auto it = std::find_if(offers.begin(), offers.end(), [](const IOfferReport::TPtr offer) {
        return offer->GetOfferAs<TFlexiblePackOffer>();
    });
    if (it == offers.end()) {
         ythrow yexception() << "TFlexiblePackDistributingBlockAction: no flexible pack offer found";
    }
    SubstGlobal(deeplink, "OFFER_ID", (*it)->GetOfferAs<TFlexiblePackOffer>()->GetOfferId());
    result["deeplink"] = std::move(deeplink);
    if (GetCloseAfterClickIfDeeplink()) {
        result["close_after_click"] = true;
    }
    return result;
}

TString TExternalLinkDistributingBlockAction::GetTypeName() {
    return "distributing_block_external_link";
}

TString TExternalLinkDistributingBlockAction::GetType() const {
    return GetTypeName();
}

TDistributingBlockAction::TPlaceReportTraits TExternalLinkDistributingBlockAction::GetCompatibilityTraitsImpl() const {
    return ReportInOffers | ReportInMainScreen;
}

bool TExternalLinkDistributingBlockAction::IsRelevant(const TOffersBuildingContext& context, const TVector<IOfferReport::TPtr>& offers, EPlaceReportTraits placeTrait) const {
    return TBase::IsRelevant(context, offers, placeTrait);
}

NDrive::TScheme TExternalLinkDistributingBlockAction::DoGetContextScheme(const NDrive::IServer* /* server */) const {
    NDrive::TScheme scheme;
    scheme.Add<TFSString>("external_link", "Внешняя ссылка").SetRequired(true);
    return scheme;
}

bool TExternalLinkDistributingBlockAction::DeserializeContextFromJson(const NJson::TJsonValue& jsonValue) {
    return NJson::ParseField(jsonValue["external_link"], ExternalLink, true);
}

NJson::TJsonValue TExternalLinkDistributingBlockAction::SerializeContextToJson() const {
    NJson::TJsonValue result;
    result["external_link"] = ExternalLink;
    return result;
}

NJson::TJsonValue TExternalLinkDistributingBlockAction::BuildContext(const NDrive::IServer& server, const TOffersBuildingContext& context, const TVector<IOfferReport::TPtr>& /* offers */, EPlaceReportTraits /* placeTrait */) const {
    TJsonFetchContext fetchContext(server, NJson::TMapBuilder
        ("user_id", context.GetUserHistoryContextRef().GetUserPermissions()->GetUserId())
    );
    fetchContext.SetOptions({"${", "}"});
    TMessagesCollector errors;

    auto fullName = IJsonContextFetcher::ProcessText("${user.full_name}", fetchContext, errors);
    auto phoneNumber = IJsonContextFetcher::ProcessText("${user.phone}", fetchContext, errors);
    auto email = IJsonContextFetcher::ProcessText("${user.email}", fetchContext, errors);
    if (errors.HasMessages()) {
        auto report = errors.GetReport();
        NDrive::TEventLog::Log("ExternalLinkDistributingBlockFailure", report);
    }
    TString link = ExternalLink;
    SubstGlobal(link, "${user.full_name}", CGIEscapeRet(fullName));
    SubstGlobal(link, "${user.phone}", CGIEscapeRet(phoneNumber));
    SubstGlobal(link, "${user.email}", CGIEscapeRet(email));
    NJson::TJsonValue result = NJson::TMapBuilder
        ("type", GetName())
        ("external_link", link);
    if (GetCloseAfterClickIfDeeplink()) {
        result["close_after_click"] = true;
    }
    return result;
}

TString TIntroscreenDistributingBlockAction::GetTypeName() {
    return "distributing_block_introscreen";
}

TString TIntroscreenDistributingBlockAction::GetType() const {
    return GetTypeName();
}

TDistributingBlockAction::TPlaceReportTraits TIntroscreenDistributingBlockAction::GetCompatibilityTraitsImpl() const {
    return ReportInOffers | ReportInMainScreen;
}

bool TIntroscreenDistributingBlockAction::IsRelevant(const TOffersBuildingContext& context, const TVector<IOfferReport::TPtr>& offers, EPlaceReportTraits placeTrait) const {
    return TBase::IsRelevant(context, offers, placeTrait);
}

NDrive::TScheme TIntroscreenDistributingBlockAction::DoGetContextScheme(const NDrive::IServer* /* server */) const {
    NDrive::TScheme scheme;
    scheme.Add<TFSString>("introscreen", "Название интроскрина").SetRequired(true);
    return scheme;
}

bool TIntroscreenDistributingBlockAction::DeserializeContextFromJson(const NJson::TJsonValue& jsonValue) {
    return NJson::ParseField(jsonValue["introscreen"], Introscreen, true);
}

NJson::TJsonValue TIntroscreenDistributingBlockAction::SerializeContextToJson() const {
    return NJson::TMapBuilder
        ("introscreen", Introscreen);
}

NJson::TJsonValue TIntroscreenDistributingBlockAction::BuildContext(const NDrive::IServer& server, const TOffersBuildingContext& /* context */, const TVector<IOfferReport::TPtr>& /* offers */, EPlaceReportTraits /* placeTrait */) const {
    auto deeplinkTemplate = server.GetSettings().GetValueDef<TString>("offers.distributing_block.introscreen_deeplink", "yandexdrive://support/_INTROSCREEN_");
    SubstGlobal(deeplinkTemplate, "_INTROSCREEN_", Introscreen);
    NJson::TJsonValue result = NJson::TMapBuilder
        ("type", GetName())
        ("deeplink", deeplinkTemplate);
    if (GetCloseAfterClickIfDeeplink()) {
        result["close_after_click"] = true;
    }
    return result;
}

TString TDeeplinkDistributingBlockAction::GetTypeName() {
    return "distributing_block_deeplink";
}

TString TDeeplinkDistributingBlockAction::GetType() const {
    return GetTypeName();
}

bool TDeeplinkDistributingBlockAction::IsRelevant(const TOffersBuildingContext& context, const TVector<IOfferReport::TPtr>& offers, EPlaceReportTraits placeTrait) const {
    return TBase::IsRelevant(context, offers, placeTrait);
}

TDistributingBlockAction::TPlaceReportTraits TDeeplinkDistributingBlockAction::GetCompatibilityTraitsImpl() const {
    return ReportInOffers | ReportInMainScreen;
}

NDrive::TScheme TDeeplinkDistributingBlockAction::DoGetContextScheme(const NDrive::IServer* /* server */) const {
    NDrive::TScheme scheme;
    scheme.Add<TFSString>("deeplink", "Диплинк").SetRequired(true);
    return scheme;
}

bool TDeeplinkDistributingBlockAction::DeserializeContextFromJson(const NJson::TJsonValue& jsonValue) {
    return NJson::ParseField(jsonValue["deeplink"], Deeplink, true);
}

NJson::TJsonValue TDeeplinkDistributingBlockAction::SerializeContextToJson() const {
    return NJson::TMapBuilder
        ("deeplink", Deeplink);
}

NJson::TJsonValue TDeeplinkDistributingBlockAction::BuildContext(const NDrive::IServer& /* server */, const TOffersBuildingContext& /* context */, const TVector<IOfferReport::TPtr>& /* offers */, EPlaceReportTraits /* placeTrait */) const {
    NJson::TJsonValue result = NJson::TMapBuilder
        ("type", GetName())
        ("deeplink", Deeplink);
    if (GetCloseAfterClickIfDeeplink()) {
        result["close_after_click"] = true;
    }
    return result;
}

TString TMapObjectDistributingBlockAction::GetTypeName() {
    return "distributing_block_map_object";
}

TString TMapObjectDistributingBlockAction::GetType() const {
    return GetTypeName();
}

bool TMapObjectDistributingBlockAction::IsRelevant(const TOffersBuildingContext& context, const TVector<IOfferReport::TPtr>& offers, EPlaceReportTraits placeTrait) const {
    return TBase::IsRelevant(context, offers, placeTrait)
        && context.GetUserHistoryContextRef().HasUserPosition();
}

TDistributingBlockAction::TPlaceReportTraits TMapObjectDistributingBlockAction::GetCompatibilityTraitsImpl() const {
    return ReportInMainScreen;
}

NDrive::TScheme TMapObjectDistributingBlockAction::DoGetContextScheme(const NDrive::IServer* server) const {
    NDrive::TScheme scheme;
    auto gLandings = server->GetDriveAPI()->GetLandingsDB()->GetCachedObjectsMap();
    scheme.Add<TFSVariants>("warning_id", "Warning id").SetVariants(NContainer::Keys(gLandings));
    return scheme;
}

bool TMapObjectDistributingBlockAction::DeserializeContextFromJson(const NJson::TJsonValue& jsonValue) {
    return NJson::ParseField(jsonValue["warning_id"], WarningId);
}

NJson::TJsonValue TMapObjectDistributingBlockAction::SerializeContextToJson() const {
    return NJson::TMapBuilder
        ("warning_id", WarningId);
}

NJson::TJsonValue TMapObjectDistributingBlockAction::BuildContext(const NDrive::IServer& /* server */, const TOffersBuildingContext& context, const TVector<IOfferReport::TPtr>& /* offers */, EPlaceReportTraits /* placeTrait */) const {
    return NJson::TMapBuilder
        ("type", GetName())
        ("introscreen_id", WarningId)
        ("coord", CalculateObjectPosition(context.GetUserHistoryContextRef().GetUserPositionRef(), context).SerializeLatLonToJson());
}

TGeoCoord TMapObjectDistributingBlockAction::CalculateObjectPosition(TGeoCoord coord, const TOffersBuildingContext& context) const {
    double offsetX = context.GetUserHistoryContextRef().GetSetting<double>("offers.distributing_block.map_object.offset.lon").GetOrElse(0.001);
    double offsetY = context.GetUserHistoryContextRef().GetSetting<double>("offers.distributing_block.map_object.offset.lat").GetOrElse(0.001);
    double randomStepX = context.GetUserHistoryContextRef().GetSetting<double>("offers.distributing_block.map_object.random_step.lon").GetOrElse(0.01);
    double randomStepY = context.GetUserHistoryContextRef().GetSetting<double>("offers.distributing_block.map_object.random_step.lat").GetOrElse(0.01);

    coord.X += offsetX;
    coord.Y += offsetY;
    coord.X += randomStepX * RandomNumber<double>();
    coord.Y += randomStepY * RandomNumber<double>();
    return coord;
}

TUserAction::TFactory::TRegistrator<TInsuranceDistributingBlockAction> TInsuranceDistributingBlockAction::Registrator(TInsuranceDistributingBlockAction::GetTypeName());
TUserAction::TFactory::TRegistrator<TPlusDistributingBlockAction> TPlusDistributingBlockAction::Registrator(TPlusDistributingBlockAction::GetTypeName());
TUserAction::TFactory::TRegistrator<TFlexiblePackDistributingBlockAction> TFlexiblePackDistributingBlockAction::Registrator(TFlexiblePackDistributingBlockAction::GetTypeName());
TUserAction::TFactory::TRegistrator<TExternalLinkDistributingBlockAction> TExternalLinkDistributingBlockAction::Registrator(TExternalLinkDistributingBlockAction::GetTypeName());
TUserAction::TFactory::TRegistrator<TIntroscreenDistributingBlockAction> TIntroscreenDistributingBlockAction::Registrator(TIntroscreenDistributingBlockAction::GetTypeName());
TUserAction::TFactory::TRegistrator<TDeeplinkDistributingBlockAction> TDeeplinkDistributingBlockAction::Registrator(TDeeplinkDistributingBlockAction::GetTypeName());
TUserAction::TFactory::TRegistrator<TMapObjectDistributingBlockAction> TMapObjectDistributingBlockAction::Registrator(TMapObjectDistributingBlockAction::GetTypeName());
