#include "fix_point.h"

#include <drive/backend/offers/actions/helpers.h>
#include <drive/backend/offers/context.h>
#include <drive/backend/offers/price/price_impl.h>
#include <drive/backend/offers/ranking/calcer.h>

#include <drive/backend/data/area_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/saas/api.h>

#include <drive/library/cpp/threading/future_cast.h>

#include <library/cpp/geobase/lookup.hpp>

#include <rtline/library/geometry/polyline.h>
#include <rtline/protos/proto_helper.h>

TFixPointOffer::TFactory::TRegistrator<TFixPointOffer> TFixPointOffer::Registrator("fix_point");

TString TFixPointOffer::DoBuildCalculationDescription(ELocalization locale, const TFullCompiledRiding& fcr, const NDrive::IServer& server) const {
    TStringBuilder sb;
    sb << TBase::DoBuildCalculationDescription(locale, fcr, server) << Endl;
    sb << "\\section{FixPoint}" << Endl;
    sb << "\\begin{enumerate}" << Endl;
    sb << "\\item UseKmForPackCalculation = " << UseKmForPackCalculation << Endl;
    sb << "\\item Route duration = " << server.GetLocalization()->FormatDuration(locale, RouteDuration, true) << Endl;
    sb << "\\item Route distance = " << server.GetLocalization()->DistanceFormatKm(locale, RouteLength / 1000) << Endl;
    sb << "\\item Additional route duration = " << server.GetLocalization()->FormatDuration(locale, AdditionalRouteDuration, true) << Endl;
    sb << "\\item Additional route distance = " << AdditionalRouteLength << Endl;
    sb << "\\item MinPrice = " << FinishMinPrice.GetOrElse(0) << Endl;
    sb << "\\item MinPriceLow = " << MinPriceLow.GetOrElse(0) << Endl;
    sb << "\\item AdditionalPrice = " << FinishAdditionalPrice.GetOrElse(0) << Endl;
    sb << "\\item PackDiscount = " << PackDiscount.GetOrElse(0) << Endl;
    sb << "\\item UseRoundedPrice = " << UseRoundedPrice << Endl;
    if (AreasDiscount) {
        sb << "\\item AreasDiscount = " << *AreasDiscount << Endl;
    } else if (RouteDuration.Seconds() && (1 - PackDiscount.GetOrElse(0)) * GetRiding().GetPrice()) {
        const double ad = 1 - 60 * GetPackPrice() / (RouteDuration.Seconds() * GetRiding().GetPrice() * (1 - PackDiscount.GetOrElse(0)));
        sb << "\\item AreasDiscount = " << ad << Endl;
    }
    sb << "\\item FeesForIncorrectFinish = " << FeesForIncorrectFinish << Endl;
    if (AcceptancePrice) {
        sb << "\\item AcceptancePrice = " << server.GetLocalization()->FormatPrice(locale, *AcceptancePrice) << Endl;
    }
    sb << "\\end{enumerate}" << Endl;
    return sb;
}

TPackOffer::TRecalcByMinutesInfo TFixPointOffer::CheckRecalcByMinutes(const TRidingInfo& rInfo, const TInstant startPack) const {
    TPackOffer::TRecalcByMinutesInfo result = TBase::CheckRecalcByMinutes(rInfo, startPack);
    if (rInfo.HasFinishCoord()) {
        auto server = NDrive::HasServer() ? &NDrive::GetServerAs<NDrive::IServer>() : nullptr;
        auto inFinishArea = IsInFinishArea(rInfo.GetFinishCoordRef(), server, FinishArea, GetTypeName()).GetOrElse(false);
        result.SetNeedRecalcNow(!inFinishArea);
        result.SetNeedRecalcInFuture(false);
    } else {
        result.SetNeedRecalcNow(false);
        result.SetNeedRecalcInFuture(true);
    }
    result.SetNeedDistancePricing(false);
    return result;
}

TFixPointOfferReport::THint::THint(const TAreaPotentialTag::TInfo& info) {
    if (info.GetStyleInfo().IsDefined()) {
        SetHintStyle(info.GetStyleInfo().GetStringRobust());
    }
    SetCoordinate(info.GetCenter());
    DiscountInfo = info.GetDiscountInfo();
    SetName(info.GetName());
    if (info.GetBorder().Size()) {
        Bound = info.GetBorder().GetRectSafe();
    }
    if (!!OfferReport) {
        OfferReport->SetIsHint(true);
    }
}

TFixPointOfferReport::THint::THint(const TUserOfferContext::TDestinationDescription& destination, TAtomicSharedPtr<TFixPointOfferReport> offer)
    : TBase(destination)
    , OfferReport(offer)
{
    if (OfferReport) {
        OfferReport->SetIsHint(true);
    }
}

NJson::TJsonValue TFixPointOfferReport::THint::SerializeToJson() const {
    NJson::TJsonValue result;
    result.InsertValue("name", GetName());
    result.InsertValue("coordinate", NJson::ToJson(GetCoordinate()));
    if (Bound) {
        result.InsertValue("bound", NJson::ToJson(Bound));
    }
    if (GetContextClient()) {
        result.InsertValue("context", GetContextClient());
    }
    if (GetHintStyle()) {
        result.InsertValue("style_info", GetHintStyle());
    }
    if (GetIcon()) {
        result.InsertValue("icon", GetIcon());
    }
    if (OfferReport) {
        auto pOffer = OfferReport->GetOfferAs<TPackOffer>();
        if (pOffer) {
            result.InsertValue("price", pOffer->GetPublicDiscountedPrice(pOffer->GetPackPrice(), "", 100));
        }
        result.InsertValue("offer_id", OfferReport->GetOffer()->GetOfferId());
    } else {
        result.InsertValue("discount_info", DiscountInfo);
    }
    return result;
}

NJson::TJsonValue TFixPointOfferReport::BuildJsonReport(ELocalization locale, NDriveSession::TReportTraits traits, const NDrive::IServer& server, const TUserPermissions& permissions) const {
    auto offer = GetOffer();
    auto fixPointOffer = std::dynamic_pointer_cast<TFixPointOffer>(offer);
    Y_ENSURE(fixPointOffer);
    auto effectiveCashback = fixPointOffer->GetEffectiveCashback(server);
    NJson::TJsonValue result = TBase::BuildJsonReport(locale, traits, server, permissions);
    result.InsertValue("is_fake", fixPointOffer->GetIsFake());
    result.InsertValue("is_hint", IsHint);
    if (effectiveCashback && !(offer && offer->GetHiddenCashback())) {
        if (!fixPointOffer->GetIsFake()) {
            result.InsertValue("cashback_prediction", NJson::ToJson(effectiveCashback));
            auto labelTemplate = permissions.GetSetting<TString>(server.GetSettings(), "offers.fix_point.cashback_info.label_template");
            if (labelTemplate && fixPointOffer) {
                NJson::TJsonValue& cashbackInfo = result["cashback_info"].SetType(NJson::JSON_MAP);
                if (const ILocalization* localization = server.GetLocalization()) {
                    SubstGlobal(*labelTemplate, "_cashback_", localization->FormatPrice(locale, *effectiveCashback));
                }
                cashbackInfo.InsertValue("label", *labelTemplate);
            }
        } else {
            auto fakeLabelTemplate = permissions.GetSetting<TString>(server.GetSettings(), "offers.fix_point.cashback_info.fake_label_template");
            if (fakeLabelTemplate) {
                NJson::TJsonValue& cashbackInfo = result["cashback_info"].SetType(NJson::JSON_MAP);
                SubstGlobal(*fakeLabelTemplate, "_cashback_percent_", ToString(fixPointOffer->GetEffectiveCashbackPercent(server)));
                cashbackInfo.InsertValue("label", *fakeLabelTemplate);
            }
        }
    }
    if (offer) {
        result.InsertValue("detailed_description", offer->FormDescriptionElement(DetailedDescription, locale, server.GetLocalization()));
    }
    if (!!Destination) {
        result.InsertValue("destination_report_info", Destination->SerializeToJson());
    } else if (!!Focus) {
        result.InsertValue("destination_report_info", NJson::JSON_MAP).InsertValue("coord", Focus->ToString());
    }
    NJson::TJsonValue& hints = result.InsertValue("hints", NJson::JSON_ARRAY);
    for (auto&& i : Hints) {
        hints.AppendValue(i.SerializeToJson());
    }
    NJson::TJsonValue& discountAreas = result.InsertValue("discount_areas", NJson::JSON_ARRAY);
    for (auto&& i : DiscountAreas) {
        discountAreas.AppendValue(i.SerializeToJson());
    }
    return result;
}

bool TFixPointOfferReport::ApplyMinPrice(const double price) {
    TFixPointOffer* fpOffer = GetOfferAs<TFixPointOffer>();
    if (!fpOffer) {
        return false;
    }
    fpOffer->SetMinPriceLow(price);
    return true;
}

void TFixPointOfferReport::ApplyInternalCorrection(const TString& areaId, const TOffersBuildingContext& /*context*/, const NDrive::IServer* server) {
    TFixPointOffer* fpOffer = GetOfferAs<TFixPointOffer>();
    if (!fpOffer) {
        return;
    }
    TMaybe<TArea> areaObject = server->GetDriveAPI()->GetAreasDB()->GetCustomObject(areaId);
    if (!areaObject) {
        return;
    }
    TCarLocationContext finishContext = TCarLocationContext::BuildByArea(fpOffer->GetObjectId(), *areaObject, *server);
    finishContext.SetNeedInternalFees(true);
    finishContext.SetGuaranteeFees(false);
    finishContext.SetTakeFeesFromOffer(false);
    const TCarLocationFeatures finishFeatures = finishContext.GetCarAreaFeatures(true, fpOffer, server);

    if (finishFeatures.HasFeeInfo() && finishFeatures.GetFeeInfoUnsafe().GetPolicy() == EDropOfferPolicy::FeesMinute) {
        const ui32 deltaResult = finishFeatures.GetFeeInfoUnsafe().GetFee();
        if (deltaResult) {
            fpOffer->MutableRiding().AddPrice(deltaResult, "flow_correction_" + areaId);
        }
    }
}

void TFixPointOfferReport::ApplyFlowCorrection(const TString& areaId, const TOffersBuildingContext& context, const NDrive::IServer* server) {
    TFixPointOffer* fpOffer = GetOfferAs<TFixPointOffer>();
    if (!fpOffer) {
        return;
    }
    const TMaybe<TGeoCoord> cStart = context.GetStartPosition();
    if (!cStart) {
        ERROR_LOG << "Cannot determinate start coord for " << fpOffer->GetObjectId() << Endl;
        return;
    }

    const auto ipc = GetFinishAreaPointContext(server, fpOffer->GetTypeName());
    const double areasDiscount = TAreaPotentialTag::CalcPriceKff(*server, *cStart, fpOffer->GetFinish(), fpOffer->GetPriceConstructorId(), ipc.GetPrecision());
    fpOffer->SetAreasDiscount(1 - areasDiscount);

    TMaybe<TArea> areaObject = server->GetDriveAPI()->GetAreasDB()->GetCustomObject(areaId);
    if (!!areaObject) {
        TCarLocationContext finishContext = TCarLocationContext::BuildByArea(fpOffer->GetObjectId(), *areaObject, *server);
        finishContext.SetGuaranteeFees(false);
        finishContext.SetTakeFeesFromOffer(false);
        const TCarLocationFeatures finishFeatures = finishContext.GetCarAreaFeatures(true, fpOffer, server);

        TCarLocationContext startContext = TCarLocationContext::BuildByCoord(fpOffer->GetObjectId(), context.OptionalOriginalRidingStart().GetOrElse(*cStart), *server);
        startContext.SetGuaranteeFees(false);
        startContext.SetTakeFeesFromOffer(false);
        const TCarLocationFeatures startFeatures = startContext.GetCarAreaFeatures(false, fpOffer, server);

        if (finishFeatures.HasFeeInfo()) {
            const TMaybe<ui32> deltaPrice = finishFeatures.GetDeltaFees(startFeatures);
            if (deltaPrice) {
                switch (finishFeatures.GetFeeInfoUnsafe().GetPolicy()) {
                    case EDropOfferPolicy::FeesFix:
                        fpOffer->SetFinishAdditionalPrice(*deltaPrice);
                        break;
                    case EDropOfferPolicy::FeesMax:
                        fpOffer->SetFinishMinPrice(*deltaPrice);
                        break;
                    case EDropOfferPolicy::FeesMinute:
                        fpOffer->MutableRiding().AddPrice(*deltaPrice, "flow_correction_" + areaId);
                        break;
                    default:
                        break;
                }
            }
        }
    }
}

TString TFixPointOfferReport::PredictDestination(const TOffersBuildingContext& context, const NDrive::IServer* server, const TCommonDestinationDetector& baseDetector) const {
    const TFixPointOffer* fpOffer = GetOfferAs<TFixPointOffer>();
    if (!fpOffer) {
        return "";
    }
    if (!context.GetStartPosition()) {
        return "";
    }
    TString areaId;
    TSet<TString> excludedZones;
    const auto actor = [&areaId, &context, &excludedZones, &baseDetector](const TDBTag& dbTag) -> bool {
        if (excludedZones.contains(dbTag.GetObjectId())) {
            return true;
        }
        const TDestinationDetectorTag* destTag = dbTag.GetTagAs<TDestinationDetectorTag>();
        if (destTag && destTag->GetModelName() != baseDetector.GetModelName()) {
            excludedZones.emplace(dbTag.GetObjectId());
            return true;
        }
        if (destTag && !destTag->GetStartZoneAttributesFilter().IsEmpty() &&
            context.GetLocationTags() && !destTag->GetStartZoneAttributesFilter().IsMatching(context.GetLocationTags())) {
            excludedZones.emplace(dbTag.GetObjectId());
            return true;
        }
        areaId = dbTag.GetObjectId();
        return false;
    };
    const auto ipc = GetFinishAreaPointContext(server, fpOffer->GetTypeName());
    server->GetDriveAPI()->GetAreasDB()->ProcessHardDBTagsInPoint<TDestinationDetectorTag>(fpOffer->GetFinish(), actor, TInstant::Zero(), ipc);
    if (!areaId) {
        server->GetDriveAPI()->GetAreasDB()->ProcessHardDBTagsInPoint<TDropAreaFeatures>(fpOffer->GetFinish(), actor, TInstant::Zero(), ipc);
    }
    if (!areaId) {
        server->GetDriveAPI()->GetAreasDB()->ProcessHardDBTagsInPoint<TAreaPotentialTag>(fpOffer->GetFinish(), actor, TInstant::Zero(), ipc);
    }
    return areaId;
}

NJson::TJsonValue TFixPointOffer::DoBuildJsonReport(const TReportOptions& options, const ICommonOfferBuilderAction* constructor, const NDrive::IServer& server) const {
    NJson::TJsonValue result = TBase::DoBuildJsonReport(options, constructor, server);
    if (!(options.Traits & NDriveSession::ReportOfferDetails)) {
        return result;
    }
    auto locale = options.Locale;
    auto localization = server.GetLocalization();
    auto ridingDuration = RouteDuration - AdditionalRouteDuration;
    NJson::InsertField(result, "riding_duration", ridingDuration.Seconds());
    if (localization) {
        NJson::InsertField(result, "localized_riding_duration", localization->FormatDuration(locale, ridingDuration));
    }
    JWRITE_DURATION(result, "walking_duration", WalkingDuration);
    JWRITE(result, "parking_in_pack", ParkingInPack);
    if (FinishAreaPublic.size() || FinishArea.size()) {
        JWRITE(result, "finish", Finish.ToString());
    }
    if (FinishAreaPublic.empty()) {
        JWRITE(result, "finish_area", TGeoCoord::SerializeVector(FinishArea));
        JWRITE(result, "finish_area_border", TGeoCoord::SerializeVectorToJsonIFace(FinishArea));
    } else {
        JWRITE(result, "finish_area", TGeoCoord::SerializeVector(FinishAreaPublic));
        JWRITE(result, "finish_area_border", TGeoCoord::SerializeVectorToJsonIFace(FinishAreaPublic));
    }
    if (DestinationContext) {
        JWRITE(result, "destination_context", DestinationContext);
    }
    if (DestinationName) {
        JWRITE(result, "destination_name", DestinationName);
    }
    return result;
}

EDriveSessionResult TFixPointOffer::DoCheckSession(const TUserPermissions& permissions, NDrive::TEntitySession& session, const NDrive::IServer* server, bool onPerform) const {
    auto base = TBase::DoCheckSession(permissions, session, server, onPerform);
    if (base != EDriveSessionResult::Success) {
        return base;
    }

    auto api = server ? server->GetDriveAPI() : nullptr;
    auto action = api->GetRolesManager()->GetAction(GetBehaviourConstructorId());
    auto constructor = action ? action->GetAs<TFixPointOfferConstructor>() : nullptr;
    auto limit = constructor ? constructor->GetFinishAreaDeviceCountLimit() : 0;
    if (limit && onPerform) {
        auto futureLocations = api->GetObjectFuturePositions();
        auto snapshots = server->GetSnapshotsManager().GetSnapshots();
        auto ids = TSet<TString>();
        for (auto&& [id, snapshot] : snapshots.ById()) {
            if (constructor->GetAreaTagsFinish().FilterWeak(snapshot.GetTagsInPoint())) {
                ids.insert(id);
                continue;
            }
            if (futureLocations.contains(id)) {
                auto futureLocation = futureLocations[id];
                auto futureLocationTags = api->GetTagsInPoint(futureLocation);
                if (constructor->GetAreaTagsFinish().FilterWeak(futureLocationTags)) {
                    ids.insert(id);
                    continue;
                }
            }
            if (ids.size() >= limit) {
                break;
            }
        }
        if (ids.size() >= limit) {
            session.SetErrorInfo("FixPointOffer::DoCheckSession", "device count in finish area exceeded", NDrive::MakeError("device_count_limit_in_finish_area"));
            return EDriveSessionResult::ResourceLocked;
        }
    }

    auto linkedAreaInfo = TFixPointCorrectionTag::GetLinkedArea(GetFinish(), *server);
    if (linkedAreaInfo && linkedAreaInfo->CarLimit && onPerform) {
        const auto& areaId = linkedAreaInfo->Area.GetIdentifier();
        const auto& snapshots = server->GetSnapshotsManager().GetSnapshots();
        const auto currentCount = snapshots.GetAreaCarsCount(areaId);
        const auto futuresCount = snapshots.GetFutureAreaCarsCount(areaId);
        if (currentCount + futuresCount > linkedAreaInfo->CarLimit) {
            session.SetErrorInfo("FixPointOffer::DoCheckSession", "device count in finish area exceeded", NDrive::MakeError("device_count_limit_in_finish_area"));
            return EDriveSessionResult::ResourceLocked;
        }
    }

    return EDriveSessionResult::Success;
}

TString TFixPointOffer::DoFormDescriptionElement(const TString& value, ELocalization locale, const ILocalization* localization) const {
    TString result = TBase::DoFormDescriptionElement(value, locale, localization);

    const TString routeDuration = localization->FormatDuration(locale, GetRouteDuration());
    SubstGlobal(result, "_RouteDuration_", routeDuration);

    const TString routeLength = localization->DistanceFormatKm(locale, GetRouteLength() / 1000);
    SubstGlobal(result, "_RouteLength_", routeLength);

    SubstGlobal(result, "_IncorrectFinishFeesPercent_", ::ToString((ui32)(GetFeesForIncorrectFinish() * 100)));
    SubstGlobal(result, "_IncorrectFinishRidingPrice_", localization->FormatPrice(locale, GetPublicOriginalPrice((1 + GetFeesForIncorrectFinish()) * GetRiding().GetPrice(), ESessionState::Riding, 1)));
    SubstGlobal(result, "_IncorrectFinishParkingPrice_", localization->FormatPrice(locale, GetPublicOriginalPrice((1 + GetFeesForIncorrectFinish()) * GetParking().GetPrice(), ESessionState::Parking, 1)));

    return result;
}

TOfferStatePtr TFixPointOffer::DoCalculate(const TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events, const TInstant& until, const TRidingInfo& ridingInfo, TOfferPricing& result) const {
    TOfferStatePtr state = TBase::DoCalculate(timeline, events, until, ridingInfo, result);
    if (!state || timeline.empty()) {
        return state;
    }

    if (timeline.back().GetTimeEvent() == IEventsSession<TCarTagHistoryEvent>::EEvent::Tag) {
        if (timeline.back().GetEventInstant() < until) {
            const THistoryDeviceSnapshot* dSnapshot = (*events[timeline.back().GetEventIndex()])->GetObjectSnapshotAs<THistoryDeviceSnapshot>();
            const auto server = NDrive::HasServer() ? &NDrive::GetServerAs<NDrive::IServer>() : nullptr;
            if (dSnapshot) {
                bool inFinishArea = IsInFinishArea(*dSnapshot, server, FinishArea, GetTypeName()).GetOrElse(false);
                if (!inFinishArea) {
                    TOfferPricing newPricing(result.GetOffer());
                    TOfferStatePtr newState = TStandartOffer::DoCalculate(timeline, events, until, ridingInfo, newPricing);
                    newPricing.MulPrices(1 + GetFeesForIncorrectFinish());
                    result = newPricing;
                    newState->SetNeedFinishFees(state->GetNeedFinishFees());
                    return newState;
                } else {
                    state->SetNeedFinishFees(false);
                }
            }
        }
    } else {
        auto packOfferState = std::dynamic_pointer_cast<TPackOfferState>(state);
        if (packOfferState) {
            auto fixPointOfferState = MakeHolder<TFixPointOfferState>(*packOfferState);
            if (NDrive::HasServer()) {
                const auto& server = NDrive::GetServerAs<NDrive::IServer>();
                const auto snapshot = server.GetSnapshotsManager().GetSnapshot(GetObjectId());
                fixPointOfferState->SetDropAvailable(IsInFinishArea(snapshot, &server, FinishArea, GetTypeName()));
            }
            return fixPointOfferState;
        }
    }
    return state;
}

bool TFixPointOffer::CheckNeedFeesOnFinish(const NDrive::IServer* server) const {
    const THistoryDeviceSnapshot dSnapshot = server->GetSnapshotsManager().GetSnapshot(GetObjectId());
    bool inFinishArea = IsInFinishArea(dSnapshot, server, FinishArea, GetTypeName()).GetOrElse(false);
    return !inFinishArea;
}

ui32 TFixPointOffer::CalcPackPrice(const NDrive::IServer* server) {
    if (InheritedProperties) {
        return InheritedProperties->PackPrice;
    }
    const double areaDiscount = 1 - AreasDiscount.GetOrElse(0);
    const double packDiscount = 1 - PackDiscount.GetOrElse(0);
    const double kfDiscount = areaDiscount * packDiscount;
    const double priceOriginal = (GetUseKmForPackCalculation() ? TFullPricesContext::CalcPackPrice(GetRouteDuration(), GetRouteLength() / 1000, Max<ui32>()) : TFullPricesContext::CalcFixPrice(GetRouteDuration(), GetRouteLength() / 1000)) * kfDiscount;
    const double priceWithAcceptance = priceOriginal + AcceptancePrice.GetOrElse(GetAcceptance().GetPrice());
    const double priceWithRestrictions = FinishAdditionalPrice.GetOrElse(0) + Max<ui32>(FinishMinPrice.GetOrElse(0), priceWithAcceptance);
    if (!MinPriceLow) {
        const double maxVariation = server->GetSettings().GetValueDef<double>("offers.fix_point.min_price_difference", 700);
        const double minPrice = server->GetSettings().GetValueDef<double>("offers.fix_point.min_price", 5342);
        MinPriceLow = minPrice + RandomNumber<double>() * maxVariation;
    }
    if (kfDiscount < 1e-5) {
        MinPriceLow = 0;
    }
    double result = Max<double>(*MinPriceLow, priceWithRestrictions);
    return UseRoundedPrice ? RoundPrice(result, 100) : result;
}

TMaybe<ui32> TFixPointOffer::GetEffectiveCashback(const NDrive::IServer& server) const {
    auto cashbackPercent = GetEffectiveCashbackPercent(server);
    if (!cashbackPercent) {
        return {};
    }
    return ICommonOffer::RoundCashback(*cashbackPercent * GetPublicDiscountedPackPrice() / 100., 100);
}

ui32 TFixPointOffer::GetPublicDiscountedPackPrice(TMaybe<ui32> overridenPackPrice, const ui32 precision) const {
    auto packPrice = overridenPackPrice.GetOrElse(GetPackPrice());
    if (!GetRiding().GetPriceModeling()) {
        return RoundPrice(ApplyDiscount(packPrice, ""), precision);
    } else {
        return RoundPrice(ApplyDiscount(packPrice, true, ""), precision);
    }
}

ui32 TFixPointOffer::GetPublicOriginalPackPrice(TMaybe<ui32> overridenPackPrice, const ui32 precision) const {
    auto packPrice = overridenPackPrice.GetOrElse(GetPackPrice());
    if (!GetRiding().GetPriceModeling()) {
        return RoundPrice(ApplyDiscount(packPrice, false, ""), precision);
    } else {
        return packPrice;
    }
}

bool TFixPointOffer::DeserializeFixPointFromProto(const NDrive::NProto::TFixPointOffer& info) {
    FeesForIncorrectFinish = info.GetFeesForIncorrectFinish();
    WalkingDuration = TDuration::Seconds(info.GetWalkingDuration());
    RouteDuration = TDuration::Seconds(info.GetRouteDuration());
    RouteLength = info.GetRouteLength();
    AdditionalRouteDuration = TDuration::Seconds(info.GetAdditionalRouteDuration());
    AdditionalRouteLength = info.GetAdditionalRouteLength();
    ParkingInPack = info.GetParkingInPack();
    if (info.HasUseKmForPackCalculation()) {
        UseKmForPackCalculation = info.GetUseKmForPackCalculation();
    }
    if (info.HasUseRoundedPrice()) {
        UseRoundedPrice = info.GetUseRoundedPrice();
    }
    if (info.HasUseMapsRouter()) {
        UseMapsRouter = info.GetUseMapsRouter();
    }
    if (info.HasUseAvoidTolls()) {
        UseAvoidTolls = info.GetUseAvoidTolls();
    }
    if (info.HasPackDiscount()) {
        PackDiscount = info.GetPackDiscount();
    }
    if (info.HasAreasDiscount()) {
        AreasDiscount = info.GetAreasDiscount();
    }
    DestinationContext = info.GetDestinationContext();
    DestinationDescription = info.GetDestinationDescription();
    DestinationName = info.GetDestinationName();
    TaxiPrice = info.GetTaxiPrice();
    if (info.HasMinPriceLow()) {
        MinPriceLow = info.GetMinPriceLow();
    }
    if (info.HasFinishMinPrice()) {
        FinishMinPrice = info.GetFinishMinPrice();
    }
    if (info.HasFinishAdditionalPrice()) {
        FinishAdditionalPrice = info.GetFinishAdditionalPrice();
    }
    if (!Finish.Deserialize(info.GetFinish())) {
        return false;
    }
    for (auto&& i : info.GetFinishArea()) {
        TGeoCoord c;
        if (!c.Deserialize(i)) {
            return false;
        }
        FinishArea.emplace_back(std::move(c));
    }
    for (auto&& i : info.GetFinishAreaIds()) {
        FinishAreaIds.emplace(i);
    }
    for (auto&& i : info.GetFinishAreaPublic()) {
        TGeoCoord c;
        if (!c.Deserialize(i)) {
            return false;
        }
        FinishAreaPublic.emplace_back(std::move(c));
    }
    if (info.HasInheritedDestination() || info.HasInheritedTimestamp() || info.HasInheritedPackPrice()) {
        InheritedProperties.ConstructInPlace();
    }
    if (info.HasInheritedDestination()) {
        if (!InheritedProperties->Destination.Deserialize(info.GetInheritedDestination())) {
            return false;
        }
    }
    if (info.HasInheritedTimestamp()) {
        InheritedProperties->Timestamp = TInstant::Seconds(info.GetInheritedTimestamp());
    }
    if (info.HasInheritedPackPrice()) {
        InheritedProperties->PackPrice = info.GetInheritedPackPrice();
    }
    for (auto&& i : info.GetRouteDurationModelInfo()) {
        RouteDurationModelInfos.push_back({i.GetName(), i.GetBefore(), i.GetAfter()});
    }
    if (info.HasAcceptancePrice()) {
        AcceptancePrice = info.GetAcceptancePrice();
    }
    for (auto&& i : info.GetAcceptancePriceModelInfo()) {
        AcceptancePriceModelInfos.push_back({i.GetName(), i.GetBefore(), i.GetAfter()});
    }
    if (info.HasIsFake()) {
        IsFake = info.GetIsFake();
    }
    return true;
}

namespace {
    // AssignProtoModelInfo helps in assigning model info to proto.
    void AssignProtoModelInfo(NDrive::NProto::TPriceModelInfo* proto, const TPriceModelInfo& value) {
        Y_ENSURE(proto);
        proto->SetName(value.Name);
        proto->SetBefore(value.Before);
        proto->SetAfter(value.After);
    }
}

void TFixPointOffer::SerializeFixPointToProto(NDrive::NProto::TFixPointOffer& info) const {
    info.SetWalkingDuration(WalkingDuration.Seconds());
    info.SetFeesForIncorrectFinish(FeesForIncorrectFinish);
    info.SetParkingInPack(ParkingInPack);
    info.SetRouteDuration(RouteDuration.Seconds());
    info.SetRouteLength(RouteLength);
    info.SetIsFake(IsFake);
    info.SetAdditionalRouteDuration(AdditionalRouteDuration.Seconds());
    info.SetAdditionalRouteLength(AdditionalRouteLength);
    info.SetUseKmForPackCalculation(UseKmForPackCalculation);
    info.SetUseMapsRouter(UseMapsRouter);
    info.SetUseAvoidTolls(UseAvoidTolls);
    info.SetUseRoundedPrice(UseRoundedPrice);
    if (!!PackDiscount) {
        info.SetPackDiscount(*PackDiscount);
    }
    if (!!AreasDiscount) {
        info.SetAreasDiscount(*AreasDiscount);
    }
    info.SetDestinationContext(DestinationContext);
    info.SetDestinationDescription(DestinationDescription);
    info.SetDestinationName(DestinationName);
    *info.MutableFinish() = Finish.Serialize();
    for (auto&& c : FinishArea) {
        *info.AddFinishArea() = c.Serialize();
    }
    for (auto&& c : FinishAreaPublic) {
        *info.AddFinishAreaPublic() = c.Serialize();
    }
    for (auto&& t : FinishAreaIds) {
        info.AddFinishAreaIds(t);
    }
    if (TaxiPrice > 0) {
        info.SetTaxiPrice(TaxiPrice);
    }
    if (MinPriceLow && *MinPriceLow) {
        info.SetMinPriceLow(*MinPriceLow);
    }
    if (FinishMinPrice && *FinishMinPrice) {
        info.SetFinishMinPrice(*FinishMinPrice);
    }
    if (FinishAdditionalPrice && *FinishAdditionalPrice) {
        info.SetFinishAdditionalPrice(*FinishAdditionalPrice);
    }
    if (InheritedProperties) {
        *info.MutableInheritedDestination() = InheritedProperties->Destination.Serialize();
        info.SetInheritedTimestamp(InheritedProperties->Timestamp.Seconds());
        info.SetInheritedPackPrice(InheritedProperties->PackPrice);
    }
    for (auto&& value : GetRouteDurationModelInfos()) {
        AssignProtoModelInfo(info.AddRouteDurationModelInfo(), value);
    }
    if (AcceptancePrice) {
        info.SetAcceptancePrice(*AcceptancePrice);
    }
    for (auto&& value : GetAcceptancePriceModelInfos()) {
        AssignProtoModelInfo(info.AddAcceptancePriceModelInfo(), value);
    }
}

void TFixPointOffer::ApplyRouteDurationModel(const NDrive::IOfferModel& model) {
    TDuration duration = TDuration::Seconds(model.Calc(MutableFeatures()));
    RouteDurationModelInfos.push_back({model.GetName(), (float)GetRouteDuration().Seconds(), (float)duration.Seconds()});
    SetRouteDuration(duration);
}

void TFixPointOffer::ApplyAcceptancePriceModel(const NDrive::IOfferModel& model) {
    auto result = model.Calc(MutableFeatures());
    if (result >= 0) {
        ui32 price = 100 * result;
        AcceptancePriceModelInfos.push_back({model.GetName(), (AcceptancePrice ? (float)*AcceptancePrice : (float)-1), (float)price});
        SetAcceptancePrice(price);
    } else {
        ClearAcceptancePrice();
    }
}

void TFixPointOfferReport::RecalculateFeatures() {
    TBase::RecalculateFeatures();
    TFixPointOffer* offer = GetOfferAs<TFixPointOffer>();
    if (offer) {
        NDrive::CalcOfferRouteDurationFeatures(offer->MutableFeatures(), offer->GetRouteDuration());
        NDrive::CalcOfferFixPointAcceptancePriceFeatures(offer->MutableFeatures(), offer->OptionalAcceptancePrice());
    }
}

void TFixPointOfferReport::DoRecalcPrices(const NDrive::IServer* server) {
    TFixPointOffer* fpOffer = GetOfferAs<TFixPointOffer>();
    if (fpOffer) {
        fpOffer->SetPackPrice(fpOffer->CalcPackPrice(server));
    }
}

TLimitedFraction<TDuration> TFixPointOfferConstructor::GetAdditionalDurationPolicy(const TOffersBuildingContext& context) const {
    TLimitedFraction<TDuration> additionalDuration(0.7, TDuration::Minutes(25), TDuration::Minutes(60));
    auto info = context.GetSetting<TString>("offers.fix_point.additional_duration");
    if (info && !additionalDuration.Deserialize(*info)) {
        NDrive::TEventLog::Log("GetAdditionalDurationPolicyError", NJson::TMapBuilder
            ("info", *info)
        );
    }
    return additionalDuration;
}

TLimitedFraction<double> TFixPointOfferConstructor::GetAdditionalDistancePolicy(const TOffersBuildingContext& context) const {
    TLimitedFraction<double> additionalDistance(0.7, 2000, 5000);
    auto info = context.GetSetting<TString>("offers.fix_point.additional_distance");
    if (info && !additionalDistance.Deserialize(*info)) {
        NDrive::TEventLog::Log("GetAdditionalDistancePolicyError", NJson::TMapBuilder
            ("info", *info)
        );
    }
    return additionalDistance;
}

TUserAction::TFactory::TRegistrator<TFixPointOfferConstructor> TFixPointOfferConstructor::Registrator(TFixPointOfferConstructor::GetTypeStatic());

bool TFixPointOfferConstructor::AddStopAreaInfo(TFixPointOffer& offer, const NDrive::IServer* server, const TOffersBuildingContext::TDestination& destination, NDrive::TInfoEntitySession& session) const {
    auto gBuildArea = destination.GetOfferContext().BuildEventGuard("add_stop_area_info");
    if (!destination.GetDestination().GetFinishArea()) {
        session.AddErrorMessage("TFixPointOfferConstructor::AddStopAreaInfo", "Not initialized finish area for destination");
        return false;
    }
    const auto& finishArea = *destination.GetDestination().GetFinishArea();
    if (!finishArea.HasFinishArea()) {
        session.AddErrorMessage("TFixPointOfferConstructor::AddStopAreaInfo", "haven't finish area");
        return false;
    }
    offer.SetWalkingDuration(finishArea.GetWalkingDuration());
    offer.MutableFinishArea() = finishArea.GetFinishAreaUnsafe();
    offer.MutableFinishAreaIds() = server->GetDriveAPI()->GetAreasDB()->GetAreaIdsInPoint(offer.GetFinish(), TInstant::Zero());
    if (finishArea.HasFinishAreaPublic()) {
        offer.MutableFinishAreaPublic() = finishArea.GetFinishAreaPublicUnsafe();
    }
    if (server->GetSettings().GetValueDef("offers.fix_point.available_area_only", false)) {
        auto gCutArea = destination.GetOfferContext().BuildEventGuard("cut_areas");
        TMap<TString, TArea> areas;
        if (server->GetDriveAPI()->GetAreasDB()->GetObjects(areas) && offer.MutableFinishArea().size()) {
            TPolyLine<TGeoCoord> line(offer.MutableFinishArea());
            double areaMax = 0;
            for (auto&& area : areas) {
                if (!area.second.GetPolyline().Size() || !area.second.GetPolyline().GetRectSafe().Cross(line.GetRectSafe())) {
                    continue;
                }
                if (CheckAreaTags(area.second, offer, server) == EDropAbility::Allow) {
                    TVector<TPolyLine<TGeoCoord>> lines = line.AreasIntersection(area.second.GetPolyline());
                    for (auto&& copy : lines) {
                        if (areaMax < copy.GetArea()) {
                            offer.MutableFinishArea() = copy.GetCoords();
                            areaMax = copy.GetArea();
                        }
                    }
                }
            }
            line = TPolyLine<TGeoCoord>(offer.MutableFinishArea());
            double areaMin = line.GetArea();
            for (auto&& area : areas) {
                if (!area.second.GetPolyline().Size() || !area.second.GetPolyline().GetRectSafe().Cross(line.GetRectSafe())) {
                    continue;
                }
                if (CheckAreaTags(area.second, offer, server) == EDropAbility::Deny) {
                    TVector<TPolyLine<TGeoCoord>> lines = line.AreasIntersection(area.second.GetPolyline(), true, false);
                    double areaMaxLocal = 0;
                    TMaybe<TVector<TGeoCoord>> maxAreaLocal;
                    for (auto&& copy : lines) {
                        if (!maxAreaLocal || areaMaxLocal < copy.GetArea()) {
                            maxAreaLocal = copy.GetCoords();
                            areaMaxLocal = copy.GetArea();
                        }
                    }
                    if (areaMin > areaMaxLocal && !!maxAreaLocal) {
                        offer.MutableFinishArea() = *maxAreaLocal;
                        areaMin = areaMaxLocal;
                    }
                }
            }
        }
    }
    if (offer.GetFinishArea().empty()) {
        session.AddErrorMessage("TFixPointOfferConstructor::AddStopAreaInfo", "empty finish area");
        return false;
    }
    return true;
}

bool TFixPointOfferConstructor::ConstructPackOffer(const TStandartOfferReport& stOfferReport, const TOffersBuildingContext& context, const NDrive::IServer* server, TVector<IOfferReport::TPtr>& result, NDrive::TInfoEntitySession& session) const {
    auto startCoord = context.GetStartPosition();
    if (!context.HasUserHistoryContext()) {
        session.AddErrorMessage("TFixPointOfferConstructor::ConstructPackOffer", "no user context");
        return false;
    }
    if (!startCoord) {
        session.AddErrorMessage("TFixPointOfferConstructor::ConstructPackOffer", "no car location");
        return false;
    }

    const auto& settings = server->GetSettings();
    auto userPermissions = context.GetUserHistoryContextRef().GetUserPermissions();
    if (context.GetUserHistoryContextUnsafe().HasUserDestination()) {
        if (context.GetDestinations().size() != 1) {
            session.AddErrorMessage("TFixPointOfferConstructor::ConstructPackOffer", "incorrect destinations for exists user_destination");
            return false;
        }
        auto expectedOffer = ConstructFixPointOffer(stOfferReport, *startCoord, context.GetDestinations().front(), context, server, session);
        if (!expectedOffer) {
            context.AddError(GetName(), expectedOffer.GetError());
            return false;
        }
        auto offer = (expectedOffer && *expectedOffer) ? expectedOffer->Get()->GetOfferAs<TFixPointOffer>() : nullptr;
        if (offer) {
            const auto& userCtx = context.GetUserHistoryContextUnsafe();
            offer->SetTaxiPrice(userCtx.GetTaxiPrice());
            result.emplace_back(*expectedOffer);
        }
        auto enableInheritance = context.GetSetting<bool>("offers.fix_point.inheritance.enabled").GetOrElse(true);
        auto previousOffer = enableInheritance ? context.GetPreviousOffer(GetName()) : nullptr;
        if (previousOffer) {
            auto eg = context.BuildEventGuard("process_previous_offer");
            auto previousFixPointOffer = std::dynamic_pointer_cast<TFixPointOffer>(previousOffer);
            if (previousFixPointOffer) {
                TFixPointOffer::TInheritedProperties inheritedProperties;
                if (previousFixPointOffer->HasInheritedProperties()) {
                    inheritedProperties = previousFixPointOffer->GetInheritedPropertiesRef();
                } else {
                    inheritedProperties.Destination = previousFixPointOffer->GetFinish();
                    inheritedProperties.PackPrice = previousFixPointOffer->GetPackPrice();
                    inheritedProperties.Timestamp = previousFixPointOffer->GetTimestamp();
                }
                auto lifetimeLimit = TUserPermissions::GetSetting<TDuration>(
                    "offers.fix_point.inheritance.lifetime_limit", settings, userPermissions
                ).GetOrElse(TDuration::Minutes(5));
                auto distanceLimit = TUserPermissions::GetSetting<double>(
                    "offers.fix_point.inheritance.distance_limit", settings, userPermissions
                ).GetOrElse(300);
                auto distance = offer->GetFinish().GetLengthTo(inheritedProperties.Destination);
                auto lifetime = offer->GetTimestamp() - inheritedProperties.Timestamp;
                bool match =
                    offer->GetInsuranceType() == previousFixPointOffer->GetInsuranceType();
                if (distance < distanceLimit && lifetime < lifetimeLimit && match) {
                    offer->SetInheritedProperties(inheritedProperties);
                    offer->SetPackPrice(offer->CalcPackPrice(server));
                }
            } else {
                NJson::TJsonValue ev = NJson::TMapBuilder
                    ("event", "process_previous_offer_failure")
                    ("offer_id", previousOffer->GetOfferId())
                    ("offer_type", previousOffer->GetTypeName())
                ;
                NDrive::TEventLog::Log("ProcessPreviousOfferFailure", ev);
                if (eg) {
                    eg->AddEvent(std::move(ev));
                }
            }
        }
        return !!offer;
    } else {
        THolder<TFixPointOfferReport> offerReport = Extend(stOfferReport);
        if (!offerReport) {
            session.AddErrorMessage("TFixPointOfferConstructor::ConstructPackOffer", "incorrect base class");
            return false;
        }
        if (context.GetDestinations().size() != context.GetUserHistoryContextUnsafe().GetHintedDestinations().size()) {
            session.AddErrorMessage("TFixPointOfferConstructor::ConstructPackOffer", "incorrect destinations for exists user_destination_hints");
            return false;
        }
        TFixPointOffer& offer = *offerReport->GetOfferAs<TFixPointOffer>();
        if (Focus) {
            offerReport->SetFocus(*Focus);
        }
        if (auto prioritizeFakes = context.GetSetting<bool>("offers.fix_point.prioritize_fakes").GetOrElse(false)) {
            offerReport->SetInternalPriority(-1);
        }
        offer.SetIsFake(true);
        offer.SetOverrunKm(offerReport->GetRerunPriceKM().GetOrElse(GetRerunPriceKM()));
        offer.MutableParking().SetPrice(offer.GetParking().GetPrice() * Max<double>(0, GetKfParkingPrice()));
        offer.SetWalkingDuration(WalkingDuration);
        offer.SetParkingInPack(ParkingInPack);
        offerReport->SetDetailedDescription(FakeDetailedDescription);
        ui32 extHintsCount = 0;
        for (auto&& destination : context.GetDestinations()) {
            auto expectedOffer = ConstructFixPointOffer(stOfferReport, *startCoord, destination, context, server, session);
            if (!expectedOffer && context.GetDestinations().size() == 1) {
                context.AddError(GetName(), expectedOffer.GetError());
            }
            if (!expectedOffer) {
                session.AddErrorMessage("FixPointOfferConstructor::ConstructPackOffer", "cannot construct offer for hinted destination " + destination.GetDestination().GetName());
                continue;
            }
            auto hintOffer = *expectedOffer;
            if (!hintOffer) {
                continue;
            }
            if (context.NeedOfferedHintsSeparately()) {
                hintOffer->SetDestination(destination.GetDestination());
                hintOffer->SetIsHint(true);
                result.emplace_back(hintOffer);
                ++extHintsCount;
            } else {
                offerReport->MutableHints().emplace_back(destination.GetDestination(), hintOffer);
            }
        }
        auto potentialInfos = TAreaPotentialTag::CalcInfos(*server, *startCoord, GetName());
        std::sort(potentialInfos.begin(), potentialInfos.end(), [](const TAreaPotentialTag::TInfo& left, const TAreaPotentialTag::TInfo& right) {
            return std::tie(left.GetPriority(), left.GetDiscountPercent()) > std::tie(right.GetPriority(), right.GetDiscountPercent());
        });
        for (auto&& info : potentialInfos) {
            if (extHintsCount + offerReport->GetHints().size() >= server->GetSettings().GetValueDef<size_t>("offers.fix_point.hints_count_max", 2)) {
                break;
            }
            if (!info.GetPublic()) {
                continue;
            }

            offerReport->MutableHints().emplace_back(info);
        }

        std::sort(potentialInfos.begin(), potentialInfos.end(), [](const TAreaPotentialTag::TInfo& left, const TAreaPotentialTag::TInfo& right) {
            return std::make_tuple(left.GetAreaSize(), left.GetDiscountPercent(), left.GetPriority()) > std::make_tuple(right.GetAreaSize(), right.GetDiscountPercent(), right.GetPriority());
        });
        for (auto&& info : potentialInfos) {
            if (!info.GetPublic()) {
                continue;
            }

            offerReport->MutableDiscountAreas().push_back(info);
        }
        if (!context.NeedOfferedHintsOnly()) {
            result.emplace_back(offerReport.Release());
        }
        return true;
    }
}

TExpected<TAtomicSharedPtr<TFixPointOfferReport>, TString> TFixPointOfferConstructor::ConstructFixPointOffer(
    const TStandartOfferReport& stOfferReport,
    const TGeoCoord& source,
    const TOffersBuildingContext::TDestination& destination,
    const TOffersBuildingContext& context,
    const NDrive::IServer* server,
    NDrive::TInfoEntitySession& session
) const {
    if (!context.HasCarId()) {
        return MakeUnexpected<TString>("incorrect_car_id");
    }
    const auto linkedAreaInfo = TFixPointCorrectionTag::GetLinkedArea(destination.GetDestination().GetCoordinate(), *server);
    const TCarLocationContext clc = linkedAreaInfo ? TCarLocationContext::BuildByArea(context.GetCarIdUnsafe(), linkedAreaInfo->Area, *server) : TCarLocationContext::BuildByCoord(context.GetCarIdUnsafe(), destination.GetDestination().GetCoordinate(), *server);
    if (linkedAreaInfo && linkedAreaInfo->CarLimit) {
        const auto& areaId = linkedAreaInfo->Area.GetIdentifier();
        const auto& snapshots = server->GetSnapshotsManager().GetSnapshots();
        const auto currentCount = snapshots.GetAreaCarsCount(areaId);
        const auto futuresCount = snapshots.GetFutureAreaCarsCount(areaId);
        if (currentCount + futuresCount >= linkedAreaInfo->CarLimit) {
            return MakeUnexpected<TString>("device_count_limit_in_finish_area");
        }
    }
    if (!GetAreaTagsFinish().Empty() && !GetAreaTagsFinish().FilterWeak(clc.GetTagsPotentially())) {
        return MakeUnexpected<TString>("incorrect_destination_tags");
    }
    if (clc.GetCarAreaFeatures(false, stOfferReport.GetOfferPtrAs<IOffer>().Get(), server).GetAllowDropDef(EDropAbility::Allow) != EDropAbility::Allow) {
        return MakeUnexpected<TString>("cannot_drop_car_in_destination");
    }

    auto optionalRoute = destination.GetRoute();
    if (!optionalRoute) {
        session.AddErrorMessage("FixPointOfferConstructor::ConstructFixPointOffer", "no router reply");
        return MakeUnexpected<TString>("router_unanswer");
    }
    if (!*optionalRoute) {
        session.AddErrorMessage("FixPointOfferConstructor::ConstructFixPointOffer", "no route");
        return MakeUnexpected<TString>("undefined_route");
    }

    THolder<TFixPointOfferReport> fpOfferReport = Extend(stOfferReport);
    if (HasFocus()) {
        fpOfferReport->SetFocus(GetFocusUnsafe());
    }
    if (!fpOfferReport) {
        session.AddErrorMessage("FixPointOfferConstructor::ConstructFixPointOffer", "incorrect base offer class");
        return MakeUnexpected<TString>("incorrect_base_offer_type");
    }
    const auto& route = optionalRoute->GetRef();
    if (route.Length > GetMaxPathDistance()) {
        return MakeUnexpected<TString>("too_far_destination");
    }

    const auto fuelDistance = Yensured(server)->GetSnapshotsManager().GetSnapshot(context.GetCarIdUnsafe()).GetFuelDistance();
    const auto fuelDistanceCoefficient = context.GetSetting<double>("offers.fix_point.fuel_distance_coefficient").GetOrElse(0);
    if (fuelDistance && fuelDistanceCoefficient != 0 && route.Length > *fuelDistance * fuelDistanceCoefficient) {
        return MakeUnexpected<TString>("not_enough_charge");
    }

    const double distanceThreshold = GetDistanceThreshold(*server);
    if (route.Length < distanceThreshold) {
        auto error = TooCloseDestinationError;
        return MakeUnexpected<TString>(std::move(error));
    }
    const double haversine = source.GetLengthTo(destination.GetDestination().GetCoordinate());
    const double haversineThreshold = GetHaversineThreshold(*server);
    if (haversine < haversineThreshold) {
        auto error = TooCloseDestinationError;
        return MakeUnexpected<TString>(std::move(error));
    }

    TFixPointOffer* offer = fpOfferReport->GetOfferAs<TFixPointOffer>();
    fpOfferReport->SetDetailedDescription(GetDetailedDescription());
    offer->SetFinish(destination.GetDestination().GetCoordinate());
    offer->SetFeesForIncorrectFinish(GetFeesForIncorrectFinish());
    offer->SetUseKmForPackCalculation(GetUseKmForPackCalculation());
    offer->SetOfferId(ICommonOffer::CreateOfferId());
    offer->SetOverrunKm(fpOfferReport->GetRerunPriceKM().GetOrElse(GetRerunPriceKM()));
    offer->SetParkingInPack(ParkingInPack);
    offer->MutableParking().SetPrice(offer->GetParking().GetPrice() * Max<double>(0, GetKfParkingPrice()));
    FeesTime.Calc(TDuration::Seconds(route.Time), offer->MutableAdditionalRouteDuration());
    FeesDistance.Calc(route.Length, offer->MutableAdditionalRouteLength());
    offer->SetRouteDuration(TDuration::Seconds(route.Time) + offer->GetAdditionalRouteDuration());
    offer->SetRouteLength(route.Length + offer->GetAdditionalRouteLength());
    offer->SetUseMapsRouter(route.Origin == NGraph::TRouter::EOrigin::MapsRouter);
    offer->SetUseAvoidTolls((route.Flags & NGraph::TRouter::EFlag::AvoidTolls) > 0);
    offer->SetUseRoundedPrice(GetUseRoundedPrice());
    if (UseDynamicPushCalculation) {
        offer->SetDistancePushThreshold(route.Length / 1000.0);
        offer->SetDurationPushThreshold(TDuration::Seconds(route.Time));
    }

    auto additionalDurationPolicy = GetAdditionalDurationPolicy(context);
    auto offerDuration = offer->GetRouteDuration() + additionalDurationPolicy.Calc(TDuration::Seconds(route.Time)).GetOrElse(TDuration::Zero());
    offer->SetDuration(offerDuration);

    auto additionalDistancePolicy = GetAdditionalDistancePolicy(context);
    auto offerDistance = offer->GetRouteLength() + additionalDistancePolicy.Calc(route.Length).GetOrElse(0);
    offer->SetMileageLimit(offerDistance / 1000.0);
    offer->SetPackDiscount(GetPackDiscount());
    offer->SetUseDeposit(context.GetSetting<bool>("offers.fix_point.use_deposit").GetOrElse(GetUseDeposit()));
    fpOfferReport->RecalcPrices(server);

    offer->SetDestinationContext(destination.GetDestination().GetContextClient());
    offer->SetDestinationDescription(destination.GetDestination().GetDescription());
    offer->SetDestinationName(destination.GetDestination().GetName());

    if (!AddStopAreaInfo(*offer, server, destination, session)) {
        return MakeUnexpected<TString>("cannot_build_stop_area_info");
    }
    TPolyLine<TGeoCoord> line(offer->GetFinishArea());
    if (IsInternalPointDisabled() && line.IsPointInternal(source)) {
        auto error = TooCloseDestinationError;
        return MakeUnexpected<TString>(std::move(error));
    }
    {
        const auto guard = context.BuildEventGuard("fix_point_features");
        const auto& coordinate = destination.GetDestination().GetCoordinate();
        const NDrive::TGeoFeatures* geoFeatures = nullptr;
        const NDrive::TGeoFeatures* geobaseFeatures = nullptr;
        const NDrive::TUserGeoFeatures* userGeoFeatures = nullptr;
        const NDrive::TUserGeoFeatures* userGeobaseFeatures = nullptr;
        const NDrive::TUserDoubleGeoFeatures* userDoubleGeoFeatures = nullptr;
        const NDrive::TUserDoubleGeoFeatures* userDoubleGeobaseFeatures = nullptr;
        {
            auto g = context.BuildEventGuard("destination_geo_features");
            geoFeatures = destination.GetDestination().GetGeoFeatures();
        }
        {
            auto g = context.BuildEventGuard("destination_geobase_features");
            geobaseFeatures = destination.GetDestination().GetGeobaseFeatures();
        }
        {
            auto g = context.BuildEventGuard("destination_user_geo_features");
            userGeoFeatures = destination.GetDestination().GetUserGeoFeatures();
        }
        {
            auto g = context.BuildEventGuard("destination_user_geobase_features");
            userGeobaseFeatures = destination.GetDestination().GetUserGeobaseFeatures();
        }
        {
            auto g = context.BuildEventGuard("destination_user_double_geo_features");
            userDoubleGeoFeatures = destination.GetUserDoubleGeoFeatures();
        }
        {
            auto g = context.BuildEventGuard("destination_user_double_geobase_features");
            userDoubleGeobaseFeatures = destination.GetUserDoubleGeobaseFeatures();
        }
        double distance = route.Length;
        double duration = route.Time;
        double score = 2;
        ui64 geobaseId = destination.GetDestination().GetGeobaseId();
        NDrive::CalcDestinationFeatures(offer->MutableFeatures(), coordinate, geoFeatures, userGeoFeatures, distance, duration);
        NDrive::CalcDestinationFeatures(offer->MutableFeatures(), geobaseId, geobaseFeatures, userGeobaseFeatures);
        NDrive::CalcDestinationFeatures(offer->MutableFeatures(), userDoubleGeoFeatures);
        NDrive::CalcDestinationFeatures(offer->MutableFeatures(), geobaseId, userDoubleGeobaseFeatures);
        NDrive::CalcBestDestinationFeatures(offer->MutableFeatures(), score);
        CalcPackOfferFeatures(*offer);
    }
    return fpOfferReport;
}

THolder<TFixPointOfferReport> TFixPointOfferConstructor::Extend(const TStandartOfferReport& standardOfferReport) const {
    return IOfferReport::Extend<TFixPointOfferReport>(standardOfferReport);
}

double TFixPointOfferConstructor::GetDistanceThreshold(const NDrive::IServer& server) const {
    return server.GetSettings().GetValue<double>("offers.fix_point.threshold_distance").GetOrElse(500);
}

double TFixPointOfferConstructor::GetHaversineThreshold(const NDrive::IServer& server) const {
    return server.GetSettings().GetValue<double>("offers.fix_point.threshold_haversine").GetOrElse(0);
}

bool TFixPointOfferConstructor::IsInternalPointDisabled() const {
    return true;
}

bool TFixPointOfferConstructor::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) {
    if (!TBase::DeserializeSpecialsFromJson(jsonValue)) {
        return false;
    }
    JREAD_DOUBLE_OPT(jsonValue, "pack_discount", PackDiscount);
    JREAD_DOUBLE_OPT(jsonValue, "fees_incorrect_finish", FeesForIncorrectFinish);

    if (!FeesDistance.DeserializeFromJson(jsonValue, "fees_distance") || !FeesTime.DeserializeFromJson(jsonValue, "fees_time")) {
        return false;
    }
    JREAD_DOUBLE_OPT(jsonValue, "finish_precision", FinishPrecision);
    JREAD_DURATION_OPT(jsonValue, "walking_duration", WalkingDuration);
    JREAD_BOOL_OPT(jsonValue, "parking_in_pack", ParkingInPack);
    JREAD_FROM_STRING_OPT(jsonValue, "walking_transport", WalkingTransport);
    JREAD_BOOL_OPT(jsonValue, "walking_area_convex", WalkingAreaConvex);
    JREAD_DOUBLE_OPT(jsonValue, "max_path_distance", MaxPathDistance);
    JREAD_STRING_OPT(jsonValue, "too_close_destination_error", TooCloseDestinationError);
    JREAD_BOOL_OPT(jsonValue, "use_km_for_pack_calculation", UseKmForPackCalculation);
    JREAD_BOOL_OPT(jsonValue, "use_rounded_price", UseRoundedPrice);

    if (jsonValue.Has("focus")) {
        TGeoCoord c;
        if (jsonValue["focus"].IsString() && c.DeserializeFromString(jsonValue["focus"].GetString())) {
            Focus = c;
        }
    }

    if (jsonValue.Has("area_finish_tags")) {
        if (!AreaTagsFinish.DeserializeFromJson(jsonValue["area_finish_tags"])) {
            return false;
        }
    }

    JREAD_FROM_STRING_OPT(jsonValue, "fake_detailed_description", FakeDetailedDescription);
    JREAD_DOUBLE_OPT(jsonValue, "kf_parking_price", KfParkingPrice);
    if (KfParkingPrice < 0) {
        return false;
    }

    return
        NJson::ParseField(jsonValue["finish_area_device_count_limit"], FinishAreaDeviceCountLimit);
}

NJson::TJsonValue TFixPointOfferConstructor::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    JWRITE(result, "pack_discount", PackDiscount);
    JWRITE(result, "fees_incorrect_finish", FeesForIncorrectFinish);

    FeesDistance.SerializeToJson(result, "fees_distance");
    FeesTime.SerializeToJson(result, "fees_time");

    JWRITE(result, "threshold_distance", 500);
    JWRITE(result, "threshold_haversine", 0);

    JWRITE(result, "finish_precision", FinishPrecision);
    JWRITE_DURATION(result, "walking_duration", WalkingDuration);
    JWRITE(result, "parking_in_pack", ParkingInPack);
    JWRITE_ENUM(result, "walking_transport", WalkingTransport);
    JWRITE(result, "walking_area_convex", WalkingAreaConvex);
    JWRITE(result, "max_path_distance", MaxPathDistance);
    JWRITE(result, "too_close_destination_error", TooCloseDestinationError);
    JWRITE(result, "use_km_for_pack_calculation", UseKmForPackCalculation);
    JWRITE(result, "use_rounded_price", UseRoundedPrice);

    if (!AreaTagsFinish.Empty()) {
        result.InsertValue("area_finish_tags", AreaTagsFinish.SerializeToJson());
    }
    if (FinishAreaDeviceCountLimit) {
        result.InsertValue("finish_area_device_count_limit", FinishAreaDeviceCountLimit);
    }

    if (Focus) {
        JWRITE(result, "focus", Focus->ToString());
    }

    JWRITE(result, "fake_detailed_description", FakeDetailedDescription);
    JWRITE(result, "kf_parking_price", KfParkingPrice);

    return result;
}

EDropAbility TFixPointOfferConstructor::CheckAreaTags(const TArea& area, const TStandartOffer& offer, const NDrive::IServer* server) const {
    if (!AreaTagsFinish.Empty()) {
        if (!AreaTagsFinish.Empty() && AreaTagsFinish.FilterWeak(area.GetTags())) {
            return EDropAbility::Allow;
        } else {
            return EDropAbility::NotAllow;
        }
    } else {
        const TCarLocationContext clc = TCarLocationContext::BuildByArea(offer.GetObjectId(), area, *server);
        const TCarLocationFeatures features = clc.GetCarAreaFeatures(false, &offer, server);
        switch (features.GetAllowDropDef(EDropAbility::NotAllow)) {
            case EDropAbility::Allow:
                return EDropAbility::Allow;
            case EDropAbility::NotAllow:
                return EDropAbility::NotAllow;
            case EDropAbility::Deny:
            case EDropAbility::DenyIncorrectData:
                return EDropAbility::Deny;
        };
    }
}

NDrive::TScheme TFixPointOfferConstructor::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    {
        auto gTab = result.StartTabGuard("client_appearance");
        result.Add<TFSText>("fake_detailed_description", "Общее описание тарифа (без финальной точки)");
        result.Add<TFSString>("too_close_destination_error", "Код ошибки для слишком короткой поездки");
    }
    {
        auto gTab = result.StartTabGuard("prices");
        result.Add<TFSNumeric>("kf_parking_price", "Коэффициент модификации цены паркинга").SetMin(0).SetMax(100).SetPrecision(2).SetDefault(1);
        result.Add<TFSNumeric>("pack_discount", "скидка на пакет (доля от поминутной цены)").SetMin(0).SetMax(1).SetPrecision(2).SetDefault(0.1);
        result.Add<TFSBoolean>("parking_in_pack", "Парковка включена в пакет", 100000).SetDefault(false);
        result.Add<TFSBoolean>("use_km_for_pack_calculation", "Расчет стоимости с учетом стоимости километра (p_park * t + p_km * l) вместо (p_r * t)");
        result.Add<TFSBoolean>("use_rounded_price", "Округлять стоимость до целых рублей");
    }
    {
        auto gTab = result.StartTabGuard("fines");
        result.Add<TFSNumeric>("fees_incorrect_finish", "штраф (доля) для некорректного завершения аренды").SetMin(0).SetMax(1).SetPrecision(2).SetDefault(0.1);
        result.Add<TFSStructure>("area_finish_tags", "Фильтр допустимых областей финиша", 100000).SetStructure(AreaTagsFinish.GetScheme(*server));
        result.Add<TFSNumeric>("finish_area_device_count_limit", "Максимальное количество машин в области финиша").SetMin(0).SetDefault(0);
    }
    {
        auto gTab = result.StartTabGuard("router");
        result.Add<TFSNumeric>("fees_distance_fraction", "Коррекция по расстоянию (доля от пути)").SetMin(0).SetMax(10).SetPrecision(2).SetDefault(0.05);
        result.Add<TFSNumeric>("fees_distance_min", "Минимальная коррекция расстояния, м").SetMin(0).SetMax(100000).SetDefault(0);
        result.Add<TFSNumeric>("fees_distance_max", "Максимальная коррекция расстояния, м").SetMin(0).SetMax(100000).SetDefault(0);

        result.Add<TFSNumeric>("fees_time_fraction", "Коррекция времени (доля от пути)").SetMin(0).SetMax(10).SetPrecision(2).SetDefault(0.05);
        result.Add<TFSDuration>("fees_time_min", "Минимальная коррекция по времени").SetDefault(TDuration::Minutes(0));
        result.Add<TFSDuration>("fees_time_max", "Максимальная коррекция по времени").SetDefault(TDuration::Minutes(0));

        result.Add<TFSNumeric>("threshold_distance", "Минимальное пороговое расстояние из A в B, м").SetDefault(500).SetDeprecated(true);
        result.Add<TFSNumeric>("threshold_haversine", "Минимальное пороговое расстояние из A в B по прямой, м").SetDefault(0).SetDeprecated(true);

        result.Add<TFSNumeric>("finish_precision", "Точность указания точки назначения (м)", 100000).SetMin(0).SetMax(10000).SetDefault(500).SetDeprecated(true);
        result.Add<TFSDuration>("walking_duration", "Допустимое время от реальной остановки машины").SetDefault(TDuration::Minutes(10)).SetDeprecated(true);
        result.Add<TFSVariants>("walking_transport", "Как добираться до машины", 100000).InitVariants<NGeoEdge::EPermitTypes>().SetDefault(::ToString(NGeoEdge::ptPedestrian)).SetDeprecated(true);
        result.Add<TFSBoolean>("walking_area_convex", "Выпуклая область", 100000).SetDefault(true).SetDeprecated(true);

        result.Add<TFSNumeric>("max_path_distance", "Максимально возможное расстояние из А в B по прямой (м)", 100000).SetMin(0).SetMax(1000000).SetDefault(100000);
        result.Add<TFSString>("focus", "Точка фокусировки в формате <lon lat>. Например: 37.5 55.6", 100000).SetRequired(false);
    }

    return result;
}

TPredestinedFixPointOfferConstructor::TFactory::TRegistrator<TPredestinedFixPointOfferConstructor> TPredestinedFixPointOfferConstructor::Registrator(TPredestinedFixPointOfferConstructor::GetTypeName());

bool TPredestinedFixPointOfferConstructor::ConstructPackOffer(
    const TStandartOfferReport& standardOfferReport,
    const TOffersBuildingContext& context,
    const NDrive::IServer* server,
    TVector<IOfferReport::TPtr>& result,
    NDrive::TInfoEntitySession& session
) const {
    auto startCoord = context.GetStartPosition();
    if (!context.HasUserHistoryContext()) {
        session.AddErrorMessage("TFixPointOfferConstructor::ConstructPackOffer", "no user context");
        return false;
    }
    if (!startCoord) {
        session.AddErrorMessage("TFixPointOfferConstructor::ConstructPackOffer", "no car location");
        return false;
    }
    auto destinationDescription = DestinationDescription;
    destinationDescription.Prefetch(context.GetUserHistoryContextUnsafe());
    auto destination = TOffersBuildingContext::TDestination(destinationDescription, context);
    auto expectedOffer = ConstructFixPointOffer(standardOfferReport, *startCoord, destination, context, server, session);
    if (!expectedOffer) {
        context.AddError(GetName(), expectedOffer.GetError());
        return false;
    }
    auto offer = (expectedOffer && *expectedOffer) ? expectedOffer->Get()->GetOffer() : nullptr;
    if (!offer) {
        return false;
    }
    result.push_back(*expectedOffer);
    return true;
}

NDrive::TScheme TPredestinedFixPointOfferConstructor::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    NDrive::TScheme& destination = result.Add<TFSStructure>("destination").SetStructure<NDrive::TScheme>();
    destination.Add<TFSString>("name");
    destination.Add<TFSJson>("coordinate", "coordinate in [LON, LAT] format");
    destination.Add<TFSText>("description", "destination description");
    return result;
}

bool TPredestinedFixPointOfferConstructor::DeserializeSpecialsFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DeserializeSpecialsFromJson(value)) {
        return false;
    }
    return
        NJson::ParseField(value["destination"], DestinationDescription, true);
}

NJson::TJsonValue TPredestinedFixPointOfferConstructor::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    result["destination"] = NJson::ToJson(DestinationDescription);
    return result;
}
