#pragma once

#include "abstract.h"
#include "pack.h"

#include <drive/backend/offers/context.h>

#include <drive/backend/areas/areas.h>
#include <drive/backend/data/area_tags.h>

#include <rtline/util/json_processing.h>
#include <rtline/util/types/expected.h>

class TFixPointOfferState: public TPackOfferState {
private:
    using TBase = TPackOfferState;

private:
    R_OPTIONAL(bool, DropAvailable);

public:
    TFixPointOfferState() = default;
    TFixPointOfferState(const TPackOfferState& state)
        : TBase(state)
    {
    }

    virtual NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer& server) const override {
        NJson::TJsonValue result = TBase::GetReport(locale, server);
        NJson::InsertField(result, "is_correct_finish_position", DropAvailable);
        return result;
    }
};

class TFixPointOffer : public TPackOffer {
private:
    using TBase = TPackOffer;

public:
    struct TInheritedProperties {
        TGeoCoord Destination;
        TInstant Timestamp;
        ui32 PackPrice = 0;
    };

private:
    static TFactory::TRegistrator<TFixPointOffer> Registrator;

private:
    R_FIELD(TDuration, RouteDuration, TDuration::Days(1));
    R_FIELD(TPriceModelInfos, RouteDurationModelInfos);
    R_FIELD(double, RouteLength, 0);
    R_FIELD(TDuration, AdditionalRouteDuration, TDuration::Zero());
    R_FIELD(double, AdditionalRouteLength, 0);
    R_OPTIONAL(double, PackDiscount);
    R_OPTIONAL(double, AreasDiscount);
    R_FIELD(double, FeesForIncorrectFinish, 0);
    R_FIELD(double, TaxiPrice, 0);
    R_OPTIONAL(TInheritedProperties, InheritedProperties);
    R_OPTIONAL(ui32, FinishMinPrice);
    R_OPTIONAL(ui32, MinPriceLow);
    R_OPTIONAL(ui32, FinishAdditionalPrice);
    R_FIELD(TGeoCoord, Finish);
    R_FIELD(TDuration, WalkingDuration, TDuration::Minutes(5));
    R_FIELD(TVector<TGeoCoord>, FinishArea);
    R_FIELD(TVector<TGeoCoord>, FinishAreaPublic);
    R_FIELD(TSet<TString>, FinishAreaIds);
    R_FIELD(bool, ParkingInPack, false);
    R_FIELD(TString, DestinationContext);
    R_FIELD(TString, DestinationDescription);
    R_FIELD(TString, DestinationName);
    R_FIELD(bool, UseKmForPackCalculation, false);
    R_FIELD(bool, UseMapsRouter, false);
    R_FIELD(bool, UseAvoidTolls, false);
    R_FIELD(bool, IsFake, false);
    R_FIELD(bool, UseRoundedPrice, false);
    // AcceptancePrice represents acceptance price for offer.
    //
    // By default, acceptance price is equal to standart offer acceptance price.
    R_OPTIONAL(ui32, AcceptancePrice);
    R_FIELD(TPriceModelInfos, AcceptancePriceModelInfos);

protected:
    virtual TString DoBuildCalculationDescription(ELocalization locale, const TFullCompiledRiding& fcr, const NDrive::IServer& server) const override;
    virtual TRecalcByMinutesInfo CheckRecalcByMinutes(const TRidingInfo& rInfo, const TInstant startPack) const override;

    virtual bool GetIsParkingIncludeInPack() const override {
        return ParkingInPack;
    }

    virtual NJson::TJsonValue DoBuildJsonReport(const TReportOptions& options, const ICommonOfferBuilderAction* constructor, const NDrive::IServer& server) const override;
    virtual EDriveSessionResult DoCheckSession(const TUserPermissions& permissions, NDrive::TEntitySession& session, const NDrive::IServer* server, bool onPerform) const override;
    virtual TString DoFormDescriptionElement(const TString& value, ELocalization locale, const ILocalization* localization) const override;

    bool DeserializeFixPointFromProto(const NDrive::NProto::TFixPointOffer& info);
    void SerializeFixPointToProto(NDrive::NProto::TFixPointOffer& info) const;

public:
    using TBase::TBase;

    static TString GetTypeNameStatic() {
        return "fix_point";
    };

    virtual TString GetTypeName() const override {
        return GetTypeNameStatic();
    }

    virtual ui32 GetPublicDiscountedPackPrice(TMaybe<ui32> overridenPackPrice = {}, const ui32 precision = 1) const override;
    virtual ui32 GetPublicOriginalPackPrice(TMaybe<ui32> overridenPackPrice = {}, const ui32 precision = 1) const override;

    virtual bool CheckNeedFeesOnFinish(const NDrive::IServer* server) const override;

    ui32 CalcPackPrice(const NDrive::IServer* server);
    TMaybe<ui32> GetEffectiveCashback(const NDrive::IServer& server) const;

    virtual TVector<TString> GetDefaultShortDescription(ELocalization locale, NDriveSession::TReportTraits traits, const ILocalization& localization) const override {
        Y_UNUSED(locale);
        Y_UNUSED(traits);
        Y_UNUSED(localization);
        return TVector<TString>();
    }

    virtual TMaybe<ui32> GetReportedAcceptancePrice() const override {
        return {};
    }

    virtual bool ShouldReportUpsale() const override {
        return false;
    }

    bool DeserializeFromProto(const NDrive::NProto::TOffer& info) override {
        if (!TBase::DeserializeFromProto(info)) {
            return false;
        }
        return DeserializeFixPointFromProto(info.GetFixPointOffer());
    }

    NDrive::NProto::TOffer SerializeToProto() const override {
        auto info = TBase::SerializeToProto();
        SerializeFixPointToProto(*info.MutableFixPointOffer());
        return info;
    }

    // ApplyRouteDurationModel applies model for route duration and appends record to model info.
    void ApplyRouteDurationModel(const NDrive::IOfferModel& model);

    // ApplyAcceptancePriceModel applies model for acceptance price and appends record to model info.
    void ApplyAcceptancePriceModel(const NDrive::IOfferModel& model);

    virtual TOfferStatePtr DoCalculate(const TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events, const TInstant& until, const TRidingInfo& ridingInfo, TOfferPricing& result) const override;

    virtual bool NeedStore(const NDrive::IServer& server) const override {
        return IsFake ?
            server.GetSettings().GetValueDef<bool>("offers.store_fake", false) :
            server.GetSettings().GetValueDef<bool>("offers.store", true);
    }
};

class TFixPointOfferReport: public TPackOfferReport {
public:
    class THint: public TUserOfferContext::TDestinationDescription {
    private:
        using TBase = TUserOfferContext::TDestinationDescription;

    private:
        TMaybe<TGeoRect> Bound;
        TString DiscountInfo;
        TAtomicSharedPtr<TFixPointOfferReport> OfferReport;

    public:
        THint(const TAreaPotentialTag::TInfo& info);
        THint(const TUserOfferContext::TDestinationDescription& destination, TAtomicSharedPtr<TFixPointOfferReport> offer);

        TAtomicSharedPtr<TFixPointOfferReport> GetOffer() const {
            return OfferReport;
        }

        NJson::TJsonValue SerializeToJson() const;
    };

    class TDiscountArea {
    private:
        TString Name;
        TVector<TGeoCoord> Border;
        TGeoCoord Center;
        NJson::TJsonValue StyleInfo;
        TGeoCoord HintPosition;
        TString DiscountInfo;

    public:
        TDiscountArea(const TAreaPotentialTag::TInfo& info) {
            Name = info.GetName();
            Border = info.GetBorder().GetCoords();
            Center = info.GetCenter();
            DiscountInfo = info.GetDiscountInfo();
            StyleInfo = info.GetStyleInfo();
            HintPosition = info.GetHintPosition();
        }

        NJson::TJsonValue SerializeToJson() const {
            NJson::TJsonValue discountArea;
            discountArea["name"] = Name;
            discountArea["border"] = NJson::ToJson(Border);
            discountArea["center"] = NJson::ToJson(Center);
            discountArea["discount_info"] = DiscountInfo;
            discountArea["hint_position"] = NJson::ToJson(HintPosition);
            if (StyleInfo.IsDefined()) {
                discountArea["style"] = StyleInfo;
            }
            return discountArea;
        }
    };

private:
    using TBase = TPackOfferReport;

private:
    R_OPTIONAL(TGeoCoord, Focus);
    R_FIELD(TVector<THint>, Hints);
    R_FIELD(TVector<TDiscountArea>, DiscountAreas);
    R_FIELD(bool, IsHint, false);
    R_FIELD(TString, DetailedDescription);
    R_OPTIONAL(TUserOfferContext::TSimpleDestination, Destination);

protected:
    virtual NJson::TJsonValue BuildJsonReport(ELocalization locale, NDriveSession::TReportTraits traits, const NDrive::IServer& server, const TUserPermissions& permissions) const override;
    virtual void DoRecalcPrices(const NDrive::IServer* server) override;

public:
    using TBase::TBase;

    using TBaseOffer = TFixPointOffer;

    TFixPointOfferReport(const TPackOfferReport& report)
        : TBase(report)
    {
    }

    bool ApplyMinPrice(const double price) override;

    virtual void ApplyInternalCorrection(const TString& areaId, const TOffersBuildingContext& /*context*/, const NDrive::IServer* server) override;
    virtual void ApplyFlowCorrection(const TString& areaId, const TOffersBuildingContext& context, const NDrive::IServer* server) override;

    virtual TString PredictDestination(const TOffersBuildingContext& context, const NDrive::IServer* server, const TCommonDestinationDetector& baseDetector) const override;

    // RecalculateFeatures recalculates features for pack offer.
    virtual void RecalculateFeatures() override;
};

class TFixPointOfferConstructor: public TBasePackOfferConstructor {
private:
    using TBase = TBasePackOfferConstructor;
private:
    R_FIELD(double, PackDiscount, 0);
    R_FIELD(double, FeesForIncorrectFinish, 0.1);

    R_FIELD(TLimitedFraction<double>, FeesDistance);
    R_FIELD(TLimitedFraction<TDuration>, FeesTime);

    R_FIELD(double, FinishPrecision, 500);
    R_FIELD(TDuration, WalkingDuration, TDuration::Minutes(10));
    R_FIELD(bool, ParkingInPack, false);
    R_FIELD(TString, FakeDetailedDescription);
    R_FIELD(double, KfParkingPrice, 1);

    R_FIELD(NGeoEdge::EPermitTypes, WalkingTransport, NGeoEdge::ptPedestrian);
    R_FIELD(bool, WalkingAreaConvex, true);
    R_FIELD(double, MaxPathDistance, 100000);
    R_OPTIONAL(TGeoCoord, Focus);
    R_FIELD(TString, TooCloseDestinationError, "too_close_destination");

    R_FIELD(TAreaTagsFilter, AreaTagsFinish);
    R_FIELD(ui32, FinishAreaDeviceCountLimit, 0);
    R_FIELD(bool, UseDynamicPushCalculation, false);
    R_FIELD(bool, UseKmForPackCalculation, false);
    R_FIELD(bool, UseRoundedPrice, false);

private:
    TLimitedFraction<TDuration> GetAdditionalDurationPolicy(const TOffersBuildingContext& context) const;
    TLimitedFraction<double> GetAdditionalDistancePolicy(const TOffersBuildingContext& context) const;

private:
    static TFactory::TRegistrator<TFixPointOfferConstructor> Registrator;

protected:
    bool AddStopAreaInfo(TFixPointOffer& offer, const NDrive::IServer* server, const TOffersBuildingContext::TDestination& destination, NDrive::TInfoEntitySession& session) const;

    virtual bool ConstructPackOffer(const TStandartOfferReport& stOffer, const TOffersBuildingContext& context, const NDrive::IServer* server, TVector<IOfferReport::TPtr>& result, NDrive::TInfoEntitySession& session) const override;
    TExpected<TAtomicSharedPtr<TFixPointOfferReport>, TString> ConstructFixPointOffer(
        const TStandartOfferReport& stOffer,
        const TGeoCoord& source,
        const TOffersBuildingContext::TDestination& destination,
        const TOffersBuildingContext& context,
        const NDrive::IServer* server,
        NDrive::TInfoEntitySession& session
    ) const;

    virtual THolder<TFixPointOfferReport> Extend(const TStandartOfferReport& standardOfferReport) const;
    virtual double GetDistanceThreshold(const NDrive::IServer& server) const;
    virtual double GetHaversineThreshold(const NDrive::IServer& server) const;
    virtual bool IsInternalPointDisabled() const;

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;
    EDropAbility CheckAreaTags(const TArea& area, const TStandartOffer& offer, const NDrive::IServer* server) const;

public:
    using TBase::TBase;

    static TString GetTypeStatic() {
        return "fixpoint_offer_builder";
    }

    static TString GetTypeName() {
        return GetTypeStatic();
    }

    virtual TString GetType() const override {
        return GetTypeStatic();
    }

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;
};

class TPredestinedFixPointOfferConstructor: public TFixPointOfferConstructor {
private:
    using TBase = TFixPointOfferConstructor;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "predestined_fixpoint_offer_builder";
    }

    void SetDestination(const TOffersBuildingContext::TDestinationDescription& value) {
        DestinationDescription = value;
    }

protected:
    virtual bool ConstructPackOffer(
        const TStandartOfferReport& standardOfferReport,
        const TOffersBuildingContext& context,
        const NDrive::IServer* server,
        TVector<IOfferReport::TPtr>& result,
        NDrive::TInfoEntitySession& session
    ) const override;

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& value) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;

private:
    TOffersBuildingContext::TDestinationDescription DestinationDescription;

private:
    static TFactory::TRegistrator<TPredestinedFixPointOfferConstructor> Registrator;
};
