#include "flexipack.h"

#include <rtline/library/json/field.h>

namespace {
    TFlexiblePackOfferBuilder::TMileageOptions DefaultMileageOptions = {
        { TDuration::Hours(2).Seconds(),    40,     { 0, 20, 40, 50, 80, 100, 150 },        TDuration::Minutes(15).Seconds() },
        { TDuration::Hours(3).Seconds(),    50,     { 0, 30, 50, 70, 100, 150, 200, 250 },  TDuration::Hours(1).Seconds() },
        { TDuration::Hours(4).Seconds(),    50,     { 0, 30, 50, 70, 100, 150, 200, 250 } },
        { TDuration::Hours(5).Seconds(),    70,     { 0, 50, 70, 100, 150, 200, 300, 400 } },
        { TDuration::Hours(6).Seconds(),    70,     { 0, 50, 70, 100, 150, 200, 300, 400 } },
        { TDuration::Hours(7).Seconds(),    100,    { 0, 70, 100, 150, 200, 250, 300, 400, 500 } },
        { TDuration::Hours(8).Seconds(),    100,    { 0, 70, 100, 150, 200, 250, 300, 400, 500 } },
        { TDuration::Hours(9).Seconds(),    100,    { 0, 70, 100, 150, 200, 250, 300, 400, 500 } },
        { TDuration::Hours(10).Seconds(),   100,    { 0, 70, 100, 150, 200, 250, 300, 400, 500 } },

        { TDuration::Days(1).Seconds(),     150,    { 0, 70, 100, 150, 200, 300, 500, 700, 1000 },  TDuration::Hours(3).Seconds() },
        { TDuration::Days(2).Seconds(),     300,    { 0, 100, 200, 300, 400, 500, 600, 800, 1000, 1300, 1500 } },
        { TDuration::Days(3).Seconds(),     300,    { 0, 100, 200, 300, 400, 500, 600, 800, 1000, 1300, 1500 } },
        { TDuration::Days(4).Seconds(),     500,    { 0, 150, 300, 500, 600, 700, 800, 1000, 1300, 1500, 2000, 2500 } },
        { TDuration::Days(5).Seconds(),     500,    { 0, 150, 300, 500, 600, 700, 800, 1000, 1300, 1500, 2000, 2500 } },
        { TDuration::Days(6).Seconds(),     600,    { 0, 200, 400, 600, 700, 800, 1000, 1300, 1500, 2000, 2500, 3000, 3500 } },
        { TDuration::Days(7).Seconds(),     600,    { 0, 200, 400, 600, 700, 800, 1000, 1300, 1500, 2000, 2500, 3000, 3500 } },
    };

    TSet<ui64> GetDurationValues(const TFlexiblePackOfferBuilder::TMileageOptions& mileageOptions) {
        TSet<ui64> result;
        for (auto&& option : mileageOptions) {
            result.insert(option.Duration);
        }
        return result;
    }

    std::pair<TMaybe<ui64>, TSet<ui64>> GetMileageValues(TDuration duration, const TFlexiblePackOfferBuilder::TMileageOptions& mileageOptions) {
        for (auto&& option : mileageOptions) {
            if (option.Duration == duration.Seconds()) {
                return { option.DefaultMileage, option.Mileages };
            }
        }
        return {};
    }

    ui64 GetRecalculateDuration(TDuration duration, const TFlexiblePackOfferBuilder::TMileageOptions& mileageOptions) {
        ui64 result = 0;
        for (auto&& option : mileageOptions) {
            if (option.Duration <= duration.Seconds()) {
                result = std::max(result, option.RecalculateDuration);
            }
        }
        return result;
    }
}

DECLARE_FIELDS_JSON_SERIALIZER(TFlexiblePackOfferBuilder::TMileageOption);

NJson::TJsonValue TFlexiblePackOffer::DoBuildJsonReport(const TReportOptions& options, const ICommonOfferBuilderAction* constructor, const NDrive::IServer& server) const {
    NJson::TJsonValue result = TBase::DoBuildJsonReport(options, constructor, server);
    if (!(options.Traits * NDriveSession::ReportOfferDetails)) {
        return result;
    }
    NJson::TJsonValue& variables = result.InsertValue("variables", NJson::JSON_ARRAY);
    auto locale = options.Locale;
    auto localization = server.GetLocalization();
    auto builder = dynamic_cast<const TFlexiblePackOfferBuilder*>(constructor);
    if (builder) {
        const auto& detailedDescription = ((GetMileageLimit() == 0) && builder->HasZeroMileageDetailedDescription())
            ? builder->GetZeroMileageDetailedDescriptionRef()
            : builder->GetDetailedDescription();
        result.InsertValue("detailed_description", FormDescriptionElement(detailedDescription, locale, server.GetLocalization()));
    }
    if (builder) {
        auto duration = builder->CreateDurationTemplate();
        if (duration.HasValues() && localization) {
            auto& namedValues = duration.OptionalNamedValues().ConstructInPlace();
            for (auto&& value : duration.GetValuesRef()) {
                auto name = localization->DaysFormat(TDuration::Seconds(value), locale);
                TOfferVariable<ui64>::TNamedValue namedValue;
                namedValue.Value = value;
                if (name.Contains(' ')) {
                    TStringBuf n = name;
                    namedValue.Name = TString{n.Before(' ')};
                    namedValue.Unit = TString{n.After(' ')};
                } else {
                    namedValue.Name = std::move(name);
                }
                namedValues.push_back(std::move(namedValue));
            }
        }
        duration.SetValue(GetDuration().Seconds());
        auto subtitle = localization ? localization->GetLocalString(locale, duration.GetSubtitle()) : duration.GetSubtitle();
        duration.SetSubtitle(FormDescriptionElement(subtitle, locale, localization));
        variables.AppendValue(duration.GetReport(locale, server));
    }
    if (builder) {
        auto mileage = builder->CreateMileageTemplate(GetDuration(), GetMileageLimit());
        mileage.SetValue(GetMileageLimit());
        auto subtitle = localization ? localization->GetLocalString(locale, mileage.GetSubtitle()) : mileage.GetSubtitle();
        mileage.SetSubtitle(FormDescriptionElement(subtitle, locale, localization));
        variables.AppendValue(mileage.GetReport(locale, server));
    }
    return result;
}

bool TFlexiblePackOfferBuilder::ConstructPackOffer(
    const TStandartOfferReport& standartOfferReport,
    const TOffersBuildingContext& context,
    const NDrive::IServer* server,
    TVector<IOfferReport::TPtr>& offers,
    NDrive::TInfoEntitySession& session
) const {
    auto poc = standartOfferReport.GetPriceOfferConstructor();
    if (!poc) {
        session.AddErrorMessage("FlexiblePackOfferBuilder::ConstructPackOffer", "PriceOfferConstructor is not defined");
        return false;
    }

    auto offerReport = IOfferReport::ExtendCustom<TPackOfferReport, TFlexiblePackOffer>(standartOfferReport);
    auto offer = offerReport ? offerReport->GetOfferAs<TFlexiblePackOffer>() : nullptr;
    if (!offer) {
        session.AddErrorMessage("FlexiblePackOfferBuilder::ConstructPackOffer", "cannot extend offer to FlexiblePackOffer");
        return false;
    }

    if (context.HasExtraDuration()) {
        offer->SetExtraDuration(context.GetExtraDurationUnsafe());
    }
    if (context.HasExtraDistance()) {
        offer->SetExtraMileageLimit(context.GetExtraDistanceUnsafe());
    }

    const auto& mileageOptions = MileageOptions ? *MileageOptions : DefaultMileageOptions;
    auto durationTemplate = CreateDurationTemplate();

    auto variables = context.GetVariables();
    auto duration = NJson::FromJson<TMaybe<TSeconds>>(variables["duration"]);
    if (duration) {
        offer->SetDuration(*duration);
    } else if (durationTemplate.HasDefaultValue()) {
        offer->SetDuration(TDuration::Seconds(durationTemplate.GetDefaultValueRef()));
    } else {
        session.AddErrorMessage("FlexiblePackOfferBuilder::ConstructPackOffer", "cannot determine duration");
        return false;
    }
    {
        auto offerDuration = offer->GetDuration();
        if (offerDuration < TDuration::Days(1)) {
            auto rounding = TDuration::Minutes(5).Seconds();
            auto pushThreshold = static_cast<i64>(std::ceil(offerDuration.SecondsFloat() * 0.1 / rounding)) * rounding;
            offer->SetRemainingDurationPushThreshold(TDuration::Seconds(pushThreshold));
        } else {
            auto rounding = TDuration::Hours(1).Seconds();
            auto pushThreshold = static_cast<i64>(std::ceil(offerDuration.SecondsFloat() * 0.05 / rounding)) * rounding;
            offer->SetRemainingDurationPushThreshold(TDuration::Seconds(pushThreshold));
        }
        auto recalculateDuration = TDuration::Seconds(GetRecalculateDuration(offer->GetDuration(), mileageOptions));
        offer->SetReturningDuration(recalculateDuration);
    }

    auto&& [defaultMileage, mileageValues] = GetMileageValues(offer->GetDuration(), mileageOptions);
    if (!defaultMileage) {
        session.AddErrorMessage("FlexiblePackOfferBuilder::ConstructPackOffer", "cannot determine default mileage");
        return false;
    }
    if (!mileageValues) {
        session.AddErrorMessage("FlexiblePackOfferBuilder::ConstructPackOffer", "cannot determine mileage values");
        return false;
    }
    auto mileageTemplate = CreateMileageTemplate(offer->GetDuration());

    auto mileage = NJson::FromJson<TMaybe<ui32>>(variables["mileage"]);
    if (mileage && !mileageValues.contains(*mileage)) {
        mileage = defaultMileage;
    }
    if (mileage) {
        offer->SetMileageLimit(*mileage);
    } else if (mileageTemplate.HasDefaultValue()) {
        offer->SetMileageLimit(mileageTemplate.GetDefaultValueRef());
    } else {
        session.AddErrorMessage("FlexiblePackOfferBuilder::ConstructPackOffer", "cannot determine mileage");
        return false;
    }
    {
        auto offerMileage = offer->GetMileageLimit();
        if (offerMileage < 100) {
            auto rounding = 5;
            auto pushThreshold = static_cast<ui32>(std::ceil(offerMileage * 0.1 / rounding)) * rounding;
            offer->SetRemainingDistancePushThreshold(pushThreshold);
        } else {
            auto rounding = 10;
            auto pushThreshold = static_cast<ui32>(std::ceil(offerMileage * 0.1 / rounding)) * rounding;
            offer->SetRemainingDistancePushThreshold(pushThreshold);
        }
    }

    auto packPrice = poc->CalcPackPrice(offer->GetDuration(), offer->GetMileageLimit());
    if (!packPrice) {
        session.AddErrorMessage("FlexiblePackOfferBuilder::ConstructPackOffer", "cannot CalcPackPrice");
        return false;
    }
    offer->SetPackPrice(*packPrice);
    offer->SetUseDeposit(GetUseDeposit());
    offer->SetDepositAmount(offer->GetPublicDiscountedPackPrice());

    auto overrunPrice = poc->CalcPackOverrunPrice(offer->GetDuration(), offer->GetMileageLimit());
    if (!overrunPrice) {
        overrunPrice = poc->GetMarket().CalcKmPrice();
    }
    offer->SetOverrunKm(overrunPrice);

    offerReport->RecalcPrices(server);
    offers.push_back(std::move(offerReport));
    return true;
}

NDrive::TScheme TFlexiblePackOfferBuilder::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSDuration>("default_duration");
    result.Add<TFSJson>("mileage_options");
    result.Add<TFSString>("zero_mileage_detailed_description", "Detailed description when no mileage included");
    return result;
}

bool TFlexiblePackOfferBuilder::DeserializeSpecialsFromJson(const NJson::TJsonValue& value) {
    return TBase::DeserializeSpecialsFromJson(value) &&
        NJson::TryFieldsFromJson(value, GetFields());
}

NJson::TJsonValue TFlexiblePackOfferBuilder::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    NJson::FieldsToJson(result, GetFields());
    return result;
}

TOfferVariable<ui64> TFlexiblePackOfferBuilder::CreateDefaultDuration() {
    TOfferVariable<ui64> result;
    result.SetId("duration");
    result.SetDefaultValue(TDuration::Hours(3).Seconds());
    result.SetTitle("flexible_pack_offer.duration.title");
    result.SetSubtitle("flexible_pack_offer.duration.subtitle");
    return result;
}

TOfferVariable<ui64> TFlexiblePackOfferBuilder::CreateDefaultMileage() {
    TOfferVariable<ui64> result;
    result.SetId("mileage");
    result.SetTitle("flexible_pack_offer.mileage.title");
    result.SetSubtitle("flexible_pack_offer.mileage.subtitle");
    result.SetUnit("units.short.km");
    return result;
}

TOfferVariable<ui64> TFlexiblePackOfferBuilder::CreateDurationTemplate() const {
    const auto& mileageOptions = MileageOptions ? *MileageOptions : DefaultMileageOptions;
    auto durationTemplate = DurationTemplate;
    auto durationValues = GetDurationValues(mileageOptions);
    durationTemplate.SetValues(std::move(durationValues));
    return durationTemplate;
}

TOfferVariable<ui64> TFlexiblePackOfferBuilder::CreateMileageTemplate(TDuration duration, TMaybe<ui32> mileage) const {
    const auto& mileageOptions = MileageOptions ? *MileageOptions : DefaultMileageOptions;
    auto mileageTemplate = MileageTemplate;
    auto&& [defaultMileage, mileageValues] = GetMileageValues(duration, mileageOptions);
    mileageTemplate.SetDefaultValue(defaultMileage);
    mileageTemplate.SetValues(mileageValues);
    if (mileage == static_cast<ui32>(0)) {
        mileageTemplate.SetSubtitle("flexible_pack_offer.zero_mileage.subtitle");
    }
    return mileageTemplate;
}

TFlexiblePackOffer::TFactory::TRegistrator<TFlexiblePackOffer> TFlexiblePackOffer::Registrator(TFlexiblePackOffer::GetTypeNameStatic());
TFlexiblePackOfferBuilder::TFactory::TRegistrator<TFlexiblePackOfferBuilder> TFlexiblePackOfferBuilder::Registrator(TFlexiblePackOfferBuilder::GetTypeName());
