#pragma once

#include "abstract.h"
#include "correctors.h"

#include <drive/backend/data/area_tags.h>

class TMinimalPriceOfferCorrector: public TSelectiveOfferCorrector {
private:
    using TBase = TSelectiveOfferCorrector;

private:
    static TFactory::TRegistrator<TMinimalPriceOfferCorrector> Registrator;

private:
    R_FIELD(TLinearScheme, PriceMinimal);
    R_FIELD(i32, Variation, 7);

public:
    virtual EOfferCorrectorResult DoApplyForOffer(IOfferReport* offer, const TVector<TDBTag>& /*tags*/, const TOffersBuildingContext& /*context*/, const TString& /*userId*/, const NDrive::IServer* /*server*/, NDrive::TInfoEntitySession& /*session*/) const override {
        const TMaybe<double> value = PriceMinimal.GetValue(Now().Seconds() % (24 * 3600));
        if (!value || !offer) {
            return EOfferCorrectorResult::Unimplemented;
        }
        if (!offer->ApplyMinPrice(*value + RandomNumber<double>() * Variation)) {
            return EOfferCorrectorResult::Unimplemented;
        }
        return EOfferCorrectorResult::Success;
    }

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override {
        NDrive::TScheme result = TBase::DoGetScheme(server);
        result.Add<TFSStructure>("price_minimal_by_time", "Корректор от времени дня").SetStructure(TLinearScheme::GetScheme());
        result.Add<TFSNumeric>("variation", "Случайная вариация (>0)").SetMin(0);
        return result;
    }

    static TString GetTypeName() {
        return "min_price_corrector";
    }

    virtual TString GetType() const override {
        return GetTypeName();
    }

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonInfo) override {
        if (!PriceMinimal.DeserializeFromJson(jsonInfo["price_minimal_by_time"])) {
            return false;
        }
        if (!TJsonProcessor::Read(jsonInfo, "variation", Variation) || Variation < 0) {
            return false;
        }
        return TBase::DeserializeSpecialsFromJson(jsonInfo);
    }

    virtual NJson::TJsonValue SerializeSpecialsToJson() const override {
        NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
        result.InsertValue("price_minimal_by_time", PriceMinimal.SerializeToJson());
        result.InsertValue("variation", Variation);
        return result;
    }
};

class TFlowsOfferCorrector
    : public TSelectiveOfferCorrector
    , public TCommonDestinationDetector
{
private:
    using TBase = TSelectiveOfferCorrector;

private:
    static TFactory::TRegistrator<TFlowsOfferCorrector> Registrator;

private:
    TSet<TString> ConflictedCorrectors;
    THolder<TNamedValuesTable<ui32, TString>> FromToFees;

public:
    using TBase::TBase;

    virtual EOfferCorrectorResult DoApplyForOffer(IOfferReport* offer, const TVector<TDBTag>& tags, const TOffersBuildingContext& context, const TString& userId, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const override;

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override {
        NDrive::TScheme result = TBase::DoGetScheme(server);
        TCommonDestinationDetector::AddScheme(result, server);
        result.Add<TFSString>("conflicted_correctors", "Применение корректоров выключает коррекцию потоков");
        TNamedValuesTable<ui32, TString>::AddToScheme(result, "fees_table", "Таблица контроля");
        return result;
    }

    static TString GetTypeName() {
        return "flows_corrector";
    }

    virtual TString GetType() const override {
        return GetTypeName();
    }

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonInfo) override {
        if (jsonInfo.Has("fees_table")) {
            THolder<TNamedValuesTable<ui32, TString>> fees(new TNamedValuesTable<ui32, TString>());
            if (!fees->DeserializeFromJson(jsonInfo["fees_table"])) {
                return false;
            }
            if (fees->IsInitialized()) {
                FromToFees.Reset(fees.Release());
            }
        }
        StringSplitter(jsonInfo["conflicted_correctors"].GetString()).SplitBySet(", ").SkipEmpty().Collect(&ConflictedCorrectors);
        return TCommonDestinationDetector::DeserializeFromJson(jsonInfo) && TBase::DeserializeSpecialsFromJson(jsonInfo);
    }

    virtual NJson::TJsonValue SerializeSpecialsToJson() const override {
        NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
        result.InsertValue("conflicted_correctors", JoinSeq(", ", ConflictedCorrectors));
        if (FromToFees) {
            result.InsertValue("fees_table", FromToFees->SerializeToJson());
        }
        TCommonDestinationDetector::SerializeToJson(result);
        return result;
    }
};

class TAbstractSDValuePredictor: public TSimpleCategoryScoreThreshold {
private:
    R_OPTIONAL(double, DetectedValue);
    R_FIELD(TString, Id);

public:
    NJson::TJsonValue SerializeToJson() const {
        NJson::TJsonValue result = NJson::JSON_MAP;
        TSimpleCategoryScoreThreshold::SerializeToJson(result);
        TJsonProcessor::Write(result, "value", DetectedValue);
        TJsonProcessor::Write(result, "id", Id);
        return result;
    }

    bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
        if (!TSimpleCategoryScoreThreshold::DeserializeFromJson(jsonInfo)) {
            return false;
        }
        if (!TJsonProcessor::Read(jsonInfo, "value", DetectedValue)) {
            return false;
        }
        if (!TJsonProcessor::Read(jsonInfo, "id", Id)) {
            return false;
        }
        return true;
    }

    static NDrive::TScheme GetScheme(const NDrive::IServer* /*server*/) {
        NDrive::TScheme result;
        TSimpleCategoryScoreThreshold::AddScheme(result);
        result.Add<TFSNumeric>("value", "Значение");
        result.Add<TFSString>("id", "Идентификатор");
        return result;
    }
};

class TOfferSDModelCorrector
    : public TSelectiveOfferCorrector
    , public TCommonCategoryScoreCalcer
{
private:
    using TBase = TSelectiveOfferCorrector;

private:
    static TFactory::TRegistrator<TOfferSDModelCorrector> Registrator;

private:
    TVector<TAbstractSDValuePredictor> Predictors;
    R_FIELD(TLinearScheme, PriceDeltaByValue);

public:
    virtual EOfferCorrectorResult DoApplyForOffer(IOfferReport* offer, const TVector<TDBTag>& tags, const TOffersBuildingContext& context, const TString& userId, const NDrive::IServer* server, NDrive::TInfoEntitySession& session) const override;

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override {
        NDrive::TScheme result = TBase::DoGetScheme(server);
        result.Add<TFSStructure>("price_modification", "Модификация цены в зависимости от значения").SetStructure(TLinearScheme::GetScheme());
        TCommonCategoryScoreCalcer::AddScheme(result, server);
        result.Add<TFSArray>("predictors", "Определение значения").SetElement(TAbstractSDValuePredictor::GetScheme(server));
        return result;
    }

    static TString GetTypeName() {
        return "offer_corrector_abstract_sd_model";
    }

    virtual TString GetType() const override {
        return GetTypeName();
    }

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonInfo) override {
        const NJson::TJsonValue* priceModificationJsonValue = nullptr;
        if (jsonInfo.GetValuePointer("price_modification", &priceModificationJsonValue)) {
            if (!PriceDeltaByValue.DeserializeFromJson(*priceModificationJsonValue)) {
                return false;
            }
        }
        const NJson::TJsonValue::TArray* modelJsonValue = nullptr;
        if (jsonInfo["predictors"].GetArrayPointer(&modelJsonValue)) {
            for (auto&& i : *modelJsonValue) {
                TAbstractSDValuePredictor predictor;
                if (!predictor.DeserializeFromJson(i)) {
                    return false;
                }
                Predictors.emplace_back(std::move(predictor));
            }
        }
        if (!TBase::DeserializeSpecialsFromJson(jsonInfo)) {
            return false;
        }
        if (!TCommonCategoryScoreCalcer::DeserializeFromJson(jsonInfo)) {
            return false;
        }
        return true;
    }

    virtual NJson::TJsonValue SerializeSpecialsToJson() const override {
        NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
        result.InsertValue("price_modification", PriceDeltaByValue.SerializeToJson());
        NJson::TJsonValue& modifiersJson = result.InsertValue("predictors", NJson::JSON_ARRAY);
        for (auto&& i : Predictors) {
            modifiersJson.AppendValue(i.SerializeToJson());
        }
        TCommonCategoryScoreCalcer::SerializeToJson(result);
        return result;
    }
};
