#include "long_term.h"

#include <drive/backend/billing/manager.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/car_attachments/registry/registry.h>
#include <drive/backend/cars/hardware.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/compiled_riding/compiled_riding.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/feedback.h>
#include <drive/backend/roles/manager.h>

#include <rtline/library/json/adapters.h>
#include <rtline/util/algorithm/datetime.h>
#include <rtline/util/types/uuid.h>

#include <library/cpp/http/misc/httpcodes.h>

const ELongTermPaymentPlan DefaultPaymentPlan = ELongTermPaymentPlan::Weekly;
const TString CancellationCostMacro = "_CancellationCost_";
const TString EarlyReturnCostMacro = "_EarlyReturnCost_";
const TString EarlyReturnSegment = "early_return";
const TString ServicingSegment = "servicing";
const TString PaymentScheduleMacro = "_PaymentSchedule_";
const TString PaymentScheduleHeaderMacro = "_PaymentScheduleHeader_";
const TString PaymentScheduleMileageMacro = "_PaymentScheduleMileage_";
const TString Tier1TagName = "long_term_standards_tier_1";
const TString Tier2TagName = "long_term_standards_tier_2";

constexpr i32 UNAVAILABLE_LIST_PRIORITY = 1000;

DECLARE_FIELDS_JSON_SERIALIZER(TLongTermCarPromoCard);
DECLARE_FIELDS_JSON_SERIALIZER(TLongTermCarPromo);
DECLARE_FIELDS_JSON_SERIALIZER(TLongTermBadge);
DECLARE_FIELDS_JSON_SERIALIZER(TLongTermPhoto);
DECLARE_FIELDS_JSON_SERIALIZER(TLongTermVoid);

template <>
NJson::TJsonValue NJson::ToJson(const ELongTermPaymentPlan& object) {
    return NJson::ToJson(NJson::Stringify(object));
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, ELongTermPaymentPlan& result) {
    return NJson::TryFromJson(value, NJson::Stringify(result));
}

template <>
NJson::TJsonValue NJson::ToJson(const ELongTermOfferImageStyle& object) {
    return NJson::ToJson(NJson::Stringify(object));
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, ELongTermOfferImageStyle& result) {
    return NJson::TryFromJson(value, NJson::Stringify(result));
}

NJson::TJsonValue TLongTermVoid::GetReport(ELocalization locale, const NDrive::IServer& server) const {
    NJson::TJsonValue result;
    auto localization = server.GetLocalization();
    result["id"] = Id;
    result["type"] = "constant";

    result["cost"] = NJson::ToJson(Cost);
    result["title"] = localization ? localization->GetLocalString(locale, Title) : Title;
    result["subtitle"] = localization ? localization->GetLocalString(locale, Subtitle) : Subtitle;
    return result;
}

NJson::TJsonValue TLongTermCarPromoCard::GetReport(ELocalization locale, const NDrive::IServer& server) const {
    NJson::TJsonValue result;
    auto localization = server.GetLocalization();
    result["title"] = localization ? localization->GetLocalString(locale, Title) : Title;
    result["message"] = localization ? localization->GetLocalString(locale, Message) : Message;
    result["image_link"] = ImageLink;
    result["details"] = localization ? localization->GetLocalString(locale, Details) : Details;
    return result;
}

NJson::TJsonValue TLongTermCarPromo::GetReport(ELocalization locale, const NDrive::IServer& server) const {
    NJson::TJsonValue result;
    auto localization = server.GetLocalization();
    result["title"] = localization ? localization->GetLocalString(locale, Title) : Title;
    result["more_info"] = localization ? localization->GetLocalString(locale, MoreInfo) : MoreInfo;
    result["cards"] = NJson::JSON_ARRAY;
    for (auto&& card : Cards) {
        result["cards"].AppendValue(card.GetReport(locale, server));
    }
    return result;
}

NJson::TJsonValue TLongTermBadge::GetReport(ELocalization locale, const NDrive::IServer& server) const {
    NJson::TJsonValue result;
    auto localization = server.GetLocalization();
    result["id"] = Id;
    result["color"] = Color;
    result["text"] = localization ? localization->GetLocalString(locale, Text) : Text;
    return result;
}

NJson::TJsonValue TLongTermExtra::GetReport(ELocalization locale, const NDrive::IServer& server) const {
    NJson::TJsonValue result;
    auto localization = server.GetLocalization();
    result["id"] = Id;
    result["title"] = localization ? localization->GetLocalString(locale, Title) : Title;
    result["subtitle"] = localization ? localization->GetLocalString(locale, Subtitle) : Subtitle;
    return result;
}

NJson::TJsonValue TLongTermPhoto::GetReport() const {
    NJson::TJsonValue result;
    result["url"] = Url;
    return result;
}

template <class E>
NJson::TJsonValue TLongTermEnum<E>::GetReport(ELocalization locale, const NDrive::IServer& server) const {
    NJson::TJsonValue result;
    auto localization = server.GetLocalization();
    result["id"] = Id;
    result["cost"] = NJson::ToJson(Cost);
    result["default_value"] = NJson::ToJson(DefaultValue);
    result["value"] = NJson::ToJson(Value);
    result["title"] = localization ? localization->GetLocalString(locale, Title) : Title;

    if (Values) {
        NJson::TJsonValue values = NJson::JSON_ARRAY;
        for (auto&& v : *Values) {
            NJson::TJsonValue value;
            auto id = TStringBuilder() << "long_term_offer." << Id << "." << v;
            value["value"] = ToString(v);
            value["title"] = localization ? localization->GetLocalString(locale, id) : id;
            values.AppendValue(std::move(value));
        }
        result["values"] = std::move(values);
    }

    return result;
}

NJson::TJsonValue TLongTermOfferState::GetReport(ELocalization locale, const NDrive::IServer& server) const {
    NJson::TJsonValue result = TBase::GetReport(locale, server);
    if (NextPayment) {
        NJson::TJsonValue payment;
        payment["timestamp"] = NextPayment->Timestamp.Seconds();
        if (NextPayment->Mileage) {
            payment["mileage"] = *NextPayment->Mileage;
        }
        payment["value"] = NextPayment->Value;
        result.InsertValue("next_payment", std::move(payment));
    }
    NJson::TJsonValue& payments = result.InsertValue("payments", NJson::JSON_ARRAY);
    for (auto&& p : Payments) {
        if (!p.Value) {
            continue;
        }
        NJson::TJsonValue payment;
        payment["timestamp"] = p.Timestamp.Seconds();
        if (p.Mileage) {
            payment["mileage"] = *p.Mileage;
        }
        payment["value"] = p.Value;
        if (p.Completed) {
            payment["completed"] = true;
        }
        payments.AppendValue(std::move(payment));
    }
    NJson::InsertField(result, "withheld_pack_price", HeldSum);
    result["current_pack_price"] = HeldSum; // TODO: remove override for DRIVEBACK-3312
    NJson::InsertField(result, "return_recalculation", ReturnRecalculation);
    NJson::InsertNonNull(result, "return_cost", ReturnCost);
    NJson::InsertNonNull(result, "return_fee", ReturnFee);
    NJson::InsertNonNull(result, "return_pack_price", ReturnPackPrice);
    NJson::InsertNonNull(result, "return_overrun_cost", ReturnOverrunCost);
    NJson::InsertNonNull(result, "chat_type", server.GetSettings().GetValueDef<TString>("long_term_chat_id", ""));
    return result;
}

TString TLongTermOfferState::FormDescriptionElement(const TString& value, const TString& currency, ELocalization locale, const ILocalization& localization) const {
    TString result = value;
    SubstGlobal(result, "_WaitingDuration_", localization.MonthsFormat(GetWaitingDuration(), locale));
    SubstGlobal(result, "_PackRemainingTime_", localization.MonthsFormat(GetPackRemainingTime(), locale));
    SubstGlobal(result, "_RemainingDuration_", localization.MonthsFormat(GetPackRemainingTime(), locale));
    SubstGlobal(result, "_RemainingTime_", localization.MonthsFormat(GetPackRemainingTime(), locale));
    SubstGlobal(result, "_UsedTime_", localization.MonthsFormat(GetUsedTime(), locale));
    SubstGlobal(result, "_ReturnCost_", GetLocalizedPrice(ReturnCost, currency, locale, localization));
    SubstGlobal(result, "_ReturnFee_", GetLocalizedPrice(ReturnFee, currency, locale, localization));
    SubstGlobal(result, "_ReturnPackPrice_", GetLocalizedPrice(ReturnPackPrice, currency, locale, localization));
    SubstGlobal(result, "_ReturnOverrunCost_", GetLocalizedPrice(ReturnOverrunCost, currency, locale, localization));
    SubstGlobal(result, "_WithheldPackPrice_", GetLocalizedPrice(HeldSum, currency, locale, localization));
    result = TBase::FormDescriptionElement(result, currency, locale, localization);
    return localization.ApplyResources(result, locale);
}

TAtomicSharedPtr<TLongTermOfferState> TLongTermOfferState::MakeFrom(TOfferStatePtr state) {
    auto baseState = std::dynamic_pointer_cast<TBase>(state);
    if (!baseState) {
        return nullptr;
    }
    return MakeAtomicShared<TLongTermOfferState>(*baseState);
}

ui32 TLongTermOffer::GetDeposit() const {
    return TStandartOffer::GetDeposit();
}

bool TLongTermOffer::DeserializeFromProto(const NDrive::NProto::TOffer& info) {
    if (!TBase::DeserializeFromProto(info)) {
        return false;
    }
    const auto& meta = info.GetLongTermOffer();
    New = meta.GetNew();
    DetailedDescription = meta.GetDetailedDescription();
    OfferImage = meta.GetOfferImage();
    OfferSmallImage = meta.GetOfferSmallImage();
    UseInsuranceTypeVariable = meta.GetUseInsuranceTypeVariable();
    Available = meta.GetAvailable();
    AvailableSince = TInstant::MicroSeconds(meta.GetAvailableSince());
    AvailableUntil = TInstant::MicroSeconds(meta.GetAvailableUntil());
    BookingTimestamp = TInstant::MicroSeconds(meta.GetBookingTimestamp());
    MinimalPeriod = TDuration::MicroSeconds(meta.GetMinimalPeriod());
    MaximalPeriod = TDuration::MicroSeconds(meta.GetMaximalPeriod());
    MaximalSince = TInstant::MicroSeconds(meta.GetMaximalSince());
    if (meta.HasDeliveryLocation()) {
        DeliveryLocation.ConstructInPlace();
        if (!DeliveryLocation->Deserialize(meta.GetDeliveryLocation())) {
            return false;
        }
    }
    DeliveryLocationName = meta.GetDeliveryLocationName();
    DeliveryTime = TInstant::MicroSeconds(meta.GetDeliveryTime());
    Since = TInstant::MicroSeconds(meta.GetSince());
    Until = TInstant::MicroSeconds(meta.GetUntil());
    DefaultDeferPeriod = TDuration::MicroSeconds(meta.GetDefaultDeferPeriod());
    DeferDelay = TDuration::MicroSeconds(meta.GetDeferDelay());
    DelayedDeferPeriod = TDuration::MicroSeconds(meta.GetDelayedDeferPeriod());
    DelayedHours = TDuration::MicroSeconds(meta.GetDelayedHours());
    ShouldDeferCommunication = meta.GetShouldDeferCommunication();
    if (meta.HasCancellationPeriod()) {
        CancellationPeriod = TDuration::MicroSeconds(meta.GetCancellationPeriod());
    }
    if (meta.HasChildSeat()) {
        ChildSeat = meta.GetChildSeat();
    }
    if (meta.HasDelivery()) {
        Delivery = meta.GetDelivery();
    }
    if (meta.HasEarlyReturnRemainder()) {
        EarlyReturnRemainder = TDuration::MicroSeconds(meta.GetEarlyReturnRemainder());
    }
    if (meta.HasMileage()) {
        Mileage = meta.GetMileage();
    }
    if (meta.HasFranchise()) {
        Franchise = meta.GetFranchise();
    }
    if (meta.HasPaymentPlan()) {
        ELongTermPaymentPlan value;
        if (!TryFromString(meta.GetPaymentPlan(), value)) {
            return false;
        }
        PaymentPlan = value;
    }
    if (meta.HasCancellationCost()) {
        CancellationCost = meta.GetCancellationCost();
    }
    if (meta.HasDurationCost()) {
        DurationCost = meta.GetDurationCost();
    }
    if (meta.HasEarlyReturnCost()) {
        EarlyReturnCost = meta.GetEarlyReturnCost();
    }
    if (meta.HasMonthlyCost()) {
        MonthlyCost = meta.GetMonthlyCost();
    }
    if (meta.HasWeeklyCost()) {
        WeeklyCost = meta.GetWeeklyCost();
    }
    if (meta.HasChildSeatCost()) {
        ChildSeatCost = meta.GetChildSeatCost();
    }
    if (meta.HasDeliveryCost()) {
        DeliveryCost = meta.GetDeliveryCost();
    }
    if (meta.HasMileageCost()) {
        MileageCost = meta.GetMileageCost();
    }
    if (meta.HasFranchiseCost()) {
        FranchiseCost = meta.GetFranchiseCost();
    }
    if (meta.HasOfferImageStyle()) {
        if (!TryFromString(meta.GetOfferImageStyle(), OfferImageStyle)) {
            return false;
        }
    }
    SetSwitchable(true);
    return true;
}

NDrive::NProto::TOffer TLongTermOffer::SerializeToProto() const {
    auto info = TBase::SerializeToProto();
    auto meta = info.MutableLongTermOffer();
    meta->SetNew(New);
    meta->SetUseInsuranceTypeVariable(UseInsuranceTypeVariable);
    if (DetailedDescription) {
        meta->SetDetailedDescription(DetailedDescription);
    }
    if (OfferImage) {
        meta->SetOfferImage(OfferImage);
    }
    if (OfferSmallImage) {
        meta->SetOfferSmallImage(OfferSmallImage);
    }
    if (OfferImageStyle != ELongTermOfferImageStyle::Default) {
        meta->SetOfferImageStyle(ToString(OfferImageStyle));
    }
    meta->SetAvailable(Available);
    meta->SetAvailableSince(AvailableSince.MicroSeconds());
    meta->SetAvailableUntil(AvailableUntil.MicroSeconds());
    meta->SetBookingTimestamp(BookingTimestamp.MicroSeconds());
    meta->SetMinimalPeriod(MinimalPeriod.MicroSeconds());
    meta->SetMaximalPeriod(MaximalPeriod.MicroSeconds());
    meta->SetMaximalSince(MaximalSince.MicroSeconds());
    meta->SetShouldDeferCommunication(ShouldDeferCommunication);
    meta->SetDefaultDeferPeriod(DefaultDeferPeriod.MicroSeconds());
    meta->SetDeferDelay(DeferDelay.MicroSeconds());
    meta->SetDelayedDeferPeriod(DelayedDeferPeriod.MicroSeconds());
    meta->SetDelayedHours(DelayedHours.MicroSeconds());
    if (CancellationPeriod) {
        meta->SetCancellationPeriod(CancellationPeriod.MicroSeconds());
    }
    if (EarlyReturnRemainder) {
        meta->SetEarlyReturnRemainder(EarlyReturnRemainder.MicroSeconds());
    }
    if (DeliveryLocation) {
        *meta->MutableDeliveryLocation() = DeliveryLocation->Serialize();
    }
    if (DeliveryLocationName) {
        meta->SetDeliveryLocationName(DeliveryLocationName);
    }
    if (Since) {
        meta->SetSince(Since.MicroSeconds());
    }
    if (Until) {
        meta->SetUntil(Until.MicroSeconds());
    }
    if (ChildSeat) {
        meta->SetChildSeat(*ChildSeat);
    }
    if (Delivery) {
        meta->SetDelivery(*Delivery);
    }
    if (Mileage) {
        meta->SetMileage(*Mileage);
    }
    if (Franchise) {
        meta->SetFranchise(*Franchise);
    }
    if (PaymentPlan) {
        meta->SetPaymentPlan(ToString(PaymentPlan));
    }
    if (DurationCost) {
        meta->SetDurationCost(DurationCost);
    }
    if (MonthlyCost) {
        meta->SetMonthlyCost(MonthlyCost);
    }
    if (WeeklyCost) {
        meta->SetWeeklyCost(WeeklyCost);
    }
    if (CancellationCost) {
        meta->SetCancellationCost(CancellationCost);
    }
    if (ChildSeatCost) {
        meta->SetChildSeatCost(*ChildSeatCost);
    }
    if (DeliveryCost) {
        meta->SetDeliveryCost(*DeliveryCost);
    }
    if (EarlyReturnCost) {
        meta->SetEarlyReturnCost(EarlyReturnCost);
    }
    if (MileageCost) {
        meta->SetMileageCost(*MileageCost);
    }
    if (FranchiseCost) {
        meta->SetFranchiseCost(*FranchiseCost);
    }
    return info;
}

TExpected<bool, NJson::TJsonValue> TLongTermOffer::CheckAgreementRequirements(ELocalization /*locale*/, const TUserPermissions::TPtr /*permissions*/, const NDrive::IServer* /*server*/) const {
    return true;
}

namespace {
    template <class T>
    auto LongTermScale(T value, TDuration duration) {
        auto multiplier = std::max(duration.SecondsFloat(), 1.0) / (365 * 24 * 60 * 60);
        return static_cast<decltype(value)>(std::ceil(multiplier * value / 1000)) * 1000;
    }
}

NJson::TJsonValue TLongTermOffer::DoBuildJsonReport(const TReportOptions& options, const ICommonOfferBuilderAction* constructor, const NDrive::IServer& server) const {
    NJson::TJsonValue result = TBase::DoBuildJsonReport(options, constructor, server);

    if (!(options.Traits & NDriveSession::ReportOfferDetails)) {
        return result;
    }

    auto locale = options.Locale;
    auto localization = server.GetLocalization();
    if (localization) {
        result.InsertValue("detailed_description", FormDescriptionElement(DetailedDescription, locale, localization));
    }

    NJson::InsertField(result, "new", New);
    NJson::InsertField(result, "offer_image", NJson::Nullable(OfferImage));
    NJson::InsertField(result, "offer_image_style", NJson::Stringify(OfferImageStyle));
    NJson::InsertField(result, "offer_small_image", NJson::Nullable(OfferSmallImage));
    NJson::InsertNonNull(result, "available_since", NJson::Seconds(AvailableSince));
    NJson::InsertNonNull(result, "available_until", NJson::Seconds(AvailableUntil));
    NJson::InsertNonNull(result, "minimal_period", NJson::Seconds(MinimalPeriod));
    NJson::InsertNonNull(result, "maximal_period", NJson::Seconds(MaximalPeriod));
    NJson::InsertNonNull(result, "maximal_since", NJson::Seconds(MaximalSince));
    NJson::InsertNonNull(result, "delivery_area", DeliveryArea);
    NJson::InsertNonNull(result, "delivery_location", DeliveryLocation);
    NJson::InsertNonNull(result, "delivery_location_name", DeliveryLocationName);
    NJson::InsertNonNull(result, "delivery_time", NJson::Seconds(DeliveryTime));
    NJson::InsertNonNull(result, "since", NJson::Seconds(Since));
    NJson::InsertNonNull(result, "until", NJson::Seconds(Until));
    NJson::InsertNonNull(result, "monthly_cost", MonthlyCost);
    NJson::InsertNonNull(result, "weekly_cost", WeeklyCost);
    NJson::InsertField(result, "available", Available);
    NJson::InsertNonNull(result, "is_group_offer", GroupOffer);

    auto duration = Until - Since;

    auto builder = dynamic_cast<const TLongTermOfferBuilder*>(constructor);
    if (builder && localization) {
        TBill prebill;
        {
            TBillRecord record;
            record.SetCost(DurationCost + DefaultMileageCost.GetOrElse(0));
            record.SetType(GetTypeName());
            record.SetTitle(NDrive::TLocalization::FormalTariffName(GetName()));
            record.SetDuration(duration);
            record.SetDetails(localization->FormatDuration(locale, record.GetDuration()));
            prebill.MutableRecords().push_back(std::move(record));
        }

        NJson::TJsonValue badges = NJson::JSON_ARRAY;
        if (false) {
            auto badge = builder->CreateAvailableBadge();
            badges.AppendValue(badge.GetReport(locale, server));
        }
        if (New) {
            auto badge = builder->CreateDefaultNewBadge();
            badges.AppendValue(badge.GetReport(locale, server));
        }
        for (auto&& badge : builder->GetBadges()) {
            badges.AppendValue(badge.GetReport(locale, server));
        }
        result.InsertValue("badges", std::move(badges));

        if (const auto& carDescription = builder->GetCarDescription()) {
            result.InsertValue("car_description", FormDescriptionElement(builder->GetCarDescription(), locale, localization));
        }
        if (const auto& carPromo = builder->GetCarPromo()) {
            result.InsertValue("car_promo", carPromo.GetReport(locale, server));
        }

        NJson::TJsonValue primary = NJson::JSON_ARRAY;
        {
            auto mileage = builder->GetMileage();
            auto multiply = [duration](auto value) {
                return LongTermScale(value, duration);
            };
            if (mileage.HasDefaultValue()) {
                mileage.SetDefaultValue(multiply(mileage.GetDefaultValueRef()));
            }
            if (mileage.HasValues()) {
                TSet<ui64> values;
                for (auto& value : mileage.GetValuesRef()) {
                    auto v = multiply(value);
                    values.insert(v);
                }
                mileage.SetValues(std::move(values));
            }
            if (mileage.HasValues() && Mileage) {
                const auto &mileageValues = mileage.GetValuesRef();
                auto valueIndex = std::distance(mileageValues.begin(), mileageValues.lower_bound(*Mileage));
                auto valueSubtitleKey = mileage.GetSubtitle();
                SubstGlobal(valueSubtitleKey, ".subtitle", "." + ToString(valueIndex + 1) + ".subtitle");
                auto valueSubtitleLocalization = localization->GetLocalString(locale, valueSubtitleKey, TString{});
                if (valueSubtitleLocalization) {
                    mileage.SetSubtitle(std::move(valueSubtitleLocalization));
                }
            }
            if (IsSwitcher()) {
                if (GetExtraMileageLimit()) {
                    auto subtitle = localization->GetLocalString(locale, "long_term_offer.with_extra_mileage.subtitle");
                    SubstGlobal(subtitle, "_ExtraMileage_", ToString(GetExtraMileageLimit()));
                    mileage.SetSubtitle(subtitle);
                } else {
                    mileage.SetSubtitle("long_term_offer.no_extra_mileage.subtitle");
                }
            }
            mileage.SetValue(Mileage);
            mileage.SetCost(ExtraMileageCost);
            primary.AppendValue(mileage.GetReport(locale, server));

            TBillRecord record;
            record.SetCost(ExtraMileageCost.GetOrElse(0));
            record.SetType(mileage.GetId());
            record.SetTitle(localization->GetLocalString(locale, mileage.GetTitle()));
            auto distance = mileage.OptionalValue().OrElse(mileage.OptionalDefaultValue()).GetOrElse(0);
            record.SetDistance(distance);
            record.SetDetails(localization->DistanceFormatKm(locale, distance, true));
            prebill.MutableRecords().push_back(std::move(record));
        }
        if (!UseInsuranceTypeVariable) {
            auto franchise = builder->CreateDefaultFranchise(server);
            franchise.SetValue(Franchise);
            franchise.SetCost(FranchiseCost);
            primary.AppendValue(franchise.GetReport(locale, server));

            TBillRecord record;
            record.SetCost(FranchiseCost.GetOrElse(0));
            record.SetType(franchise.GetId());
            record.SetTitle(localization->GetLocalString(locale, franchise.GetTitle()));
            auto value = franchise.OptionalValue().OrElse(franchise.OptionalDefaultValue()).GetOrElse(0);
            record.SetDetails(localization->FormatPrice(locale, 100 * value, { "units.short.rub" }));
            prebill.MutableRecords().push_back(std::move(record));
        } else {
            const auto& insuranceTypeValue = (Franchise == 0u) ? FullInsuranceType : StandartInsuranceType;
            auto insuranceType = builder->CreateInsuranceType(server, GetMileageLimit());
            insuranceType.SetValue(insuranceTypeValue);
            insuranceType.SetCost(FranchiseCost);
            primary.AppendValue(insuranceType.GetReport(locale, server));

            TBillRecord record;
            record.SetCost(FranchiseCost.GetOrElse(0));
            record.SetType(insuranceType.GetId());
            record.SetTitle(localization->GetLocalString(locale, insuranceType.GetTitle()));
            prebill.MutableRecords().push_back(std::move(record));
        }
        result.InsertValue("primary", std::move(primary));

        NJson::TJsonValue secondary = NJson::JSON_ARRAY;
        if (Delivery.GetOrElse(true)) {
            auto delivery = builder->GetDelivery();
            secondary.AppendValue(delivery.GetReport(locale, server));

            TBillRecord record;
            record.SetCost(DeliveryCost.GetOrElse(0));
            record.SetType(delivery.GetId());
            record.SetTitle(localization->GetLocalString(locale, delivery.GetTitle()));
            record.SetDetails(DeliveryLocationName);
            prebill.MutableRecords().push_back(std::move(record));
        }
        if (builder->GetAllowChildSeat() || ChildSeatCost) {
            auto childSeat = builder->GetChildSeat();
            childSeat.SetValue(ChildSeat);
            childSeat.SetCost(ChildSeatCost);
            childSeat.SetPredictedCost(builder->GetChildSeatCost());
            secondary.AppendValue(childSeat.GetReport(locale, server));

            if (ChildSeatCost) {
                TBillRecord record;
                record.SetCost(*ChildSeatCost);
                record.SetType(childSeat.GetId());
                record.SetTitle(localization->GetLocalString(locale, childSeat.GetTitle()));
                prebill.MutableRecords().push_back(std::move(record));
            }
        }
        result.InsertValue("secondary", std::move(secondary));

        NJson::TJsonValue extras = NJson::JSON_ARRAY;
        for (auto&& extra : builder->GetExtras()) {
            extras.AppendValue(extra.GetReport(locale, server));
        }
        result.InsertValue("extras", std::move(extras));

        NJson::TJsonValue photos = NJson::JSON_NULL;
        for (auto&& photo : builder->GetPhotos()) {
            photos.AppendValue(photo.GetReport());
        }
        if (photos.IsDefined()) {
            result.InsertValue("photos", std::move(photos));
        }

        NJson::TJsonValue paymentPlan = NJson::JSON_NULL;
        {
            auto pp = builder->GetPaymentPlan();
            pp.SetValue(PaymentPlan);
            auto effective = PaymentPlan.OrElse(pp.OptionalDefaultValue()).GetOrElse(DefaultPaymentPlan);
            switch (effective) {
            case ELongTermPaymentPlan::Monthly:
                pp.SetCost(MonthlyCost);
                break;
            case ELongTermPaymentPlan::OneTime:
                pp.SetCost(GetPackPrice());
                break;
            case ELongTermPaymentPlan::Weekly:
                pp.SetCost(WeeklyCost);
                break;
            case ELongTermPaymentPlan::Custom:
                pp.SetCost(GetPackPriceQuantum());
                break;
            }
            pp.SetTitle("long_term.offer.payment_plan." + ToString(effective) + ".title");
            paymentPlan = pp.GetReport(locale, server);
        }
        result.InsertValue("payment_plan", std::move(paymentPlan));

        {
            TBillRecord record;
            record.SetCost(GetPackPrice());
            record.SetType(TBillRecord::TotalType);
            record.SetTitle(NDrive::TLocalization::TotalBillHeader());
            prebill.MutableRecords().push_back(std::move(record));
        }
        result.InsertValue("prebill", prebill.GetReport(locale, NDriveSession::ReportAll, server));
    }

    NJson::InsertNonNull(result, "group_offer_id", GroupOfferId);

    return result;
}

TLongTermOffer::TRecalcByMinutesInfo TLongTermOffer::CheckRecalcByMinutes(const TRidingInfo& rInfo, const TInstant startPack) const {
    Y_UNUSED(rInfo);
    Y_UNUSED(startPack);
    return {
        false,
        false,
        false,
        false
    };
}

TString TLongTermOffer::DoFormDescriptionElement(const TString& value, ELocalization locale, const ILocalization* localization) const {
    auto result = TBase::DoFormDescriptionElement(value, locale, localization);
    if (result.Contains(CancellationCostMacro) && localization) {
        auto cancellationCost = localization->FormatPrice(locale, CancellationCost);
        SubstGlobal(result, CancellationCostMacro, cancellationCost);
    }
    if (result.Contains(EarlyReturnCostMacro) && localization) {
        auto earlyReturnCost = localization->FormatPrice(locale, EarlyReturnCost);
        SubstGlobal(result, EarlyReturnCostMacro, earlyReturnCost);
    }
    if (result.Contains(PaymentScheduleHeaderMacro) && localization) {
        TString localizationId;
        if (GetUseMileageForSchedule()) {
            localizationId = "long_term_offer.schedule.header.with_mileage";
        } else {
            localizationId = "long_term_offer.schedule.header.no_mileage";
        }
        auto paymentScheduleHeader = localization->GetLocalString(locale, localizationId);
        SubstGlobal(result, PaymentScheduleHeaderMacro, paymentScheduleHeader);
    }
    if (result.Contains(PaymentScheduleMacro) && localization) {
        auto paymentScheduleReport = GetPaymentScheduleReport(locale, *localization);
        SubstGlobal(result, PaymentScheduleMacro, paymentScheduleReport);
    }
    return result;
}

TOfferStatePtr TLongTermOffer::DoCalculate(const TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events, const TInstant& until, const TRidingInfo& ridingInfo, TOfferPricing& result) const {
    auto packOfferPricing = result;
    auto state = TBase::DoCalculate(timeline, events, until, ridingInfo, packOfferPricing);
    if (!state) {
        return nullptr;
    }
    auto longTermState = TLongTermOfferState::MakeFrom(state);
    if (!longTermState) {
        return nullptr;
    }

    auto now = ModelingNow();
    auto cutoff = std::min(now, until);
    auto schedule = GetOrBuildPaymentSchedule();
    auto nextPayment = schedule.GetNextQuantum(cutoff, longTermState->GetMileage());
    if (nextPayment) {
        TLongTermOfferState::TPayment payment;
        payment.Timestamp = nextPayment->Timestamp;
        payment.Mileage = nextPayment->Mileage;
        payment.Value = GetPublicDiscountedPrice(nextPayment->Value);
        longTermState->SetNextPayment(std::move(payment));
    }
    auto server = NDrive::HasServer() ? &NDrive::GetServerAs<NDrive::IServer>() : nullptr;
    auto api = server ? server->GetDriveAPI() : nullptr;

    auto payments = api && api->HasBillingManager() ? api->GetBillingManager().GetPaymentsManager().GetObject(GetOfferId(), TInstant::Zero()) : Nothing();
    TMaybe<ui64> heldSum = payments ? payments->GetHeldSum() : TMaybe<ui64>();
    ui64 paymentsSum = 0;
    for (auto&& quantum : schedule.GetQuanta()) {
        if (!quantum.Value) {
            continue;
        }
        TLongTermOfferState::TPayment payment;
        payment.Timestamp = quantum.Timestamp;
        payment.Mileage = quantum.Mileage;
        payment.Value = GetPublicDiscountedPrice(quantum.Value);
        paymentsSum += payment.Value;
        if (quantum.ShouldBePaid(cutoff, longTermState->GetMileage())) {
            if (!heldSum || paymentsSum <= *heldSum) {
                payment.Completed = true;
                longTermState->SetHeldSum(paymentsSum);
            }
        }
        longTermState->MutablePayments().push_back(std::move(payment));
    }

    auto constructor = api ? api->GetRolesManager()->GetAction(GetBehaviourConstructorId()) : Nothing();
    auto taggedObject = api ? api->GetTagsManager().GetDeviceTags().GetObject(GetObjectId()) : Nothing();

    auto currentStage = TString();
    for (auto&& tl : timeline) {
        if (tl.GetTimeEvent() == NEventsSession::EEvent::Tag && tl.GetEventInstant() < until) {
            Y_ENSURE_BT(static_cast<size_t>(tl.GetEventIndex()) < events.size());
            auto ev = events[tl.GetEventIndex()];
            if (ev && *ev) {
                currentStage = ev->GetData()->GetName();
            }
        }
    }
    if (currentStage == TChargableTag::Prereservation) {
        result.AddSegmentInfo(currentStage, 0, 0);
        return longTermState;
    }
    if (currentStage == TLongTermOfferHolderTag::Preparation) {
        result.AddSegmentInfo(currentStage, GetCancellationCost(), GetCancellationCost());
        return longTermState;
    }
    auto optionalStart = ridingInfo.GetSegmentStart(TChargableTag::Reservation);
    if (!optionalStart) {
        return nullptr;
    }
    auto start = *optionalStart;
    auto finish = ridingInfo.GetLastInstant();
    auto duration = finish - start;
    duration = duration - GetExtraDuration();

    auto ridingStart = ridingInfo.GetSegmentStart(TChargableTag::Riding);
    if (!ridingStart && !ridingInfo.GetIsSwitch() && duration < CancellationPeriod) {
        result.AddSegmentInfo(TLongTermOfferHolderTag::Preparation, GetCancellationCost(), GetCancellationCost());
        return longTermState;
    }

    ui64 returnCost = 0;
    ui64 wholePackCost = GetPackPrice();
    {
        ui64 packCost = 0;
        ui64 mileageLimit = 0;
        ui64 earlyReturnCost = 0;
        auto recalculate = [&] (auto value) {
            if (GetDuration() >= TDuration::Days(1) && duration > TDuration::Days(1)) {
                return (value * duration.Days()) / GetDuration().Days();
            } else {
                return (value * duration.Seconds()) / std::max<ui64>(1, GetDuration().Seconds());
            }
        };

        if (GetUseMileageForSchedule()) {
            ui64 wholeMileage = GetMileageLimit() + GetExtraMileageLimit();
            double distanceTravelled = longTermState->GetMileage();
            double distanceTravelledWithoutExtra = std::max(distanceTravelled - GetExtraMileageLimit(), 0.0);
            double timeFraction = recalculate(1.0);
            double finalFraction = timeFraction;
            if (GetMileageLimit() > 0) {
                auto mileageFraction = std::min(distanceTravelledWithoutExtra / GetMileageLimit(), 1.0);
                finalFraction = std::max(finalFraction, mileageFraction);
            }

            packCost = static_cast<i64>(finalFraction * wholePackCost);
            mileageLimit = wholeMileage;
            if (longTermState->GetMileage() < mileageLimit && wholePackCost > packCost) {
                earlyReturnCost = std::min<i64>(GetEarlyReturnCost(), wholePackCost - packCost);
            }
        } else {
            packCost = recalculate(wholePackCost);
            mileageLimit = recalculate(GetMileageLimit()) + GetExtraMileageLimit();
            earlyReturnCost = GetEarlyReturnCost();
        }

        result.AddSegmentInfo("pack", packCost, packCost, duration);
        longTermState->SetReturnPackPrice(packCost);
        returnCost += packCost;

        if (longTermState->GetMileage() > mileageLimit) {
            auto overrun = longTermState->GetMileage() - mileageLimit;
            auto overrunCost = overrun * GetOverrunKm();
            result.AddSegmentInfo("overrun", overrunCost, overrunCost, TDuration::Zero(), overrun);
            longTermState->SetReturnOverrunCost(overrunCost);
            returnCost += overrunCost;
        }

        bool addEarlyReturn = true;
        if (api) {
            auto disableEarlyReturnTagName = server->GetSettings().GetValue<TString>("offers.long_term.disable_early_return_tag").GetOrElse("long_term_without_early_return");
            auto expectedUser = api->GetTagsManager().GetUserTags().GetCachedObject(GetUserId());
            bool disableEarlyReturn = expectedUser && expectedUser->GetTag(disableEarlyReturnTagName).Defined();
            addEarlyReturn = !disableEarlyReturn;
        }
        if (addEarlyReturn) {
            result.AddSegmentInfo(EarlyReturnSegment, earlyReturnCost, earlyReturnCost);
            longTermState->SetReturnFee(earlyReturnCost);
            returnCost += earlyReturnCost;
        }

        if (longTermState->GetServicingDuration()) {
            result.AddSegmentInfo(ServicingSegment, 0, 0, longTermState->GetServicingDuration());
        }
    }
    longTermState->SetReturnCost(returnCost);
    longTermState->SetMaterialized(true);

    bool returnRecalculation = !ridingInfo.GetIsSwitch() && duration + EarlyReturnRemainder < GetDuration();
    longTermState->SetReturnRecalculation(returnRecalculation);
    if (ridingInfo.GetIsFinished() && returnRecalculation) {
        return longTermState;
    }
    bool upsaleable = GetSwitchable();
    if (upsaleable && constructor) {
        auto builder = constructor->GetAs<IOfferBuilderAction>();
        upsaleable = builder ? !builder->GetSwitchOfferTagsList().empty() : false;
    }
    if (upsaleable && taggedObject) {
        auto disableSwitchTagName = server->GetSettings().GetValue<TString>("offers.long_term.disable_switch_tag").GetOrElse("long_term_prolongation_forbidden");
        auto disableSwitchTag = taggedObject->GetTag(disableSwitchTagName);
        upsaleable = !disableSwitchTag;
    }
    if (finish + EarlyReturnRemainder > start + GetTotalDuration()) {
        longTermState->SetUpsaleable(upsaleable);
    }
    result = std::move(packOfferPricing);
    return longTermState;
}

void TLongTermOffer::FillBillPack(TBill& bill, const TOfferPricing& pricing, TOfferStatePtr segmentState, ELocalization locale, const NDrive::IServer* server) const {
    TBase::FillBillPack(bill, pricing, segmentState, locale, server);
    auto localization = server ? server->GetLocalization() : nullptr;
    auto& records = bill.MutableRecords();

    auto er = pricing.GetPrices().find(EarlyReturnSegment);
    if (er != pricing.GetPrices().end()) {
        const TOfferSegment& segment = er->second;

        auto titleKey = "session." + er->first + ".bill.title";
        auto title = localization ? localization->GetLocalString(locale, titleKey) : titleKey;

        TBillRecord record;
        record.SetCost(segment.GetOriginalPrice());
        record.SetDuration(segment.GetDuration());
        record.SetTitle(title);
        record.SetType(EarlyReturnSegment);
        records.push_back(std::move(record));
    }
}

void TLongTermOffer::FillBillPricing(TBill& bill, const TOfferPricing& pricing, TOfferStatePtr segmentState, ELocalization locale, const NDrive::IServer* server) const {
    auto localization = server ? server->GetLocalization() : nullptr;
    auto longTermState = std::dynamic_pointer_cast<TLongTermOfferState>(segmentState);
    if (longTermState && !longTermState->IsMaterialized()) {
        for (auto&& [stage, segment] : pricing.GetPrices()) {
            auto titleKey = "session." + stage + ".bill.title";
            auto title = localization ? localization->GetLocalString(locale, titleKey) : titleKey;
            TBillRecord record;
            record.SetId(stage);
            record.SetType(stage);
            record.SetCost(segment.GetPrice());
            record.SetDistance(segment.GetDistance());
            record.SetDuration(segment.GetDuration());
            record.SetTitle(std::move(title));
            bill.MutableRecords().push_back(std::move(record));
        }
    } else {
        TBase::FillBillPricing(bill, pricing, segmentState, locale, server);
    }
}

void TLongTermOffer::OnBooking(const TString& deviceId, const TString& /*mobilePaymethodId*/) {
    TBase::OnBooking(deviceId);
    auto now = ModelingNow();
    if (!BookingTimestamp) {
        BookingTimestamp = now;
    }
    if (!HasPackPriceSchedule() && GetTargetHolderTag().empty()) {
        FillPaymentSchedule(now);
    }
}

TString TLongTermOffer::GetPaymentScheduleReport(ELocalization locale, const ILocalization& localization) const {
    return GetPaymentScheduleReport(GetOrBuildPaymentSchedule(), locale, localization);
}

TString TLongTermOffer::GetPaymentScheduleReport(const TPriceSchedule& schedule, ELocalization locale, const ILocalization& localization) {
    TStringBuilder result;
    for (auto&& [timestamp, mileage, value] : schedule.GetQuanta()) {
        result << "<tr>";
        result << "<td>" << timestamp.FormatLocalTime("%d.%m.%Y") << "</td>";
        if (mileage) {
            TString mileageString;
            if (*mileage == 0) {
                mileageString = localization.GetLocalString(locale, "long_term_offer.schedule.mileage_0_table_column", "");
            } else {
                mileageString = localization.GetLocalString(locale, "long_term_offer.schedule.mileage_table_column", "");
                SubstGlobal(mileageString, PaymentScheduleMileageMacro, ToString(*mileage));
                if (!mileageString) {
                    mileageString = ToString(*mileage);
                }
            }
            result << "<td>" << mileageString << "</td>";
        }
        result << "<td>" << localization.FormatPrice(locale, value, { "units.short.rub" }) << "</td>";
        result << "</tr>";
    }
    return result;
}

TPriceSchedule TLongTermOffer::BuildPaymentSchedule(TInstant firstPaymentBySchedule) const {
    auto realFirstPayment = BookingTimestamp ? BookingTimestamp : ModelingNow();
    auto paymentPlan = PaymentPlan.GetOrElse(DefaultPaymentPlan);
    auto wholeCost = static_cast<i64>(GetPackPrice());

    TPriceSchedule result(GetUseMileageForSchedule());
    auto addQuantum = [&](TInstant timestamp, ui32 mileage, i64 sum) {
        Y_ENSURE_BT(sum > 0, sum);
        Y_ENSURE_BT(sum < std::numeric_limits<ui32>::max(), sum);
        result.AddQuantum(timestamp, mileage, sum);
    };

    auto roundUpTo10 = [](double value) {
        return static_cast<ui32>(std::ceil(value / 10) * 10);
    };

    i64 quantumCost;
    std::function<TInstant(TInstant)> addQuantumDuration;

    switch (paymentPlan) {
    case ELongTermPaymentPlan::Monthly:
        quantumCost = MonthlyCost;
        addQuantumDuration = [&](TInstant timestamp) { return AddMonths(timestamp, 1); };
        break;
    case ELongTermPaymentPlan::Weekly:
        quantumCost = WeeklyCost;
        addQuantumDuration = [&](TInstant timestamp) { return timestamp + TDuration::Days(7); };
        break;
    case ELongTermPaymentPlan::OneTime:
        quantumCost = wholeCost;
        addQuantumDuration = [&](TInstant timestamp) { return timestamp + GetDuration(); };
        break;
    case ELongTermPaymentPlan::Custom:
        quantumCost = GetPackPriceQuantum();
        addQuantumDuration = [&](TInstant timestamp) { return timestamp + GetPackPriceQuantumPeriod(); };
        break;
    }
    Y_ENSURE_BT(quantumCost > 0);
    double wholeMileage = GetMileageLimit() + GetExtraMileageLimit();
    double quantumMileage = (1.0 * (quantumCost - (quantumCost > 1 ? 1 : 0)) / wholeCost) * GetMileageLimit();

    auto remainingCost = wholeCost;
    double currentMileage = GetExtraMileageLimit();
    auto currentTimestamp = firstPaymentBySchedule;

    bool firstQuantum = true;
    while (remainingCost > 0) {
        auto timestamp = firstQuantum ? realFirstPayment : currentTimestamp;
        addQuantum(timestamp, roundUpTo10(currentMileage), std::min(remainingCost, quantumCost));
        remainingCost -= quantumCost;
        currentTimestamp = addQuantumDuration(currentTimestamp);
        currentMileage += quantumMileage;
        currentMileage = std::min<double>(currentMileage, wholeMileage);
        firstQuantum = false;
    }
    return result;
}

TPriceSchedule TLongTermOffer::GetOrBuildPaymentSchedule() const {
    if (HasPackPriceSchedule()) {
        return GetPackPriceScheduleRef();
    } else {
        auto timestamp = std::max(BookingTimestamp, Since);
        return BuildPaymentSchedule(timestamp);
    }
}

void TLongTermOffer::FillPaymentSchedule(TInstant timestamp) {
    SetPackPriceSchedule(BuildPaymentSchedule(timestamp));
}

TLongTermBadge TLongTermOfferBuilder::CreateDefaultNewBadge() {
    TLongTermBadge result;
    result.SetId("new");
    result.SetColor("#FF9C41");
    result.SetText("long_term_offer.new.text");
    return result;
}

TLongTermVariable<bool> TLongTermOfferBuilder::CreateDefaultChildSeat() {
    TLongTermVariable<bool> result;
    result.SetId("child_seat");
    result.SetTitle("long_term_offer.child_seat.title");
    result.SetSubtitle("long_term_offer.child_seat.subtitle");
    result.SetDefaultValue(false);
    return result;
}

TLongTermVariable<ui64> TLongTermOfferBuilder::CreateDefaultMileage() {
    TLongTermVariable<ui64> result;
    result.SetId("mileage");
    result.SetTitle("long_term_offer.mileage.title");
    result.SetSubtitle("long_term_offer.mileage.subtitle");
    result.SetDefaultValue(25000);
    result.SetMaximalValue(25000 * 3);
    result.SetMinimalValue(24000 / 3);
    result.SetValues(TSet<ui64>{
        8000,
        10000,
        12500,
        16000,
        25000,
        37500,
        50000,
        62500,
        75000,
    });
    result.SetUnit("units.short.km");
    return result;
}

TLongTermVariable<ui64> TLongTermOfferBuilder::CreateDefaultFranchise(const NDrive::IServer& server) {
    auto defaultValue = server.GetSettings().GetValue<ui64>("offers.long_term.franchise.default_value").GetOrElse(30000);
    TLongTermVariable<ui64> result;
    result.SetId("franchise");
    result.SetTitle("long_term_offer.franchise.title");
    result.SetSubtitle("long_term_offer.franchise.subtitle");
    result.SetDefaultValue(defaultValue);
    result.SetValues(TSet<ui64>{
        0,
        defaultValue,
    });
    result.SetUnit("units.short.rub");
    return result;
}

TLongTermVoid TLongTermOfferBuilder::CreateDefaultDelivery() {
    TLongTermVoid result;
    result.SetId("delivery");
    result.SetTitle("long_term_offer.delivery.title");
    result.SetSubtitle("long_term_offer.delivery.subtitle");
    return result;
}

TLongTermExtras TLongTermOfferBuilder::CreateDefaultExtras() {
    TLongTermExtras result;
    {
        TLongTermExtra washing;
        washing.SetId("washing");
        washing.SetTitle("long_term_offer.washing.title");
        washing.SetSubtitle("long_term_offer.washing.subtitle");
        result.push_back(std::move(washing));
    }
    {
        TLongTermExtra fueling;
        fueling.SetId("fueling");
        fueling.SetTitle("long_term_offer.fueling.title");
        fueling.SetSubtitle("long_term_offer.fueling.subtitle");
        result.push_back(std::move(fueling));
    }
    return result;
}

TLongTermEnum<ELongTermPaymentPlan> TLongTermOfferBuilder::CreateDefaultPaymentPlan() {
    TLongTermEnum<ELongTermPaymentPlan> result;
    result.SetId("payment_plan");
    result.SetDefaultValue(DefaultPaymentPlan);
    result.SetValues({
        ELongTermPaymentPlan::OneTime,
        ELongTermPaymentPlan::Weekly,
        ELongTermPaymentPlan::Monthly,
    });
    return result;
}

TLongTermBadge TLongTermOfferBuilder::CreateAvailableBadge() const {
    TLongTermBadge result;
    result.SetId("available");
    result.SetColor("#6B7AFF");
    result.SetText("CreateAvailableBadge");
    return result;
}

TMaybe<TTagsFilter> TLongTermOfferBuilder::CreateCarTagsFilter() const {
    if (CarTagsFilter) {
        return TTagsFilter::BuildFromString(CarTagsFilter);
    } else {
        return {};
    }
}

TMaybe<TBaseAreaTagsFilter> TLongTermOfferBuilder::GetDeliveryAreaFilter() const {
    if (DeliveryAreaTagsFilter) {
        return TBaseAreaTagsFilter::BuildFromString(DeliveryAreaTagsFilter);
    } else {
        return {};
    }
}

TLongTermVariable<TString> TLongTermOfferBuilder::CreateInsuranceType(const NDrive::IServer& server, ui64 mileage) const {
    auto franchise = CreateDefaultFranchise(server);
    auto defaultFranchiseValue = franchise.OptionalDefaultValue().GetOrElse(0);
    TLongTermVariable<TString> result;
    result.SetId(InsuranceTypeId);
    result.SetTitle("long_term_offer." + InsuranceTypeId + ".title");
    result.SetSubtitle("long_term_offer." + InsuranceTypeId + ".subtitle");
    result.SetDefaultValue(StandartInsuranceType);
    result.OptionalNamedValues().ConstructInPlace();
    {
        auto& v = result.MutableNamedValuesRef().emplace_back();
        v.Value = StandartInsuranceType;
        v.Cost = 0;
        v.Name = "long_term_offer." + InsuranceTypeId + "." + StandartInsuranceType + ".name";
        v.Link = "long_term_offer." + InsuranceTypeId + "." + StandartInsuranceType + ".link";
        v.LinkTitle = "long_term_offer." + InsuranceTypeId + "." + StandartInsuranceType + ".link_title";
        v.Subtitle = "long_term_offer." + InsuranceTypeId + "." + StandartInsuranceType + ".subtitle";
    }
    {
        auto& v = result.MutableNamedValuesRef().emplace_back();
        v.Value = FullInsuranceType;
        v.Cost = std::ceil((0.0 - defaultFranchiseValue) * FranchiseAlpha * mileage);
        v.Name = "long_term_offer." + InsuranceTypeId + "." + FullInsuranceType + ".name";
        v.Link = "long_term_offer." + InsuranceTypeId + "." + FullInsuranceType + ".link";
        v.LinkTitle = "long_term_offer." + InsuranceTypeId + "." + FullInsuranceType + ".link_title";
        v.Subtitle = "long_term_offer." + InsuranceTypeId + "." + FullInsuranceType + ".subtitle";
    }
    return result;
}

TVector<TTaggedObject> TLongTermOfferBuilder::GetCars(const NDrive::IServer& server) const {
    if (!CarTagsFilter) {
        return {};
    }

    const auto& deviceTagManager = server.GetDriveAPI()->GetTagsManager().GetDeviceTags();
    auto filter = TTagsFilter::BuildFromString(CarTagsFilter);
    auto prefilteredObjects = deviceTagManager.PrefilterObjects(filter);

    if (!prefilteredObjects) {
        return {};
    }

    TVector<TTaggedObject> result;
    for (auto&& objectId : *prefilteredObjects) {
        auto object = deviceTagManager.GetObject(objectId);
        if (!object) {
            continue;
        }
        if (!filter.IsMatching(*object)) {
            continue;
        }

        result.push_back(std::move(*object));
    }

    return result;
}

TLongTermOfferBuilder::TCandidates TLongTermOfferBuilder::FindCandidates(const NDrive::IServer& server) const {
    auto sessionBuilder = server.GetDriveAPI()->GetTagsManager().GetDeviceTags().GetSessionsBuilder("billing");

    TCandidates result;
    for (auto&& object : GetCars(server)) {
        bool tier1 = object.HasTag(Tier1TagName);
        bool tier2 = object.HasTag(Tier2TagName);
        if (!tier1 && !tier2) {
            continue;
        }

        if (sessionBuilder) {
            auto session = sessionBuilder->GetLastObjectSession(object.GetId());
            auto billingSession = std::dynamic_pointer_cast<TBillingSession>(session);
            if (billingSession && !billingSession->GetClosed()) {
                auto offer = billingSession->GetCurrentOffer();
                if (offer->GetTypeName() == TLongTermOffer::GetTypeNameStatic()) {
                    continue;
                }
            }
        }

        if (tier1) {
            result.Tier1.push_back(std::move(object));
            continue;
        }
        if (tier2) {
            result.Tier2.push_back(std::move(object));
            continue;
        }
    }

    return result;
}

TInstant TLongTermOfferBuilder::GetCarLongTermUntil(const TString& carId, const NDrive::IServer& server) {
    auto &carAttachments = server.GetDriveAPI()->GetCarAttachmentAssignments();
    TCarGenericAttachment registryAttachment;
    if (!carAttachments.TryGetAttachmentOfType(carId, EDocumentAttachmentType::CarRegistryDocument, registryAttachment)) {
        return {};
    }
    auto registryDocument = Yensured(std::dynamic_pointer_cast<TCarRegistryDocument>(registryAttachment.GetImpl()));
    auto carUntil = registryDocument->GetAvailableForLongTermUntil();
    return carUntil;
}

TMaybe<TInstant> TLongTermOfferBuilder::GetAvailableInLongTermUntil(const NDrive::IServer& server) const {
    if (!AvailableUntilByCarInfo || !CarTagsFilter) {
        return TInstant::Max();
    }

    TInstant until;
    for (auto&& object : GetCars(server)) {
        auto carUntil = GetCarLongTermUntil(object.GetId(), server);
        if (!carUntil) {
            return TInstant::Max();
        }
        until = std::max(until, carUntil);
    }
    if (until) {
        return until;
    }
    return Nothing();
}

bool TLongTermOfferBuilder::HasGrouppingParent(const NDrive::IServer& server) const {
    auto maybeAncestor = server.GetDriveAPI()->GetRolesManager()->GetAction(GetActionParent());
    if (!maybeAncestor) {
        return false;
    }
    auto ancestor = maybeAncestor->GetAs<TLongTermOfferBuilder>();
    return ancestor && ancestor->IsGroupOfferBuilder();
}

bool TLongTermOfferBuilder::IsGroupOfferBuilderStrict(const NDrive::IServer& server) const {
    return IsGroupOfferBuilder() && !HasGrouppingParent(server);
}

EOfferCorrectorResult TLongTermOfferBuilder::BuildGroupOffer(
        const TUserPermissions& permissions,
        TVector<IOfferReport::TPtr>& offers,
        const TOffersBuildingContext& context,
        const NDrive::IServer* server,
        NDrive::TInfoEntitySession& session
) const {
    if (!server) {
        return EOfferCorrectorResult::Unimplemented;
    }

    bool groupOffersEnabled = permissions.GetSetting<bool>(server->GetSettings(), "offers.long_term.group_offers.enabled").GetOrElse(false);
    if (!groupOffersEnabled) {
        return EOfferCorrectorResult::Unimplemented;
    }

    TVector<IOfferReport::TPtr> groupOfferReports;
    auto contextForGroup = context;
    contextForGroup.SetForceNonGroupOffer(true);
    DoBuildOffers(permissions, groupOfferReports, contextForGroup, server, session);
    if (groupOfferReports.size() != 1) {
        return EOfferCorrectorResult::Unimplemented;
    }
    auto groupOfferReport = groupOfferReports[0];
    auto groupOffer = groupOfferReport->GetOfferAs<TLongTermOffer>();
    Y_ENSURE(groupOffer);
    groupOffer->SetBookable(false);
    groupOffer->SetGroupOffer(true);

    // TODO: remove me when client does not need this
    // https://st.yandex-team.ru/DRIVEBACK-4596
    groupOffer->SetGroupOfferId(groupOffer->GetOfferId());

    TVector<IOfferReport::TPtr> childOffers;
    auto actions = permissions.GetOfferBuilders();

    auto contextForInner = context;
    contextForInner.SetBuildingFromGroupOffer(true);

    bool anyAvailable = false;
    for (auto&& action : actions) {
        if (action->GetActionParent() != GetName()) {
            continue;
        }
        if (auto longTermAction = std::dynamic_pointer_cast<const TLongTermOfferBuilder>(action)) {
            for (auto& report : longTermAction->BuildOffers(contextForInner, session)) {
                if (auto longTermOffer = report->GetOfferAs<TLongTermOffer>()) {
                    longTermOffer->SetGroupOfferId(groupOffer->GetOfferId());
                    if (longTermOffer->IsAvailable()) {
                        anyAvailable = true;
                    }
                    childOffers.push_back(report);
                }
            }
        }
    }

    if (childOffers.empty()) {
        return EOfferCorrectorResult::Unimplemented;
    }

    if (!anyAvailable) {
        groupOffer->MakeUnavailable();
        groupOfferReport->SetListPriority(UNAVAILABLE_LIST_PRIORITY);
    }

    offers.push_back(groupOfferReport);
    offers.insert(offers.end(), childOffers.begin(), childOffers.end());

    return EOfferCorrectorResult::Success;
}

EOfferCorrectorResult TLongTermOfferBuilder::SetOfferTimeInfo(TLongTermOffer& offer, const TOffersBuildingContext& context, const NDrive::IServer& server) const {
    auto now = context.GetRequestStartTime();

    auto availableSince = std::max(AvailableSince, now + AvailableAfter);
    auto maximalSince = now + MaximalSinceOffset;
    if (context.HasLongTermSince()) {
        availableSince = context.GetLongTermSinceRef();
        maximalSince = context.GetLongTermSinceRef();
    }

    auto availableUntil = TInstant::Max();
    if (auto availableInLongTermUntil = GetAvailableInLongTermUntil(server)) {
        availableUntil = std::min(availableUntil, *availableInLongTermUntil);
    } else {
        return EOfferCorrectorResult::Unimplemented;
    }
    if (AvailableUntil) {
        availableUntil = std::min(availableUntil, AvailableUntil);
    }

    auto maximalPeriod = TDuration::Max();
    if (MaximalPeriod) {
        maximalPeriod = std::min(maximalPeriod, MaximalPeriod);
    }
    maximalPeriod = std::min(maximalPeriod, availableUntil - availableSince);
    maximalPeriod = TDuration::Days(maximalPeriod.Days());

    auto minimalPeriod = MinimalPeriod;
    minimalPeriod = TDuration::Days(minimalPeriod.Days());

    if (maximalPeriod < minimalPeriod) {
        return EOfferCorrectorResult::Unimplemented;
    }

    auto variables = context.GetVariables();
    auto defaultSince = availableSince;
    auto optionalSince = NJson::FromJson<TMaybe<TInstant>>(variables["since"]);
    auto since = optionalSince.GetOrElse(defaultSince);

    auto defaultUntil = since + minimalPeriod;
    auto optionalUntil = NJson::FromJson<TMaybe<TInstant>>(variables["until"]);
    auto until = optionalUntil.GetOrElse(defaultUntil);

    auto duration = until - since;

    duration = std::min(duration, maximalPeriod);
    duration = std::max(duration, minimalPeriod);
    duration = TDuration::Days(duration.Days());

    until = since + duration;

    if (since < availableSince) {
        since = availableSince;
    }
    if (since > maximalSince) {
        since = maximalSince;
    }
    until = since + duration;

    if (until > availableUntil) {
        until = availableUntil;
        since = availableUntil - duration;
    }

    Y_ENSURE(since >= availableSince);
    Y_ENSURE(since <= maximalSince);
    Y_ENSURE(until <= availableUntil);
    Y_ENSURE(until - since == duration);
    Y_ENSURE(duration <= maximalPeriod);
    Y_ENSURE(duration >= minimalPeriod);

    offer.SetAvailableSince(availableSince);
    offer.SetMaximalSince(maximalSince);
    offer.SetAvailableUntil(availableUntil == TInstant::Max() ? TInstant() : availableUntil);
    offer.SetMaximalPeriod(maximalPeriod == TDuration::Max() ? TDuration() : maximalPeriod);
    offer.SetMinimalPeriod(minimalPeriod);

    offer.SetSince(since);
    offer.SetUntil(until);
    offer.SetDuration(duration);

    return EOfferCorrectorResult::Success;
}

EOfferCorrectorResult TLongTermOfferBuilder::DoBuildOffers(
        const TUserPermissions& permissions,
        TVector<IOfferReport::TPtr>& offers,
        const TOffersBuildingContext& context,
        const NDrive::IServer* server,
        NDrive::TInfoEntitySession& session
) const {
    if (!server) {
        return EOfferCorrectorResult::Unimplemented;
    }
    if (context.HasCarId() && !context.HasLongTermSince()) {
        return EOfferCorrectorResult::Unimplemented;
    }
    if (ForSwitch && !context.IsSwitching() && !context.IsReplacingCar()) {
        return EOfferCorrectorResult::Unimplemented;
    }

    if (IsGroupOfferBuilderStrict(*server) && !context.IsForceNonGroupOffer()) {
        return BuildGroupOffer(permissions, offers, context, server, session);
    }

    bool groupOffersEnabled = permissions.GetSetting<bool>(server->GetSettings(), "offers.long_term.group_offers.enabled").GetOrElse(false);
    if (groupOffersEnabled && HasGrouppingParent(*server) && !context.IsBuildingFromGroupOffer()) {
        return EOfferCorrectorResult::Unimplemented;
    }

    auto now = context.GetRequestStartTime();
    auto variables = context.GetVariables();

    auto offer = MakeAtomicShared<TLongTermOffer>();

    offer->SetCancellationPeriod(CancellationPeriod);
    offer->SetCriticalDistanceRemainder(CriticalDistanceRemainder);
    offer->SetCriticalDurationRemainder(CriticalDurationRemainder);
    offer->SetEarlyReturnRemainder(EarlyReturnRemainder);
    offer->SetDetailedDescription(DetailedDescription);
    offer->SetNew(New);
    offer->SetOfferImage(OfferImage);
    offer->SetOfferImageStyle(OfferImageStyle);
    offer->SetOfferSmallImage(OfferSmallImage);
    offer->SetObjectModel(CarModel);
    offer->SetSubName(Subname);
    offer->SetUserId(context.GetUserHistoryContextUnsafe().GetUserId());
    offer->SetOverrunKm(OverrunPrice);
    offer->MutableParking().SetPrice(OvertimePrice);
    offer->MutableRiding().SetPrice(OvertimePrice);
    offer->SetShouldDeferCommunication(ShouldDeferCommunication);
    offer->SetDeferDelay(DeferDelay);
    offer->SetDelayedDeferPeriod(DelayedDeferPeriod);
    offer->SetDefaultDeferPeriod(DefaultDeferPeriod);
    offer->SetDelayedHours(DelayedHours);
    offer->SetSwitchable(true);

    i64 cost = 0;

    if (auto result = SetOfferTimeInfo(*offer, context, *server); result != EOfferCorrectorResult::Success) {
        return result;
    }
    const auto duration = offer->GetDuration();

    if (duration) {
        auto durationCostFloat = BaseCost + duration.Days() * DurationAlpha;
        auto durationCost = static_cast<i64>(std::ceil(durationCostFloat));
        offer->SetDurationCost(durationCost);
        cost += durationCost;
    }

    auto deliveryLocation = NJson::FromJson<TMaybe<TGeoCoord>>(variables["delivery_location"]);
    if (deliveryLocation) {
        if (!CheckDeliveryLocation(*deliveryLocation, *server)) {
            session.SetCode(HTTP_BAD_REQUEST);
            session.SetErrorInfo(
                "TLongTermOfferBuilder::DoBuildOffers",
                TStringBuilder() << "cannot deliver to " << deliveryLocation,
                NDrive::MakeError("bad_delivery_location")
            );
            return EOfferCorrectorResult::BuildingProblems;
        }
        offer->SetDeliveryLocation(deliveryLocation);
    }
    auto deliveryLocationName = NJson::FromJson<TMaybe<TString>>(variables["delivery_location_name"]);
    if (deliveryLocationName) {
        offer->SetDeliveryLocationName(*deliveryLocationName);
    }
    if (context.HasUserHistoryContext()) {
        auto userDestination = context.GetUserHistoryContextRef().GetUserDestinationPtr();
        auto userFinishArea = userDestination ? userDestination->GetFinishArea() : nullptr;
        if (userFinishArea) {
            offer->SetDeliveryArea(userFinishArea->OptionalFinishArea());
        }
    }

    auto childSeat = NJson::FromJson<TMaybe<bool>>(variables[ChildSeat.GetId()]);
    if (AllowChildSeat && childSeat) {
        offer->SetChildSeat(childSeat);
        if (*childSeat) {
            auto childSeatCost = static_cast<i64>(std::ceil(ChildSeatCost));
            offer->SetChildSeatCost(childSeatCost);
            cost += childSeatCost;
        }
    }
    i64 defaultMileage = Mileage.HasDefaultValue() ? LongTermScale(Mileage.GetDefaultValueRef(), duration) : 0;
    i64 mileage = NJson::FromJson<TMaybe<ui64>>(variables[Mileage.GetId()]).GetOrElse(defaultMileage);
    if (Mileage.HasValues()) {
        auto values = TSet<ui64>();
        for (auto&& value : Mileage.GetValuesRef()) {
            values.insert(LongTermScale(value, duration));
        }
        if (!values.contains(mileage)) {
            mileage = defaultMileage;
        }
    }
    if (mileage) {
        offer->SetMileage(mileage);
        offer->SetMileageLimit(mileage);

        auto defaultMileageCostFloat = defaultMileage * MileageAlpha;
        auto defaultMileageCost = static_cast<i64>(std::ceil(defaultMileageCostFloat));

        auto extraMileage = mileage - defaultMileage;
        auto extraMileageAlpha = extraMileage > 0 ? ExtraMileageAlpha : UnderMileageAlpha;
        auto extraMileageCostFloat = extraMileage * extraMileageAlpha;

        auto mileageCostFloat = mileage * MileageAlpha + extraMileageCostFloat;
        auto mileageCost = static_cast<i64>(std::ceil(mileageCostFloat));

        offer->SetMileageCost(mileageCost);
        offer->SetDefaultMileageCost(defaultMileageCost);
        if (extraMileage) {
            offer->SetExtraMileageCost(mileageCost - defaultMileageCost);
        }
        cost += mileageCost;
    }
    offer->SetUseMileageForSchedule(permissions.GetSetting<bool>(server->GetSettings(), "offer.long_term.use_mileage_for_schedule").GetOrElse(false));

    // Franchise/Insurance

    auto defaultFranchiseVariable = CreateDefaultFranchise(*server);

    auto franchise = NJson::FromJson<TMaybe<ui64>>(variables[defaultFranchiseVariable.GetId()]);
    auto insuranceType = NJson::FromJson<TMaybe<TString>>(variables[InsuranceTypeId]);

    if (context.IsProlonging()) {
        auto session = Yensured(context.GetBillingSession());
        if (session->GetClosed()) {
            return EOfferCorrectorResult::Problems;
        }
        auto currentOffer = Yensured(std::dynamic_pointer_cast<TLongTermOffer>(session->GetCurrentOffer()));
        if (!insuranceType) {
            insuranceType = currentOffer->GetInsuranceType();
        }
    }

    i64 defaultFranchise = defaultFranchiseVariable.OptionalDefaultValue().GetOrElse(0);
    if (!franchise) {
        if (insuranceType == FullInsuranceType) {
            franchise = 0;
        }
    }
    if (franchise) {
        offer->SetFranchise(franchise);
        auto franchiseCostFloat = (static_cast<i64>(*franchise) - defaultFranchise) * FranchiseAlpha * mileage;
        auto franchiseCost = static_cast<i64>(std::ceil(franchiseCostFloat));
        offer->SetFranchiseCost(franchiseCost);
        cost += franchiseCost;
    } else {
        offer->SetFranchise(defaultFranchise);
        offer->SetFranchiseCost(0);
    }
    if (franchise && *franchise == 0) {
        offer->SetInsuranceType(FullInsuranceType);
    }
    offer->SetUseInsuranceTypeVariable(context.GetSetting<bool>("offers.long_term.use_insurance_type_variable").GetOrElse(false));

    auto paymentPlan = NJson::FromJson<TMaybe<ELongTermPaymentPlan>>(variables[PaymentPlan.GetId()]);
    if (paymentPlan) {
        offer->SetPaymentPlan(paymentPlan);
    }

    if (cost >= 0) {
        auto days = duration.Days();
        auto months = GetMonthsDelta(offer->GetSince(), offer->GetUntil());
        auto monthlyCostFloat = 0.0;
        if (months > 0) {
            auto lastPaymentTimestamp = AddMonths(offer->GetSince(), months);
            auto tail = offer->GetUntil() - lastPaymentTimestamp;
            auto lastPayment = 1.0 * cost * tail.Seconds() / duration.Seconds();
            monthlyCostFloat = (cost - lastPayment) / months;
        } else {
            monthlyCostFloat = cost;
        }
        auto monthlyCost = static_cast<i64>(std::ceil(monthlyCostFloat));
        auto weeklyCostFloat = days > 7 ? 7.0 * cost / days : cost;
        auto weeklyCost = static_cast<i64>(std::ceil(weeklyCostFloat));
        offer->SetCancellationCost(CancellationCost);
        offer->SetEarlyReturnCost(EarlyReturnCost);
        offer->SetPackPrice(cost);
        offer->SetMonthlyCost(monthlyCost);
        offer->SetWeeklyCost(weeklyCost);
    }

    offer->SetUseDeposit(true);

    if (PackPriceQuantumPeriod) {
        auto quantum = offer->GetPackPrice() * PackPriceQuantumPeriod.Seconds() / std::max<ui64>(1, duration.Seconds());
        offer->SetDepositAmount(quantum);
        offer->SetPackPriceQuantum(quantum);
        offer->SetPackPriceQuantumPeriod(PackPriceQuantumPeriod);
        offer->SetPaymentDiscretization(quantum);
        offer->SetPaymentPlan(ELongTermPaymentPlan::Custom);
    } else {
        switch (offer->OptionalPaymentPlan().GetOrElse(DefaultPaymentPlan)) {
        case ELongTermPaymentPlan::Monthly:
            offer->SetDepositAmount(offer->GetMonthlyCost());
            offer->SetPackPriceQuantum(offer->GetMonthlyCost());
            offer->SetPackPriceQuantumByMonths(true);
            offer->SetPaymentDiscretization(offer->GetMonthlyCost());
            break;
        case ELongTermPaymentPlan::OneTime:
            offer->SetDepositAmount(offer->GetPackPrice());
            break;
        case ELongTermPaymentPlan::Weekly:
            offer->SetDepositAmount(offer->GetWeeklyCost());
            offer->SetPackPriceQuantum(offer->GetWeeklyCost());
            offer->SetPackPriceQuantumPeriod(TDuration::Days(7));
            offer->SetPaymentDiscretization(offer->GetWeeklyCost());
            break;
        case ELongTermPaymentPlan::Custom:
            ythrow yexception() << "cannot set Custom payment plan externally";
        }
    }
    if (GetPaymentDiscretization() != DefaultPaymentDiscretization) {
        offer->SetPaymentDiscretization(GetPaymentDiscretization());
    }
    if (context.HasExtraDuration()) {
        offer->SetExtraDuration(context.GetExtraDurationUnsafe());
    }
    if (context.HasExtraDistance()) {
        offer->SetExtraMileageLimit(context.GetExtraDistanceUnsafe());
    }

    if (context.HasCarId()) {
        offer->SetObjectId(context.GetCarIdRef());
        offer->FillPaymentSchedule(offer->GetSince());
    } else {
        offer->SetTargetHolderTag(OfferHolderTag);
    }

    if (context.IsReplacingCar()) {
        auto session = Yensured(context.GetBillingSession());
        if (session->GetClosed()) {
            return EOfferCorrectorResult::Success;
        }
        auto currentOffer = Yensured(std::dynamic_pointer_cast<TLongTermOffer>(session->GetCurrentOffer()));
        auto compilation = Yensured(session->GetCompilationAs<TBillingSession::TBillingCompilation>());
        auto packOfferState = Yensured(compilation->GetCurrentOfferStateAs<TPackOfferState>());
        auto newOffer = Yensured(std::dynamic_pointer_cast<TLongTermOffer>(currentOffer->Clone()));
        // auto startTimestamp = ModelingNow();
        // TInstant prev;
        // bool foundPrev = false;
        // TPriceSchedule result;
        // auto add = [&](TInstant timestamp, i64 sum) {
        //     Y_ENSURE_BT(sum > 0, sum);
        //     Y_ENSURE_BT(sum < std::numeric_limits<ui32>::max(), sum);
        //     result.AddQuantum(timestamp, sum);
        // };
        // for (auto&& [timestamp, value] : currentOffer->GetPackPriceScheduleRef().GetQuanta()) {
        //     if (timestamp > startTimestamp) {
        //         if (!foundPrev) {
        //             foundPrev = true;
        //             add(startTimestamp, value * (timestamp - startTimestamp) / (timestamp - prev));
        //         }
        //         add(timestamp, value);
        //     } else {
        //         prev = timestamp;
        //     }
        // }
        // newOffer->SetPackPriceSchedule(result);

        auto fullPrice  = currentOffer->GetPackPrice() - packOfferState->GetServicingOmittedPrice();
        auto prices = compilation->GetCurrentOfferPricing();
        auto packPricePtr = prices ? prices->GetPrices("pack") : nullptr;
        auto paidPrice = packPricePtr? packPricePtr->GetOriginalPrice() : 0;
        ui32 usedExtraMileage = std::min<ui32>(packOfferState->GetMileage(), currentOffer->GetExtraMileageLimit());
        ui32 remainingExtraMileage = currentOffer->GetExtraMileageLimit() - usedExtraMileage;
        ui32 remainingMainMileage = std::ceil(packOfferState->GetRemainingDistance()) - remainingExtraMileage;

        newOffer->SetPackPrice(fullPrice - paidPrice);
        newOffer->SetExtraMileageLimit(remainingExtraMileage);
        newOffer->SetMileage(remainingMainMileage);
        newOffer->SetMileageLimit(remainingMainMileage);

        TDuration usedDuration = currentOffer->GetTotalDuration() - packOfferState->GetPackRemainingTime() - packOfferState->GetServicingDuration();
        TDuration usedExtraDuration = std::min(usedDuration, currentOffer->GetExtraDuration());
        TDuration remainingExtraDuration = currentOffer->GetExtraDuration() - usedExtraDuration;
        TDuration remainingMainDuration = packOfferState->GetPackRemainingTime() - remainingExtraDuration;
        newOffer->SetExtraDuration(remainingExtraDuration);
        newOffer->SetDuration(remainingMainDuration);

        newOffer->SetSince(now + TDuration::Days(1));

        // from constructed offer
        newOffer->SetOfferId(offer->GetOfferId());
        newOffer->SetTimestamp(offer->GetTimestamp());
        newOffer->SetDeadline(offer->GetDeadline());
        newOffer->SetBookingTimestamp(offer->GetBookingTimestamp());
        newOffer->SetDeliveryLocation(offer->OptionalDeliveryLocation());
        newOffer->SetDeliveryLocationName(offer->GetDeliveryLocationName());
        newOffer->SetDeliveryArea(offer->OptionalDeliveryArea());
        newOffer->SetObjectId(offer->GetObjectId());
        newOffer->SetTargetHolderTag(offer->GetTargetHolderTag());

        newOffer->ClearPackPriceSchedule();

        offer = newOffer;
    }

    if ((!context.IsSwitching() || context.IsReplacingCar()) && !offer->HasDeliveryLocation()) {
        offer->SetBookable(false);
    }

    offer->SetAgreement(NDrive::GetOfferAgreement(Agreement, context, offer->GetInsuranceType()));

    auto report = MakeAtomicShared<TLongTermOfferReport>(offer, nullptr);

    if (Autohide && !IsGroupOfferBuilderStrict(*server) && !context.IsSwitching()) {
        auto guard = context.BuildEventGuard("autohide");
        auto candidates = FindCandidates(*server);
        auto candidatesScore = candidates.Tier1.size() + 0.5 * candidates.Tier2.size();

        auto tx = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        auto existing = TLongTermOfferHolderTag::RestoreOfferHolderTags(GetName(), *server, tx);
        if (!existing) {
            session.MergeErrorMessages(tx.GetMessages(), "tx");
            return EOfferCorrectorResult::Problems;
        }
        bool hide = existing->size() >= candidatesScore;
        bool reportAutohideStats = context.GetSetting<bool>("offers.long_term.report_autohide_stats").GetOrElse(false);
        NJson::TJsonValue ev = NJson::TMapBuilder
            ("event", "autohide_stats")
            ("hide", hide)
            ("name", GetName())
            ("existing", existing->size())
            ("tier1", candidates.Tier1.size())
            ("tier2", candidates.Tier2.size())
            ("score", candidatesScore)
        ;
        if (reportAutohideStats) {
            session.AddErrorMessage("long_term_autohide", ev.GetStringRobust());
        }
        if (guard) {
            guard->AddEvent(std::move(ev));
        }
        if (hide) {
            bool useAvailabilityFlag = context.GetSetting<bool>("offers.long_term.use_availability_flag_for_autohide").GetOrElse(false);
            if (useAvailabilityFlag) {
                offer->MakeUnavailable();
                report->SetListPriority(UNAVAILABLE_LIST_PRIORITY);
            } else {
                return EOfferCorrectorResult::Unimplemented;
            }
        }
    }

    TVector<TDBTag> tags;
    for (auto&& tagName : OfferTags) {
        TDBTag tag;
        tag.SetData(MakeAtomicShared<TFeedbackTraceTag>(tagName));
        tag.SetObjectId(offer->GetOfferId());
        tag.SetTagId(NUtil::CreateUUID());
        tags.push_back(tag);
    }
    report->SetTags(std::move(tags));
    offers.push_back(report);
    return EOfferCorrectorResult::Success;
}

EOfferCorrectorResult TLongTermOfferBuilder::DoCheckOfferConditions(const TOffersBuildingContext& context, const TUserPermissions& permissions) const {
    Y_UNUSED(context);
    Y_UNUSED(permissions);
    return EOfferCorrectorResult::Success;
}

NDrive::TScheme TLongTermOfferBuilder::DoGetScheme(const NDrive::IServer* server) const {
    auto models = server ? server->GetDriveDatabase().GetModelsDB().GetModelCodes() : TSet<TString>();
    auto offerHolderTags = server
        ? server->GetDriveDatabase().GetTagsManager().GetTagsMeta().GetRegisteredTagNames({ TLongTermOfferHolderTag::Type() })
        : TSet<TString>();
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSBoolean>("group_offer_builder", "Генератор групп офферов");
    result.Add<TFSBoolean>("for_switch", "For switch (prolongation)");
    result.Add<TFSBoolean>("new", "New car");
    result.Add<TFSBoolean>("allow_child_seat", "Allow child seat");
    result.Add<TFSVariants>("car_model", "Car model").SetVariants(models);
    result.Add<TFSString>("car_description", "Car description");
    result.Add<TFSString>("car_tags_filter", "Cars tags filter in standard format");
    result.Add<TFSString>("delivery_area_tags_filter", "Delivery area tags filter (comma separated)");
    result.Add<TFSString>("detailed_description", "Detailed description");
    result.Add<TFSVariants>("offer_holder_tag", "Offer holder tag").SetVariants(offerHolderTags);
    result.Add<TFSString>("offer_image", "Offer image");
    result.Add<TFSString>("offer_small_image", "Offer small image");
    result.Add<TFSVariants>("offer_image_style", "Offer image style").SetVariants(GetEnumAllValues<ELongTermOfferImageStyle>());
    result.Add<TFSJson>("offer_tags", "Offer tags");
    result.Add<TFSString>("subname", "Подзаголовок");
    result.Add<TFSString>("replace_offer_building_action", "Offer builder при замене");
    result.Add<TFSDuration>("available_after", "Available after").SetDefault(AvailableAfter);
    result.Add<TFSString>("available_until", "Available until");
    result.Add<TFSBoolean>("available_until_by_car_info", "Вычислять максимальную дату аренды по информации из гаража");
    result.Add<TFSDuration>("cancellation_period", "Cancellation period after car delivery").SetDefault(CancellationPeriod);
    result.Add<TFSNumeric>("critical_distance_remainder", "Critical distance remainder").SetDefault(CriticalDistanceRemainder);
    result.Add<TFSDuration>("critical_duration_remainder", "Critical duration remainder").SetDefault(CriticalDurationRemainder);
    result.Add<TFSDuration>("early_return_remainder", "Early return remainder").SetDefault(EarlyReturnRemainder);
    result.Add<TFSDuration>("minimal_period", "Minimal period").SetDefault(MinimalPeriod);
    result.Add<TFSDuration>("maximal_period", "Maximal period").SetDefault(MaximalPeriod);
    result.Add<TFSDuration>("maximal_since_offset", "Maximal since offset").SetDefault(MaximalSinceOffset);
    result.Add<TFSDuration>("pack_price_quantum_period", "Pack price quantum period (for testing only)").SetDefault(PackPriceQuantumPeriod);
    result.Add<TFSNumeric>("base_cost", "Base cost").SetDefault(BaseCost);
    result.Add<TFSNumeric>("cancellation_cost", "Cancellation cost").SetDefault(CancellationCost);
    result.Add<TFSNumeric>("child_seat_cost", "Child seat cost").SetDefault(ChildSeatCost);
    result.Add<TFSNumeric>("early_return_cost", "Early return cost").SetDefault(EarlyReturnCost);
    result.Add<TFSNumeric>("duration_alpha", "Duration alpha").SetDefault(DurationAlpha);
    result.Add<TFSNumeric>("franchise_alpha", "Franchise alpha").SetDefault(FranchiseAlpha);
    result.Add<TFSNumeric>("mileage_alpha", "Mileage alpha").SetDefault(MileageAlpha);
    result.Add<TFSNumeric>("extra_mileage_alpha", "ExtraMileage alpha").SetDefault(ExtraMileageAlpha);
    result.Add<TFSNumeric>("under_mileage_alpha", "UnderMileage alpha").SetDefault(UnderMileageAlpha);
    result.Add<TFSNumeric>("overrun_price", "Overrun price").SetDefault(OverrunPrice);
    result.Add<TFSNumeric>("overtime_price", "Overtime price").SetDefault(OvertimePrice);

    result.Add<TFSBoolean>("autohide", "Autohide offer");
    result.Add<TFSBoolean>("should_defer", "Defer communication tag");
    result.Add<TFSDuration>("defer_delay", "If rent starts after more than").SetDefault(DeferDelay);
    result.Add<TFSDuration>("delayed_defer_period", "Defer request for").SetDefault(DelayedDeferPeriod);
    result.Add<TFSDuration>("default_defer_period", "Otherwise defer request for").SetDefault(DefaultDeferPeriod);
    result.Add<TFSDuration>("delayed_hours", "Start at UTC hours").SetDefault(DelayedHours);

    {
        NDrive::TScheme badges;
        badges.Add<TFSString>("id", "Identifier");
        badges.Add<TFSString>("color", "Color");
        badges.Add<TFSString>("text", "Text");
        result.Add<TFSArray>("badges", "Offer badges").SetElement(std::move(badges));
    }
    {
        NDrive::TScheme photos;
        photos.Add<TFSString>("url", "Url");
        result.Add<TFSArray>("photos", "Offer photos").SetElement(std::move(photos));
    }
    {
        NDrive::TScheme mileage;
        mileage.Add<TFSNumeric>("default_value", "Default value");
        mileage.Add<TFSNumeric>("minimal_value", "Minimal value");
        mileage.Add<TFSNumeric>("maximal_value", "Maximal value");
        mileage.Add<TFSArray>("values", "Possible values").SetElement<TFSNumeric>();
        mileage.Add<TFSString>("subtitle", "Subtitle");
        result.Add<TFSStructure>("mileage").SetStructure(std::move(mileage));
    }
    {
        NDrive::TScheme carPromoCard;
        carPromoCard.Add<TFSString>("title", "Title");
        carPromoCard.Add<TFSString>("message", "Message");
        carPromoCard.Add<TFSString>("image_link", "Image link");
        carPromoCard.Add<TFSString>("details", "Details");
        NDrive::TScheme carPromo;
        carPromo.Add<TFSString>("title", "Title");
        carPromo.Add<TFSString>("more_info", "More info");
        carPromo.Add<TFSArray>("cards", "Cards").SetElement(std::move(carPromoCard));
        result.Add<TFSStructure>("car_promo", "Car promo").SetStructure(std::move(carPromo));
    }
    {
        auto gTab = result.StartTabGuard("client_appearance");
        result.Add<TFSString>("agreement", "Ссылка на акт приемки-передачи");
    }

    return result;
}

bool TLongTermOfferBuilder::DeserializeSpecialsFromJson(const NJson::TJsonValue& value) {
    bool result = TBase::DeserializeSpecialsFromJson(value) &&
        NJson::TryFieldsFromJson(value, GetFields());
    if (!value["under_mileage_alpha"].IsDefined()) {
        SetUnderMileageAlpha(GetExtraMileageAlpha());
    }
    return result;
}

NJson::TJsonValue TLongTermOfferBuilder::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    NJson::FieldsToJson(result, GetFields());
    return result;
}

TLongTermOffer::TFactory::TRegistrator<TLongTermOffer> TLongTermOffer::Registrator(TLongTermOffer::GetTypeNameStatic());
TLongTermOfferBuilder::TFactory::TRegistrator<TLongTermOfferBuilder> TLongTermOfferBuilder::Registrator(TLongTermOfferBuilder::GetTypeName());
