#pragma once

#include "abstract.h"
#include "pack.h"
#include "types.h"

#include <drive/backend/data/long_term.h>

#include <rtline/util/types/field.h>

enum class ELongTermPaymentPlan {
    OneTime,
    Weekly,
    Monthly,
    Custom,
};

enum class ELongTermOfferImageStyle {
    Default,
    Render,
};

template <class E>
struct TLongTermEnum {
public:
    R_OPTIONAL(E, Value);
    R_OPTIONAL(E, DefaultValue);
    R_OPTIONAL(TSet<E>, Values);
    R_OPTIONAL(i64, Cost);
    R_FIELD(TString, Id);
    R_FIELD(TString, Title);

public:
    NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer& server) const;

    DECLARE_FIELDS(
        Field(DefaultValue, "default_value"),
        Field(Value, "value"),
        Field(Values, "values"),
        Field(Cost, "cost"),
        Field(Id, "id"),
        Field(Title, "title")
    )
};

template <class T>
using TLongTermVariable = TOfferVariable<T>;

struct TLongTermVoid {
public:
    R_OPTIONAL(i64, Cost);
    R_FIELD(TString, Id);
    R_FIELD(TString, Title);
    R_FIELD(TString, Subtitle);

public:
    NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer& server) const;

    DECLARE_FIELDS(
        Field(Cost, "cost"),
        Field(Id, "id"),
        Field(Title, "title"),
        Field(Subtitle, "subtitle")
    )
};

struct TLongTermCarPromoCard {
public:
    R_FIELD(TString, Title);
    R_FIELD(TString, Message);
    R_FIELD(TString, ImageLink);
    R_FIELD(TString, Details);

public:
    NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer& server) const;

    DECLARE_FIELDS(
        Field(Title, "title"),
        Field(Message, "message"),
        Field(ImageLink, "image_link"),
        Field(Details, "details")
    )
};

struct TLongTermCarPromo {
public:
    R_FIELD(TString, Title);
    R_FIELD(TString, MoreInfo);
    R_FIELD(TVector<TLongTermCarPromoCard>, Cards);

public:
    explicit operator bool() const {
        return Title || MoreInfo || !Cards.empty();
    }

    NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer& server) const;

    DECLARE_FIELDS(
        Field(Title, "title"),
        Field(MoreInfo, "more_info"),
        Field(Cards, "cards")
    )
};

struct TLongTermBadge {
public:
    R_FIELD(TString, Id);
    R_FIELD(TString, Color);
    R_FIELD(TString, Text);

public:
    NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer& server) const;

    DECLARE_FIELDS(
        Field(Id, "id"),
        Field(Color, "color"),
        Field(Text, "text")
    )
};
using TLongTermBadges = TVector<TLongTermBadge>;

struct TLongTermExtra {
public:
    R_FIELD(TString, Id);
    R_FIELD(TString, Title);
    R_FIELD(TString, Subtitle);

public:
    NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer& server) const;

    DECLARE_FIELDS(
        Field(Id, "id"),
        Field(Title, "title"),
        Field(Subtitle, "subtitle")
    )
};
using TLongTermExtras = TVector<TLongTermExtra>;

struct TLongTermPhoto {
public:
    R_FIELD(TString, Url);

public:
    NJson::TJsonValue GetReport() const;

    DECLARE_FIELDS(
        Field(Url, "url")
    )
};
using TLongTermPhotos = TVector<TLongTermPhoto>;

class TLongTermOfferState: public TPackOfferState {
private:
    using TBase = TPackOfferState;

public:
    struct TPayment {
        TInstant Timestamp;
        TMaybe<ui32> Mileage;
        ui32 Value = 0;
        bool Completed = false;
    };
    using TPayments = TVector<TPayment>;

public:
    R_OPTIONAL(TPayment, NextPayment);
    R_FIELD(TPayments, Payments);
    R_FIELD(ui64, HeldSum, 0);
    R_FIELD(ui32, ReturnCost, 0);
    R_FIELD(ui32, ReturnFee, 0);
    R_FIELD(ui32, ReturnPackPrice, 0);
    R_FIELD(ui32, ReturnOverrunCost, 0);
    R_FIELD(bool, ReturnRecalculation, false);
    R_FIELD(bool, Materialized, false);

public:
    TLongTermOfferState(const TBase& other)
        : TBase(other)
    {
    }

    NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer& server) const override;

    virtual TString FormDescriptionElement(const TString& value, const TString& currency, ELocalization locale, const ILocalization& localization) const override;

public:
    static TAtomicSharedPtr<TLongTermOfferState> MakeFrom(TOfferStatePtr state);
};

class TLongTermOffer: public TPackOffer {
private:
    using TBase = TPackOffer;

public:
    using TBase::TBase;

    static TString GetTypeNameStatic() {
        return "long_term";
    }
    virtual TString GetTypeName() const override {
        return GetTypeNameStatic();
    }
    virtual bool GetReportedSwitchable() const override {
        return false;
    }
    virtual TMaybe<TInstant> GetStart() const override {
        return Since ? MakeMaybe(Since) : Nothing();
    }
    virtual bool HasFueling() const override {
        return false;
    }
    virtual bool ShouldReportUpsale() const override {
        return false;
    }

    virtual ui32 GetDeposit() const override;

    TPriceSchedule BuildPaymentSchedule(TInstant start) const;
    TPriceSchedule GetOrBuildPaymentSchedule() const;
    void FillPaymentSchedule(TInstant timestamp);

    bool DeserializeFromProto(const NDrive::NProto::TOffer& info) override;
    NDrive::NProto::TOffer SerializeToProto() const override;

    TExpected<bool, NJson::TJsonValue> CheckAgreementRequirements(ELocalization /*locale*/, const TUserPermissions::TPtr /*permissions*/, const NDrive::IServer* /*server*/) const override;

protected:
    virtual TRecalcByMinutesInfo CheckRecalcByMinutes(const TRidingInfo& rInfo, const TInstant startPack) const override;
    virtual NJson::TJsonValue DoBuildJsonReport(const TReportOptions& options, const ICommonOfferBuilderAction* constructor, const NDrive::IServer& server) const override;
    virtual TOfferStatePtr DoCalculate(const TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events, const TInstant& until, const TRidingInfo& ridingInfo, TOfferPricing& result) const override;
    virtual TString DoFormDescriptionElement(const TString& value, ELocalization locale, const ILocalization* localization) const override;
    virtual void FillBillPack(TBill& bill, const TOfferPricing& pricing, TOfferStatePtr segmentState, ELocalization locale, const NDrive::IServer* server) const override;
    virtual void FillBillPricing(TBill& bill, const TOfferPricing& pricing, TOfferStatePtr segmentState, ELocalization locale, const NDrive::IServer* server) const override;
    virtual void OnBooking(const TString& deviceId, const TString& mobilePaymethodId = "") override;

private:
    TString GetPaymentScheduleReport(ELocalization locale, const ILocalization& localization) const;
    static TString GetPaymentScheduleReport(const TPriceSchedule& schedule, ELocalization locale, const ILocalization& localization);

private:
    R_FIELD(bool, New, false);
    R_FIELD(TString, DetailedDescription);
    R_FIELD(TString, OfferImage);
    R_FIELD(TString, OfferSmallImage);
    R_FIELD(ELongTermOfferImageStyle, OfferImageStyle, ELongTermOfferImageStyle::Default);
    R_FIELD(bool, UseInsuranceTypeVariable, false);

    R_FIELD(bool, Available, true);
    R_FIELD(TInstant, AvailableSince);
    R_FIELD(TInstant, AvailableUntil);
    R_FIELD(TInstant, BookingTimestamp);
    R_FIELD(TInstant, MaximalSince);
    R_FIELD(TDuration, CancellationPeriod, TDuration::Days(1));
    R_FIELD(TDuration, EarlyReturnRemainder, TDuration::Days(7));
    R_FIELD(TDuration, MinimalPeriod);
    R_FIELD(TDuration, MaximalPeriod);

    R_OPTIONAL(TVector<TGeoCoord>, DeliveryArea);
    R_OPTIONAL(TGeoCoord, DeliveryLocation);
    R_FIELD(TString, DeliveryLocationName);
    R_FIELD(TInstant, DeliveryTime);
    R_FIELD(TInstant, Since);
    R_FIELD(TInstant, Until);
    R_FIELD(i64, CancellationCost, 0);
    R_FIELD(i64, EarlyReturnCost, 0);
    R_FIELD(i64, DurationCost, 0);
    R_FIELD(i64, MonthlyCost, 0);
    R_FIELD(i64, WeeklyCost, 0);

    R_OPTIONAL(bool, ChildSeat);
    R_OPTIONAL(i64, ChildSeatCost);
    R_OPTIONAL(bool, Delivery);
    R_OPTIONAL(i64, DeliveryCost);
    R_OPTIONAL(ui64, Mileage);
    R_OPTIONAL(i64, MileageCost);
    R_OPTIONAL(i64, DefaultMileageCost);
    R_OPTIONAL(i64, ExtraMileageCost);
    R_OPTIONAL(ui64, Franchise);
    R_OPTIONAL(i64, FranchiseCost);
    R_OPTIONAL(ELongTermPaymentPlan, PaymentPlan);

    R_FIELD(bool, ShouldDeferCommunication, false);
    R_FIELD(TDuration, DeferDelay, TDuration::Days(1));
    R_FIELD(TDuration, DefaultDeferPeriod, TDuration::Hours(1));
    R_FIELD(TDuration, DelayedDeferPeriod, TDuration::Days(1));
    R_FIELD(TDuration, DelayedHours, TDuration::Hours(9));

    R_FIELD(bool, GroupOffer, false);
    R_FIELD(TString, GroupOfferId);

public:
    void MakeUnavailable() {
        SetAvailable(false);
        SetBookable(false);
    }

private:
    static TFactory::TRegistrator<TLongTermOffer> Registrator;
};

using TLongTermOfferReport = TPackOfferReport;

class TLongTermOfferBuilder
    : public IOfferBuilderAction
    , public NDrive::IDeliveryAreaFilter
{
private:
    using TBase = IOfferBuilderAction;

public:
    struct TCandidates {
        TVector<TTaggedObject> Tier1;
        TVector<TTaggedObject> Tier2;
    };

public:
    static TLongTermBadge CreateDefaultNewBadge();
    static TLongTermVariable<bool> CreateDefaultChildSeat();
    static TLongTermVariable<ui64> CreateDefaultMileage();
    static TLongTermVariable<ui64> CreateDefaultFranchise(const NDrive::IServer& server);
    static TLongTermVoid CreateDefaultDelivery();
    static TLongTermExtras CreateDefaultExtras();
    static TLongTermEnum<ELongTermPaymentPlan> CreateDefaultPaymentPlan();
    static TInstant GetCarLongTermUntil(const TString& carId, const NDrive::IServer& server);

public:
    using TBase::TBase;

    TLongTermBadge CreateAvailableBadge() const;
    TMaybe<TTagsFilter> CreateCarTagsFilter() const;
    TMaybe<TBaseAreaTagsFilter> GetDeliveryAreaFilter() const override;
    TLongTermVariable<TString> CreateInsuranceType(const NDrive::IServer& server, ui64 mileage) const;

    TVector<TTaggedObject> GetCars(const NDrive::IServer& server) const;
    TCandidates FindCandidates(const NDrive::IServer& server) const;

    bool HasGrouppingParent(const NDrive::IServer& server) const;
    bool IsGroupOfferBuilderStrict(const NDrive::IServer& server) const;

    static TString GetTypeName() {
        return "long_term_offer_builder";
    }

    virtual TString GetType() const override {
        return GetTypeName();
    }

protected:
    virtual EOfferCorrectorResult DoBuildOffers(
        const TUserPermissions& permissions,
        TVector<IOfferReport::TPtr>& offers,
        const TOffersBuildingContext& context,
        const NDrive::IServer* server,
        NDrive::TInfoEntitySession& session
    ) const override;
    virtual EOfferCorrectorResult DoCheckOfferConditions(
        const TOffersBuildingContext& context,
        const TUserPermissions& permissions
    ) const override;

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& value) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;

private:
    R_FIELD(TString, CarModel);
    R_FIELD(TString, CarDescription);
    R_FIELD(TLongTermCarPromo, CarPromo);
    R_FIELD(TString, CarTagsFilter);
    R_FIELD(TString, DeliveryAreaTagsFilter);
    R_FIELD(TString, DetailedDescription);
    R_FIELD(TString, OfferHolderTag, TLongTermOfferHolderTag::Type());
    R_FIELD(TString, OfferImage);
    R_FIELD(TString, OfferSmallImage);
    R_FIELD(TString, Subname);
    R_FIELD(TString, ReplaceOfferBuildingAction);
    R_FIELD(TSet<TString>, OfferTags);
    R_FIELD(TDuration, AvailableAfter, TDuration::Days(2));
    R_FIELD(TInstant, AvailableSince);
    R_FIELD(TInstant, AvailableUntil);
    R_FIELD(bool, AvailableUntilByCarInfo, false);
    R_FIELD(TDuration, CancellationPeriod, TDuration::Days(1));
    R_FIELD(TDuration, EarlyReturnRemainder, TDuration::Days(7));
    R_FIELD(TDuration, MinimalPeriod, TDuration::Days(31));
    R_FIELD(TDuration, MaximalPeriod, TDuration::Days(365));
    R_FIELD(TDuration, MaximalSinceOffset, TDuration::Days(365));
    R_FIELD(TDuration, PackPriceQuantumPeriod);
    R_FIELD(double, CriticalDistanceRemainder, 100);
    R_FIELD(TDuration, CriticalDurationRemainder, TDuration::Days(7));
    R_FIELD(ELongTermOfferImageStyle, OfferImageStyle, ELongTermOfferImageStyle::Default);
    R_FIELD(bool, ForSwitch, false);
    R_FIELD(bool, New, false);
    R_FIELD(bool, AllowChildSeat, true);
    R_FIELD(bool, GroupOfferBuilder, false);
    R_FIELD(TString, Agreement);

    R_FIELD(double, BaseCost, 100000);
    R_FIELD(double, CancellationCost, 4000);
    R_FIELD(double, ChildSeatCost, 4242);
    R_FIELD(double, EarlyReturnCost, 100500);
    R_FIELD(double, DurationAlpha, 1);
    R_FIELD(double, FranchiseAlpha, -0.1);
    R_FIELD(double, MileageAlpha, 1);
    R_FIELD(double, ExtraMileageAlpha, 0.1);
    R_FIELD(double, UnderMileageAlpha, 0.05);
    R_FIELD(ui32, OverrunPrice, 100);
    R_FIELD(ui32, OvertimePrice, 200);

    R_FIELD(bool, Autohide, false);
    R_FIELD(bool, ShouldDeferCommunication, false);
    R_FIELD(TDuration, DefaultDeferPeriod, TDuration::Hours(1));
    R_FIELD(TDuration, DeferDelay, TDuration::Days(1));
    R_FIELD(TDuration, DelayedDeferPeriod, TDuration::Days(1));
    R_FIELD(TDuration, DelayedHours, TDuration::Hours(9));

    R_FIELD(TLongTermBadges, Badges);
    R_FIELD(TLongTermPhotos, Photos);
    R_FIELD(TLongTermVariable<bool>, ChildSeat, CreateDefaultChildSeat());
    R_FIELD(TLongTermVariable<ui64>, Mileage, CreateDefaultMileage());
    R_FIELD(TLongTermVoid, Delivery, CreateDefaultDelivery());
    R_FIELD(TLongTermExtras, Extras, CreateDefaultExtras());
    R_FIELD(TLongTermEnum<ELongTermPaymentPlan>, PaymentPlan, CreateDefaultPaymentPlan());

public:
    DECLARE_FIELDS(
        Field(CarModel, "car_model"),
        Field(CarDescription, "car_description"),
        Field(CarPromo, "car_promo"),
        Field(CarTagsFilter, "car_tags_filter"),
        Field(DeliveryAreaTagsFilter, "delivery_area_tags_filter"),
        Field(DetailedDescription, "detailed_description"),
        Field(OfferHolderTag, "offer_holder_tag"),
        Field(OfferImage, "offer_image"),
        Field(OfferImageStyle, "offer_image_style"),
        Field(OfferSmallImage, "offer_small_image"),
        Field(Subname, "subname"),
        Field(ReplaceOfferBuildingAction, "replace_offer_building_action"),
        Field(OfferTags, "offer_tags"),
        Field(NJson::Hr(AvailableAfter), "available_after"),
        Field(AvailableSince, "available_since"),
        Field(AvailableUntil, "available_until"),
        Field(AvailableUntilByCarInfo, "available_until_by_car_info"),
        Field(NJson::Hr(CancellationPeriod), "cancellation_period"),
        Field(CriticalDistanceRemainder, "critical_distance_remainder"),
        Field(NJson::Hr(CriticalDurationRemainder), "critical_duration_remainder"),
        Field(NJson::Hr(EarlyReturnRemainder), "early_return_remainder"),
        Field(NJson::Hr(MinimalPeriod), "minimal_period"),
        Field(NJson::Hr(MaximalPeriod), "maximal_period"),
        Field(NJson::Hr(MaximalSinceOffset), "maximal_since_offset"),
        Field(NJson::Hr(PackPriceQuantumPeriod), "pack_price_quantum_period"),
        Field(ForSwitch, "for_switch"),
        Field(New, "new"),
        Field(AllowChildSeat, "allow_child_seat"),
        Field(GroupOfferBuilder, "group_offer_builder"),
        Field(Agreement, "agreement"),

        Field(Badges, "badges"),
        Field(Photos, "photos"),
        Field(Mileage, "mileage"),

        Field(BaseCost, "base_cost"),
        Field(CancellationCost, "cancellation_cost"),
        Field(ChildSeatCost, "child_seat_cost"),
        Field(EarlyReturnCost, "early_return_cost"),
        Field(DurationAlpha, "duration_alpha"),
        Field(FranchiseAlpha, "franchise_alpha"),
        Field(MileageAlpha, "mileage_alpha"),
        Field(ExtraMileageAlpha, "extra_mileage_alpha"),
        Field(UnderMileageAlpha, "under_mileage_alpha"),
        Field(OverrunPrice, "overrun_price"),
        Field(OvertimePrice, "overtime_price"),

        Field(Autohide, "autohide"),
        Field(ShouldDeferCommunication, "should_defer"),
        Field(NJson::Hr(DefaultDeferPeriod), "default_defer_period"),
        Field(NJson::Hr(DeferDelay), "defer_delay"),
        Field(NJson::Hr(DelayedDeferPeriod), "delayed_defer_period"),
        Field(NJson::Hr(DelayedHours), "delayed_hours")
    )

private:
    EOfferCorrectorResult BuildGroupOffer(
            const TUserPermissions& permissions,
            TVector<IOfferReport::TPtr>& offers,
            const TOffersBuildingContext& context,
            const NDrive::IServer* server,
            NDrive::TInfoEntitySession& session
    ) const;
    EOfferCorrectorResult SetOfferTimeInfo(TLongTermOffer& offer, const TOffersBuildingContext& context, const NDrive::IServer& server) const;
    TMaybe<TInstant> GetAvailableInLongTermUntil(const NDrive::IServer& server) const;

    static TFactory::TRegistrator<TLongTermOfferBuilder> Registrator;
};
