#include "offer_price.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/models/storage.h>
#include <drive/backend/roles/manager.h>

TUserAction::TFactory::TRegistrator<TPriceOfferConstructor> TPriceOfferConstructor::Registrator(TPriceOfferConstructor::GetTypeStatic());

bool TPriceOfferConstructor::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) {
    if (jsonValue.Has("market")) {
        if (!Market.DeserializeFromJson(jsonValue["market"])) {
            return false;
        }
    } else {
        if (!Market.DeserializeFromJson(jsonValue)) {
            return false;
        }
    }
    if (jsonValue.Has("equilibrium") && !Equilibrium.DeserializeFromJson(jsonValue["equilibrium"])) {
        return false;
    }
    if (!TJsonProcessor::Read(jsonValue, "payment_discretization", PaymentDiscretization)) {
        return false;
    }
    if (jsonValue["object_tags_filter"].IsString() && !ObjectTagsFilter.DeserializeFromString(jsonValue["object_tags_filter"].GetString())) {
        return false;
    }
    if (jsonValue["object_location_tags_filter"].IsString() && !ObjectLocationTagsFilter.DeserializeFromString(jsonValue["object_location_tags_filter"].GetString())) {
        return false;
    }
    return TBase::DeserializeSpecialsFromJson(jsonValue);
}

NJson::TJsonValue TPriceOfferConstructor::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    result.InsertValue("market", Market.SerializeToJson());
    result.InsertValue("equilibrium", Equilibrium.SerializeToJson());
    result.InsertValue("payment_discretization", PaymentDiscretization);
    result.InsertValue("object_tags_filter", ObjectTagsFilter.ToString());
    result.InsertValue("object_location_tags_filter", ObjectLocationTagsFilter.ToString());
    return result;
}

NDrive::TScheme TPriceOfferConstructor::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSStructure>("market").SetStructure(TMarketPriceCalculator::GetScheme(server));
    result.Add<TFSStructure>("equilibrium").SetStructure(TEquilibriumPriceCalculator::GetScheme(server));

    result.Add<TFSNumeric>("payment_discretization", "Дискретизация при снятии денег с карты(копейки)").SetPrecision(0).SetDefault(DefaultPaymentDiscretization);

    result.Add<TFSString>("object_tags_filter", "Фильтр ценообразования по тегам объекта в стандартном формате");
    result.Add<TFSString>("object_location_tags_filter", "Фильтр ценообразования по тегам локации объекта в стандартном формате");
    return result;
}

TFullPricesContext TPriceOfferConstructor::BuildFullContext() const {
    TFullPricesContext result;
    result.MutableAcceptance() = Market.GetAcceptanceCost().BuildPriceContext();
    result.MutableKm() = Market.GetKmPrice().BuildPriceContext();
    result.MutableRiding() = Market.GetRidingPrice().BuildPriceContext();
    result.MutableParking() = Market.GetParkingPrice().BuildPriceContext();
    return result;
}

TMaybe<ui32> TPriceOfferConstructor::CalcPackPrice(TDuration duration, ui32 mileage) const {
    if (!Market.GetPackPriceMatrix().empty()) {
        auto price = Market.GetPackPriceMatrix().FindPtr(std::make_pair(duration.Seconds(), mileage));
        if (price) {
            return *price;
        } else {
            return {};
        }
    }

    auto durationPrice = Market.GetPackDurationPrice().FindPtr(duration.Seconds());
    auto mileagePrice = Market.GetPackMileagePrice().FindPtr(mileage);
    if (durationPrice && mileagePrice) {
        return *durationPrice + *mileagePrice;
    } else {
        return {};
    }
}

TMaybe<ui32> TPriceOfferConstructor::CalcPackOverrunPrice(TDuration duration, ui32 mileage) const {
    const auto& matrix = Market.GetPackOverrunPriceMatrix();
    if (matrix.empty()) {
        return {};
    }

    auto key = std::make_pair(duration.Seconds(), mileage);
    auto lk = LowerKey(matrix, key);
    if (lk == matrix.end()) {
        return {};
    }
    return lk->second;
}

TVector<TString> TPriceOfferConstructor::GetNames(const NDrive::IServer* server) {
    if (!server) {
        return {};
    }
    auto driveApi = server->GetDriveAPI();
    if (!driveApi) {
        return {};
    }
    auto rolesManager = driveApi->GetRolesManager();
    if (!rolesManager) {
        return {};
    }

    return rolesManager->GetActionsDB().GetActionNamesWithType<TPriceOfferConstructor>(/*reportDeprecated=*/false);
}

NJson::TJsonValue IOfferReport::BuildJsonReport(ELocalization locale, NDriveSession::TReportTraits traits, const NDrive::IServer& server, const TUserPermissions& permissions) const {
    Y_UNUSED(permissions);
    ICommonOffer::TReportOptions reportOptions(locale, traits);
    NJson::TJsonValue result = Offer->BuildJsonReport(reportOptions, server);
    NJson::TJsonValue& prices = result["prices"];
    result.InsertValue("list_priority", ListPriority.GetOrElse(0));
    if (DepositInfoText) {
        result.InsertValue("deposit_info", Offer->FormDescriptionElement(DepositInfoText, locale, server.GetLocalization()));
    }
    if (!InsurancePrices.empty()) {
        prices.InsertValue("insurance_prices", NJson::ToJson(NJson::Dictionary(InsurancePrices)));
    }
    return result;
}

bool TPriceCorrectionLimits::Check(const double oldPrice, const double newPrice, TString* message) const {
    if (MinValue && newPrice < *MinValue) {
        if (message) {
            *message = "Cannot take new price (min): " + ::ToString(newPrice) + " < " + ::ToString(*MinValue);
        }
        return false;
    }
    if (MaxValue && newPrice >= *MaxValue) {
        if (message) {
            *message = "Cannot take new price (max): " + ::ToString(newPrice) + " >= " + ::ToString(*MaxValue);
        }
        return false;
    }
    if (MaxDelta && Abs<double>(newPrice - oldPrice) >= *MaxDelta) {
        if (message) {
            *message = "Cannot take new price (delta): " + ::ToString(Abs<double>(newPrice - oldPrice)) + " >= " + ::ToString(*MaxDelta);
        }
        return false;
    }
    return true;
}

NDrive::TScheme TPriceCorrectionLimits::GetScheme(const NDrive::IServer* server) {
    Y_UNUSED(server);
    NDrive::TScheme result;
    result.Add<TFSNumeric>("min_value", "Минимальное допустимое значение");
    result.Add<TFSNumeric>("max_value", "Максимальное допустимое значение");
    result.Add<TFSNumeric>("max_delta", "Максимальное шаговое изменение");
    result.Add<TFSNumeric>("corrector", "Корректор цены от себестоимости").SetMin(0.1);
    return result;
}
