#pragma once

#include <drive/backend/offers/offers/abstract.h>
#include <drive/backend/offers/price/model.h>
#include <drive/backend/offers/price/price.h>

#include <drive/backend/actions/abstract/action.h>

#include <drive/library/cpp/scheme/scheme.h>

class TPriceCorrectionLimits {
private:
    R_OPTIONAL(double, Corrector);
    R_OPTIONAL(ui32, MinValue);
    R_OPTIONAL(ui32, MaxValue);
    R_OPTIONAL(ui32, MaxDelta);

public:
    bool Check(const double oldPrice, const double newPrice, TString* message) const;

    NJson::TJsonValue SerializeToJson() const {
        NJson::TJsonValue result = NJson::JSON_MAP;
        TJsonProcessor::Write(result, "min_value", MinValue);
        TJsonProcessor::Write(result, "max_value", MaxValue);
        TJsonProcessor::Write(result, "max_delta", MaxDelta);
        TJsonProcessor::Write(result, "corrector", Corrector);
        return result;
    }

    bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
        if (!TJsonProcessor::Read(jsonInfo, "min_value", MinValue)) {
            return false;
        }
        if (!TJsonProcessor::Read(jsonInfo, "max_value", MaxValue)) {
            return false;
        }
        if (!TJsonProcessor::Read(jsonInfo, "max_delta", MaxDelta)) {
            return false;
        }
        if (!TJsonProcessor::Read(jsonInfo, "corrector", Corrector)) {
            return false;
        }
        if (Corrector.GetOrElse(1) < 1e-1) {
            return false;
        }
        return true;
    }

    static NDrive::TScheme GetScheme(const NDrive::IServer* server);
};

class TPriceOfferConstructor: public TUserAction {
private:
    using TBase = TUserAction;

private:
    R_FIELD(TMarketPriceCalculator, Market);
    R_FIELD(TEquilibriumPriceCalculator, Equilibrium);

    R_READONLY(ui32, PaymentDiscretization, DefaultPaymentDiscretization);

    R_FIELD(TTagsFilter, ObjectTagsFilter);
    R_FIELD(TTagsFilter, ObjectLocationTagsFilter);

private:
    static TFactory::TRegistrator<TPriceOfferConstructor> Registrator;

protected:
    bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) override;
    NJson::TJsonValue SerializeSpecialsToJson() const override;

    NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

public:
    TFullPricesContext BuildFullContext() const;
    TMaybe<ui32> CalcPackPrice(TDuration duration, ui32 mileage) const;
    TMaybe<ui32> CalcPackOverrunPrice(TDuration duration, ui32 mileage) const;

    virtual TString GetType() const override {
        return GetTypeStatic();
    }

    static TString GetTypeStatic() {
        return "price_offer_constructor";
    }

    static TVector<TString> GetNames(const NDrive::IServer* server);
};

class TCommonCategoryScoreThreshold;
class TCommonDestinationDetector;

class TPredictedDuration {
    R_OPTIONAL(TDuration, Duration);
    R_OPTIONAL(TString, PredictorId);
public:
    TPredictedDuration() = default;
    TPredictedDuration(const TDuration d)
        : Duration(d)
    {
    }
};

class IOfferReport: public TFullPricesContextBuilder {
public:
    using TInsurancePrices = TMap<TString, i64>;
    using TLocalizations = TMap<TString, TString>;

private:
    R_FIELD(TInsurancePrices, InsurancePrices);
    R_FIELD(TLocalizations, Localizations);
    R_FIELD(TString, DepositInfoText);
    R_OPTIONAL(i32, ListPriority);
    R_OPTIONAL(i32, InternalPriority);
    R_OPTIONAL(TVector<TDBTag>, Tags);
    R_OPTIONAL(TDuration, WalkingDuration);
    R_FIELD(bool, Visible, true);

    ICommonOffer::TPtr Offer;
    TAtomicSharedPtr<const TPriceOfferConstructor> PriceOfferConstructor;

public:
    using TPtr = TAtomicSharedPtr<IOfferReport>;

public:
    virtual TString PredictDestination(const TOffersBuildingContext& /*context*/, const NDrive::IServer* /*server*/, const TCommonDestinationDetector& /*otherDetector*/) const {
        return "";
    }

    virtual bool ApplyMinPrice(const double /*price*/) {
        return false;
    }

    virtual void ApplyFlowCorrection(const TString& /*areaId*/, const TOffersBuildingContext& /*context*/, const NDrive::IServer* /*server*/) {
    }

    virtual void ApplyInternalCorrection(const TString& /*areaId*/, const TOffersBuildingContext& /*context*/, const NDrive::IServer* /*server*/) {
    }

    virtual TMaybe<ui32> GetRerunPriceKM() const {
        if (GetPriceOfferConstructor()) {
            return GetPriceOfferConstructor()->GetMarket().CalcKmPrice();
        }
        return {};
    }

    virtual ~IOfferReport() = default;

    IOfferReport(const IOfferReport& from) = default;
    IOfferReport(IOfferReport&& from) = default;

    template <class To, class Offer, class From>
    static THolder<To> ExtendCustom(const From& from) {
        if (!from.GetOffer()) {
            return nullptr;
        }
        auto fromOffer = from.template GetOfferAs<typename From::TBaseOffer>();
        if (!fromOffer) {
            return nullptr;
        }
        auto result = MakeHolder<To>(from);
        result->Offer = MakeAtomicShared<Offer>(*fromOffer);
        result->PriceOfferConstructor = from.PriceOfferConstructor;
        return result;
    }

    template <class To, class From>
    static THolder<To> Extend(const From& from) {
        return ExtendCustom<To, typename To::TBaseOffer, From>(from);
    }

    IOfferReport(ICommonOffer::TPtr offer, TAtomicSharedPtr<const TPriceOfferConstructor> priceOfferConstructor)
        : Offer(offer)
        , PriceOfferConstructor(priceOfferConstructor)
    {
    }

    explicit operator bool() const {
        return Offer != nullptr;
    }

    TAtomicSharedPtr<const TPriceOfferConstructor> GetPriceOfferConstructor() const {
        return PriceOfferConstructor;
    }

    auto GetConstructionId() const {
        auto offer = GetOfferAs<IOffer>();
        return Yensured(offer)->GetConstructionId();
    }

    ICommonOffer::TPtr GetOffer() const {
        return Offer;
    }
    template <class T>
    T* GetOfferAs() {
        return dynamic_cast<T*>(Offer.Get());
    }
    template <class T>
    const T* GetOfferAs() const {
        return dynamic_cast<T*>(Offer.Get());
    }

    template <class T>
    typename T::TPtr GetOfferPtrAs() {
        return std::dynamic_pointer_cast<T>(Offer);
    }
    template <class T>
    typename T::TConstPtr GetOfferPtrAs() const {
        return std::dynamic_pointer_cast<T>(Offer);
    }

    void SetOffer(ICommonOffer::TPtr value) {
        Offer = std::move(value);
    }
    void SetPriceOfferConstructor(TAtomicSharedPtr<const TPriceOfferConstructor> value) {
        PriceOfferConstructor = std::move(value);
    }

    virtual NJson::TJsonValue BuildJsonReport(ELocalization locale, NDriveSession::TReportTraits traits, const NDrive::IServer& server, const TUserPermissions& permissions) const;
};

class TFakeOfferReport: public IOfferReport {
protected:
    virtual void DoRecalcPrices(const NDrive::IServer* /*server*/) override {
    }
    virtual TFullPricesContext* GetFullPricesContext() override {
        return nullptr;
    }

public:
    TFakeOfferReport(ICommonOffer::TPtr offer)
        : IOfferReport(offer, nullptr)
    {
    }

    virtual void RecalculateFeatures() override {
    }
};
